import React, { useState, useEffect } from "react";

declare global {
  interface Window {
    google: any;
  }
}
import { Plus, MapPin } from "lucide-react";
import { Button } from "../../ui/button";
import { Input } from "../../ui/input";
import { Label } from "../../ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../../ui/select";
import { Textarea } from "../../ui/textarea";
import { DataTable } from "./components/DataTable";
import { SearchFilters } from "./components/SearchFilters";
import { Pagination } from "./components/Pagination";
import { FormModal } from "./components/FormModal";

import { PlacesAutocomplete } from "./components/PlacesAutocomplete";
import { getZoningChanges, createZoningChange, updateZoningChange, deleteZoningChange, ZoningChange } from "../../../api/hyperlocalEconomic";
import { toast } from "sonner";

export function ZoningChanges() {
  const [changes, setChanges] = useState<ZoningChange[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState("");
  const [cityFilter, setCityFilter] = useState("all");
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalItems, setTotalItems] = useState(0);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingChange, setEditingChange] = useState<ZoningChange | null>(null);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isGoogleMapsReady, setIsGoogleMapsReady] = useState(false);

  useEffect(() => {
    const checkGoogleMaps = () => {
      if (window.google?.maps) {
        setIsGoogleMapsReady(true);
      } else {
        setTimeout(checkGoogleMaps, 100);
      }
    };
    checkGoogleMaps();
  }, []);

  const [formData, setFormData] = useState({
    name: "",
    type: "",
    description: "",
    status: "planned",
    expected_year: "",
    latitude: 28.6139,
    longitude: 77.2090,
    radius_meters: "1000",
    areaSearch: ""
  });

  const itemsPerPage = 10;

  const columns = [
    { key: "name", label: "Name", sortable: true },
    { key: "type", label: "Type", sortable: true },
    { key: "status", label: "Status", sortable: true },
    { key: "expected_year", label: "Expected Year", sortable: true },
    { key: "description", label: "Description", sortable: true }
  ];

  const cityOptions = [
    { value: "Delhi", label: "Delhi" },
    { value: "Mumbai", label: "Mumbai" },
    { value: "Bangalore", label: "Bangalore" },
    { value: "Chennai", label: "Chennai" },
    { value: "Hyderabad", label: "Hyderabad" },
    { value: "Pune", label: "Pune" },
    { value: "Kolkata", label: "Kolkata" },
    { value: "Ahmedabad", label: "Ahmedabad" },
    { value: "Noida", label: "Noida" },
    { value: "Gurugram", label: "Gurugram" }
  ];

  const zoningTypes = [
    "Residential",
    "Commercial",
    "Industrial",
    "Mixed Use",
    "Agricultural",
    "Institutional",
    "Recreational",
    "Transportation",
    "Utilities",
    "Green Belt"
  ];

  const fetchChanges = async () => {
    try {
      setLoading(true);
      const response = await getZoningChanges(currentPage, itemsPerPage, searchQuery, cityFilter);
      setChanges(response.data);
      setTotalPages(Math.ceil(response.total / itemsPerPage));
      setTotalItems(response.total);
    } catch (error) {
      toast.error("Failed to fetch zoning changes");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchChanges();
  }, [currentPage, searchQuery, cityFilter]);

  const handleAdd = () => {
    setEditingChange(null);
    setFormData({
      name: "",
      type: "",
      description: "",
      status: "planned",
      expected_year: "",
      latitude: 28.6139,
      longitude: 77.2090,
      radius_meters: "1000",
      areaSearch: ""
    });
    setIsModalOpen(true);
  };

  const handleEdit = (change: ZoningChange) => {
    setEditingChange(change);
    setFormData({
      name: change.name,
      type: change.type,
      description: change.description || "",
      status: change.status,
      expected_year: change.expected_year?.toString() || "",
      latitude: change.latitude,
      longitude: change.longitude,
      radius_meters: change.radius_meters.toString(),
      areaSearch: ""
    });
    setIsModalOpen(true);
  };

  const handleDelete = async (change: ZoningChange) => {
    if (!confirm("Are you sure you want to delete this zoning change?")) return;
    
    try {
      await deleteZoningChange(change.id);
      toast.success("Zoning change deleted successfully");
      fetchChanges();
    } catch (error) {
      toast.error("Failed to delete zoning change");
    }
  };

  const handleSubmit = async () => {
    try {
      // Validate required fields
      if (!formData.name.trim()) {
        toast.error("Name is required");
        return;
      }
      if (!formData.type.trim()) {
        toast.error("Type is required");
        return;
      }
      if (!formData.radius_meters || parseInt(formData.radius_meters) < 100) {
        toast.error("Radius must be at least 100 meters");
        return;
      }
      
      setIsSubmitting(true);
      
      const changeData = {
        name: formData.name.trim(),
        type: formData.type,
        description: formData.description.trim() || null,
        status: formData.status,
        expected_year: formData.expected_year ? parseInt(formData.expected_year) : null,
        latitude: formData.latitude,
        longitude: formData.longitude,
        radius_meters: parseInt(formData.radius_meters)
      };

      if (editingChange) {
        await updateZoningChange(editingChange.id, changeData);
        toast.success("Zoning change updated successfully");
      } else {
        await createZoningChange(changeData);
        toast.success("Zoning change created successfully");
      }

      setIsModalOpen(false);
      fetchChanges();
    } catch (error) {
      toast.error("Failed to save zoning change");
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleClearFilters = () => {
    setSearchQuery("");
    setCityFilter("all");
    setCurrentPage(1);
  };

  const filters = [
    {
      key: "city",
      label: "City",
      options: cityOptions,
      value: cityFilter,
      onChange: setCityFilter
    }
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <MapPin className="h-8 w-8 text-green-600" />
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Zoning Changes</h1>
            <p className="text-gray-600">Track zoning modifications and their impact on property values</p>
          </div>
        </div>
        <Button onClick={handleAdd} className="!bg-blue-600 !hover:bg-blue-700 !text-white border-0" style={{backgroundColor: '#2563eb', color: 'white'}}>
          <Plus className="h-4 w-4 mr-2" />
          Add Zoning Change
        </Button>
      </div>

      <SearchFilters
        searchQuery={searchQuery}
        onSearchChange={setSearchQuery}
        filters={filters}
        onClearFilters={handleClearFilters}
        placeholder="Search zoning changes..."
      />

      <DataTable
        columns={columns}
        data={changes}
        onEdit={handleEdit}
        onDelete={handleDelete}
        loading={loading}
      />

      {!loading && totalPages > 1 && (
        <Pagination
          currentPage={currentPage}
          totalPages={totalPages}
          onPageChange={setCurrentPage}
          totalItems={totalItems}
          itemsPerPage={itemsPerPage}
        />
      )}

      <FormModal
        isOpen={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        title={editingChange ? "Edit Zoning Change" : "Add Zoning Change"}
        onSubmit={handleSubmit}
        isSubmitting={isSubmitting}
        size="lg"
      >
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="md:col-span-2">
            <Label htmlFor="areaSearch">Search Location</Label>
            <PlacesAutocomplete
              value={formData.areaSearch}
              onChange={(value) => setFormData({ ...formData, areaSearch: value })}
              onPlaceSelect={(coordinates, address) => {
                setFormData({
                  ...formData,
                  latitude: coordinates.lat,
                  longitude: coordinates.lng,
                  name: formData.name || address.split(',')[0]
                });
              }}
              placeholder="Search for location..."
            />
          </div>

          <div className="md:col-span-2">
            <Label htmlFor="name">Name *</Label>
            <Input
              id="name"
              value={formData.name}
              onChange={(e) => setFormData({ ...formData, name: e.target.value })}
              placeholder="Enter zoning change name"
              required
            />
          </div>

          <div>
            <Label htmlFor="type">Type *</Label>
            <Select value={formData.type} onValueChange={(value) => setFormData({ ...formData, type: value })}>
              <SelectTrigger>
                <SelectValue placeholder="Select type" />
              </SelectTrigger>
              <SelectContent>
                {zoningTypes.map((type) => (
                  <SelectItem key={type} value={type}>
                    {type}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label htmlFor="status">Status *</Label>
            <Select value={formData.status} onValueChange={(value) => setFormData({ ...formData, status: value })}>
              <SelectTrigger>
                <SelectValue placeholder="Select status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="planned">Planned</SelectItem>
                <SelectItem value="under_construction">Under Construction</SelectItem>
                <SelectItem value="completed">Completed</SelectItem>
                <SelectItem value="cancelled">Cancelled</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label htmlFor="expected_year">Expected Year</Label>
            <Input
              id="expected_year"
              type="number"
              value={formData.expected_year}
              onChange={(e) => setFormData({ ...formData, expected_year: e.target.value })}
              placeholder="Enter expected year"
              min="2024"
              max="2050"
            />
          </div>

          <div>
            <Label htmlFor="radius_meters">Radius (meters) *</Label>
            <Input
              id="radius_meters"
              type="number"
              value={formData.radius_meters}
              onChange={(e) => setFormData({ ...formData, radius_meters: e.target.value })}
              placeholder="Enter radius in meters"
              required
              min="100"
              max="50000"
            />
          </div>

          <div className="md:col-span-2">
            <Label htmlFor="description">Description</Label>
            <Textarea
              id="description"
              value={formData.description}
              onChange={(e) => setFormData({ ...formData, description: e.target.value })}
              placeholder="Enter description"
              rows={3}
            />
          </div>

          <div>
            <Label htmlFor="latitude">Latitude *</Label>
            <Input
              id="latitude"
              type="number"
              step="any"
              value={formData.latitude}
              onChange={(e) => setFormData({ 
                ...formData, 
                latitude: parseFloat(e.target.value) || 0
              })}
              placeholder="Enter latitude"
              required
            />
          </div>

          <div>
            <Label htmlFor="longitude">Longitude *</Label>
            <Input
              id="longitude"
              type="number"
              step="any"
              value={formData.longitude}
              onChange={(e) => setFormData({ 
                ...formData, 
                longitude: parseFloat(e.target.value) || 0
              })}
              placeholder="Enter longitude"
              required
            />
          </div>
        </div>
      </FormModal>
    </div>
  );
}