import React, { useState, useEffect } from "react";
import {
  TrendingUp,
  TrendingDown,
  Minus,
  MapPin,
  Activity,
} from "lucide-react";

interface CityTrend {
  city: string;
  region: string;
  total_listings: number;
  active_listings: number;
  sold_count: number;
  avg_price: number;
  avg_price_formatted: string;
  price_change_30d: number;
  demand_supply_ratio: number;
  conversion_rate: number;
  market_velocity: string;
  trend_direction: string;
  avg_days_on_market: number;
}

const CityTrends: React.FC = () => {
  const [data, setData] = useState<CityTrend[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchCityTrends();
  }, []);

  const fetchCityTrends = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch("/api/admin_market_trends/city-trends");
      const result = await response.json();

      if (result.success) {
        setData(result.data.city_trends);
      } else {
        setError("Failed to fetch city trends data");
      }
    } catch (err) {
      setError("Error connecting to server");
    } finally {
      setLoading(false);
    }
  };

  const getTrendIcon = (direction: string) => {
    switch (direction) {
      case "up":
        return <TrendingUp className="w-4 h-4 text-green-500" />;
      case "down":
        return <TrendingDown className="w-4 h-4 text-red-500" />;
      default:
        return <Minus className="w-4 h-4 text-gray-500" />;
    }
  };

  const getVelocityColor = (velocity: string) => {
    switch (velocity) {
      case "Fast":
        return "bg-green-100 text-green-800";
      case "Moderate":
        return "bg-yellow-100 text-yellow-800";
      case "Slow":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/4 mb-4"></div>
          <div className="space-y-4">
            {[...Array(3)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-32 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">{error}</p>
          <button
            onClick={fetchCityTrends}
            className="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">
          City Trends Analysis
        </h2>
        <p className="text-gray-600">
          Market performance and trends across different cities
        </p>
      </div>

      <div className="space-y-6">
        {data.map((city) => (
          <div
            key={city.city}
            className="bg-white border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow"
          >
            {/* Header */}
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center gap-3">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <MapPin className="w-5 h-5 text-blue-600" />
                </div>
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    {city.city}
                  </h3>
                  <p className="text-sm text-gray-600">{city.region}</p>
                </div>
              </div>

              <div className="flex items-center gap-2">
                {getTrendIcon(city.trend_direction)}
                <span
                  className={`px-3 py-1 rounded-full text-sm font-medium ${getVelocityColor(
                    city.market_velocity,
                  )}`}
                >
                  {city.market_velocity}
                </span>
              </div>
            </div>

            {/* Stats Grid */}
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
              <div className="text-center">
                <div className="text-2xl font-bold text-blue-600">
                  {city.total_listings}
                </div>
                <div className="text-xs text-gray-600">Total Listings</div>
              </div>

              <div className="text-center">
                <div className="text-2xl font-bold text-green-600">
                  {city.active_listings}
                </div>
                <div className="text-xs text-gray-600">Active</div>
              </div>

              <div className="text-center">
                <div className="text-2xl font-bold text-purple-600">
                  {city.sold_count}
                </div>
                <div className="text-xs text-gray-600">Sold</div>
              </div>

              <div className="text-center">
                <div className="text-2xl font-bold text-orange-600">
                  {city.avg_price_formatted}
                </div>
                <div className="text-xs text-gray-600">Avg Price</div>
              </div>

              <div className="text-center">
                <div
                  className={`text-2xl font-bold ${
                    city.price_change_30d > 0
                      ? "text-green-600"
                      : city.price_change_30d < 0
                        ? "text-red-600"
                        : "text-gray-600"
                  }`}
                >
                  {city.price_change_30d > 0 ? "+" : ""}
                  {city.price_change_30d}%
                </div>
                <div className="text-xs text-gray-600">30d Change</div>
              </div>

              <div className="text-center">
                <div className="text-2xl font-bold text-indigo-600">
                  {city.conversion_rate}%
                </div>
                <div className="text-xs text-gray-600">Conversion</div>
              </div>
            </div>

            {/* Extra Info */}
            <div className="mt-4 pt-4 border-t border-gray-100">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
                <div className="flex justify-between">
                  <span className="text-gray-600">Demand-Supply Ratio:</span>
                  <span className="font-medium">
                    {city.demand_supply_ratio}
                  </span>
                </div>

                <div className="flex justify-between">
                  <span className="text-gray-600">Avg Days on Market:</span>
                  <span className="font-medium">
                    {city.avg_days_on_market} days
                  </span>
                </div>

                <div className="flex justify-between">
                  <span className="text-gray-600">Market Trend:</span>
                  <span className="font-medium capitalize flex items-center gap-1">
                    {getTrendIcon(city.trend_direction)}
                    {city.trend_direction}
                  </span>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {data.length === 0 && !loading && (
        <div className="text-center py-12">
          <Activity className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No city trends data available</p>
        </div>
      )}
    </div>
  );
};

export default CityTrends;
