import React, { useState, useEffect } from "react";
import { CheckCircle, TrendingUp, Database } from "lucide-react";

interface QualityMetrics {
  total_records: number;
  valid_records: number;
  invalid_records: number;
  data_completeness_score: number;
  accuracy_score: number;
  duplicate_properties: number;
  missing_prices: number;
  invalid_locations: number;
  price_outliers: number;
  missing_descriptions: number;
  quality_trends: QualityTrend[];
  quality_issues: QualityIssue[];
}

interface QualityTrend {
  date: string;
  completeness_score: number;
  invalid_count: number;
  daily_records: number;
  accuracy_score: number;
}

interface QualityIssue {
  type: string;
  count: number;
  severity: string;
}

const DataQuality: React.FC = () => {
  const [data, setData] = useState<QualityMetrics | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchDataQuality();
  }, []);

  const fetchDataQuality = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch(
        "/api/admin_market_trends/data-quality-metrics",
      );
      const result = await response.json();

      if (result.success) {
        setData(result.data.quality_metrics);
      } else {
        setError("Failed to fetch data quality metrics");
      }
    } catch (err) {
      setError("Error connecting to server");
    } finally {
      setLoading(false);
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 90) return "text-blue-600";
    if (score >= 70) return "text-yellow-600";
    return "text-red-600";
  };

  const getScoreBackground = (score: number) => {
    if (score >= 90) return "bg-blue-100";
    if (score >= 70) return "bg-yellow-100";
    return "bg-red-100";
  };

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case "High":
        return "bg-red-100 text-red-800";
      case "Medium":
        return "bg-yellow-100 text-yellow-800";
      case "Low":
        return "bg-blue-100 text-blue-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/3 mb-4"></div>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
            {[...Array(3)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-32 rounded-lg"></div>
            ))}
          </div>
          <div className="space-y-4">
            {[...Array(3)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-24 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">{error}</p>
          <button
            onClick={fetchDataQuality}
            className="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    );
  }

  if (!data) return null;

  return (
    <div className="p-6">
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">
          Data Quality Assessment
        </h2>
        <p className="text-gray-600">
          Comprehensive analysis of data completeness and accuracy
        </p>
      </div>

      {/* Score Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <div
          className={`rounded-lg p-6 ${getScoreBackground(data.data_completeness_score)}`}
        >
          <div className="flex items-center justify-between mb-4">
            <CheckCircle
              className={`w-8 h-8 ${getScoreColor(data.data_completeness_score)}`}
            />
            <span
              className={`text-3xl font-bold ${getScoreColor(data.data_completeness_score)}`}
            >
              {data.data_completeness_score}%
            </span>
          </div>
          <h3 className="text-lg font-semibold text-gray-900 mb-2">
            Data Completeness
          </h3>
          <p className="text-sm text-gray-600">
            {data.valid_records.toLocaleString()} of{" "}
            {data.total_records.toLocaleString()} records are complete
          </p>
        </div>

        <div
          className={`rounded-lg p-6 ${getScoreBackground(data.accuracy_score)}`}
        >
          <div className="flex items-center justify-between mb-4">
            <TrendingUp
              className={`w-8 h-8 ${getScoreColor(data.accuracy_score)}`}
            />
            <span
              className={`text-3xl font-bold ${getScoreColor(data.accuracy_score)}`}
            >
              {data.accuracy_score}%
            </span>
          </div>
          <h3 className="text-lg font-semibold text-gray-900 mb-2">
            Data Accuracy
          </h3>
          <p className="text-sm text-gray-600">
            Based on duplicate detection and outlier analysis
          </p>
        </div>

        <div className="bg-blue-100 rounded-lg p-6">
          <div className="flex items-center justify-between mb-4">
            <Database className="w-8 h-8 text-blue-600" />
            <span className="text-3xl font-bold text-blue-600">
              {data.total_records.toLocaleString()}
            </span>
          </div>
          <h3 className="text-lg font-semibold text-gray-900 mb-2">
            Total Records
          </h3>
          <p className="text-sm text-gray-600">
            {data.invalid_records.toLocaleString()} records need attention
          </p>
        </div>
      </div>

      {/* Issues */}
      <div className="mb-8">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">
          Quality Issues Breakdown
        </h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {data.quality_issues.map((issue, index) => (
            <div
              key={index}
              className="bg-white border border-gray-200 rounded-lg p-4"
            >
              <div className="flex items-center justify-between mb-2">
                <h4 className="font-medium text-gray-900">{issue.type}</h4>
                <span
                  className={`px-2 py-1 rounded-full text-xs font-medium ${getSeverityColor(issue.severity)}`}
                >
                  {issue.severity}
                </span>
              </div>
              <div className="text-2xl font-bold text-gray-900 mb-1">
                {issue.count.toLocaleString()}
              </div>
              <div className="text-sm text-gray-600">
                {((issue.count / data.total_records) * 100).toFixed(1)}% of
                total records
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Trends */}
      <div className="bg-white border border-gray-200 rounded-lg p-6 mb-8">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">
          Quality Trends (Last 7 Days)
        </h3>
        <div className="space-y-3">
          {data.quality_trends.slice(0, 7).map((trend, index) => (
            <div key={index} className="flex items-center justify-between">
              <span className="text-sm text-gray-600">
                {new Date(trend.date).toLocaleDateString()}
              </span>
              <div className="flex items-center gap-4">
                <span
                  className={`text-sm font-medium ${getScoreColor(trend.completeness_score)}`}
                >
                  {trend.completeness_score}%
                </span>
                <span className="text-sm text-gray-500">
                  {trend.daily_records} records
                </span>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};

export default DataQuality;
