import React, { useState, useEffect } from "react";
import { BarChart3, AlertCircle, CheckCircle } from "lucide-react";

interface DemandSupplyData {
  locality: string;
  city: string;
  total_supply: number;
  active_demand: number;
  active_supply: number;
  sold_count: number;
  conversion_rate: number;
  demand_supply_ratio: number;
  imbalance_level: string;
  avg_price: number;
  avg_price_formatted: string;
  price_change_30d: number;
  price_trend: string;
  market_velocity: string;
  recommendation: string;
}

const DemandSupplyAnalysis: React.FC = () => {
  const [data, setData] = useState<DemandSupplyData[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchDemandSupplyData();
  }, []);

  const fetchDemandSupplyData = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch(
        "/api/admin_market_trends/demand-supply-analysis",
      );
      const result = await response.json();

      if (result.success) {
        setData(result.data.analysis);
      } else {
        setError("Failed to fetch demand-supply data");
      }
    } catch (err) {
      setError("Error connecting to server");
    } finally {
      setLoading(false);
    }
  };

  const getImbalanceColor = (level: string) => {
    switch (level) {
      case "High Demand":
        return "bg-red-100 text-red-800";
      case "High Surplus":
        return "bg-blue-100 text-blue-800";
      case "Balanced":
        return "bg-green-100 text-green-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getImbalanceIcon = (level: string) => {
    switch (level) {
      case "High Demand":
        return <AlertCircle className="w-4 h-4" />;
      case "Balanced":
        return <CheckCircle className="w-4 h-4" />;
      default:
        return <BarChart3 className="w-4 h-4" />;
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/3 mb-4"></div>
          <div className="space-y-4">
            {[...Array(5)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-24 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">{error}</p>
          <button
            onClick={fetchDemandSupplyData}
            className="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">
          Demand-Supply Analysis
        </h2>
        <p className="text-gray-600">
          Market balance analysis with actionable recommendations
        </p>
      </div>

      <div className="overflow-x-auto">
        <table className="min-w-full bg-white border border-gray-200 rounded-lg">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                Locality
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                Supply
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                Demand
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                Ratio
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                Status
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                Avg Price
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                Conversion
              </th>
            </tr>
          </thead>
          <tbody className="divide-y divide-gray-200">
            {data.map((item, index) => (
              <tr key={index} className="hover:bg-gray-50">
                <td className="px-6 py-4">
                  <div className="text-sm font-medium text-gray-900">
                    {item.locality}
                  </div>
                  <div className="text-sm text-gray-500">{item.city}</div>
                </td>
                <td className="px-6 py-4">
                  <div className="text-sm text-gray-900">
                    {item.total_supply}
                  </div>
                  <div className="text-xs text-gray-500">
                    {item.active_supply} active
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="text-sm text-gray-900">
                    {item.active_demand}
                  </div>
                  <div className="text-xs text-gray-500">
                    {item.sold_count} sold
                  </div>
                </td>
                <td className="px-6 py-4 font-medium text-gray-900">
                  {item.demand_supply_ratio}
                </td>
                <td className="px-6 py-4">
                  <span
                    className={`inline-flex items-center gap-1 px-2.5 py-0.5 rounded-full text-xs font-medium ${getImbalanceColor(
                      item.imbalance_level,
                    )}`}
                  >
                    {getImbalanceIcon(item.imbalance_level)}
                    {item.imbalance_level}
                  </span>
                </td>
                <td className="px-6 py-4">
                  <div className="text-sm text-gray-900">
                    {item.avg_price_formatted}
                  </div>
                  <div
                    className={`text-xs ${
                      item.price_change_30d > 0
                        ? "text-green-600"
                        : item.price_change_30d < 0
                          ? "text-red-600"
                          : "text-gray-500"
                    }`}
                  >
                    {item.price_change_30d > 0 ? "+" : ""}
                    {item.price_change_30d}% (30d)
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="text-sm text-gray-900">
                    {item.conversion_rate}%
                  </div>
                  <div className="text-xs text-gray-500">
                    {item.market_velocity}
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {data.length > 0 && (
        <div className="mt-8">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">
            Market Recommendations
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {data.slice(0, 6).map((item, index) => (
              <div
                key={index}
                className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-4"
              >
                <h4 className="font-medium text-gray-900 mb-2">
                  {item.locality}
                </h4>
                <p className="text-sm text-gray-600">{item.recommendation}</p>
              </div>
            ))}
          </div>
        </div>
      )}

      {data.length === 0 && !loading && (
        <div className="text-center py-12">
          <BarChart3 className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No demand-supply data available</p>
        </div>
      )}
    </div>
  );
};

export default DemandSupplyAnalysis;
