import React, { useState, useEffect } from "react";
import { BarChart3, Target } from "lucide-react";

interface AccuracyMetric {
  model_type: string;
  forecast_period: string;
  accuracy_score: number;
  mean_absolute_error: number;
  predicted_values: number[];
  actual_values: number[];
  last_updated: string;
  data_points_used: number;
  model_confidence: string;
}

interface OverallPerformance {
  average_accuracy: number;
  total_data_points: number;
  model_reliability: string;
  last_model_update: string;
  data_source: string;
}

const ForecastAccuracy: React.FC = () => {
  const [data, setData] = useState<AccuracyMetric[]>([]);
  const [overall, setOverall] = useState<OverallPerformance | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchForecastAccuracy();
  }, []);

  const fetchForecastAccuracy = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch(
        "/api/admin_market_trends/forecast-accuracy",
      );
      const result = await response.json();

      if (result.success) {
        setData(result.data.accuracy_metrics);
        setOverall(result.data.overall_model_performance);
      } else {
        setError("Failed to fetch forecast accuracy data");
      }
    } catch (err) {
      setError("Error connecting to server");
    } finally {
      setLoading(false);
    }
  };

  const getAccuracyColor = (score: number) => {
    if (score >= 90) return "text-green-600";
    if (score >= 80) return "text-blue-600";
    if (score >= 70) return "text-yellow-600";
    return "text-red-600";
  };

  const getAccuracyBackground = (score: number) => {
    if (score >= 90) return "bg-green-100";
    if (score >= 80) return "bg-blue-100";
    if (score >= 70) return "bg-yellow-100";
    return "bg-red-100";
  };

  const getConfidenceColor = (confidence: string) => {
    switch (confidence) {
      case "High":
        return "bg-green-100 text-green-800";
      case "Medium":
        return "bg-yellow-100 text-yellow-800";
      case "Low":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/3 mb-4"></div>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
            {[...Array(4)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-48 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">{error}</p>
          <button
            onClick={fetchForecastAccuracy}
            className="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">
          Forecast Accuracy
        </h2>
        <p className="text-gray-600">
          ML model performance and prediction accuracy metrics
        </p>
      </div>

      {overall && (
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-6 mb-8">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">
            Overall Model Performance
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center">
              <div
                className={`text-3xl font-bold ${getAccuracyColor(overall.average_accuracy)}`}
              >
                {overall.average_accuracy}%
              </div>
              <div className="text-sm text-gray-600">Average Accuracy</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-blue-600">
                {overall.total_data_points.toLocaleString()}
              </div>
              <div className="text-sm text-gray-600">Data Points</div>
            </div>
            <div className="text-center">
              <div
                className={`text-lg font-bold px-3 py-1 rounded-full ${getConfidenceColor(overall.model_reliability)}`}
              >
                {overall.model_reliability}
              </div>
              <div className="text-sm text-gray-600 mt-1">Reliability</div>
            </div>
            <div className="text-center">
              <div className="text-sm text-gray-900 font-medium">
                {new Date(overall.last_model_update).toLocaleDateString()}
              </div>
              <div className="text-sm text-gray-600">Last Updated</div>
            </div>
          </div>
        </div>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        {data.map((model, index) => (
          <div
            key={index}
            className="bg-white border border-gray-200 rounded-lg p-6"
          >
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center gap-3">
                <div
                  className={`p-2 rounded-lg ${getAccuracyBackground(model.accuracy_score)}`}
                >
                  <Target
                    className={`w-5 h-5 ${getAccuracyColor(model.accuracy_score)}`}
                  />
                </div>
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    {model.model_type}
                  </h3>
                  <p className="text-sm text-gray-600">
                    {model.forecast_period} forecast
                  </p>
                </div>
              </div>
              <span
                className={`px-3 py-1 rounded-full text-sm font-medium ${getConfidenceColor(model.model_confidence)}`}
              >
                {model.model_confidence}
              </span>
            </div>

            <div className="grid grid-cols-2 gap-4 mb-4 text-center">
              <div>
                <div
                  className={`text-2xl font-bold ${getAccuracyColor(model.accuracy_score)}`}
                >
                  {model.accuracy_score}%
                </div>
                <div className="text-xs text-gray-600">Accuracy Score</div>
              </div>
              <div>
                <div className="text-2xl font-bold text-gray-900">
                  {model.mean_absolute_error.toLocaleString()}
                </div>
                <div className="text-xs text-gray-600">Mean Abs Error</div>
              </div>
            </div>

            <div className="text-xs text-gray-500 border-t border-gray-100 pt-3">
              Last updated: {new Date(model.last_updated).toLocaleDateString()}
            </div>
          </div>
        ))}
      </div>

      {data.length === 0 && !loading && (
        <div className="text-center py-12">
          <BarChart3 className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No forecast accuracy data available</p>
        </div>
      )}
    </div>
  );
};

export default ForecastAccuracy;
