import React, { useState, useEffect } from "react";
import { PieChart, TrendingUp, TrendingDown } from "lucide-react";

interface MarketShift {
  period: string;
  conversion_rate: number;
  previous_conversion_rate: number;
  conversion_change: number;
  avg_days_on_market: number;
  price_movement: number;
  volume_change: number;
  market_phase: string;
  total_listings: number;
  total_sold: number;
  avg_price: number;
  key_insights: string[];
}

const MarketShifts: React.FC = () => {
  const [data, setData] = useState<MarketShift[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchMarketShifts();
  }, []);

  const fetchMarketShifts = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch("/api/admin_market_trends/market-shifts");
      const result = await response.json();

      if (result.success) {
        setData(result.data.shifts);
      } else {
        setError("Failed to fetch market shifts data");
      }
    } catch (err) {
      setError("Error connecting to server");
    } finally {
      setLoading(false);
    }
  };

  const getPhaseColor = (phase: string) => {
    switch (phase) {
      case "Growth":
        return "bg-green-100 text-green-800";
      case "Decline":
        return "bg-red-100 text-red-800";
      case "Stable":
        return "bg-blue-100 text-blue-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/3 mb-4"></div>
          <div className="space-y-4">
            {[...Array(6)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-32 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">{error}</p>
          <button
            onClick={fetchMarketShifts}
            className="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">
          Market Shifts Analysis
        </h2>
        <p className="text-gray-600">
          Historical market trends and conversion rate analysis
        </p>
      </div>

      <div className="space-y-6">
        {data.map((shift, index) => (
          <div
            key={index}
            className="bg-white border border-gray-200 rounded-lg p-6"
          >
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900">
                {shift.period}
              </h3>
              <span
                className={`px-3 py-1 rounded-full text-sm font-medium ${getPhaseColor(shift.market_phase)}`}
              >
                {shift.market_phase} Phase
              </span>
            </div>

            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4 mb-6">
              <div className="text-center">
                <div className="text-2xl font-bold text-blue-600">
                  {shift.conversion_rate}%
                </div>
                <div className="text-xs text-gray-600">Conversion Rate</div>
              </div>

              <div className="text-center">
                <div
                  className={`text-2xl font-bold flex items-center justify-center gap-1 ${
                    shift.conversion_change > 0
                      ? "text-green-600"
                      : shift.conversion_change < 0
                        ? "text-red-600"
                        : "text-gray-600"
                  }`}
                >
                  {shift.conversion_change > 0 && (
                    <TrendingUp className="w-5 h-5" />
                  )}
                  {shift.conversion_change < 0 && (
                    <TrendingDown className="w-5 h-5" />
                  )}
                  {shift.conversion_change > 0 ? "+" : ""}
                  {shift.conversion_change}%
                </div>
                <div className="text-xs text-gray-600">Change</div>
              </div>

              <div className="text-center">
                <div className="text-2xl font-bold text-orange-600">
                  {shift.avg_days_on_market}
                </div>
                <div className="text-xs text-gray-600">Days on Market</div>
              </div>

              <div className="text-center">
                <div
                  className={`text-2xl font-bold ${
                    shift.price_movement > 0
                      ? "text-green-600"
                      : shift.price_movement < 0
                        ? "text-red-600"
                        : "text-gray-600"
                  }`}
                >
                  {shift.price_movement > 0 ? "+" : ""}
                  {shift.price_movement}%
                </div>
                <div className="text-xs text-gray-600">Price Movement</div>
              </div>

              <div className="text-center">
                <div className="text-2xl font-bold text-purple-600">
                  {shift.total_listings}
                </div>
                <div className="text-xs text-gray-600">Total Listings</div>
              </div>

              <div className="text-center">
                <div className="text-2xl font-bold text-indigo-600">
                  {shift.total_sold}
                </div>
                <div className="text-xs text-gray-600">Properties Sold</div>
              </div>
            </div>

            <div className="border-t border-gray-100 pt-4">
              <h4 className="font-medium text-gray-900 mb-3">Key Insights</h4>
              <ul className="space-y-2">
                {shift.key_insights.map((insight, idx) => (
                  <li
                    key={idx}
                    className="flex items-start gap-2 text-sm text-gray-600"
                  >
                    <div className="w-1.5 h-1.5 bg-blue-500 rounded-full mt-2 flex-shrink-0"></div>
                    {insight}
                  </li>
                ))}
              </ul>
            </div>
          </div>
        ))}
      </div>

      {data.length === 0 && !loading && (
        <div className="text-center py-12">
          <PieChart className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No market shifts data available</p>
        </div>
      )}
    </div>
  );
};

export default MarketShifts;
