import React, { useState, useEffect } from "react";
import {
  Building,
  Users,
  UserCheck,
  Calendar,
  Activity,
  TrendingUp,
} from "lucide-react";

interface PlatformData {
  total_properties: number;
  active_properties: number;
  total_users: number;
  total_agents: number;
  avg_days_on_market: number;
  platform_health_score: number;
}

const PlatformOverview: React.FC = () => {
  const [data, setData] = useState<PlatformData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchPlatformOverview();
  }, []);

  const fetchPlatformOverview = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch(
        "/api/admin_market_trends/platform-overview",
      );
      const result = await response.json();

      if (result.success) {
        setData(result.data.overview);
      } else {
        setError("Failed to fetch platform data");
      }
    } catch (err) {
      setError("Error connecting to server");
    } finally {
      setLoading(false);
    }
  };

  const getHealthScoreColor = (score: number) => {
    if (score >= 80) return "text-green-600 bg-green-100";
    if (score >= 60) return "text-yellow-600 bg-yellow-100";
    return "text-red-600 bg-red-100";
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {[...Array(6)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-32 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">{error}</p>
          <button
            onClick={fetchPlatformOverview}
            className="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    );
  }

  const metrics = [
    {
      title: "Total Properties",
      value: data?.total_properties || 0,
      icon: Building,
      color: "text-blue-600 bg-blue-100",
    },
    {
      title: "Active Properties",
      value: data?.active_properties || 0,
      icon: Activity,
      color: "text-green-600 bg-green-100",
    },
    {
      title: "Total Users",
      value: data?.total_users || 0,
      icon: Users,
      color: "text-purple-600 bg-purple-100",
    },
    {
      title: "Total Agents",
      value: data?.total_agents || 0,
      icon: UserCheck,
      color: "text-indigo-600 bg-indigo-100",
    },
    {
      title: "Avg Days on Market",
      value: data?.avg_days_on_market || 0,
      icon: Calendar,
      color: "text-orange-600 bg-orange-100",
      suffix: " days",
    },
    {
      title: "Platform Health Score",
      value: data?.platform_health_score || 0,
      icon: TrendingUp,
      color: getHealthScoreColor(data?.platform_health_score || 0),
      suffix: "/100",
    },
  ];

  return (
    <div className="p-6">
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">
          Platform Overview
        </h2>
        <p className="text-gray-600">
          Key performance indicators and platform health metrics
        </p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {metrics.map((metric, index) => {
          const Icon = metric.icon;
          return (
            <div
              key={index}
              className="bg-white border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow"
            >
              <div className="flex items-center justify-between mb-4">
                <div className={`p-3 rounded-lg ${metric.color}`}>
                  <Icon className="w-6 h-6" />
                </div>
              </div>

              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">
                  {metric.title}
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {metric.value.toLocaleString()}
                  {metric.suffix || ""}
                </p>
              </div>
            </div>
          );
        })}
      </div>

      {data && (
        <div className="mt-8 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">
            Platform Health Analysis
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">
                {data.total_properties > 0
                  ? (
                      (data.active_properties / data.total_properties) *
                      100
                    ).toFixed(1)
                  : 0}
                %
              </div>
              <div className="text-sm text-gray-600">Active Rate</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">
                {(data.total_users * 0.1).toFixed(1)}%
              </div>
              <div className="text-sm text-gray-600">User Engagement</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-purple-600">
                {Math.max(0, 100 - data.avg_days_on_market)}%
              </div>
              <div className="text-sm text-gray-600">Market Velocity</div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default PlatformOverview;
