import React, { useState, useEffect } from "react";
import { AlertTriangle, Eye, CheckCircle, Clock, XCircle } from "lucide-react";

interface PriceAnomaly {
  property_id: string;
  city: string;
  locality: string;
  location: string;
  listed_price: number;
  expected_price: number;
  deviation_percent: number;
  anomaly_type: string;
  severity: string;
  detected_at: string;
  status: string;
  property_type: string;
}

const PriceAnomalies: React.FC = () => {
  const [data, setData] = useState<PriceAnomaly[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [filter, setFilter] = useState("all");

  useEffect(() => {
    fetchPriceAnomalies();
  }, []);

  const fetchPriceAnomalies = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch("/api/admin_market_trends/price-anomalies");
      const result = await response.json();

      if (result.success) {
        setData(result.data.anomalies);
      } else {
        setError("Failed to fetch price anomalies data");
      }
    } catch (err) {
      setError("Error connecting to server");
    } finally {
      setLoading(false);
    }
  };

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case "High":
        return "bg-red-100 text-red-800 border-red-200";
      case "Medium":
        return "bg-yellow-100 text-yellow-800 border-yellow-200";
      case "Low":
        return "bg-blue-100 text-blue-800 border-blue-200";
      default:
        return "bg-gray-100 text-gray-800 border-gray-200";
    }
  };

  const getSeverityIcon = (severity: string) => {
    switch (severity) {
      case "High":
        return <AlertTriangle className="w-4 h-4" />;
      case "Medium":
        return <Eye className="w-4 h-4" />;
      case "Low":
        return <CheckCircle className="w-4 h-4" />;
      default:
        return <AlertTriangle className="w-4 h-4" />;
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "Open":
        return <AlertTriangle className="w-4 h-4 text-red-500" />;
      case "Under Review":
        return <Clock className="w-4 h-4 text-yellow-500" />;
      case "Resolved":
        return <CheckCircle className="w-4 h-4 text-green-500" />;
      case "False Positive":
        return <XCircle className="w-4 h-4 text-gray-500" />;
      default:
        return <AlertTriangle className="w-4 h-4 text-red-500" />;
    }
  };

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)}Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(1)}L`;
    return `₹${price.toLocaleString()}`;
  };

  const filteredData = data.filter((anomaly) => {
    if (filter === "all") return true;
    return anomaly.severity.toLowerCase() === filter;
  });

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse space-y-4">
          {[...Array(5)].map((_, i) => (
            <div key={i} className="bg-gray-200 h-24 rounded-lg"></div>
          ))}
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">{error}</p>
          <button
            onClick={fetchPriceAnomalies}
            className="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">
          Price Anomaly Detection
        </h2>
        <p className="text-gray-600">
          Identify unusual pricing patterns and potential market manipulation
        </p>
      </div>

      {/* Filter Tabs */}
      <div className="mb-6 border-b border-gray-200">
        <nav className="flex space-x-8">
          {["all", "high", "medium", "low"].map((severity) => (
            <button
              key={severity}
              onClick={() => setFilter(severity)}
              className={`${
                filter === severity
                  ? "border-red-500 text-red-600"
                  : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              } whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm capitalize`}
            >
              {severity === "all" ? "All Anomalies" : `${severity} Severity`}
            </button>
          ))}
        </nav>
      </div>

      <div className="space-y-4">
        {filteredData.map((anomaly, index) => (
          <div
            key={index}
            className={`border rounded-lg p-6 ${getSeverityColor(anomaly.severity)}`}
          >
            <div className="flex items-start justify-between mb-4">
              <div className="flex items-center gap-3">
                {getSeverityIcon(anomaly.severity)}
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    {anomaly.property_id}
                  </h3>
                  <p className="text-sm text-gray-600">{anomaly.location}</p>
                </div>
              </div>
              <div className="flex items-center gap-2">
                {getStatusIcon(anomaly.status)}
                <span className="text-sm font-medium">{anomaly.status}</span>
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-4">
              <div>
                <div className="text-sm text-gray-600">Listed Price</div>
                <div className="text-lg font-semibold text-gray-900">
                  {formatPrice(anomaly.listed_price)}
                </div>
              </div>
              <div>
                <div className="text-sm text-gray-600">Expected Price</div>
                <div className="text-lg font-semibold text-gray-900">
                  {formatPrice(anomaly.expected_price)}
                </div>
              </div>
              <div>
                <div className="text-sm text-gray-600">Deviation</div>
                <div
                  className={`text-lg font-semibold ${
                    anomaly.deviation_percent > 25
                      ? "text-red-600"
                      : anomaly.deviation_percent > 15
                        ? "text-yellow-600"
                        : "text-blue-600"
                  }`}
                >
                  {anomaly.deviation_percent.toFixed(1)}%
                </div>
              </div>
              <div>
                <div className="text-sm text-gray-600">Anomaly Type</div>
                <div className="text-lg font-semibold text-gray-900">
                  {anomaly.anomaly_type}
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {filteredData.length === 0 && (
        <div className="text-center py-12">
          <AlertTriangle className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No price anomalies found</p>
        </div>
      )}
    </div>
  );
};

export default PriceAnomalies;
