
import React, { useState } from 'react';
import { adminService } from '../../../api/adminService';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { AlertCircle, Lock, Server } from 'lucide-react';

interface AddConnectorModalProps {
    open: boolean;
    onClose: () => void;
    onSuccess: () => void;
}

export const AddConnectorModal: React.FC<AddConnectorModalProps> = ({ open, onClose, onSuccess }) => {
    const [provider, setProvider] = useState('google');
    const [clientId, setClientId] = useState('');
    const [clientSecret, setClientSecret] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const handleSubmit = async () => {
        if (!clientId || !clientSecret) {
            setError('All fields are required');
            return;
        }

        try {
            setLoading(true);
            setError(null);
            await adminService.connectors.addConfig({
                provider,
                client_id: clientId,
                client_secret: clientSecret
            });
            onSuccess();
            onClose();
            setClientId('');
            setClientSecret('');
        } catch (err: any) {
            setError(err.response?.data?.error || 'Failed to add connector configuration');
        } finally {
            setLoading(false);
        }
    };

    return (
        <Dialog open={open} onOpenChange={onClose}>
            <DialogContent className="sm:max-w-[450px] rounded-2xl p-6 bg-white shadow-xl border-gray-100">
                <DialogHeader>
                    <DialogTitle className="flex items-center gap-2 text-xl font-bold">
                        <div className="bg-blue-50 p-2 rounded-lg">
                            <Server className="w-5 h-5 text-blue-600" />
                        </div>
                        Add Connector
                    </DialogTitle>
                </DialogHeader>

                <div className="space-y-4 py-4">
                    {error && (
                        <div className="bg-red-50 text-red-600 text-sm p-3 rounded-lg flex items-center gap-2">
                            <AlertCircle className="w-4 h-4" /> {error}
                        </div>
                    )}

                    <div className="space-y-2">
                        <Label className="text-xs font-bold uppercase text-gray-500">Provider</Label>
                        <Select value={provider} onValueChange={setProvider}>
                            <SelectTrigger className="rounded-xl border-gray-200"><SelectValue /></SelectTrigger>
                            <SelectContent>
                                <SelectItem value="google">Google Classroom</SelectItem>
                                <SelectItem value="microsoft">Microsoft Teams</SelectItem>
                                <SelectItem value="canvas">Canvas LMS</SelectItem>
                                <SelectItem value="zoom">Zoom</SelectItem>
                                <SelectItem value="slack">Slack</SelectItem>
                                <SelectItem value="salesforce">Salesforce</SelectItem>
                                <SelectItem value="hubspot">HubSpot</SelectItem>
                                <SelectItem value="dropbox">Dropbox</SelectItem>
                                <SelectItem value="github">GitHub</SelectItem>
                                <SelectItem value="notion">Notion</SelectItem>
                            </SelectContent>
                        </Select>
                    </div>

                    <div className="space-y-2">
                        <Label className="text-xs font-bold uppercase text-gray-500">Client ID</Label>
                        <Input value={clientId} onChange={(e) => setClientId(e.target.value)} className="rounded-xl border-gray-200" placeholder="From Developer Portal" />
                    </div>

                    <div className="space-y-2">
                        <Label className="text-xs font-bold uppercase text-gray-500">Client Secret</Label>
                        <Input type="password" value={clientSecret} onChange={(e) => setClientSecret(e.target.value)} className="rounded-xl border-gray-200" placeholder="••••••••" />
                        <p className="text-[10px] text-gray-400">Stored securely in Vault.</p>
                    </div>
                </div>

                <DialogFooter className="gap-2 sm:justify-between">
                    <Button variant="ghost" onClick={onClose} className="rounded-xl font-bold text-gray-500">Cancel</Button>
                    <Button onClick={handleSubmit} disabled={loading} className="bg-[#0056D2] hover:bg-blue-700 text-white font-bold rounded-xl px-6">
                        {loading ? 'Saving...' : 'Save Configuration'}
                    </Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
};
