
import React from 'react';
import { useOAuthFlow } from '../../../hooks/useOAuthFlow';
import { ExternalLink, Settings, Trash2, CheckCircle, Cloud, Database, Box, Slack, Video } from 'lucide-react';

interface ConnectorCardProps {
    provider: any;
    connector?: any;
    config?: any;
    onRefresh: () => void;
    onConfigure: () => void;
    onDeleteConfig: (id: number) => void;
    onViewSync: () => void;
    onConnectSuccess: () => void;
}

export const ConnectorCard: React.FC<ConnectorCardProps> = ({
    provider,
    connector,
    config,
    onRefresh,
    onConfigure,
    onDeleteConfig,
    onViewSync,
    onConnectSuccess
}) => {
    const { initiateOAuth, loading: authLoading } = useOAuthFlow(provider.id);
    // Backend returns 'status' field: 'active', 'pending_oauth', 'inactive'
    const isConnected = !!connector && connector.status === 'active';
    const isConfigured = !!config && (config.is_active !== false);

    const handleConnect = async () => {
        if (!isConfigured) {
            onConfigure();
            return;
        }

        const url = await initiateOAuth();
        if (url) {
            const width = 600;
            const height = 700;
            const left = window.screen.width / 2 - width / 2;
            const top = window.screen.height / 2 - height / 2;

            window.open(
                url,
                'oauth_popup',
                `width=${width},height=${height},left=${left},top=${top},scrollbars=yes,status=yes`
            );

            // Listen for success message from popup
            const handleMessage = (event: MessageEvent) => {
                // Ideally verify origin for security
                if (event.data.type === 'OAUTH_SUCCESS' && event.data.provider === provider.id) {
                    window.removeEventListener('message', handleMessage);
                    onRefresh();
                    onConnectSuccess();
                }
            };
            window.addEventListener('message', handleMessage);
        }
    };

    const getBrandStyles = (id: string) => {
        switch (id) {
            case 'salesforce-crm': return { bg: 'bg-blue-50', iconColor: 'text-blue-500', Icon: Cloud };
            case 'zoho-crm': return { bg: 'bg-yellow-50', iconColor: 'text-yellow-600', Icon: Database };
            case 'google-workspace': return { bg: 'bg-green-50', iconColor: 'text-green-600', Icon: Cloud };
            case 'hubspot': return { bg: 'bg-orange-50', iconColor: 'text-orange-500', Icon: Database };
            case 'mailchimp': return { bg: 'bg-yellow-50', iconColor: 'text-yellow-700', Icon: Box };
            case 'slack': return { bg: 'bg-purple-50', iconColor: 'text-purple-600', Icon: Slack };
            case 'zillow': return { bg: 'bg-blue-50', iconColor: 'text-blue-600', Icon: Database };
            case 'docusign': return { bg: 'bg-indigo-50', iconColor: 'text-indigo-600', Icon: Box };
            case 'quickbooks': return { bg: 'bg-green-50', iconColor: 'text-green-500', Icon: Database };
            case 'calendly': return { bg: 'bg-blue-50', iconColor: 'text-blue-400', Icon: Video };
            default: return { bg: 'bg-gray-50', iconColor: 'text-gray-500', Icon: Settings };
        }
    };

    const styles = getBrandStyles(provider.id);
    const Icon = styles.Icon;

    return (
        <div className={`relative rounded-2xl border border-gray-100 shadow-sm hover:shadow-md transition-all p-6 bg-white group`}>
            <div className={`absolute top-4 right-4 w-2.5 h-2.5 rounded-full ${isConnected ? 'bg-green-500 animate-pulse' : 'bg-gray-300'}`} />

            <div className="flex justify-between items-start mb-4">
                <div className={`w-12 h-12 rounded-xl flex items-center justify-center ${styles.bg} ${styles.iconColor}`}>
                    <Icon size={24} />
                </div>
                {isConfigured && (
                    <button onClick={() => onDeleteConfig(config.id)} className="p-2 text-gray-400 hover:text-red-500 hover:bg-red-50 rounded-lg transition-colors opacity-0 group-hover:opacity-100">
                        <Trash2 size={16} />
                    </button>
                )}
            </div>

            <h3 className="text-lg font-bold text-gray-900 mb-1">{provider.name}</h3>
            <p className="text-sm text-gray-500 mb-6 h-10 leading-snug">{provider.description}</p>

            {isConnected ? (
                <button
                    onClick={onViewSync}
                    className="w-full bg-white border border-gray-200 text-gray-700 font-bold py-3 rounded-xl hover:bg-gray-50 flex items-center justify-center gap-2"
                >
                    <ExternalLink size={16} /> View Logs
                </button>
            ) : (
                <button
                    onClick={handleConnect}
                    disabled={authLoading}
                    className={`w-full py-3 rounded-xl font-bold flex items-center justify-center gap-2 transition-all
                    ${isConfigured
                            ? 'bg-[#0056D2] text-white hover:bg-blue-700'
                            : 'bg-[#0056D2] text-white hover:bg-blue-700'
                        }`}
                >
                    {authLoading ? 'Connecting...' : isConfigured ? 'Connect App' : 'Configure First'}
                </button>
            )}

            {isConnected && (
                <div className="mt-4 pt-4 border-t border-gray-100 flex items-center gap-2 text-xs font-medium text-green-600">
                    <CheckCircle size={12} /> Connected
                </div>
            )}
        </div>
    );
};
