
import React from 'react';
import { useConnectorAPI } from '../../../hooks/useConnectorAPI';
import { adminService } from '../../../api/adminService';
import { RefreshCw, CheckCircle, XCircle, Clock, Loader2 } from 'lucide-react';
import { Button } from "@/components/ui/button";

export const SyncJobMonitor: React.FC = () => {
    const { syncJobs, connectors, loading, fetchSyncJobs } = useConnectorAPI();
    const connectedApps = connectors.filter(c => c.is_connected === true);

    const handleSync = async (app: any) => {
        if (confirm(`Start full sync for ${app.name}?`)) {
            try {
                await adminService.connectors.triggerSync(app.id);
                fetchSyncJobs();
            } catch (e) {
                alert('Failed to start sync');
            }
        }
    };

    if (loading && syncJobs.length === 0) return <div className="p-8 text-center text-gray-500">Loading history...</div>;

    return (
        <div className="space-y-6">
            {connectedApps.length > 0 && (
                <div className="bg-green-50/50 p-4 rounded-xl border border-green-100">
                    <h4 className="text-sm font-bold text-green-800 mb-3 flex items-center gap-2">
                        <span className="w-2 h-2 rounded-full bg-green-500 animate-pulse" /> Active Integrations
                    </h4>
                    <div className="flex flex-wrap gap-3">
                        {connectedApps.map(app => (
                            <div key={app.id} className="bg-white px-3 py-2 rounded-lg border border-green-200 shadow-sm flex items-center gap-3">
                                <span className="text-sm font-medium text-gray-700">{app.name}</span>
                                <Button size="sm" variant="ghost" className="h-7 text-xs text-white bg-[#0056D2] hover:bg-blue-700 hover:text-white" onClick={() => handleSync(app)}>
                                    <RefreshCw className="w-3 h-3 mr-1" /> Sync
                                </Button>
                            </div>
                        ))}
                    </div>
                </div>
            )}

            <div className="border rounded-xl overflow-hidden">
                <table className="w-full text-left text-sm">
                    <thead className="bg-gray-50 border-b">
                        <tr>
                            <th className="px-6 py-3 font-semibold text-black">Job Type</th>
                            <th className="px-6 py-3 font-semibold text-black">Status</th>
                            <th className="px-6 py-3 font-semibold text-black">Records</th>
                            <th className="px-6 py-3 font-semibold text-black">Time</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100">
                        {syncJobs.length === 0 ? (
                            <tr><td colSpan={4} className="px-6 py-8 text-center text-gray-400">No sync jobs found.</td></tr>
                        ) : (
                            syncJobs.map((job) => (
                                <tr key={job.id} className="hover:bg-gray-50/50">
                                    <td className="px-6 py-4 font-medium text-gray-900">{job.job_type}</td>
                                    <td className="px-6 py-4">
                                        <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-bold capitalize gap-1
                                            ${job.status === 'completed' ? 'bg-green-100 text-green-700' :
                                                job.status === 'failed' ? 'bg-red-100 text-red-700' :
                                                    job.status === 'running' ? 'bg-blue-100 text-blue-700' : 'bg-gray-100 text-gray-700'}`}>
                                            {job.status === 'completed' && <CheckCircle size={10} />}
                                            {job.status === 'failed' && <XCircle size={10} />}
                                            {job.status === 'running' && <Loader2 size={10} className="animate-spin" />}
                                            {job.status === 'pending' && <Clock size={10} />}
                                            {job.status}
                                        </span>
                                    </td>
                                    <td className="px-6 py-4 text-gray-500">
                                        {job.processed_records} / {job.total_records || '-'}
                                    </td>
                                    <td className="px-6 py-4 text-gray-400 text-xs">
                                        {new Date(job.created_at).toLocaleString()}
                                    </td>
                                </tr>
                            ))
                        )}
                    </tbody>
                </table>
            </div>
        </div>
    );
};
