import React, { useState, useEffect } from 'react';
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { AlertCircle, AlertTriangle, Info, CheckCircle, Loader2 } from 'lucide-react';
import { adminService } from '../../../api/adminService';

export const SystemAlertsPanel: React.FC = () => {
    const [alerts, setAlerts] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const fetchAlerts = async () => {
            try {
                // Fetch recent logs and filter for alerts (warnings/errors)
                const res = await adminService.system.getLogs(50);
                const logs = Array.isArray(res.data) ? res.data : [];

                // Filter only warnings and errors/criticals
                // And map them to alert format
                const activeAlerts = logs.filter((log: any) =>
                    ['error', 'critical', 'warning'].includes(log.severity?.toLowerCase())
                ).map((log: any) => ({
                    id: log.id,
                    type: log.severity?.toLowerCase() === 'error' ? 'critical' : log.severity?.toLowerCase(),
                    title: log.source ? log.source.toUpperCase() : 'SYSTEM ALERT',
                    message: log.message,
                    time: new Date(log.created_at).toLocaleString()
                }));

                setAlerts(activeAlerts);
            } catch (error) {
                console.error("Failed to fetch alerts:", error);
            } finally {
                setLoading(false);
            }
        };

        fetchAlerts();
    }, []);

    if (loading) {
        return <div className="flex justify-center p-8 text-gray-500"><Loader2 className="animate-spin mr-2" /> Loading Alerts...</div>;
    }

    if (alerts.length === 0) {
        return (
            <div className="flex flex-col items-center justify-center p-8 text-gray-400 bg-gray-50 rounded-xl border border-dashed border-gray-200">
                <CheckCircle className="h-8 w-8 text-green-500 mb-2" />
                <p>No active alerts. System is healthy.</p>
            </div>
        );
    }

    return (
        <div className="space-y-4">
            {alerts.map(alert => (
                <Alert key={alert.id} variant={alert.type === 'critical' ? 'destructive' : 'default'}
                    className={`border-l-4 ${alert.type === 'critical' ? 'border-l-red-500 bg-red-50' :
                        alert.type === 'warning' ? 'border-l-orange-500 bg-orange-50 border-orange-200' :
                            'border-l-blue-500 bg-blue-50 border-blue-200 text-blue-900'
                        }`}
                >
                    <div className="flex items-start gap-3">
                        {alert.type === 'critical' && <AlertCircle className="h-5 w-5" />}
                        {alert.type === 'warning' && <AlertTriangle className="h-5 w-5 text-orange-600" />}
                        {alert.type === 'info' && <CheckCircle className="h-5 w-5 text-blue-600" />}

                        <div className="flex-1">
                            <AlertTitle className={`font-bold ${alert.type === 'critical' ? 'text-red-900' :
                                alert.type === 'warning' ? 'text-orange-900' : 'text-blue-900'
                                }`}>{alert.title}</AlertTitle>
                            <AlertDescription className={`mt-1 text-xs font-medium ${alert.type === 'critical' ? 'text-red-700' :
                                alert.type === 'warning' ? 'text-orange-700' : 'text-blue-700'
                                }`}>
                                {alert.message}
                            </AlertDescription>
                        </div>
                        <span className="text-[10px] text-gray-400 font-mono whitespace-nowrap pt-1">
                            {alert.time}
                        </span>
                    </div>
                </Alert>
            ))}
        </div>
    );
};
