
import React, { useState, useEffect } from 'react';
import { adminService } from '../../../api/adminService';
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { RefreshCw, Search, Filter, AlertTriangle, AlertCircle, Info, Download } from 'lucide-react';

export const SystemLogsPanel: React.FC = () => {
    const [allLogs, setAllLogs] = useState<any[]>([]); // Store full fetched logs
    const [logs, setLogs] = useState<any[]>([]); // Store filtered/paginated logs
    const [loading, setLoading] = useState(false);
    const [filter, setFilter] = useState({ level: 'all', search: '' });
    const [page, setPage] = useState(1);
    const ITEMS_PER_PAGE = 10;
    const [totalPages, setTotalPages] = useState(1);

    const fetchLogs = async () => {
        setLoading(true);
        try {
            // Fetch unified system logs (Audit + Webhook + Alerts)
            const res = await adminService.system.getLogs(100);
            const data = Array.isArray(res.data) ? res.data : [];
            setAllLogs(data);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchLogs();
    }, []);

    // Client-side filtering and pagination
    useEffect(() => {
        let filtered = allLogs;

        if (filter.level !== 'all') {
            filtered = filtered.filter(log => log.severity?.toLowerCase() === filter.level.toLowerCase());
        }

        if (filter.search) {
            const lowerSearch = filter.search.toLowerCase();
            filtered = filtered.filter(log =>
                log.message?.toLowerCase().includes(lowerSearch) ||
                log.source?.toLowerCase().includes(lowerSearch) ||
                log.id?.toLowerCase().includes(lowerSearch)
            );
        }

        setTotalPages(Math.ceil(filtered.length / ITEMS_PER_PAGE) || 1);
        const start = (page - 1) * ITEMS_PER_PAGE;
        setLogs(filtered.slice(start, start + ITEMS_PER_PAGE));
    }, [allLogs, filter, page]);

    const getLevelBadge = (level: string) => {
        switch (level.toLowerCase()) {
            case 'error': return <Badge variant="destructive" className="flex items-center gap-1"><AlertCircle size={10} /> ERROR</Badge>;
            case 'warning': return <Badge className="bg-orange-100 text-orange-700 hover:bg-orange-200 border-none flex items-center gap-1"><AlertTriangle size={10} /> WARN</Badge>;
            default: return <Badge variant="secondary" className="flex items-center gap-1"><Info size={10} /> INFO</Badge>;
        }
    };

    return (
        <div className="space-y-4">
            <div className="flex flex-col md:flex-row gap-4 justify-between items-center bg-gray-50 p-4 rounded-xl border border-gray-100">
                <div className="flex gap-2 w-full md:w-auto">
                    <div className="relative">
                        <Search className="absolute left-3 top-2.5 h-4 w-4 text-gray-400" />
                        <Input
                            placeholder="Search logs..."
                            className="pl-9 bg-white rounded-xl w-[250px]"
                            value={filter.search}
                            onChange={(e) => setFilter({ ...filter, search: e.target.value })}
                        />
                    </div>
                    <Select value={filter.level} onValueChange={(val) => setFilter({ ...filter, level: val })}>
                        <SelectTrigger className="w-[150px] bg-white rounded-xl">
                            <Filter className="w-4 h-4 mr-2 text-gray-400" />
                            <SelectValue placeholder="Level" />
                        </SelectTrigger>
                        <SelectContent>
                            <SelectItem value="all">All Levels</SelectItem>
                            <SelectItem value="info">Info</SelectItem>
                            <SelectItem value="warning">Warning</SelectItem>
                            <SelectItem value="error">Error</SelectItem>
                        </SelectContent>
                    </Select>
                </div>
                <div className="flex gap-2">
                    <Button variant="outline" className="rounded-xl border-gray-200" onClick={fetchLogs}>
                        <RefreshCw className={`w-4 h-4 mr-2 ${loading ? 'animate-spin' : ''}`} /> Refresh
                    </Button>
                    <Button variant="outline" className="rounded-xl border-gray-200">
                        <Download className="w-4 h-4 mr-2" /> Export
                    </Button>
                </div>
            </div>

            <div className="rounded-xl border border-gray-200 overflow-hidden bg-white shadow-sm">
                <Table>
                    <TableHeader className="bg-gray-50">
                        <TableRow>
                            <TableHead className="w-[180px] font-bold text-gray-600">Timestamp</TableHead>
                            <TableHead className="w-[100px] font-bold text-gray-600">Level</TableHead>
                            <TableHead className="font-bold text-gray-600">Message</TableHead>
                            <TableHead className="w-[150px] font-bold text-gray-600">Source</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {logs.length === 0 && !loading ? (
                            <TableRow>
                                <TableCell colSpan={4} className="h-24 text-center text-gray-500">
                                    No logs found matching your criteria.
                                </TableCell>
                            </TableRow>
                        ) : (
                            logs.map((log, index) => (
                                <TableRow key={log.id || index} className="hover:bg-gray-50/50">
                                    <TableCell className="font-mono text-xs text-gray-500">
                                        {new Date(log.created_at || log.timestamp).toLocaleString()}
                                    </TableCell>
                                    <TableCell>{getLevelBadge(log.level)}</TableCell>
                                    <TableCell className="font-medium text-gray-700">{log.message || log.action}</TableCell>
                                    <TableCell className="text-gray-500 text-sm">{log.source || log.user?.email || 'System'}</TableCell>
                                </TableRow>
                            ))
                        )}
                    </TableBody>
                </Table>
            </div>

            {/* Pagination controls could go here */}
            <div className="flex justify-between items-center px-2">
                <p className="text-xs text-gray-400">Showing {logs.length} results</p>
                <div className="flex gap-2">
                    <Button
                        variant="ghost"
                        size="sm"
                        disabled={page <= 1}
                        onClick={() => setPage(p => Math.max(1, p - 1))}
                    >
                        Previous
                    </Button>
                    <Button
                        variant="ghost"
                        size="sm"
                        disabled={page >= totalPages}
                        onClick={() => setPage(p => p + 1)}
                    >
                        Next
                    </Button>
                </div>
            </div>
        </div>
    );
};
