import React, { useState, useEffect } from 'react';
import { platformConfigAPI, PlatformConfig, ConfigAuditLog } from '../../api/platformConfig';
import { toast, Toaster } from 'react-hot-toast';

const PlatformSettings: React.FC = () => {
  const [configs, setConfigs] = useState<PlatformConfig[]>([]);
  const [auditLogs, setAuditLogs] = useState<ConfigAuditLog[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  const [editingConfig, setEditingConfig] = useState<string | null>(null);
  const [editValue, setEditValue] = useState<string>('');
  const [showAuditLogs, setShowAuditLogs] = useState(false);

  const categories = [
    { value: 'all', label: 'All Settings' },
    { value: 'general', label: 'General' },
    { value: 'business', label: 'Business' },
    { value: 'features', label: 'Features' },
    { value: 'email', label: 'Email' },
    { value: 'payment', label: 'Payment' },
  ];

  useEffect(() => {
    fetchConfigs();
  }, [selectedCategory]);

  const fetchConfigs = async () => {
    try {
      setLoading(true);
      const category = selectedCategory === 'all' ? undefined : selectedCategory;
      const data = await platformConfigAPI.getAllConfigs(category);
      setConfigs(data.configs);
    } catch (error: any) {
      toast.error(error.response?.data?.detail || 'Failed to fetch configurations');
    } finally {
      setLoading(false);
    }
  };

  const fetchAuditLogs = async () => {
    try {
      const data = await platformConfigAPI.getAuditLogs(undefined, 50);
      setAuditLogs(data.logs);
      setShowAuditLogs(true);
    } catch (error: any) {
      toast.error('Failed to fetch audit logs');
    }
  };

  const handleEdit = (config: PlatformConfig) => {
    setEditingConfig(config.config_key);
    setEditValue(config.config_value);
  };

  const handleSave = async (configKey: string) => {
    try {
      await platformConfigAPI.updateConfig(configKey, editValue);
      toast.success('Configuration updated successfully');
      setEditingConfig(null);
      fetchConfigs();
    } catch (error: any) {
      toast.error(error.response?.data?.detail || 'Failed to update configuration');
    }
  };

  const handleCancel = () => {
    setEditingConfig(null);
    setEditValue('');
  };

  const handleReload = async () => {
    try {
      const result = await platformConfigAPI.reloadConfigs();
      toast.success(result.message);
    } catch (error: any) {
      toast.error('Failed to reload configurations');
    }
  };

  const getCategoryIcon = (category: string) => {
    const icons: Record<string, string> = {
      general: '⚙️',
      business: '💼',
      features: '🎛️',
      email: '📧',
      payment: '💳',
    };
    return icons[category] || '📋';
  };

  const formatValue = (config: PlatformConfig) => {
    if (config.config_type === 'boolean') {
      return config.config_value === 'true' ? '✅ Enabled' : '❌ Disabled';
    }
    if (config.config_type === 'number') {
      if (config.config_key.includes('rate') || config.config_key.includes('commission')) {
        return `${config.config_value}%`;
      }
      if (config.config_key.includes('price') || config.config_key.includes('fee') || config.config_key.includes('amount')) {
        return `₹${parseFloat(config.config_value).toLocaleString()}`;
      }
      return config.config_value;
    }
    return config.config_value;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-xl">Loading configurations...</div>
      </div>
    );
  }

  return (
    <>
      <Toaster position="top-right" />
      <div className="p-6 max-w-7xl mx-auto">
      <div className="mb-6">
        <h1 className="text-3xl font-bold text-gray-900 mb-2">Platform Settings</h1>
        <p className="text-gray-600">Manage platform-wide configurations and settings</p>
      </div>

      <div className="bg-white rounded-lg shadow-sm p-4 mb-6 flex items-center justify-between">
        <div className="flex items-center space-x-4">
          <label className="text-sm font-medium text-gray-700">Category:</label>
          <select
            value={selectedCategory}
            onChange={(e) => setSelectedCategory(e.target.value)}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          >
            {categories.map((cat) => (
              <option key={cat.value} value={cat.value}>
                {cat.label}
              </option>
            ))}
          </select>
        </div>
        <div className="flex space-x-3">
          <button
            onClick={fetchAuditLogs}
            className="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors"
          >
            📜 View Audit Logs
          </button>
          <button
            onClick={handleReload}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            🔄 Reload Cache
          </button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        {configs.map((config) => (
          <div key={config.id} className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
            <div className="flex items-start justify-between mb-4">
              <div className="flex items-center space-x-2">
                <span className="text-2xl">{getCategoryIcon(config.category)}</span>
                <div>
                  <h3 className="font-semibold text-gray-900 capitalize">
                    {config.config_key.replace(/_/g, ' ')}
                  </h3>
                  <p className="text-sm text-gray-500">{config.description}</p>
                </div>
              </div>
              <span className="px-2 py-1 text-xs font-medium bg-gray-100 text-gray-600 rounded">
                {config.category}
              </span>
            </div>

            {editingConfig === config.config_key ? (
              <div className="space-y-3">
                {config.config_type === 'boolean' ? (
                  <select
                    value={editValue}
                    onChange={(e) => setEditValue(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="true">Enabled</option>
                    <option value="false">Disabled</option>
                  </select>
                ) : (
                  <input
                    type={config.config_type === 'number' ? 'number' : 'text'}
                    value={editValue}
                    onChange={(e) => setEditValue(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  />
                )}
                <div className="flex gap-3 mt-3">
                  <button
                    onClick={() => handleSave(config.config_key)}
                    className="flex-1 bg-green-500 hover:bg-green-600 text-white py-3 px-6 rounded-lg font-semibold text-base flex items-center justify-center gap-2 transition-all duration-200 hover:shadow-lg active:scale-95"
                  >
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                    </svg>
                    <span>Save</span>
                  </button>
                  <button
                    onClick={handleCancel}
                    className="flex-1 bg-red-600 hover:bg-red-700 text-white py-3 px-6 rounded-lg font-semibold text-base flex items-center justify-center gap-2 transition-all duration-200 hover:shadow-lg active:scale-95"
                  >
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                    </svg>
                    <span>Cancel</span>
                  </button>
                </div>
              </div>
            ) : (
              <div className="flex items-center justify-between">
                <div className="text-lg font-semibold text-gray-900">{formatValue(config)}</div>
                <button
                  onClick={() => handleEdit(config)}
                  className="px-4 py-2 bg-blue-50 text-blue-600 rounded-lg hover:bg-blue-100 transition-colors"
                >
                  ✏️ Edit
                </button>
              </div>
            )}

            {config.updated_by && (
              <div className="mt-3 pt-3 border-t border-gray-100 text-xs text-gray-500">
                Last updated by {config.updated_by} on {new Date(config.updated_at).toLocaleString()}
              </div>
            )}
          </div>
        ))}
      </div>

      {showAuditLogs && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[80vh] overflow-hidden">
            <div className="p-6 border-b border-gray-200 flex items-center justify-between">
              <h2 className="text-2xl font-bold text-gray-900">Audit Logs</h2>
              <button
                onClick={() => setShowAuditLogs(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                ✕
              </button>
            </div>
            <div className="p-6 overflow-y-auto max-h-[60vh]">
              <div className="space-y-4">
                {auditLogs.map((log) => (
                  <div key={log.id} className="bg-gray-50 rounded-lg p-4 border border-gray-200">
                    <div className="flex items-start justify-between mb-2">
                      <div className="font-semibold text-gray-900 capitalize">
                        {log.config_key.replace(/_/g, ' ')}
                      </div>
                      <div className="text-sm text-gray-500">
                        {new Date(log.changed_at).toLocaleString()}
                      </div>
                    </div>
                    <div className="flex items-center space-x-4 text-sm">
                      <div>
                        <span className="text-gray-500">Old:</span>{' '}
                        <span className="font-medium text-red-600">{log.old_value || 'N/A'}</span>
                      </div>
                      <span className="text-gray-400">→</span>
                      <div>
                        <span className="text-gray-500">New:</span>{' '}
                        <span className="font-medium text-green-600">{log.new_value}</span>
                      </div>
                    </div>
                    <div className="mt-2 text-xs text-gray-500">
                      Changed by {log.changed_by} {log.ip_address && `from ${log.ip_address}`}
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
    </>
  );
};

export default PlatformSettings;
