import React, { useState, useEffect } from 'react';
import { Download, TrendingUp, DollarSign, BarChart3, MapPin, Calendar, Users, Target, Loader2, AlertCircle, Home, Shield, Star, ArrowUp, ArrowDown, Filter, Search, RefreshCw, Calculator, Building, Zap } from 'lucide-react';
import { api } from '../../api/client';
import { rentalYieldApi } from '../../api/rentalYield';

interface PropertyInput {
  city: string;
  locality: string;
  property_type: 'apartment' | 'independent_house' | 'villa';
  bhk_config: string;
  built_area: number;
  purchase_price: number;
  furnishing: 'unfurnished' | 'semi_furnished' | 'fully_furnished';
  age_years: number;
}

interface RentalYieldResult {
  property_details: {
    location: string;
    type: string;
    built_area: number;
    purchase_price: number;
  };
  rental_analysis: {
    base_rent_per_sqft: number;
    adjusted_rent_per_sqft: number;
    estimated_monthly_rent: {
      min: number;
      max: number;
      average: number;
    };
    annual_rental_income: number;
    rental_yield_percentage: number;
    market_comparison: {
      locality_average_yield: number;
      performance: string;
    };
  };
  market_insights: {
    demand_level: string;
    rental_growth_trend: string;
    occupancy_rate: number;
  };
  investment_recommendation: {
    grade: string;
    score: number;
    reasons: string[];
  };
  adjustments_applied: {
    furnishing_bonus: number;
    age_factor: number;
    area_premium: number;
  };
}

interface LocalityYield {
  locality: string;
  current_yield: number;
  avg_property_price: number;
  avg_monthly_rent: number;
  property_count: number;
  avg_area_sqft: number;
  price_per_sqft: number;
  projected_cagr: number;
  rental_growth_rate: number;
  risk_score: number;
  investment_grade: string;
  market_tier: string;
  liquidity_score: number;
  growth_potential: string;
  rental_demand_score: number;
}

interface RentalYieldData {
  report_metadata: {
    generated_at: string;
    city: string;
    property_type: string;
    investment_horizon_years: number;
  };
  executive_summary: {
    average_rental_yield: string;
    best_performing_locality: string;
    projected_5_year_cagr: string;
    investment_grade: string;
    market_sentiment: string;
    total_properties_analyzed: number;
  };
  rental_yield_analysis: {
    city_overview: {
      current_avg_yield: string;
      yield_range: { min: string; max: string };
      market_maturity: string;
      rental_demand_strength: string;
    };
    locality_breakdown: LocalityYield[];
  };
  investment_recommendations: {
    top_recommendations: LocalityYield[];
  };
  roi_projections: {
    summary: {
      best_roi_locality: string;
      average_roi: number;
      average_annualized_return: number;
    };
  };
}

interface DynamicRentalYieldProps {
  city?: string;
  propertyType?: string;
}

export function DynamicRentalYield({ city = 'Bangalore', propertyType = 'Apartment' }: DynamicRentalYieldProps) {
  const [data, setData] = useState<RentalYieldData | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [selectedCity, setSelectedCity] = useState(city);
  const [selectedPropertyType, setSelectedPropertyType] = useState(propertyType);
  const [searchTerm, setSearchTerm] = useState('');
  const [sortBy, setSortBy] = useState<'yield' | 'price' | 'cagr'>('yield');
  const [sortOrder, setSortOrder] = useState<'asc' | 'desc'>('desc');
  const [showCalculator, setShowCalculator] = useState(true);
  const [calculationResult, setCalculationResult] = useState<RentalYieldResult | null>(null);
  const [calculatorLoading, setCalculatorLoading] = useState(false);

  const [propertyInput, setPropertyInput] = useState<PropertyInput>({
    city: 'Bangalore',
    locality: '',
    property_type: 'apartment',
    bhk_config: '2BHK',
    built_area: 1200,
    purchase_price: 5000000,
    furnishing: 'semi_furnished',
    age_years: 2
  });

  const cities = ['Bangalore', 'Mumbai', 'Delhi', 'Chennai', 'Hyderabad', 'Pune', 'Kolkata', 'Ahmedabad'];
  const propertyTypes = ['apartment', 'independent_house', 'villa'];
  const bhkConfigs = ['1BHK', '2BHK', '3BHK', '4BHK', '5BHK'];
  const furnishingOptions = ['unfurnished', 'semi_furnished', 'fully_furnished'];

  useEffect(() => {
    if (!showCalculator) {
      fetchRentalYieldData();
    }
  }, [selectedCity, selectedPropertyType, showCalculator]);

  const calculateRentalYield = async () => {
    try {
      setCalculatorLoading(true);
      setError(null);
      
      const response = await api.post('/rental-yield/calculate', {
        location: {
          city: propertyInput.city,
          locality: propertyInput.locality
        },
        property: {
          type: propertyInput.property_type,
          bhk_config: propertyInput.bhk_config,
          built_area: propertyInput.built_area,
          purchase_price: propertyInput.purchase_price,
          furnishing: propertyInput.furnishing,
          age_years: propertyInput.age_years
        }
      });
      
      if (response.data.success) {
        setCalculationResult(response.data.data);
      } else {
        throw new Error('Calculation failed');
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Calculation failed');
    } finally {
      setCalculatorLoading(false);
    }
  };

  const fetchRentalYieldData = async () => {
    try {
      setLoading(true);
      setError(null);
      
      const result = await rentalYieldApi.getAnalysis({
        city: selectedCity,
        property_type: selectedPropertyType,
        investment_horizon: 5
      });
      
      if (result.success && result.data) {
        setData(result.data);
      } else {
        throw new Error('Invalid response format');
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch data');
      console.error('Error fetching rental yield data:', err);
    } finally {
      setLoading(false);
    }
  };

  const downloadReport = () => {
    if (!data) return;

    const reportData = {
      ...data,
      downloaded_at: new Date().toISOString()
    };

    const blob = new Blob([JSON.stringify(reportData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `rental-yield-${selectedCity}-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const filteredAndSortedLocalities = () => {
    if (!data) return [];
    
    let localities = data.rental_yield_analysis.locality_breakdown.filter(locality =>
      locality.locality.toLowerCase().includes(searchTerm.toLowerCase())
    );

    localities.sort((a, b) => {
      let aValue, bValue;
      switch (sortBy) {
        case 'yield':
          aValue = a.current_yield;
          bValue = b.current_yield;
          break;
        case 'price':
          aValue = a.avg_property_price;
          bValue = b.avg_property_price;
          break;
        case 'cagr':
          aValue = a.projected_cagr;
          bValue = b.projected_cagr;
          break;
        default:
          return 0;
      }
      
      return sortOrder === 'desc' ? bValue - aValue : aValue - bValue;
    });

    return localities;
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-indigo-50">
        <div className="flex items-center justify-center p-12">
          <div className="text-center">
            <Loader2 className="w-12 h-12 animate-spin text-[#0056D2] mx-auto mb-4" />
            <h3 className="text-xl font-semibold text-gray-900 mb-2">Analyzing Rental Yields</h3>
            <p className="text-gray-600">Fetching real-time market data for {selectedCity}...</p>
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-indigo-50 p-8">
        <div className="max-w-2xl mx-auto">
          <div className="bg-white border border-red-200 rounded-2xl p-8 shadow-lg">
            <div className="flex items-center gap-4 mb-6">
              <div className="bg-red-100 p-3 rounded-full">
                <AlertCircle className="w-8 h-8 text-red-600" />
              </div>
              <div>
                <h3 className="text-2xl font-bold text-red-900">Unable to Load Data</h3>
                <p className="text-red-700 mt-1">We encountered an issue while fetching the rental yield analysis</p>
              </div>
            </div>
            <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
              <p className="text-red-800 font-medium">{error}</p>
            </div>
            <div className="flex gap-3">
              <button 
                onClick={fetchRentalYieldData}
                className="flex items-center gap-2 px-6 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors font-medium"
              >
                <RefreshCw className="w-4 h-4" />
                Retry Analysis
              </button>
              <button 
                onClick={() => window.location.reload()}
                className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
              >
                Refresh Page
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (!data) return null;

  const { executive_summary, rental_yield_analysis, investment_recommendations, roi_projections } = data;

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-indigo-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="space-y-8">
          {/* Enhanced Header with Controls */}
          <div className="bg-white rounded-2xl shadow-xl p-8 border border-gray-200">
            <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-6">
              <div className="flex-1">
                <div className="flex items-center gap-3 mb-2">
                  <div className="bg-gradient-to-r from-[#0056D2] to-blue-600 p-2 rounded-lg">
                    <BarChart3 className="w-6 h-6 text-white" />
                  </div>
                  <h1 className="text-4xl font-bold bg-gradient-to-r from-gray-900 to-gray-700 bg-clip-text text-transparent">
                    Live Rental Yield Analysis
                  </h1>
                </div>
                <p className="text-gray-600 text-lg">
                  Real-time market insights • {data.report_metadata.total_properties_analyzed} properties analyzed
                </p>
              </div>
              
              <div className="flex flex-col sm:flex-row gap-4">
                <button
                  onClick={downloadReport}
                  className="flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-[#0056D2] to-blue-600 text-white rounded-xl hover:from-blue-700 hover:to-blue-700 transition-all duration-200 shadow-lg hover:shadow-xl font-medium"
                >
                  <Download className="w-5 h-5" />
                  Download Report
                </button>
                <button
                  onClick={fetchRentalYieldData}
                  className="flex items-center gap-2 px-6 py-3 border border-gray-300 text-gray-700 rounded-xl hover:bg-gray-50 transition-colors font-medium"
                >
                  <RefreshCw className="w-5 h-5" />
                  Refresh Data
                </button>
              </div>
            </div>
            
            {/* Filter Controls */}
            <div className="mt-8 pt-6 border-t border-gray-200">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">City</label>
                  <select
                    value={selectedCity}
                    onChange={(e) => setSelectedCity(e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent transition-colors"
                  >
                    {cities.map(city => (
                      <option key={city} value={city}>{city}</option>
                    ))}
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
                  <select
                    value={selectedPropertyType}
                    onChange={(e) => setSelectedPropertyType(e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent transition-colors"
                  >
                    {propertyTypes.map(type => (
                      <option key={type} value={type}>{type}</option>
                    ))}
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Search Localities</label>
                  <div className="relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                    <input
                      type="text"
                      placeholder="Search by locality name..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent transition-colors"
                    />
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Enhanced Executive Summary Cards */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <div className="group bg-gradient-to-br from-emerald-50 to-green-100 p-6 rounded-2xl border border-green-200 hover:shadow-lg transition-all duration-300 hover:-translate-y-1">
              <div className="flex items-center justify-between mb-4">
                <div className="bg-green-500 p-3 rounded-xl shadow-lg group-hover:scale-110 transition-transform duration-300">
                  <DollarSign className="w-6 h-6 text-white" />
                </div>
                <div className="flex items-center gap-1">
                  <Star className="w-4 h-4 text-green-600 fill-current" />
                  <span className="text-xs font-bold text-green-700 bg-green-200 px-2 py-1 rounded-full">AVG YIELD</span>
                </div>
              </div>
              <p className="text-3xl font-bold text-green-900 mb-1">{executive_summary.average_rental_yield}</p>
              <p className="text-sm text-green-700 font-medium">Current Market Average</p>
              <div className="mt-3 flex items-center gap-1 text-green-600">
                <ArrowUp className="w-4 h-4" />
                <span className="text-xs font-medium">Market Leading</span>
              </div>
            </div>

            <div className="group bg-gradient-to-br from-blue-50 to-indigo-100 p-6 rounded-2xl border border-blue-200 hover:shadow-lg transition-all duration-300 hover:-translate-y-1">
              <div className="flex items-center justify-between mb-4">
                <div className="bg-blue-500 p-3 rounded-xl shadow-lg group-hover:scale-110 transition-transform duration-300">
                  <TrendingUp className="w-6 h-6 text-white" />
                </div>
                <div className="flex items-center gap-1">
                  <Star className="w-4 h-4 text-blue-600 fill-current" />
                  <span className="text-xs font-bold text-blue-700 bg-blue-200 px-2 py-1 rounded-full">GROWTH</span>
                </div>
              </div>
              <p className="text-3xl font-bold text-blue-900 mb-1">{executive_summary.projected_5_year_cagr}</p>
              <p className="text-sm text-blue-700 font-medium">5-Year CAGR Projection</p>
              <div className="mt-3 flex items-center gap-1 text-blue-600">
                <TrendingUp className="w-4 h-4" />
                <span className="text-xs font-medium">Strong Growth</span>
              </div>
            </div>

            <div className="group bg-gradient-to-br from-purple-50 to-violet-100 p-6 rounded-2xl border border-purple-200 hover:shadow-lg transition-all duration-300 hover:-translate-y-1">
              <div className="flex items-center justify-between mb-4">
                <div className="bg-purple-500 p-3 rounded-xl shadow-lg group-hover:scale-110 transition-transform duration-300">
                  <Target className="w-6 h-6 text-white" />
                </div>
                <div className="flex items-center gap-1">
                  <Star className="w-4 h-4 text-purple-600 fill-current" />
                  <span className="text-xs font-bold text-purple-700 bg-purple-200 px-2 py-1 rounded-full">GRADE</span>
                </div>
              </div>
              <p className="text-3xl font-bold text-purple-900 mb-1">{executive_summary.investment_grade}</p>
              <p className="text-sm text-purple-700 font-medium">Investment Grade</p>
              <div className="mt-3 flex items-center gap-1 text-purple-600">
                <Shield className="w-4 h-4" />
                <span className="text-xs font-medium">Quality Assured</span>
              </div>
            </div>

            <div className="group bg-gradient-to-br from-orange-50 to-amber-100 p-6 rounded-2xl border border-orange-200 hover:shadow-lg transition-all duration-300 hover:-translate-y-1">
              <div className="flex items-center justify-between mb-4">
                <div className="bg-orange-500 p-3 rounded-xl shadow-lg group-hover:scale-110 transition-transform duration-300">
                  <BarChart3 className="w-6 h-6 text-white" />
                </div>
                <div className="flex items-center gap-1">
                  <Star className="w-4 h-4 text-orange-600 fill-current" />
                  <span className="text-xs font-bold text-orange-700 bg-orange-200 px-2 py-1 rounded-full">SENTIMENT</span>
                </div>
              </div>
              <p className="text-2xl font-bold text-orange-900 mb-1">{executive_summary.market_sentiment}</p>
              <p className="text-sm text-orange-700 font-medium">Market Outlook</p>
              <div className="mt-3 flex items-center gap-1 text-orange-600">
                <TrendingUp className="w-4 h-4" />
                <span className="text-xs font-medium">Positive Trend</span>
              </div>
            </div>
          </div>

          {/* Enhanced City Overview */}
          <div className="bg-white rounded-2xl shadow-xl p-8 border border-gray-200">
            <div className="flex items-center gap-3 mb-6">
              <div className="bg-gradient-to-r from-[#0056D2] to-blue-600 p-3 rounded-xl">
                <MapPin className="w-6 h-6 text-white" />
              </div>
              <div>
                <h3 className="text-2xl font-bold text-gray-900">{selectedCity} Market Overview</h3>
                <p className="text-gray-600">Comprehensive market analysis for {selectedPropertyType.toLowerCase()}s</p>
              </div>
            </div>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
              <div className="text-center p-4 bg-gray-50 rounded-xl">
                <p className="text-sm text-gray-600 mb-2">Average Yield</p>
                <p className="text-3xl font-bold text-gray-900">{rental_yield_analysis.city_overview.current_avg_yield}</p>
                <div className="mt-2 flex items-center justify-center gap-1 text-green-600">
                  <ArrowUp className="w-4 h-4" />
                  <span className="text-xs font-medium">Above National Avg</span>
                </div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-xl">
                <p className="text-sm text-gray-600 mb-2">Yield Range</p>
                <p className="text-xl font-semibold text-gray-900">
                  {rental_yield_analysis.city_overview.yield_range.min} - {rental_yield_analysis.city_overview.yield_range.max}
                </p>
                <div className="mt-2">
                  <span className="text-xs font-medium text-blue-600 bg-blue-100 px-2 py-1 rounded-full">Wide Range</span>
                </div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-xl">
                <p className="text-sm text-gray-600 mb-2">Market Maturity</p>
                <p className="text-xl font-semibold text-gray-900">{rental_yield_analysis.city_overview.market_maturity}</p>
                <div className="mt-2">
                  <span className="text-xs font-medium text-purple-600 bg-purple-100 px-2 py-1 rounded-full">Stable</span>
                </div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-xl">
                <p className="text-sm text-gray-600 mb-2">Rental Demand</p>
                <p className="text-xl font-semibold text-gray-900">{rental_yield_analysis.city_overview.rental_demand_strength}</p>
                <div className="mt-2">
                  <span className="text-xs font-medium text-orange-600 bg-orange-100 px-2 py-1 rounded-full">Strong</span>
                </div>
              </div>
            </div>
          </div>

          {/* Enhanced Top Recommendations */}
          <div className="bg-white rounded-2xl shadow-xl p-8 border border-gray-200">
            <div className="flex items-center justify-between mb-6">
              <div className="flex items-center gap-3">
                <div className="bg-gradient-to-r from-green-500 to-emerald-600 p-3 rounded-xl">
                  <Target className="w-6 h-6 text-white" />
                </div>
                <div>
                  <h3 className="text-2xl font-bold text-gray-900">Top Investment Recommendations</h3>
                  <p className="text-gray-600">Best performing localities for rental yield</p>
                </div>
              </div>
              <div className="text-right">
                <p className="text-sm text-gray-600">Updated</p>
                <p className="text-sm font-medium text-gray-900">Just now</p>
              </div>
            </div>
            <div className="space-y-4">
              {investment_recommendations.top_recommendations.slice(0, 5).map((locality, index) => (
                <div key={index} className="group flex items-center justify-between p-6 bg-gradient-to-r from-gray-50 to-white rounded-xl border border-gray-200 hover:shadow-lg hover:border-[#0056D2] transition-all duration-300">
                  <div className="flex items-center gap-4 flex-1">
                    <div className="relative">
                      <div className="bg-gradient-to-r from-[#0056D2] to-blue-600 text-white rounded-full w-12 h-12 flex items-center justify-center font-bold text-lg shadow-lg">
                        {index + 1}
                      </div>
                      {index === 0 && (
                        <div className="absolute -top-1 -right-1 bg-yellow-400 rounded-full p-1">
                          <Star className="w-3 h-3 text-yellow-800 fill-current" />
                        </div>
                      )}
                    </div>
                    <div className="flex-1">
                      <h4 className="font-bold text-lg text-gray-900 group-hover:text-[#0056D2] transition-colors">{locality.locality}</h4>
                      <div className="flex items-center gap-4 mt-1">
                        <span className="text-sm text-gray-600 flex items-center gap-1">
                          <Home className="w-4 h-4" />
                          ₹{(locality.avg_property_price / 100000).toFixed(1)}L avg price
                        </span>
                        <span className="text-sm text-gray-600">
                          {locality.property_count} properties
                        </span>
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center gap-8">
                    <div className="text-center">
                      <p className="text-sm text-gray-600 mb-1">Yield</p>
                      <div className="flex items-center gap-1">
                        <p className="text-2xl font-bold text-green-600">{locality.current_yield.toFixed(2)}%</p>
                        <ArrowUp className="w-4 h-4 text-green-600" />
                      </div>
                    </div>
                    <div className="text-center">
                      <p className="text-sm text-gray-600 mb-1">Grade</p>
                      <span className={`px-4 py-2 rounded-full text-sm font-bold shadow-sm ${
                        locality.investment_grade.startsWith('A') ? 'bg-green-100 text-green-800 border border-green-200' :
                        locality.investment_grade.startsWith('B') ? 'bg-blue-100 text-blue-800 border border-blue-200' :
                        'bg-yellow-100 text-yellow-800 border border-yellow-200'
                      }`}>
                        {locality.investment_grade}
                      </span>
                    </div>
                    <div className="text-center">
                      <p className="text-sm text-gray-600 mb-1">Risk</p>
                      <div className="flex items-center gap-1">
                        <div className={`w-3 h-3 rounded-full ${
                          locality.risk_score <= 40 ? 'bg-green-500' :
                          locality.risk_score <= 60 ? 'bg-yellow-500' :
                          'bg-red-500'
                        }`}></div>
                        <p className={`text-sm font-semibold ${
                          locality.risk_score <= 40 ? 'text-green-600' :
                          locality.risk_score <= 60 ? 'text-yellow-600' :
                          'text-red-600'
                        }`}>
                          {locality.risk_score <= 40 ? 'Low' : locality.risk_score <= 60 ? 'Medium' : 'High'}
                        </p>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Enhanced Detailed Locality Analysis */}
          <div className="bg-white rounded-2xl shadow-xl p-8 border border-gray-200">
            <div className="flex items-center justify-between mb-6">
              <div className="flex items-center gap-3">
                <div className="bg-gradient-to-r from-indigo-500 to-purple-600 p-3 rounded-xl">
                  <BarChart3 className="w-6 h-6 text-white" />
                </div>
                <div>
                  <h3 className="text-2xl font-bold text-gray-900">Detailed Locality Analysis</h3>
                  <p className="text-gray-600">Comprehensive breakdown of all localities</p>
                </div>
              </div>
              <div className="flex items-center gap-3">
                <select
                  value={sortBy}
                  onChange={(e) => setSortBy(e.target.value as 'yield' | 'price' | 'cagr')}
                  className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                >
                  <option value="yield">Sort by Yield</option>
                  <option value="price">Sort by Price</option>
                  <option value="cagr">Sort by CAGR</option>
                </select>
                <button
                  onClick={() => setSortOrder(sortOrder === 'desc' ? 'asc' : 'desc')}
                  className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                >
                  {sortOrder === 'desc' ? <ArrowDown className="w-4 h-4" /> : <ArrowUp className="w-4 h-4" />}
                </button>
              </div>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b-2 border-gray-200 bg-gray-50">
                    <th className="text-left py-4 px-4 text-sm font-bold text-gray-700 rounded-tl-lg">Locality</th>
                    <th className="text-right py-4 px-4 text-sm font-bold text-gray-700">Avg Price</th>
                    <th className="text-right py-4 px-4 text-sm font-bold text-gray-700">Monthly Rent</th>
                    <th className="text-right py-4 px-4 text-sm font-bold text-gray-700">Yield</th>
                    <th className="text-right py-4 px-4 text-sm font-bold text-gray-700">CAGR</th>
                    <th className="text-center py-4 px-4 text-sm font-bold text-gray-700">Tier</th>
                    <th className="text-center py-4 px-4 text-sm font-bold text-gray-700 rounded-tr-lg">Grade</th>
                  </tr>
                </thead>
                <tbody>
                  {filteredAndSortedLocalities().slice(0, 15).map((locality, index) => (
                    <tr key={index} className="border-b border-gray-100 hover:bg-blue-50 transition-colors group">
                      <td className="py-4 px-4">
                        <div className="flex items-center gap-3">
                          <div className="bg-blue-100 p-2 rounded-lg group-hover:bg-blue-200 transition-colors">
                            <Home className="w-4 h-4 text-blue-600" />
                          </div>
                          <div>
                            <span className="font-semibold text-gray-900 group-hover:text-[#0056D2] transition-colors">{locality.locality}</span>
                            <p className="text-xs text-gray-500 mt-1">{locality.property_count} properties</p>
                          </div>
                        </div>
                      </td>
                      <td className="text-right py-4 px-4">
                        <span className="font-semibold text-gray-900">₹{(locality.avg_property_price / 100000).toFixed(1)}L</span>
                        <p className="text-xs text-gray-500 mt-1">₹{locality.price_per_sqft}/sqft</p>
                      </td>
                      <td className="text-right py-4 px-4">
                        <span className="font-semibold text-gray-700">₹{(locality.avg_monthly_rent / 1000).toFixed(0)}K</span>
                      </td>
                      <td className="text-right py-4 px-4">
                        <div className="flex items-center justify-end gap-1">
                          <span className="font-bold text-green-600 text-lg">{locality.current_yield.toFixed(2)}%</span>
                          <ArrowUp className="w-3 h-3 text-green-600" />
                        </div>
                      </td>
                      <td className="text-right py-4 px-4">
                        <span className="font-bold text-blue-600">{locality.projected_cagr.toFixed(1)}%</span>
                      </td>
                      <td className="text-center py-4 px-4">
                        <span className={`px-3 py-1 rounded-full text-xs font-bold ${
                          locality.market_tier === 'Premium' ? 'bg-purple-100 text-purple-800 border border-purple-200' :
                          locality.market_tier === 'Mid-Premium' ? 'bg-blue-100 text-blue-800 border border-blue-200' :
                          locality.market_tier === 'Mid-Market' ? 'bg-green-100 text-green-800 border border-green-200' :
                          'bg-gray-100 text-gray-800 border border-gray-200'
                        }`}>
                          {locality.market_tier}
                        </span>
                      </td>
                      <td className="text-center py-4 px-4">
                        <span className={`px-3 py-1 rounded-full text-xs font-bold shadow-sm ${
                          locality.investment_grade.startsWith('A') ? 'bg-green-100 text-green-800 border border-green-200' :
                          locality.investment_grade.startsWith('B') ? 'bg-blue-100 text-blue-800 border border-blue-200' :
                          'bg-yellow-100 text-yellow-800 border border-yellow-200'
                        }`}>
                          {locality.investment_grade}
                        </span>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            {filteredAndSortedLocalities().length > 15 && (
              <div className="mt-6 text-center">
                <p className="text-gray-600 mb-4">Showing top 15 of {filteredAndSortedLocalities().length} localities</p>
                <button className="px-6 py-3 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors font-medium">
                  View All Localities
                </button>
              </div>
            )}
          </div>

          {/* Enhanced ROI Summary */}
          <div className="bg-gradient-to-r from-[#0056D2] via-blue-600 to-indigo-700 text-white p-8 rounded-2xl shadow-2xl relative overflow-hidden">
            <div className="absolute inset-0 bg-gradient-to-r from-blue-600/20 to-transparent"></div>
            <div className="relative z-10">
              <div className="flex items-center gap-3 mb-8">
                <div className="bg-white/20 backdrop-blur-sm p-3 rounded-xl">
                  <TrendingUp className="w-8 h-8 text-white" />
                </div>
                <div>
                  <h3 className="text-3xl font-bold mb-2">5-Year ROI Projections</h3>
                  <p className="text-blue-100">Comprehensive return analysis for {selectedCity}</p>
                </div>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="bg-white/10 backdrop-blur-sm p-6 rounded-xl border border-white/20 hover:bg-white/15 transition-colors">
                  <div className="flex items-center gap-2 mb-3">
                    <Target className="w-5 h-5 text-blue-200" />
                    <p className="text-blue-100 text-sm font-medium">Best ROI Locality</p>
                  </div>
                  <p className="text-3xl font-bold mb-2">{roi_projections.summary.best_roi_locality}</p>
                  <div className="flex items-center gap-1 text-blue-200">
                    <Star className="w-4 h-4 fill-current" />
                    <span className="text-sm">Top Performer</span>
                  </div>
                </div>
                <div className="bg-white/10 backdrop-blur-sm p-6 rounded-xl border border-white/20 hover:bg-white/15 transition-colors">
                  <div className="flex items-center gap-2 mb-3">
                    <BarChart3 className="w-5 h-5 text-blue-200" />
                    <p className="text-blue-100 text-sm font-medium">Average ROI</p>
                  </div>
                  <p className="text-3xl font-bold mb-2">{roi_projections.summary.average_roi.toFixed(1)}%</p>
                  <div className="flex items-center gap-1 text-blue-200">
                    <ArrowUp className="w-4 h-4" />
                    <span className="text-sm">Market Average</span>
                  </div>
                </div>
                <div className="bg-white/10 backdrop-blur-sm p-6 rounded-xl border border-white/20 hover:bg-white/15 transition-colors">
                  <div className="flex items-center gap-2 mb-3">
                    <DollarSign className="w-5 h-5 text-blue-200" />
                    <p className="text-blue-100 text-sm font-medium">Annualized Return</p>
                  </div>
                  <p className="text-3xl font-bold mb-2">{roi_projections.summary.average_annualized_return.toFixed(1)}%</p>
                  <div className="flex items-center gap-1 text-blue-200">
                    <TrendingUp className="w-4 h-4" />
                    <span className="text-sm">Compound Growth</span>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Enhanced Best Performing Locality Highlight */}
          <div className="bg-gradient-to-br from-emerald-50 via-green-50 to-teal-50 border-2 border-green-200 rounded-2xl p-8 shadow-lg relative overflow-hidden">
            <div className="absolute top-0 right-0 w-32 h-32 bg-green-200/30 rounded-full -translate-y-16 translate-x-16"></div>
            <div className="absolute bottom-0 left-0 w-24 h-24 bg-emerald-200/30 rounded-full translate-y-12 -translate-x-12"></div>
            <div className="relative z-10">
              <div className="flex items-center gap-4 mb-6">
                <div className="bg-gradient-to-r from-green-500 to-emerald-600 text-white rounded-2xl p-4 shadow-lg">
                  <Target className="w-8 h-8" />
                </div>
                <div>
                  <h3 className="text-2xl font-bold text-green-900 mb-1">Best Performing Locality</h3>
                  <p className="text-green-700 font-medium">Highest rental yield opportunity in {selectedCity}</p>
                </div>
                <div className="ml-auto">
                  <div className="bg-yellow-400 text-yellow-900 px-3 py-1 rounded-full text-sm font-bold flex items-center gap-1">
                    <Star className="w-4 h-4 fill-current" />
                    TOP PICK
                  </div>
                </div>
              </div>
              <div className="bg-white rounded-xl p-6 shadow-sm border border-green-100">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-4xl font-bold text-green-900 mb-2">{executive_summary.best_performing_locality}</p>
                    <p className="text-gray-600 text-lg">This locality offers the best rental yield opportunities based on current market data and future projections.</p>
                  </div>
                  <div className="text-right">
                    <div className="bg-green-100 rounded-lg p-4">
                      <p className="text-sm text-green-700 mb-1">Expected Yield</p>
                      <p className="text-2xl font-bold text-green-900">8.5%+</p>
                    </div>
                  </div>
                </div>
                <div className="mt-6 flex gap-4">
                  <button className="flex-1 bg-green-600 text-white py-3 px-6 rounded-lg hover:bg-green-700 transition-colors font-medium">
                    View Properties
                  </button>
                  <button className="flex-1 border border-green-600 text-green-600 py-3 px-6 rounded-lg hover:bg-green-50 transition-colors font-medium">
                    Get Detailed Report
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
