import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { TrendingUp, Brain, Target, Download, Share } from 'lucide-react';
import { MarketInsights } from '@/api/forecast';

interface InsightsCardProps {
  insights: MarketInsights;
}

export const InsightsCard: React.FC<InsightsCardProps> = ({ insights }) => {
  const getSentimentColor = (color: string) => {
    switch (color) {
      case 'green': return 'bg-green-100 text-green-800 border-green-200';
      case 'yellow': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'red': return 'bg-red-100 text-red-800 border-red-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getConfidenceLevel = (score: number) => {
    if (score >= 80) return { level: 'High', color: 'text-green-600' };
    if (score >= 60) return { level: 'Medium', color: 'text-yellow-600' };
    return { level: 'Low', color: 'text-red-600' };
  };

  const confidenceLevel = getConfidenceLevel(insights.confidence_score);

  return (
    <div className="space-y-4">
      {/* AI Insights Card */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Brain className="w-5 h-5 text-blue-600" />
            AI Market Insights
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          {/* Market Sentiment */}
          <div>
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium text-gray-600">Market Sentiment</span>
              <Badge className={getSentimentColor(insights.sentiment_color)}>
                {insights.sentiment}
              </Badge>
            </div>
          </div>

          {/* Confidence Score */}
          <div>
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium text-gray-600">Prediction Confidence</span>
              <span className={`text-sm font-medium ${confidenceLevel.color}`}>
                {confidenceLevel.level}
              </span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div
                className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                style={{ width: `${insights.confidence_score}%` }}
              ></div>
            </div>
            <p className="text-xs text-gray-500 mt-1">{insights.confidence_score}% accuracy</p>
          </div>

          {/* Recommendation */}
          <div className="bg-gray-50 p-3 rounded-lg">
            <div className="flex items-start gap-2">
              <Target className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
              <div>
                <p className="text-sm font-medium text-gray-900 mb-1">Investment Recommendation</p>
                <p className="text-sm text-gray-600">{insights.recommendation}</p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Key Factors Card */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="w-5 h-5 text-green-600" />
            Market Factors
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            <div className="flex items-center justify-between">
              <span className="text-sm text-gray-600">Infrastructure Development</span>
              <Badge variant="outline" className="text-green-600 border-green-200">Positive</Badge>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm text-gray-600">Demand Trend</span>
              <Badge variant="outline" className="text-blue-600 border-blue-200">Stable</Badge>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm text-gray-600">Price Volatility</span>
              <Badge variant="outline" className="text-yellow-600 border-yellow-200">Moderate</Badge>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm text-gray-600">Investment Risk</span>
              <Badge variant="outline" className="text-green-600 border-green-200">Low</Badge>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Actions Card */}
      <Card>
        <CardHeader>
          <CardTitle>Actions</CardTitle>
        </CardHeader>
        <CardContent className="space-y-3">
          <Button variant="outline" className="w-full justify-start" size="sm">
            <Download className="w-4 h-4 mr-2" />
            Download Report
          </Button>
          <Button variant="outline" className="w-full justify-start" size="sm">
            <Share className="w-4 h-4 mr-2" />
            Share Analysis
          </Button>
        </CardContent>
      </Card>
    </div>
  );
};