import React, { useState, useEffect } from "react";
import {
  BarChart3,
  DollarSign,
  TrendingUp,
  MapPin,
  Target,
  Activity,
  PieChart,
  Users,
  Building,
  ArrowUpRight,
  ArrowDownRight,
  Eye,
  Calendar,
  Star,
  Shield,
  Zap,
  Award,
  Download,
  FileText,
  TrendingDown,
  AlertTriangle,
  CheckCircle,
} from "lucide-react";
import {
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  BarChart,
  Bar,
  PieChart as RechartsPieChart,
  Pie,
  Cell,
  AreaChart,
  Area,
} from "recharts";
import { Card, CardContent, CardHeader, CardTitle } from "../ui/card";
import { Badge } from "../ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../ui/tabs";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../ui/select";
import { PropertyHeatmap } from "../heatmap/PropertyHeatmap";
import { getProperties } from "../../api/properties";
import { Button } from "../ui/button";
import { toast } from "sonner";
import jsPDF from "jspdf";

export function InvestmentInsights() {
  const [selectedCity, setSelectedCity] = useState("Bangalore");
  const [activeTab, setActiveTab] = useState("report");
  const [marketData, setMarketData] = useState([]);
  const [localityData, setLocalityData] = useState([]);
  const [propertyData, setPropertyData] = useState([
    {
      id: 1,
      title: "Modern Apartment in Koramangala",
      location: "Koramangala",
      roi: 21.5,
      potential: "High",
    },
    {
      id: 2,
      title: "Luxury Villa in Whitefield",
      location: "Whitefield",
      roi: 18.7,
      potential: "High",
    },
    {
      id: 3,
      title: "Premium Flat in Indiranagar",
      location: "Indiranagar",
      roi: 16.2,
      potential: "Medium",
    },
  ]);
  const [allProperties, setAllProperties] = useState([]);
  const [pieData, setPieData] = useState([]);
  const [heatmapProperties, setHeatmapProperties] = useState([]);
  const [marketMetrics, setMarketMetrics] = useState({
    growth: 0,
    yield: 0,
    safetyScore: 0,
    avgPrice: 0,
    momentum: "Stable",
  });
  const [loading, setLoading] = useState(true);
  const [reportData, setReportData] = useState({
    executiveSummary: "",
    marketOverview: "",
    investmentRecommendations: [],
    riskAnalysis: "",
    futureOutlook: "",
    generatedDate: new Date().toISOString().split("T")[0],
  });

  useEffect(() => {
    const fetchRealData = async () => {
      setLoading(true);
      try {
        // Use the existing properties API function
        const data = await getProperties();
        let properties = data.properties || [];

        console.log("Fetched properties:", properties.length);
        if (properties.length > 0)
          console.log("Sample property:", properties[0]);
        console.log("Selected city:", selectedCity);

        // Filter by city if selected (more flexible matching)
        const cityProperties =
          selectedCity === "All"
            ? properties
            : properties.filter((p) => {
                const propertyCity = (p.city || "").toLowerCase();
                const selectedCityLower = selectedCity.toLowerCase();
                
                // Special case for Bangalore - include both Bangalore and Bengaluru
                if (selectedCity === "Bangalore") {
                  return (
                    propertyCity.includes("bangalore") ||
                    propertyCity.includes("bengaluru") ||
                    propertyCity === "bangalore" ||
                    propertyCity === "bengaluru"
                  );
                }
                
                return (
                  propertyCity.includes(selectedCityLower) ||
                  propertyCity === selectedCityLower ||
                  selectedCityLower.includes(propertyCity)
                );
              });

        console.log("City filtered properties:", cityProperties.length);
        setAllProperties(cityProperties);

        // Always show some properties even if city filter doesn't match
        const propertiesForCalculation =
          cityProperties.length > 0 ? cityProperties : properties.slice(0, 10);

        // Calculate real metrics
        calculateMarketMetrics(propertiesForCalculation);
        calculateLocalityData(propertiesForCalculation);
        calculatePropertyROI(propertiesForCalculation);
        calculateMarketTrends(propertiesForCalculation);
        calculatePropertyMix(propertiesForCalculation);

        // Set heatmap properties (properties with coordinates)
        const heatmapProps = propertiesForCalculation
          .filter((p) => p.latitude && p.longitude)
          .map((p) => ({
            id: p.id,
            latitude: parseFloat(p.latitude),
            longitude: parseFloat(p.longitude),
            sale_price: p.sale_price || 0,
            monthly_rent: p.monthly_rent || 0,
            property_type: p.property_type,
            listing_type: p.listing_type,
            city: p.city,
          }));
        setHeatmapProperties(heatmapProps);

        // Generate comprehensive report data after all calculations
        setTimeout(() => generateReportData(propertiesForCalculation), 100);
      } catch (error) {
        console.error("Failed to fetch real property data:", error);
        // Fallback to sample data if API fails
        const fallbackProperties = [
          {
            id: 1,
            title: "Modern Apartment in Koramangala",
            city: "Bangalore",
            locality: "Koramangala",
            property_type: "apartment",
            listing_type: "sell",
            sale_price: 15000000,
            monthly_rent: 50000,
            built_up_area: 1500,
          },
          {
            id: 2,
            title: "Luxury Villa in Whitefield",
            city: "Bangalore",
            locality: "Whitefield",
            property_type: "villa",
            listing_type: "sell",
            sale_price: 25000000,
            monthly_rent: 75000,
            built_up_area: 2500,
          },
          {
            id: 3,
            title: "Premium Flat in Indiranagar",
            city: "Bangalore",
            locality: "Indiranagar",
            property_type: "apartment",
            listing_type: "sell",
            sale_price: 12000000,
            monthly_rent: 40000,
            built_up_area: 1200,
          },
        ];

        setAllProperties(fallbackProperties);
        calculateMarketMetrics(fallbackProperties);
        calculateLocalityData(fallbackProperties);
        calculatePropertyROI(fallbackProperties);
        calculateMarketTrends(fallbackProperties);
        calculatePropertyMix(fallbackProperties);
        // Generate report data after all calculations
        setTimeout(() => generateReportData(fallbackProperties), 100);
      } finally {
        setLoading(false);
      }
    };

    fetchRealData();
  }, [selectedCity]);

  const calculateMarketMetrics = (properties) => {
    if (!properties.length) {
      setMarketMetrics({
        growth: "12.5",
        yield: "4.2",
        safetyScore: 85,
        avgPrice: 6200,
        momentum: "Rising",
      });
      return;
    }

    const prices = properties
      .filter((p) => p.sale_price > 0)
      .map((p) => p.sale_price);
    const areas = properties
      .filter((p) => p.built_up_area > 0)
      .map((p) => p.built_up_area);
    const rents = properties
      .filter((p) => p.monthly_rent > 0)
      .map((p) => p.monthly_rent);

    const avgPrice = prices.length
      ? prices.reduce((a, b) => a + b, 0) / prices.length
      : 15000000;
    const avgArea = areas.length
      ? areas.reduce((a, b) => a + b, 0) / areas.length
      : 1500;
    const avgRent = rents.length
      ? rents.reduce((a, b) => a + b, 0) / rents.length
      : 50000;

    const pricePerSqft = avgPrice / avgArea;
    const rentalYield =
      avgRent > 0 && avgPrice > 0 ? ((avgRent * 12) / avgPrice) * 100 : 4.2;
    const growth = 8 + properties.length * 0.5 + Math.random() * 5; // Growth based on market activity
    const safetyScore = Math.min(95, 70 + properties.length * 2); // Score based on market activity

    setMarketMetrics({
      growth: growth.toFixed(1),
      yield: rentalYield.toFixed(1),
      safetyScore: Math.round(safetyScore),
      avgPrice: Math.round(pricePerSqft),
      momentum: growth > 12 ? "Rising" : growth > 8 ? "Stable" : "Declining",
    });
  };

  const calculateLocalityData = (properties) => {
    if (!properties.length) {
      setLocalityData([]);
      return;
    }

    const localityMap = new Map();

    properties.forEach((property) => {
      const locality = property.locality || property.city || "Unknown";
      if (!localityMap.has(locality)) {
        localityMap.set(locality, {
          name: locality,
          properties: [],
          totalPrice: 0,
          totalRent: 0,
        });
      }

      const data = localityMap.get(locality);
      data.properties.push(property);
      if (property.sale_price) data.totalPrice += property.sale_price;
      if (property.monthly_rent) data.totalRent += property.monthly_rent;
    });

    const localities = Array.from(localityMap.values())
      .filter((l) => l.properties.length >= 1)
      .map((locality) => {
        const avgPrice =
          locality.totalPrice /
            locality.properties.filter((p) => p.sale_price).length || 0;
        const avgRent =
          locality.totalRent /
            locality.properties.filter((p) => p.monthly_rent).length || 0;
        const appreciation = 12 + Math.random() * 15;
        const rentalYield =
          avgRent && avgPrice
            ? ((avgRent * 12) / avgPrice) * 100
            : 3 + Math.random() * 3;
        const score = Math.min(
          100,
          60 + locality.properties.length * 2 + Math.random() * 20
        );

        return {
          name: locality.name,
          appreciation: parseFloat(appreciation.toFixed(1)),
          yield: parseFloat(rentalYield.toFixed(1)),
          score: Math.round(score),
        };
      })
      .sort((a, b) => b.score - a.score)
      .slice(0, 6);

    setLocalityData(localities);
  };

  const calculatePropertyROI = (properties) => {
    console.log("Calculating ROI for properties:", properties.length);

    if (!properties || properties.length === 0) {
      setPropertyData([]);
      return;
    }

    // Calculate comprehensive ROI for real properties
    const roiProperties = properties
      .map((property) => {
        const salePrice = property.sale_price || 0;
        const monthlyRent = property.monthly_rent || 0;
        const builtUpArea =
          property.built_up_area || property.carpet_area || 1000;
        const locality = (
          property.locality ||
          property.city ||
          ""
        ).toLowerCase();
        const propertyType = property.property_type || "apartment";
        const isRental = property.listing_type === "rent";

        let rentalYield = 0;
        let capitalAppreciation = 0;
        let totalROI = 0;
        let pricePerSqft = 0;

        if (salePrice > 0) {
          pricePerSqft = salePrice / builtUpArea;

          // Calculate rental yield if rent data available
          if (monthlyRent > 0) {
            const annualRent = monthlyRent * 12;
            rentalYield = (annualRent / salePrice) * 100;
          } else {
            // Estimate rental yield based on location and property type
            if (
              locality.includes("koramangala") ||
              locality.includes("indiranagar")
            )
              rentalYield = 3.5;
            else if (
              locality.includes("whitefield") ||
              locality.includes("electronic city")
            )
              rentalYield = 4.2;
            else if (
              locality.includes("hebbal") ||
              locality.includes("marathahalli")
            )
              rentalYield = 3.8;
            else rentalYield = 3.2;
          }

          // Calculate capital appreciation based on location factors
          if (
            locality.includes("koramangala") ||
            locality.includes("indiranagar")
          )
            capitalAppreciation = 12;
          else if (
            locality.includes("whitefield") ||
            locality.includes("electronic city")
          )
            capitalAppreciation = 10;
          else if (
            locality.includes("hebbal") ||
            locality.includes("marathahalli")
          )
            capitalAppreciation = 8;
          else if (
            locality.includes("sarjapur") ||
            locality.includes("outer ring road")
          )
            capitalAppreciation = 9;
          else capitalAppreciation = 7;

          // Adjust for property type
          if (
            propertyType.includes("villa") ||
            propertyType.includes("independent")
          )
            capitalAppreciation += 1;
          else if (
            propertyType.includes("commercial") ||
            propertyType.includes("office")
          ) {
            capitalAppreciation += 2;
            rentalYield += 1;
          }

          totalROI = rentalYield + capitalAppreciation;
        }

        // Calculate investment grade
        let investmentGrade = "C";
        let riskLevel = "High";
        if (totalROI >= 15) {
          investmentGrade = "A+";
          riskLevel = "Low";
        } else if (totalROI >= 12) {
          investmentGrade = "A";
          riskLevel = "Low";
        } else if (totalROI >= 10) {
          investmentGrade = "B+";
          riskLevel = "Medium";
        } else if (totalROI >= 8) {
          investmentGrade = "B";
          riskLevel = "Medium";
        } else {
          investmentGrade = "C";
          riskLevel = "High";
        }

        return {
          id: property.id,
          title:
            property.title ||
            `${
              propertyType.charAt(0).toUpperCase() + propertyType.slice(1)
            } in ${property.locality || property.city}`,
          location: `${property.locality || property.city || "Unknown"}, ${
            property.city || "Unknown"
          }`,
          roi: isRental ? 0 : parseFloat(totalROI.toFixed(1)),
          rentalYield: parseFloat(rentalYield.toFixed(1)),
          capitalAppreciation: parseFloat(capitalAppreciation.toFixed(1)),
          pricePerSqft: Math.round(pricePerSqft),
          salePrice: salePrice,
          monthlyRent: monthlyRent,
          builtUpArea: builtUpArea,
          propertyType: propertyType,
          potential: isRental
            ? "Rental"
            : totalROI >= 15
            ? "High"
            : totalROI >= 10
            ? "Medium"
            : "Low",
          investmentGrade: investmentGrade,
          riskLevel: riskLevel,
          isRental: isRental,
          locality: property.locality || property.city,
        };
      })
      .filter((p) => !p.isRental) // Focus on investment properties
      .sort((a, b) => b.roi - a.roi)
      .slice(0, 12); // Show more properties

    console.log("ROI Properties calculated:", roiProperties.length);
    setPropertyData(roiProperties);
  };

  const calculateMarketTrends = (properties) => {
    const currentYear = new Date().getFullYear();
    const trends = [];

    // Base prices for trend calculation
    const basePrice = properties.length
      ? properties.reduce((sum, p) => sum + (p.sale_price || 15000000), 0) /
        properties.length
      : 15000000;

    for (let i = 5; i >= 0; i--) {
      const year = currentYear - i;
      const yearMultiplier = 1 - i * 0.08; // 8% growth per year
      const price = (basePrice * yearMultiplier) / 1000000; // Convert to millions
      const demand = Math.min(100, 50 + (5 - i) * 10 + Math.random() * 15);

      trends.push({
        year: year.toString(),
        price: Math.round(price),
        demand: Math.round(demand),
      });
    }

    setMarketData(trends);
  };

  const calculatePropertyMix = (properties) => {
    if (!properties.length) {
      setPieData([
        { name: "Residential", value: 65, color: "#0056D2" },
        { name: "Commercial", value: 25, color: "#10B981" },
        { name: "Industrial", value: 10, color: "#F59E0B" },
      ]);
      return;
    }

    const typeCount = properties.reduce((acc, property) => {
      const type = property.property_type || "residential";
      const category =
        type.includes("commercial") ||
        type.includes("office") ||
        type.includes("shop")
          ? "Commercial"
          : type.includes("industrial") || type.includes("warehouse")
          ? "Industrial"
          : "Residential";
      acc[category] = (acc[category] || 0) + 1;
      return acc;
    }, {});

    const total = Object.values(typeCount).reduce((a, b) => a + b, 0) || 1;
    const colors = {
      Residential: "#0056D2",
      Commercial: "#10B981",
      Industrial: "#F59E0B",
    };

    let pieChartData = Object.entries(typeCount).map(([name, count]) => ({
      name,
      value: Math.round((count / total) * 100),
      color: colors[name] || "#6B7280",
    }));

    // Ensure we have at least some data
    if (pieChartData.length === 0) {
      pieChartData = [{ name: "Residential", value: 100, color: "#0056D2" }];
    }

    setPieData(pieChartData);
  };

  const generateReportData = (properties) => {
    const avgPrice = properties.length
      ? properties.reduce((sum, p) => sum + (p.sale_price || 15000000), 0) /
        properties.length
      : 15000000;
    const totalProperties = properties.length;
    const avgROI = parseFloat(marketMetrics.growth) || 12.5;
    const safetyScore = marketMetrics.safetyScore || 85;

    const executiveSummary = `The ${selectedCity} real estate market demonstrates strong fundamentals with ${totalProperties} active properties analyzed. Current market conditions show an average ROI of ${avgROI}% with a safety score of ${safetyScore}/100, indicating a stable investment environment. The market exhibits ${marketMetrics.momentum.toLowerCase()} momentum with sustained demand across residential and commercial segments.`;

    const marketOverview = `Market Analysis for ${selectedCity}:\n\n• Average Property Price: ₹${(
      avgPrice / 100000
    ).toFixed(
      1
    )} Lakhs\n• Price per Sq.Ft: ₹${marketMetrics.avgPrice.toLocaleString()}\n• Rental Yield: ${
      marketMetrics.yield
    }% annually\n• Market Growth: +${
      marketMetrics.growth
    }% (5-year CAGR)\n• Active Listings: ${totalProperties} properties\n• Market Momentum: ${
      marketMetrics.momentum
    }\n\nThe market shows consistent appreciation with strong fundamentals driven by infrastructure development, employment growth, and urbanization trends.`;

    // Generate city-specific recommendations based on actual locality data
    const getAreaRecommendations = () => {
      if (localityData.length >= 3) {
        return [
          {
            type: "High Priority",
            area: localityData[0].name,
            roi: localityData[0].appreciation,
            reason: "Premium location with excellent connectivity and high appreciation potential",
            risk: "Low",
            timeframe: "2-3 years",
          },
          {
            type: "Medium Priority",
            area: localityData[1].name,
            roi: localityData[1].appreciation,
            reason: "IT corridor with steady rental demand and capital appreciation",
            risk: "Medium",
            timeframe: "3-5 years",
          },
          {
            type: "Long-term",
            area: localityData[2].name,
            roi: localityData[2].appreciation,
            reason: "Emerging area with infrastructure development and growth potential",
            risk: "Medium",
            timeframe: "5-7 years",
          },
        ];
      }
      
      // Use actual properties from selected city to generate recommendations
      const cityLocalities = new Map();
      properties.forEach(p => {
        const loc = p.locality || p.city || "Unknown";
        if (!cityLocalities.has(loc)) {
          cityLocalities.set(loc, { count: 0, totalPrice: 0, totalRent: 0 });
        }
        const data = cityLocalities.get(loc);
        data.count++;
        data.totalPrice += p.sale_price || 0;
        data.totalRent += p.monthly_rent || 0;
      });
      
      const topLocalities = Array.from(cityLocalities.entries())
        .map(([name, data]) => {
          const avgPrice = data.totalPrice / data.count;
          const avgRent = data.totalRent / data.count;
          const roi = avgRent && avgPrice ? ((avgRent * 12) / avgPrice) * 100 + 10 : 15;
          return { name, roi: parseFloat(roi.toFixed(1)) };
        })
        .sort((a, b) => b.roi - a.roi)
        .slice(0, 3);
      
      if (topLocalities.length >= 3) {
        return [
          {
            type: "High Priority",
            area: topLocalities[0].name,
            roi: topLocalities[0].roi,
            reason: "Premium location with excellent connectivity and high appreciation potential",
            risk: "Low",
            timeframe: "2-3 years",
          },
          {
            type: "Medium Priority",
            area: topLocalities[1].name,
            roi: topLocalities[1].roi,
            reason: "IT corridor with steady rental demand and capital appreciation",
            risk: "Medium",
            timeframe: "3-5 years",
          },
          {
            type: "Long-term",
            area: topLocalities[2].name,
            roi: topLocalities[2].roi,
            reason: "Emerging area with infrastructure development and growth potential",
            risk: "Medium",
            timeframe: "5-7 years",
          },
        ];
      }
      
      return [];
    };
    
    const recommendations = getAreaRecommendations();

    const riskAnalysis = `Risk Assessment:\n\n• Market Risk: LOW - Stable demand and supply dynamics\n• Liquidity Risk: LOW - High transaction volumes and buyer interest\n• Regulatory Risk: LOW - Transparent policies and RERA compliance\n• Economic Risk: MEDIUM - Dependent on economic growth and employment\n• Interest Rate Risk: MEDIUM - Sensitive to monetary policy changes\n\nOverall Risk Rating: LOW TO MEDIUM\n\nMitigation Strategies:\n- Diversify across multiple localities\n- Focus on properties with strong rental potential\n- Consider infrastructure development timelines\n- Monitor interest rate trends and policy changes`;

    const futureOutlook = `Market Outlook (2024-2027):\n\n• Expected Growth: 12-15% annually\n• Infrastructure Impact: Metro expansion and IT park development\n• Demand Drivers: Population growth, urbanization, employment opportunities\n• Supply Dynamics: Controlled supply with quality developments\n• Investment Climate: Favorable with government support\n\nKey Catalysts:\n- Infrastructure projects completion\n- IT sector expansion\n- Improved connectivity\n- Policy support for real estate\n\nRecommended Investment Strategy:\n- Entry Point: Q4 2024 - Q1 2025\n- Target Areas: IT corridors and metro connectivity zones\n- Investment Horizon: 3-5 years for optimal returns\n- Portfolio Allocation: 60% residential, 30% commercial, 10% emerging areas`;

    setReportData({
      executiveSummary,
      marketOverview,
      investmentRecommendations: recommendations,
      riskAnalysis,
      futureOutlook,
      generatedDate: new Date().toISOString().split("T")[0],
    });
  };

  const downloadReport = () => {
    try {
      const pdf = new jsPDF();

      // Add blue header background like in the image
      pdf.setFillColor(52, 86, 209); // Blue color #3456D1
      pdf.rect(0, 0, 210, 60, "F");

      // Add NAL logo
      const logoImg = new Image();
      logoImg.onload = () => {
        // Add logo on blue background
        pdf.addImage(logoImg, "PNG", 20, 15, 25, 25);

        // Add white text on blue background
        pdf.setTextColor(255, 255, 255); // White text
        pdf.setFontSize(20);
        pdf.setFont("helvetica", "bold");
        pdf.text("PROPERTY VALUATION REPORT", 55, 30);

        pdf.setFontSize(16);
        pdf.setFont("helvetica", "normal");
        pdf.text("Professional Property Assessment & Market Analysis", 55, 42);

        // Add NAL India watermark
        pdf.setGState(new pdf.GState({ opacity: 0.1 }));
        pdf.setTextColor(128, 128, 128);
        pdf.setFontSize(50);
        pdf.setFont("helvetica", "bold");
        const pageWidth = pdf.internal.pageSize.width;
        const pageHeight = pdf.internal.pageSize.height;
        pdf.text("NAL INDIA", pageWidth / 2, pageHeight / 2, {
          align: "center",
          angle: 45,
        });

        // Reset opacity and color for content
        pdf.setGState(new pdf.GState({ opacity: 1 }));
        pdf.setTextColor(0, 0, 0); // Black text

        let yPos = 80;

        // Executive Summary
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("EXECUTIVE SUMMARY", 20, yPos);
        yPos += 10;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        const summaryLines = pdf.splitTextToSize(
          reportData.executiveSummary,
          170
        );
        pdf.text(summaryLines, 20, yPos);
        yPos += summaryLines.length * 5 + 10;

        // Market Metrics
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("KEY METRICS", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(`Market Growth: ${marketMetrics.growth}%`, 20, yPos);
        pdf.text(`Rental Yield: ${marketMetrics.yield}%`, 100, yPos);
        yPos += 8;
        pdf.text(`Safety Score: ${marketMetrics.safetyScore}/100`, 20, yPos);
        pdf.text(`Market Status: ${marketMetrics.momentum}`, 100, yPos);
        yPos += 15;

        // Investment Recommendations
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("INVESTMENT RECOMMENDATIONS", 20, yPos);
        yPos += 15;

        reportData.investmentRecommendations.forEach((rec, index) => {
          pdf.setFontSize(12);
          pdf.setFont("helvetica", "bold");
          pdf.text(`${index + 1}. ${rec.area} (${rec.type})`, 20, yPos);
          yPos += 8;

          pdf.setFontSize(10);
          pdf.setFont("helvetica", "normal");
          pdf.text(
            `Expected ROI: ${rec.roi}% | Risk: ${rec.risk} | Timeframe: ${rec.timeframe}`,
            25,
            yPos
          );
          yPos += 6;

          const reasonLines = pdf.splitTextToSize(rec.reason, 165);
          pdf.text(reasonLines, 25, yPos);
          yPos += reasonLines.length * 4 + 8;
        });

        // Add new page if needed
        if (yPos > 200) {
          pdf.addPage();
          yPos = 30;
        }

        // Risk Analysis Section
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("RISK ANALYSIS", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          "• Market Risk: LOW - Stable demand and supply dynamics",
          20,
          yPos
        );
        yPos += 6;
        pdf.text("• Liquidity Risk: LOW - High transaction volumes", 20, yPos);
        yPos += 6;
        pdf.text(
          "• Economic Risk: MEDIUM - Dependent on economic growth",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Interest Rate Risk: MEDIUM - Sensitive to policy changes",
          20,
          yPos
        );
        yPos += 15;

        // Market Outlook
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("MARKET OUTLOOK", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          "• Expected Growth: 12-15% annually over next 3 years",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Infrastructure Impact: Metro expansion driving appreciation",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• IT Sector Growth: Continued job creation and demand",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Government Support: Favorable policies for real estate",
          20,
          yPos
        );
        yPos += 15;

        // Disclaimer
        pdf.setFontSize(12);
        pdf.setFont("helvetica", "bold");
        pdf.text("DISCLAIMER", 20, yPos);
        yPos += 10;

        pdf.setFontSize(9);
        pdf.setFont("helvetica", "normal");
        const disclaimerText =
          "This report is for informational purposes only and should not be considered as financial advice. Market conditions can change rapidly, and past performance does not guarantee future results. Please consult with qualified financial advisors before making investment decisions.";
        const disclaimerLines = pdf.splitTextToSize(disclaimerText, 170);
        pdf.text(disclaimerLines, 20, yPos);

        // Footer with NAL branding
        pdf.setFillColor(52, 86, 209);
        pdf.rect(0, 275, 210, 22, "F");
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(10);
        pdf.setFont("helvetica", "bold");
        pdf.text("NAL INDIA - New Age Land", 20, 285);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          "www.nglindia.com | info@nglindia.com | Professional Real Estate Services",
          20,
          292
        );

        // Save PDF
        pdf.save(
          `Investment_Report_${selectedCity}_${reportData.generatedDate}.pdf`
        );

        toast.success(
          `Investment report for ${selectedCity} downloaded successfully!`,
          {
            description: `Report saved as PDF with NAL branding`,
          }
        );
      };

      logoImg.src = "/src/assets/logo.jpg";
    } catch (error) {
      console.error("Error downloading PDF report:", error);
      toast.error("Failed to download PDF report. Please try again.");
    }
  };

  const downloadDetailedReport = () => {
    try {
      const pdf = new jsPDF();

      // Add blue header background like in the image
      pdf.setFillColor(52, 86, 209); // Blue color #3456D1
      pdf.rect(0, 0, 210, 60, "F");

      // Add NAL logo
      const logoImg = new Image();
      logoImg.onload = () => {
        // Add logo on blue background
        pdf.addImage(logoImg, "PNG", 20, 15, 25, 25);

        // Add white text on blue background
        pdf.setTextColor(255, 255, 255); // White text
        pdf.setFontSize(20);
        pdf.setFont("helvetica", "bold");
        pdf.text("PROPERTY VALUATION REPORT", 55, 30);

        pdf.setFontSize(16);
        pdf.setFont("helvetica", "normal");
        pdf.text("Professional Property Assessment & Market Analysis", 55, 42);

        pdf.setFontSize(12);
        pdf.text(
          `Detailed Analysis - Generated on: ${reportData.generatedDate}`,
          55,
          52
        );

        // Add NAL India watermark on every page
        const addWatermark = () => {
          pdf.setGState(new pdf.GState({ opacity: 0.1 }));
          pdf.setTextColor(128, 128, 128);
          pdf.setFontSize(50);
          pdf.setFont("helvetica", "bold");
          const pageWidth = pdf.internal.pageSize.width;
          const pageHeight = pdf.internal.pageSize.height;
          pdf.text("NAL INDIA", pageWidth / 2, pageHeight / 2, {
            align: "center",
            angle: 45,
          });

          // Reset opacity and color for content
          pdf.setGState(new pdf.GState({ opacity: 1 }));
          pdf.setTextColor(0, 0, 0); // Black text
        };

        addWatermark();

        let yPos = 80;

        // Market Metrics Summary
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("MARKET METRICS SUMMARY", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          `• Market Growth Rate: ${marketMetrics.growth}% (5-Year CAGR)`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Average Rental Yield: ${marketMetrics.yield}% annually`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Investment Safety Score: ${marketMetrics.safetyScore}/100`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Average Price per Sq.Ft: ₹${marketMetrics.avgPrice.toLocaleString()}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(`• Market Momentum: ${marketMetrics.momentum}`, 20, yPos);
        yPos += 6;
        pdf.text(
          `• Total Properties Analyzed: ${allProperties.length}`,
          20,
          yPos
        );
        yPos += 20;

        // Top Performing Localities
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("TOP PERFORMING LOCALITIES", 20, yPos);
        yPos += 15;

        localityData.slice(0, 5).forEach((locality, index) => {
          pdf.setFontSize(12);
          pdf.setFont("helvetica", "bold");
          pdf.text(`${index + 1}. ${locality.name}`, 20, yPos);
          yPos += 8;

          pdf.setFontSize(10);
          pdf.setFont("helvetica", "normal");
          pdf.text(`• Appreciation Rate: ${locality.appreciation}%`, 25, yPos);
          yPos += 5;
          pdf.text(`• Rental Yield: ${locality.yield}%`, 25, yPos);
          yPos += 5;
          pdf.text(`• Investment Score: ${locality.score}/100`, 25, yPos);
          yPos += 10;
        });

        // Add new page with watermark
        pdf.addPage();
        addWatermark();
        yPos = 30;

        // Property ROI Analysis
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("PROPERTY ROI ANALYSIS", 20, yPos);
        yPos += 15;

        propertyData.slice(0, 6).forEach((property, index) => {
          pdf.setFontSize(11);
          pdf.setFont("helvetica", "bold");
          pdf.text(`${index + 1}. ${property.title}`, 20, yPos);
          yPos += 7;

          pdf.setFontSize(10);
          pdf.setFont("helvetica", "normal");
          pdf.text(`• Location: ${property.location}`, 25, yPos);
          yPos += 5;
          pdf.text(`• Expected ROI: ${property.roi}%`, 25, yPos);
          yPos += 5;
          pdf.text(`• Investment Grade: ${property.investmentGrade}`, 25, yPos);
          yPos += 10;
        });

        // Investment Strategy
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("INVESTMENT STRATEGY", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text("IMMEDIATE OPPORTUNITIES (0-6 months)", 20, yPos);
        yPos += 8;
        pdf.text(
          "• Focus on pre-launch and under-construction projects",
          25,
          yPos
        );
        yPos += 5;
        pdf.text("• Target IT corridor properties for rental income", 25, yPos);
        yPos += 10;

        pdf.text("SHORT-TERM STRATEGY (6 months - 2 years)", 20, yPos);
        yPos += 8;
        pdf.text("• Diversify across 2-3 localities", 25, yPos);
        yPos += 5;
        pdf.text(
          "• Balance between capital appreciation and rental yield",
          25,
          yPos
        );
        yPos += 15;

        // Risk Assessment
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("RISK ASSESSMENT", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text("Overall Risk Rating: LOW TO MEDIUM", 20, yPos);
        yPos += 8;
        pdf.text(
          "Market Stability: High with consistent growth patterns",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "Liquidity: Excellent due to high demand and transaction volume",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "Regulatory Environment: Stable with RERA compliance",
          20,
          yPos
        );
        yPos += 15;

        // Add new page for additional content with watermark
        pdf.addPage();
        addWatermark();
        yPos = 30;

        // Financing Options
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("FINANCING OPTIONS", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text("Home Loans:", 20, yPos);
        yPos += 8;
        pdf.text("• Interest Rates: 8.5% - 9.5% per annum", 25, yPos);
        yPos += 6;
        pdf.text(
          "• Loan-to-Value: Up to 90% for properties under ₹45L",
          25,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Tenure: Up to 30 years with flexible EMI options",
          25,
          yPos
        );
        yPos += 12;

        pdf.text("Tax Benefits:", 20, yPos);
        yPos += 8;
        pdf.text(
          "• Section 80C: Principal repayment deduction up to ₹1.5L",
          25,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Section 24: Interest deduction up to ₹2L annually",
          25,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Capital Gains: LTCG benefits after 2 years holding",
          25,
          yPos
        );
        yPos += 15;

        // Next Steps
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("RECOMMENDED NEXT STEPS", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          "1. Conduct detailed due diligence on shortlisted properties",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "2. Engage with local real estate experts and legal advisors",
          20,
          yPos
        );
        yPos += 6;
        pdf.text("3. Secure pre-approval for financing options", 20, yPos);
        yPos += 6;
        pdf.text(
          "4. Monitor market trends and timing for optimal entry",
          20,
          yPos
        );
        yPos += 6;
        pdf.text("5. Develop a systematic investment approach", 20, yPos);
        yPos += 15;

        // Disclaimer
        pdf.setFontSize(12);
        pdf.setFont("helvetica", "bold");
        pdf.text("DISCLAIMER", 20, yPos);
        yPos += 10;

        pdf.setFontSize(9);
        pdf.setFont("helvetica", "normal");
        const disclaimerText =
          "This report is prepared based on available market data and analysis. It is intended for informational purposes only and should not be construed as investment advice. Market conditions are subject to change, and past performance does not guarantee future results. Investors are advised to conduct independent research and consult qualified financial and legal advisors.";
        const disclaimerLines = pdf.splitTextToSize(disclaimerText, 170);
        pdf.text(disclaimerLines, 20, yPos);

        // Footer with NAL branding on all pages
        const addFooter = () => {
          pdf.setFillColor(52, 86, 209);
          pdf.rect(0, 275, 210, 22, "F");
          pdf.setTextColor(255, 255, 255);
          pdf.setFontSize(10);
          pdf.setFont("helvetica", "bold");
          pdf.text("NAL INDIA - New Age Land", 20, 285);
          pdf.setFont("helvetica", "normal");
          pdf.text(
            "www.nglindia.com | info@nglindia.com | Professional Real Estate Services",
            20,
            292
          );
        };

        // Add footer to all pages
        const totalPages = pdf.internal.getNumberOfPages();
        for (let i = 1; i <= totalPages; i++) {
          pdf.setPage(i);
          addFooter();
        }

        // Save PDF
        pdf.save(
          `Detailed_Investment_Analysis_${selectedCity}_${reportData.generatedDate}.pdf`
        );

        toast.success(`Detailed investment analysis downloaded successfully!`, {
          description: `Comprehensive PDF report saved with NAL branding`,
        });
      };

      logoImg.src = "/src/assets/logo.jpg";
    } catch (error) {
      console.error("Error downloading detailed PDF report:", error);
      toast.error("Failed to download detailed PDF report. Please try again.");
    }
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center py-4">
            <h1 className="text-2xl font-bold text-gray-900">
              Investment Insights
            </h1>
            <Select value={selectedCity} onValueChange={setSelectedCity}>
              <SelectTrigger className="w-48">
                <SelectValue placeholder="Select City" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="Bangalore">Bangalore</SelectItem>
                <SelectItem value="Mumbai">Mumbai</SelectItem>
                <SelectItem value="Delhi">Delhi</SelectItem>
                <SelectItem value="Hyderabad">Hyderabad</SelectItem>
                <SelectItem value="Chennai">Chennai</SelectItem>
                <SelectItem value="Pune">Pune</SelectItem>
                <SelectItem value="Kolkata">Kolkata</SelectItem>
                <SelectItem value="Ahmedabad">Ahmedabad</SelectItem>
                <SelectItem value="Gurgaon">Gurgaon</SelectItem>
                <SelectItem value="Noida">Noida</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList className="grid w-full grid-cols-2 max-w-lg mb-8">
            <TabsTrigger value="report">Investment Report</TabsTrigger>
            <TabsTrigger value="property">Property ROI</TabsTrigger>
          </TabsList>

          {/* Investment Report */}
          <TabsContent value="report" className="space-y-8">
            {/* Report Header with Download Options */}
            <Card className="bg-gradient-to-r from-blue-50 to-indigo-50 border-blue-200">
              <CardContent className="p-6">
                <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                  <div>
                    <h2 className="text-2xl font-bold text-gray-900 mb-2">
                      Investment Report - {selectedCity}
                    </h2>
                    <p className="text-gray-600 mb-4">
                      Comprehensive market analysis with real-time data,
                      investment recommendations, and risk assessment. Generated
                      on {reportData.generatedDate}
                    </p>
                    <div className="flex flex-wrap gap-2">
                      <Badge className="bg-green-100 text-green-800">
                        <CheckCircle className="w-3 h-3 mr-1" />
                        Real-time Data
                      </Badge>
                      <Badge className="bg-blue-100 text-blue-800">
                        <Shield className="w-3 h-3 mr-1" />
                        Risk Analyzed
                      </Badge>
                      <Badge className="bg-purple-100 text-purple-800">
                        <TrendingUp className="w-3 h-3 mr-1" />
                        ROI Optimized
                      </Badge>
                    </div>
                  </div>
                  <div className="flex flex-col sm:flex-row gap-3">
                    <Button
                      onClick={downloadReport}
                      className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white flex items-center gap-2"
                    >
                      <Download className="w-4 h-4" />
                      Download Report
                    </Button>
                    <Button
                      onClick={downloadDetailedReport}
                      variant="outline"
                      className="border-[#0056D2] text-[#0056D2] hover:bg-[#0056D2] hover:text-white flex items-center gap-2"
                    >
                      <FileText className="w-4 h-4" />
                      Detailed Analysis
                    </Button>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Executive Summary */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <FileText className="w-5 h-5 text-[#0056D2]" />
                  Executive Summary
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="bg-gradient-to-r from-gray-50 to-blue-50 p-6 rounded-lg border border-gray-200">
                  <p className="text-gray-700 leading-relaxed mb-4">
                    {reportData.executiveSummary}
                  </p>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mt-6">
                    <div className="text-center p-3 bg-white rounded-lg shadow-sm">
                      <div className="text-2xl font-bold text-green-600">
                        {marketMetrics.growth}%
                      </div>
                      <div className="text-sm text-gray-600">Expected ROI</div>
                    </div>
                    <div className="text-center p-3 bg-white rounded-lg shadow-sm">
                      <div className="text-2xl font-bold text-blue-600">
                        {marketMetrics.safetyScore}/100
                      </div>
                      <div className="text-sm text-gray-600">Safety Score</div>
                    </div>
                    <div className="text-center p-3 bg-white rounded-lg shadow-sm">
                      <div className="text-2xl font-bold text-purple-600">
                        {allProperties.length}
                      </div>
                      <div className="text-sm text-gray-600">
                        Properties Analyzed
                      </div>
                    </div>
                    <div className="text-center p-3 bg-white rounded-lg shadow-sm">
                      <div className="text-2xl font-bold text-orange-600">
                        {marketMetrics.momentum}
                      </div>
                      <div className="text-sm text-gray-600">Market Status</div>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Investment Recommendations */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Target className="w-5 h-5 text-[#0056D2]" />
                  Investment Recommendations
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {reportData.investmentRecommendations.map((rec, index) => (
                    <div
                      key={index}
                      className="p-6 border rounded-xl bg-gradient-to-br from-white to-gray-50 hover:shadow-lg transition-shadow"
                    >
                      <div className="flex items-center justify-between mb-4">
                        <Badge
                          className={`${
                            rec.type === "High Priority"
                              ? "bg-green-100 text-green-800"
                              : rec.type === "Medium Priority"
                              ? "bg-blue-100 text-blue-800"
                              : "bg-purple-100 text-purple-800"
                          }`}
                        >
                          {rec.type}
                        </Badge>
                        <div className="text-right">
                          <div className="text-2xl font-bold text-green-600">
                            {rec.roi}%
                          </div>
                          <div className="text-xs text-gray-600">
                            Expected ROI
                          </div>
                        </div>
                      </div>
                      <h4 className="font-bold text-lg text-gray-900 mb-2">
                        {rec.area}
                      </h4>
                      <p className="text-sm text-gray-600 mb-4">{rec.reason}</p>
                      <div className="space-y-2">
                        <div className="flex justify-between text-sm">
                          <span className="text-gray-600">Risk Level:</span>
                          <span
                            className={`font-medium ${
                              rec.risk === "Low"
                                ? "text-green-600"
                                : "text-yellow-600"
                            }`}
                          >
                            {rec.risk}
                          </span>
                        </div>
                        <div className="flex justify-between text-sm">
                          <span className="text-gray-600">Timeframe:</span>
                          <span className="font-medium text-blue-600">
                            {rec.timeframe}
                          </span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
            {/* Key Metrics Overview */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-blue-600 text-sm font-medium">
                        Market Growth
                      </p>
                      <p className="text-2xl font-bold text-blue-900">
                        +{marketMetrics.growth}%
                      </p>
                      <p className="text-xs text-blue-700 flex items-center gap-1">
                        <ArrowUpRight className="w-3 h-3" />
                        5-Year CAGR
                      </p>
                    </div>
                    <div className="p-3 bg-blue-200 rounded-full">
                      <TrendingUp className="w-6 h-6 text-blue-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-green-600 text-sm font-medium">
                        Rental Yield
                      </p>
                      <p className="text-2xl font-bold text-green-900">
                        {marketMetrics.yield}%
                      </p>
                      <p className="text-xs text-green-700 flex items-center gap-1">
                        <DollarSign className="w-3 h-3" />
                        Annual Return
                      </p>
                    </div>
                    <div className="p-3 bg-green-200 rounded-full">
                      <Target className="w-6 h-6 text-green-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-purple-600 text-sm font-medium">
                        Safety Score
                      </p>
                      <p className="text-2xl font-bold text-purple-900">
                        {marketMetrics.safetyScore}/100
                      </p>
                      <p className="text-xs text-purple-700 flex items-center gap-1">
                        <Shield className="w-3 h-3" />
                        Investment Grade
                      </p>
                    </div>
                    <div className="p-3 bg-purple-200 rounded-full">
                      <Award className="w-6 h-6 text-purple-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-orange-600 text-sm font-medium">
                        Market Activity
                      </p>
                      <p className="text-2xl font-bold text-orange-900">
                        {marketMetrics.momentum}
                      </p>
                      <p className="text-xs text-orange-700 flex items-center gap-1">
                        <Activity className="w-3 h-3" />
                        Demand Level
                      </p>
                    </div>
                    <div className="p-3 bg-orange-200 rounded-full">
                      <Zap className="w-6 h-6 text-orange-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Market Analysis & Risk Assessment */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <AlertTriangle className="w-5 h-5 text-[#0056D2]" />
                    Risk Assessment Matrix
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-4 bg-green-50 rounded-lg border border-green-200">
                      <div className="flex items-center gap-3">
                        <div className="w-3 h-3 bg-green-500 rounded-full"></div>
                        <span className="font-medium text-green-900">
                          Market Risk
                        </span>
                      </div>
                      <Badge className="bg-green-100 text-green-800">LOW</Badge>
                    </div>
                    <div className="flex items-center justify-between p-4 bg-green-50 rounded-lg border border-green-200">
                      <div className="flex items-center gap-3">
                        <div className="w-3 h-3 bg-green-500 rounded-full"></div>
                        <span className="font-medium text-green-900">
                          Liquidity Risk
                        </span>
                      </div>
                      <Badge className="bg-green-100 text-green-800">LOW</Badge>
                    </div>
                    <div className="flex items-center justify-between p-4 bg-yellow-50 rounded-lg border border-yellow-200">
                      <div className="flex items-center gap-3">
                        <div className="w-3 h-3 bg-yellow-500 rounded-full"></div>
                        <span className="font-medium text-yellow-900">
                          Economic Risk
                        </span>
                      </div>
                      <Badge className="bg-yellow-100 text-yellow-800">
                        MEDIUM
                      </Badge>
                    </div>
                    <div className="flex items-center justify-between p-4 bg-yellow-50 rounded-lg border border-yellow-200">
                      <div className="flex items-center gap-3">
                        <div className="w-3 h-3 bg-yellow-500 rounded-full"></div>
                        <span className="font-medium text-yellow-900">
                          Interest Rate Risk
                        </span>
                      </div>
                      <Badge className="bg-yellow-100 text-yellow-800">
                        MEDIUM
                      </Badge>
                    </div>
                    <div className="mt-4 p-4 bg-blue-50 rounded-lg">
                      <div className="flex items-center gap-2 mb-2">
                        <Shield className="w-4 h-4 text-blue-600" />
                        <span className="font-semibold text-blue-900">
                          Overall Risk Rating
                        </span>
                      </div>
                      <div className="text-2xl font-bold text-blue-600">
                        LOW TO MEDIUM
                      </div>
                      <p className="text-sm text-blue-700 mt-1">
                        Suitable for conservative to moderate investors
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Calendar className="w-5 h-5 text-[#0056D2]" />
                    Investment Timeline & Strategy
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-6">
                    <div className="relative">
                      <div className="absolute left-4 top-8 bottom-0 w-0.5 bg-gray-200"></div>
                      <div className="flex items-start gap-4">
                        <div className="w-8 h-8 bg-green-500 rounded-full flex items-center justify-center text-white text-sm font-bold">
                          1
                        </div>
                        <div>
                          <h4 className="font-semibold text-gray-900">
                            Q4 2024 - Entry Phase
                          </h4>
                          <p className="text-sm text-gray-600">
                            Optimal buying window with pre-launch opportunities
                          </p>
                          <Badge className="mt-1 bg-green-100 text-green-800">
                            Recommended
                          </Badge>
                        </div>
                      </div>
                    </div>
                    <div className="relative">
                      <div className="absolute left-4 top-8 bottom-0 w-0.5 bg-gray-200"></div>
                      <div className="flex items-start gap-4">
                        <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center text-white text-sm font-bold">
                          2
                        </div>
                        <div>
                          <h4 className="font-semibold text-gray-900">
                            Q1-Q2 2025 - Growth Phase
                          </h4>
                          <p className="text-sm text-gray-600">
                            Infrastructure completion driving appreciation
                          </p>
                          <Badge className="mt-1 bg-blue-100 text-blue-800">
                            High Growth
                          </Badge>
                        </div>
                      </div>
                    </div>
                    <div className="relative">
                      <div className="flex items-start gap-4">
                        <div className="w-8 h-8 bg-purple-500 rounded-full flex items-center justify-center text-white text-sm font-bold">
                          3
                        </div>
                        <div>
                          <h4 className="font-semibold text-gray-900">
                            2026-2027 - Maturity Phase
                          </h4>
                          <p className="text-sm text-gray-600">
                            Stable returns with exit opportunities
                          </p>
                          <Badge className="mt-1 bg-purple-100 text-purple-800">
                            Exit Window
                          </Badge>
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Charts Section */}
            <div className="grid grid-cols-1 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <BarChart3 className="w-5 h-5 text-[#0056D2]" />
                    Price Appreciation Trend & Market Performance
                  </CardTitle>
                  <p className="text-sm text-gray-600 mt-1">
                    Historical price trends showing consistent growth with
                    projected future performance
                  </p>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={300}>
                    <AreaChart data={marketData}>
                      <defs>
                        <linearGradient
                          id="colorPrice"
                          x1="0"
                          y1="0"
                          x2="0"
                          y2="1"
                        >
                          <stop
                            offset="5%"
                            stopColor="#0056D2"
                            stopOpacity={0.3}
                          />
                          <stop
                            offset="95%"
                            stopColor="#0056D2"
                            stopOpacity={0}
                          />
                        </linearGradient>
                      </defs>
                      <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                      <XAxis dataKey="year" stroke="#666" />
                      <YAxis stroke="#666" />
                      <Tooltip
                        contentStyle={{
                          backgroundColor: "white",
                          border: "1px solid #e5e7eb",
                          borderRadius: "8px",
                          boxShadow: "0 4px 6px -1px rgba(0, 0, 0, 0.1)",
                        }}
                        formatter={(value, name) => [
                          name === "price" ? `₹${value}M` : `${value}%`,
                          name === "price" ? "Avg Price" : "Demand Index",
                        ]}
                      />
                      <Area
                        type="monotone"
                        dataKey="price"
                        stroke="#0056D2"
                        strokeWidth={3}
                        fillOpacity={1}
                        fill="url(#colorPrice)"
                      />
                    </AreaChart>
                  </ResponsiveContainer>
                  <div className="mt-4 grid grid-cols-3 gap-4 text-center">
                    <div className="p-3 bg-green-50 rounded-lg">
                      <div className="text-lg font-bold text-green-600">
                        +{marketMetrics.growth}%
                      </div>
                      <div className="text-xs text-gray-600">5-Year CAGR</div>
                    </div>
                    <div className="p-3 bg-blue-50 rounded-lg">
                      <div className="text-lg font-bold text-blue-600">
                        {marketMetrics.yield}%
                      </div>
                      <div className="text-xs text-gray-600">Rental Yield</div>
                    </div>
                    <div className="p-3 bg-purple-50 rounded-lg">
                      <div className="text-lg font-bold text-purple-600">
                        {marketMetrics.safetyScore}
                      </div>
                      <div className="text-xs text-gray-600">Safety Score</div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Market Insights & Future Outlook */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Activity className="w-5 h-5 text-[#0056D2]" />
                    Demand vs Supply Analysis
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={250}>
                    <BarChart data={marketData}>
                      <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                      <XAxis dataKey="year" stroke="#666" />
                      <YAxis stroke="#666" />
                      <Tooltip
                        contentStyle={{
                          backgroundColor: "white",
                          border: "1px solid #e5e7eb",
                          borderRadius: "8px",
                        }}
                        formatter={(value) => [`${value}%`, "Demand Index"]}
                      />
                      <Bar
                        dataKey="demand"
                        fill="#0056D2"
                        radius={[4, 4, 0, 0]}
                      />
                    </BarChart>
                  </ResponsiveContainer>
                  <div className="mt-4 p-3 bg-blue-50 rounded-lg">
                    <p className="text-sm text-blue-700">
                      <strong>Market Balance:</strong> Healthy demand-supply
                      ratio with controlled inventory levels supporting price
                      stability.
                    </p>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="w-5 h-5 text-[#0056D2]" />
                    Future Market Outlook
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="p-4 bg-gradient-to-r from-green-50 to-green-100 rounded-lg border border-green-200">
                      <div className="flex items-center gap-2 mb-2">
                        <ArrowUpRight className="w-4 h-4 text-green-600" />
                        <span className="font-semibold text-green-900">
                          Growth Drivers
                        </span>
                      </div>
                      <ul className="text-sm text-green-700 space-y-1">
                        <li>• Infrastructure development projects</li>
                        <li>• IT sector expansion and job creation</li>
                        <li>• Metro connectivity improvements</li>
                        <li>• Government policy support</li>
                      </ul>
                    </div>

                    <div className="p-4 bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg border border-blue-200">
                      <div className="flex items-center gap-2 mb-2">
                        <Target className="w-4 h-4 text-blue-600" />
                        <span className="font-semibold text-blue-900">
                          Key Projections
                        </span>
                      </div>
                      <div className="grid grid-cols-2 gap-3 text-sm">
                        <div>
                          <div className="font-bold text-blue-600">12-15%</div>
                          <div className="text-blue-700">Annual Growth</div>
                        </div>
                        <div>
                          <div className="font-bold text-blue-600">
                            3-5 Years
                          </div>
                          <div className="text-blue-700">Optimal Hold</div>
                        </div>
                      </div>
                    </div>

                    <div className="p-4 bg-gradient-to-r from-purple-50 to-purple-100 rounded-lg border border-purple-200">
                      <div className="flex items-center gap-2 mb-2">
                        <Award className="w-4 h-4 text-purple-600" />
                        <span className="font-semibold text-purple-900">
                          Investment Grade
                        </span>
                      </div>
                      <div className="text-2xl font-bold text-purple-600">
                        A+
                      </div>
                      <p className="text-sm text-purple-700">
                        Excellent fundamentals with strong growth potential
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Property Mix Analysis */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <PieChart className="w-5 h-5 text-[#0056D2]" />
                  Property Portfolio Distribution
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                  <div>
                    <ResponsiveContainer width="100%" height={250}>
                      <RechartsPieChart>
                        <Tooltip
                          contentStyle={{
                            backgroundColor: "white",
                            border: "1px solid #e5e7eb",
                            borderRadius: "8px",
                          }}
                          formatter={(value) => [`${value}%`, "Market Share"]}
                        />
                        <Pie
                          data={pieData}
                          dataKey="value"
                          cx="50%"
                          cy="50%"
                          outerRadius={80}
                          label={({ name, value }) => `${name}: ${value}%`}
                        >
                          {pieData.map((entry, index) => (
                            <Cell key={`cell-${index}`} fill={entry.color} />
                          ))}
                        </Pie>
                      </RechartsPieChart>
                    </ResponsiveContainer>
                  </div>
                  <div className="space-y-4">
                    {pieData.map((item, index) => (
                      <div
                        key={index}
                        className="flex items-center justify-between p-4 bg-gray-50 rounded-lg"
                      >
                        <div className="flex items-center gap-3">
                          <div
                            className="w-4 h-4 rounded-full"
                            style={{ backgroundColor: item.color }}
                          ></div>
                          <span className="font-medium text-gray-900">
                            {item.name}
                          </span>
                        </div>
                        <div className="text-right">
                          <div className="font-bold text-gray-900">
                            {item.value}%
                          </div>
                          <div className="text-xs text-gray-600">
                            Market Share
                          </div>
                        </div>
                      </div>
                    ))}
                    <div className="mt-4 p-4 bg-blue-50 rounded-lg">
                      <h4 className="font-semibold text-blue-900 mb-2">
                        Portfolio Recommendation
                      </h4>
                      <p className="text-sm text-blue-700">
                        Diversify across residential (60%), commercial (30%),
                        and emerging sectors (10%) for optimal risk-adjusted
                        returns.
                      </p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Investor Dashboard */}
          <TabsContent value="dashboard" className="space-y-6">
            {/* Dashboard Overview */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <Card className="bg-gradient-to-br from-indigo-50 to-indigo-100 border-indigo-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-indigo-600 text-sm font-medium">
                        Portfolio Value
                      </p>
                      <p className="text-2xl font-bold text-indigo-900">
                        ₹
                        {(
                          allProperties.reduce(
                            (sum, p) => sum + (p.sale_price || 0),
                            0
                          ) / 10000000
                        ).toFixed(1)}
                        Cr
                      </p>
                      <p className="text-xs text-indigo-700 flex items-center gap-1">
                        <ArrowUpRight className="w-3 h-3" />
                        +12.5% this month
                      </p>
                    </div>
                    <div className="p-3 bg-indigo-200 rounded-full">
                      <Building className="w-6 h-6 text-indigo-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-emerald-50 to-emerald-100 border-emerald-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-emerald-600 text-sm font-medium">
                        Active Properties
                      </p>
                      <p className="text-2xl font-bold text-emerald-900">
                        {allProperties.length}
                      </p>
                      <p className="text-xs text-emerald-700 flex items-center gap-1">
                        <Eye className="w-3 h-3" />2 new this week
                      </p>
                    </div>
                    <div className="p-3 bg-emerald-200 rounded-full">
                      <Users className="w-6 h-6 text-emerald-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-amber-50 to-amber-100 border-amber-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-amber-600 text-sm font-medium">
                        Monthly ROI
                      </p>
                      <p className="text-2xl font-bold text-amber-900">
                        {marketMetrics.growth}%
                      </p>
                      <p className="text-xs text-amber-700 flex items-center gap-1">
                        <Star className="w-3 h-3" />
                        Above average
                      </p>
                    </div>
                    <div className="p-3 bg-amber-200 rounded-full">
                      <Target className="w-6 h-6 text-amber-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Top Performing Localities */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <TrendingUp className="w-5 h-5 text-[#0056D2]" />
                  Top Investment Opportunities
                </CardTitle>
              </CardHeader>
              <CardContent>
                {loading ? (
                  <div className="text-center py-8">Loading...</div>
                ) : (
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    {localityData.map((locality, index) => (
                      <div
                        key={index}
                        className="p-6 border rounded-xl bg-gradient-to-br from-white to-gray-50 hover:shadow-lg transition-shadow"
                      >
                        <div className="flex items-center justify-between mb-4">
                          <h4 className="font-bold text-lg text-gray-900">
                            {locality.name}
                          </h4>
                          <Badge
                            className={`${
                              locality.score >= 90
                                ? "bg-green-100 text-green-800"
                                : "bg-blue-100 text-blue-800"
                            }`}
                          >
                            {locality.score >= 90 ? "Premium" : "High Growth"}
                          </Badge>
                        </div>
                        <div className="space-y-3">
                          <div className="flex justify-between items-center">
                            <span className="text-sm text-gray-600 flex items-center gap-2">
                              <ArrowUpRight className="w-4 h-4 text-green-600" />
                              Appreciation
                            </span>
                            <span className="text-green-600 font-bold text-lg">
                              {locality.appreciation}%
                            </span>
                          </div>
                          <div className="flex justify-between items-center">
                            <span className="text-sm text-gray-600 flex items-center gap-2">
                              <DollarSign className="w-4 h-4 text-blue-600" />
                              Rental Yield
                            </span>
                            <span className="text-blue-600 font-bold text-lg">
                              {locality.yield}%
                            </span>
                          </div>
                          <div className="flex justify-between items-center">
                            <span className="text-sm text-gray-600 flex items-center gap-2">
                              <Award className="w-4 h-4 text-purple-600" />
                              Investment Score
                            </span>
                            <span className="text-purple-600 font-bold text-lg">
                              {locality.score}/100
                            </span>
                          </div>
                        </div>
                        <div className="mt-4 pt-4 border-t">
                          <div className="w-full bg-gray-200 rounded-full h-2">
                            <div
                              className="bg-gradient-to-r from-blue-500 to-green-500 h-2 rounded-full transition-all duration-500"
                              style={{ width: `${locality.score}%` }}
                            ></div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Market Activity & Trends */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Activity className="w-5 h-5 text-[#0056D2]" />
                    Weekly Market Movers
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-4 bg-gradient-to-r from-green-50 to-green-100 rounded-xl border border-green-200">
                      <div className="flex items-center gap-3">
                        <div className="p-2 bg-green-200 rounded-full">
                          <TrendingUp className="w-4 h-4 text-green-700" />
                        </div>
                        <div>
                          <span className="font-semibold text-gray-900">
                            Koramangala
                          </span>
                          <p className="text-xs text-gray-600">
                            Premium locality
                          </p>
                        </div>
                      </div>
                      <div className="text-right">
                        <span className="text-green-600 font-bold text-lg">
                          +3.2%
                        </span>
                        <p className="text-xs text-gray-600">This week</p>
                      </div>
                    </div>
                    <div className="flex items-center justify-between p-4 bg-gradient-to-r from-blue-50 to-blue-100 rounded-xl border border-blue-200">
                      <div className="flex items-center gap-3">
                        <div className="p-2 bg-blue-200 rounded-full">
                          <TrendingUp className="w-4 h-4 text-blue-700" />
                        </div>
                        <div>
                          <span className="font-semibold text-gray-900">
                            Whitefield
                          </span>
                          <p className="text-xs text-gray-600">IT corridor</p>
                        </div>
                      </div>
                      <div className="text-right">
                        <span className="text-blue-600 font-bold text-lg">
                          +2.8%
                        </span>
                        <p className="text-xs text-gray-600">This week</p>
                      </div>
                    </div>
                    <div className="flex items-center justify-between p-4 bg-gradient-to-r from-purple-50 to-purple-100 rounded-xl border border-purple-200">
                      <div className="flex items-center gap-3">
                        <div className="p-2 bg-purple-200 rounded-full">
                          <ArrowDownRight className="w-4 h-4 text-purple-700" />
                        </div>
                        <div>
                          <span className="font-semibold text-gray-900">
                            Electronic City
                          </span>
                          <p className="text-xs text-gray-600">Tech hub</p>
                        </div>
                      </div>
                      <div className="text-right">
                        <span className="text-purple-600 font-bold text-lg">
                          +1.5%
                        </span>
                        <p className="text-xs text-gray-600">This week</p>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Calendar className="w-5 h-5 text-[#0056D2]" />
                    Investment Timeline
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-start gap-4">
                      <div className="w-3 h-3 bg-green-500 rounded-full mt-2"></div>
                      <div>
                        <p className="font-semibold text-gray-900">Q4 2024</p>
                        <p className="text-sm text-gray-600">
                          Peak buying season begins
                        </p>
                        <p className="text-xs text-green-600 font-medium">
                          Optimal entry point
                        </p>
                      </div>
                    </div>
                    <div className="flex items-start gap-4">
                      <div className="w-3 h-3 bg-blue-500 rounded-full mt-2"></div>
                      <div>
                        <p className="font-semibold text-gray-900">Q1 2025</p>
                        <p className="text-sm text-gray-600">
                          Infrastructure projects launch
                        </p>
                        <p className="text-xs text-blue-600 font-medium">
                          Growth catalyst
                        </p>
                      </div>
                    </div>
                    <div className="flex items-start gap-4">
                      <div className="w-3 h-3 bg-purple-500 rounded-full mt-2"></div>
                      <div>
                        <p className="font-semibold text-gray-900">Q2 2025</p>
                        <p className="text-sm text-gray-600">
                          Expected price appreciation
                        </p>
                        <p className="text-xs text-purple-600 font-medium">
                          15-20% gains
                        </p>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Property ROI */}
          <TabsContent value="property" className="space-y-6">
            {/* ROI Performance Metrics */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card className="bg-gradient-to-br from-emerald-50 to-emerald-100 border-emerald-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-emerald-600 text-sm font-medium">
                        Average ROI
                      </p>
                      <p className="text-3xl font-bold text-emerald-900">
                        {marketMetrics.growth}%
                      </p>
                      <p className="text-xs text-emerald-700 flex items-center gap-1">
                        <ArrowUpRight className="w-3 h-3" />
                        +2.3% vs last year
                      </p>
                    </div>
                    <div className="p-3 bg-emerald-200 rounded-full">
                      <Target className="w-6 h-6 text-emerald-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-blue-600 text-sm font-medium">
                        Capital Growth
                      </p>
                      <p className="text-3xl font-bold text-blue-900">
                        {marketMetrics.growth}%
                      </p>
                      <p className="text-xs text-blue-700 flex items-center gap-1">
                        <TrendingUp className="w-3 h-3" />
                        5-Year CAGR
                      </p>
                    </div>
                    <div className="p-3 bg-blue-200 rounded-full">
                      <BarChart3 className="w-6 h-6 text-blue-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-purple-600 text-sm font-medium">
                        Rental Yield
                      </p>
                      <p className="text-3xl font-bold text-purple-900">
                        {marketMetrics.yield}%
                      </p>
                      <p className="text-xs text-purple-700 flex items-center gap-1">
                        <DollarSign className="w-3 h-3" />
                        Annual return
                      </p>
                    </div>
                    <div className="p-3 bg-purple-200 rounded-full">
                      <Building className="w-6 h-6 text-purple-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-orange-600 text-sm font-medium">
                        Risk Rating
                      </p>
                      <p className="text-3xl font-bold text-orange-900">Low</p>
                      <p className="text-xs text-orange-700 flex items-center gap-1">
                        <Shield className="w-3 h-3" />
                        Grade A investment
                      </p>
                    </div>
                    <div className="p-3 bg-orange-200 rounded-full">
                      <Award className="w-6 h-6 text-orange-700" />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* ROI Analysis Charts */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <BarChart3 className="w-5 h-5 text-[#0056D2]" />
                    ROI Performance Trend
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={300}>
                    <AreaChart data={marketData}>
                      <defs>
                        <linearGradient
                          id="colorROI"
                          x1="0"
                          y1="0"
                          x2="0"
                          y2="1"
                        >
                          <stop
                            offset="5%"
                            stopColor="#10B981"
                            stopOpacity={0.3}
                          />
                          <stop
                            offset="95%"
                            stopColor="#10B981"
                            stopOpacity={0}
                          />
                        </linearGradient>
                      </defs>
                      <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                      <XAxis dataKey="year" stroke="#666" />
                      <YAxis stroke="#666" />
                      <Tooltip
                        contentStyle={{
                          backgroundColor: "white",
                          border: "1px solid #e5e7eb",
                          borderRadius: "8px",
                          boxShadow: "0 4px 6px -1px rgba(0, 0, 0, 0.1)",
                        }}
                      />
                      <Area
                        type="monotone"
                        dataKey="demand"
                        stroke="#10B981"
                        strokeWidth={3}
                        fillOpacity={1}
                        fill="url(#colorROI)"
                      />
                    </AreaChart>
                  </ResponsiveContainer>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <PieChart className="w-5 h-5 text-[#0056D2]" />
                    ROI Distribution
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-4 bg-gradient-to-r from-green-50 to-green-100 rounded-lg">
                      <div className="flex items-center gap-3">
                        <div className="w-4 h-4 bg-green-500 rounded-full"></div>
                        <span className="font-medium">High ROI (&gt;15%)</span>
                      </div>
                      <span className="font-bold text-green-600">65%</span>
                    </div>
                    <div className="flex items-center justify-between p-4 bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg">
                      <div className="flex items-center gap-3">
                        <div className="w-4 h-4 bg-blue-500 rounded-full"></div>
                        <span className="font-medium">Medium ROI (10-15%)</span>
                      </div>
                      <span className="font-bold text-blue-600">25%</span>
                    </div>
                    <div className="flex items-center justify-between p-4 bg-gradient-to-r from-yellow-50 to-yellow-100 rounded-lg">
                      <div className="flex items-center gap-3">
                        <div className="w-4 h-4 bg-yellow-500 rounded-full"></div>
                        <span className="font-medium">
                          Stable ROI (&lt;10%)
                        </span>
                      </div>
                      <span className="font-bold text-yellow-600">10%</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Comprehensive Property Analysis */}
            <div className="space-y-6">
              {/* ROI Summary Cards */}
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
                  <CardContent className="p-4 text-center">
                    <div className="text-2xl font-bold text-green-600">
                      {propertyData.length
                        ? (
                            propertyData.reduce((sum, p) => sum + p.roi, 0) /
                            propertyData.length
                          ).toFixed(1)
                        : "0"}
                      %
                    </div>
                    <div className="text-sm text-green-700">
                      Average Total ROI
                    </div>
                  </CardContent>
                </Card>
                <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
                  <CardContent className="p-4 text-center">
                    <div className="text-2xl font-bold text-blue-600">
                      {propertyData.length
                        ? (
                            propertyData.reduce(
                              (sum, p) => sum + p.rentalYield,
                              0
                            ) / propertyData.length
                          ).toFixed(1)
                        : "0"}
                      %
                    </div>
                    <div className="text-sm text-blue-700">
                      Average Rental Yield
                    </div>
                  </CardContent>
                </Card>
                <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
                  <CardContent className="p-4 text-center">
                    <div className="text-2xl font-bold text-purple-600">
                      {propertyData.length
                        ? (
                            propertyData.reduce(
                              (sum, p) => sum + p.capitalAppreciation,
                              0
                            ) / propertyData.length
                          ).toFixed(1)
                        : "0"}
                      %
                    </div>
                    <div className="text-sm text-purple-700">
                      Capital Appreciation
                    </div>
                  </CardContent>
                </Card>
                <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
                  <CardContent className="p-4 text-center">
                    <div className="text-2xl font-bold text-orange-600">
                      ₹
                      {propertyData.length
                        ? Math.round(
                            propertyData.reduce(
                              (sum, p) => sum + p.pricePerSqft,
                              0
                            ) / propertyData.length
                          ).toLocaleString()
                        : "0"}
                    </div>
                    <div className="text-sm text-orange-700">
                      Avg Price/Sqft
                    </div>
                  </CardContent>
                </Card>
              </div>

              {/* Detailed Property ROI Analysis */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Star className="w-5 h-5 text-[#0056D2]" />
                    Property ROI Analysis - {selectedCity}
                  </CardTitle>
                  <p className="text-sm text-gray-600 mt-1">
                    Comprehensive ROI analysis based on rental yield and capital
                    appreciation potential
                  </p>
                </CardHeader>
                <CardContent>
                  {loading ? (
                    <div className="text-center py-8">
                      Loading property data...
                    </div>
                  ) : propertyData.length === 0 ? (
                    <div className="text-center py-8">
                      <div className="text-gray-500 mb-2">
                        No investment properties found for {selectedCity}
                      </div>
                      <div className="text-sm text-gray-400">
                        Try selecting a different city or check property
                        availability
                      </div>
                    </div>
                  ) : (
                    <div className="space-y-4">
                      {propertyData.map((property) => (
                        <div
                          key={property.id}
                          className="p-6 border rounded-xl bg-gradient-to-br from-white to-gray-50 hover:shadow-lg transition-all duration-300"
                        >
                          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                            {/* Property Info */}
                            <div className="lg:col-span-1">
                              <div className="flex items-start justify-between mb-3">
                                <div className="p-2 bg-blue-100 rounded-lg">
                                  <Building className="w-5 h-5 text-blue-600" />
                                </div>
                                <Badge
                                  className={`${
                                    property.investmentGrade === "A+"
                                      ? "bg-green-100 text-green-800"
                                      : property.investmentGrade === "A"
                                      ? "bg-blue-100 text-blue-800"
                                      : property.investmentGrade === "B+"
                                      ? "bg-yellow-100 text-yellow-800"
                                      : "bg-gray-100 text-gray-800"
                                  }`}
                                >
                                  Grade {property.investmentGrade}
                                </Badge>
                              </div>
                              <h4 className="font-bold text-lg text-gray-900 mb-2">
                                {property.title}
                              </h4>
                              <p className="text-sm text-gray-600 mb-2 flex items-center gap-1">
                                <MapPin className="w-4 h-4" />
                                {property.location}
                              </p>
                              <div className="text-sm text-gray-600 space-y-1">
                                <div>
                                  Type:{" "}
                                  {property.propertyType
                                    .charAt(0)
                                    .toUpperCase() +
                                    property.propertyType.slice(1)}
                                </div>
                                <div>
                                  Area: {property.builtUpArea.toLocaleString()}{" "}
                                  sqft
                                </div>
                                <div>
                                  Price: ₹
                                  {(property.salePrice / 100000).toFixed(1)}L
                                </div>
                              </div>
                            </div>

                            {/* ROI Breakdown */}
                            <div className="lg:col-span-1">
                              <h5 className="font-semibold text-gray-900 mb-3">
                                ROI Breakdown
                              </h5>
                              <div className="space-y-3">
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Total ROI
                                  </span>
                                  <span className="text-xl font-bold text-green-600">
                                    {property.roi}%
                                  </span>
                                </div>
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Rental Yield
                                  </span>
                                  <span className="font-semibold text-blue-600">
                                    {property.rentalYield}%
                                  </span>
                                </div>
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Capital Appreciation
                                  </span>
                                  <span className="font-semibold text-purple-600">
                                    {property.capitalAppreciation}%
                                  </span>
                                </div>
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Price per Sqft
                                  </span>
                                  <span className="font-semibold text-gray-900">
                                    ₹{property.pricePerSqft.toLocaleString()}
                                  </span>
                                </div>
                              </div>
                            </div>

                            {/* Investment Metrics */}
                            <div className="lg:col-span-1">
                              <h5 className="font-semibold text-gray-900 mb-3">
                                Investment Metrics
                              </h5>
                              <div className="space-y-3">
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Risk Level
                                  </span>
                                  <Badge
                                    className={`${
                                      property.riskLevel === "Low"
                                        ? "bg-green-100 text-green-800"
                                        : property.riskLevel === "Medium"
                                        ? "bg-yellow-100 text-yellow-800"
                                        : "bg-red-100 text-red-800"
                                    }`}
                                  >
                                    {property.riskLevel}
                                  </Badge>
                                </div>
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Potential
                                  </span>
                                  <Badge
                                    className={`${
                                      property.potential === "High"
                                        ? "bg-green-100 text-green-800"
                                        : property.potential === "Medium"
                                        ? "bg-blue-100 text-blue-800"
                                        : "bg-gray-100 text-gray-800"
                                    }`}
                                  >
                                    {property.potential}
                                  </Badge>
                                </div>
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Monthly Rent
                                  </span>
                                  <span className="font-semibold text-gray-900">
                                    {property.monthlyRent
                                      ? `₹${(
                                          property.monthlyRent / 1000
                                        ).toFixed(0)}K`
                                      : "Est. ₹" +
                                        Math.round(
                                          (property.salePrice *
                                            property.rentalYield) /
                                            1200000
                                        ) +
                                        "K"}
                                  </span>
                                </div>
                                <div className="flex justify-between items-center">
                                  <span className="text-sm text-gray-600">
                                    Payback Period
                                  </span>
                                  <span className="font-semibold text-gray-900">
                                    {Math.round(100 / property.rentalYield)}{" "}
                                    years
                                  </span>
                                </div>
                              </div>
                            </div>
                          </div>

                          {/* ROI Progress Bar */}
                          <div className="mt-4 pt-4 border-t">
                            <div className="flex justify-between text-sm text-gray-600 mb-2">
                              <span>ROI Performance</span>
                              <span>{property.roi}% Total Return</span>
                            </div>
                            <div className="w-full bg-gray-200 rounded-full h-3">
                              <div
                                className={`h-3 rounded-full transition-all duration-500 ${
                                  property.roi >= 15
                                    ? "bg-gradient-to-r from-green-400 to-green-600"
                                    : property.roi >= 10
                                    ? "bg-gradient-to-r from-blue-400 to-blue-600"
                                    : "bg-gradient-to-r from-yellow-400 to-yellow-600"
                                }`}
                                style={{
                                  width: `${Math.min(property.roi * 5, 100)}%`,
                                }}
                              ></div>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Market Heatmap */}
          <TabsContent value="heatmap" className="space-y-6">
            {/* Market Overview */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
              <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
                <CardContent className="p-6 text-center">
                  <div className="flex items-center justify-center mb-3">
                    <div className="p-3 bg-green-200 rounded-full">
                      <TrendingUp className="w-6 h-6 text-green-700" />
                    </div>
                  </div>
                  <div className="text-2xl font-bold text-green-900 mb-2">
                    {marketMetrics.momentum}
                  </div>
                  <div className="text-sm text-green-700 font-medium">
                    Market Momentum
                  </div>
                  <div className="text-xs text-green-600 mt-1">
                    +15% this quarter
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
                <CardContent className="p-6 text-center">
                  <div className="flex items-center justify-center mb-3">
                    <div className="p-3 bg-blue-200 rounded-full">
                      <DollarSign className="w-6 h-6 text-blue-700" />
                    </div>
                  </div>
                  <div className="text-2xl font-bold text-blue-900 mb-2">
                    ₹{marketMetrics.avgPrice.toLocaleString()}
                  </div>
                  <div className="text-sm text-blue-700 font-medium">
                    Avg Price/sqft
                  </div>
                  <div className="text-xs text-blue-600 mt-1">
                    +8% YoY growth
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
                <CardContent className="p-6 text-center">
                  <div className="flex items-center justify-center mb-3">
                    <div className="p-3 bg-orange-200 rounded-full">
                      <Activity className="w-6 h-6 text-orange-700" />
                    </div>
                  </div>
                  <div className="text-2xl font-bold text-orange-900 mb-2">
                    High
                  </div>
                  <div className="text-sm text-orange-700 font-medium">
                    Demand Level
                  </div>
                  <div className="text-xs text-orange-600 mt-1">
                    Above average
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
                <CardContent className="p-6 text-center">
                  <div className="flex items-center justify-center mb-3">
                    <div className="p-3 bg-purple-200 rounded-full">
                      <Shield className="w-6 h-6 text-purple-700" />
                    </div>
                  </div>
                  <div className="text-2xl font-bold text-purple-900 mb-2">
                    {marketMetrics.safetyScore}/100
                  </div>
                  <div className="text-sm text-purple-700 font-medium">
                    Safety Score
                  </div>
                  <div className="text-xs text-purple-600 mt-1">
                    Investment grade
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Interactive Heatmap */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <MapPin className="w-5 h-5 text-[#0056D2]" />
                  Interactive Investment Heatmap - {selectedCity}
                </CardTitle>
                <p className="text-sm text-gray-600 mt-2">
                  Explore property density and price distribution across
                  different areas. Click on zones to view detailed information.
                </p>
              </CardHeader>
              <CardContent>
                <div className="bg-gray-50 rounded-lg p-4">
                  <PropertyHeatmap properties={heatmapProperties} />
                </div>
              </CardContent>
            </Card>

            {/* Heatmap Insights */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Eye className="w-5 h-5 text-[#0056D2]" />
                    Key Insights
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-start gap-3 p-4 bg-blue-50 rounded-lg">
                      <div className="p-2 bg-blue-200 rounded-full">
                        <TrendingUp className="w-4 h-4 text-blue-700" />
                      </div>
                      <div>
                        <h4 className="font-semibold text-blue-900">
                          High Density Zones
                        </h4>
                        <p className="text-sm text-blue-700">
                          Central areas show 3x higher property concentration
                          with premium pricing
                        </p>
                      </div>
                    </div>
                    <div className="flex items-start gap-3 p-4 bg-green-50 rounded-lg">
                      <div className="p-2 bg-green-200 rounded-full">
                        <Target className="w-4 h-4 text-green-700" />
                      </div>
                      <div>
                        <h4 className="font-semibold text-green-900">
                          Emerging Areas
                        </h4>
                        <p className="text-sm text-green-700">
                          Peripheral zones offer 25% better ROI potential with
                          lower entry costs
                        </p>
                      </div>
                    </div>
                    <div className="flex items-start gap-3 p-4 bg-purple-50 rounded-lg">
                      <div className="p-2 bg-purple-200 rounded-full">
                        <Award className="w-4 h-4 text-purple-700" />
                      </div>
                      <div>
                        <h4 className="font-semibold text-purple-900">
                          Investment Hotspots
                        </h4>
                        <p className="text-sm text-purple-700">
                          IT corridors and metro connectivity drive highest
                          appreciation rates
                        </p>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <BarChart3 className="w-5 h-5 text-[#0056D2]" />
                    Price Heat Distribution
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex justify-between items-center p-3 bg-red-50 rounded-lg border border-red-200">
                      <div className="flex items-center gap-3">
                        <div className="w-4 h-4 bg-red-500 rounded-full"></div>
                        <span className="font-medium text-red-900">
                          Premium (&gt;Rs.8000/sqft)
                        </span>
                      </div>
                      <span className="font-bold text-red-600">15%</span>
                    </div>
                    <div className="flex justify-between items-center p-3 bg-orange-50 rounded-lg border border-orange-200">
                      <div className="flex items-center gap-3">
                        <div className="w-4 h-4 bg-orange-500 rounded-full"></div>
                        <span className="font-medium text-orange-900">
                          High (Rs.6000-8000/sqft)
                        </span>
                      </div>
                      <span className="font-bold text-orange-600">35%</span>
                    </div>
                    <div className="flex justify-between items-center p-3 bg-yellow-50 rounded-lg border border-yellow-200">
                      <div className="flex items-center gap-3">
                        <div className="w-4 h-4 bg-yellow-500 rounded-full"></div>
                        <span className="font-medium text-yellow-900">
                          Medium (Rs.4000-6000/sqft)
                        </span>
                      </div>
                      <span className="font-bold text-yellow-600">35%</span>
                    </div>
                    <div className="flex justify-between items-center p-3 bg-green-50 rounded-lg border border-green-200">
                      <div className="flex items-center gap-3">
                        <div className="w-4 h-4 bg-green-500 rounded-full"></div>
                        <span className="font-medium text-green-900">
                          Affordable (&lt;Rs.4000/sqft)
                        </span>
                      </div>
                      <span className="font-bold text-green-600">15%</span>
                    </div>
                  </div>
                  <div className="mt-4 p-3 bg-gray-50 rounded-lg">
                    <p className="text-xs text-gray-600">
                      <strong>Note:</strong> Heatmap data is updated in
                      real-time based on current market transactions and
                      property listings.
                    </p>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}