import React, { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { 
  Shield, 
  FileText, 
  Scale, 
  CreditCard, 
  MapPin, 
  CheckCircle,
  AlertTriangle,
  Clock,
  Upload,
  Download,
  Star,
  TrendingUp,
  Users,
  Building,
  Search,
  Eye,
  Lock,
  Award
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { toast } from 'sonner';
// import { legalHealthAPI } from '@/services/legalHealthAPI';

const LegalHealth = () => {
  const [reports, setReports] = useState([
    {
      id: '1',
      report_number: 'LHR-2024-001',
      property_address: '123 MG Road, Bangalore, Karnataka 560001',
      property_type: 'apartment',
      report_status: 'completed',
      overall_score: 85,
      risk_level: 'low',
      recommendation: 'Safe to invest',
      created_at: '2024-01-15T10:30:00Z',
      completed_at: '2024-01-22T16:45:00Z'
    },
    {
      id: '2', 
      report_number: 'LHR-2024-002',
      property_address: '456 Brigade Road, Bangalore, Karnataka 560025',
      property_type: 'villa',
      report_status: 'in_progress',
      overall_score: 0,
      risk_level: 'unknown',
      created_at: '2024-01-20T14:20:00Z'
    }
  ]);
  const [selectedReport, setSelectedReport] = useState(null);
  const [loading, setLoading] = useState(false);
  const [dashboardStats, setDashboardStats] = useState({
    total_reports: 2,
    completed_reports: 1,
    in_progress_reports: 1,
    risk_distribution: { low: 1, medium: 0, high: 0, critical: 0 }
  });
  const [showCreateDialog, setShowCreateDialog] = useState(false);

  // Verification modules configuration
  const verificationModules = [
    {
      id: 'ownership',
      title: 'Ownership Verification',
      description: 'Complete verification of current and historical ownership with title clarity assessment.',
      icon: Shield,
      color: 'bg-blue-500',
      checks: [
        'Current owner identity verification and authorization',
        'Chain of ownership tracking from original allotment',
        'Title deed authenticity and registration verification',
        'Power of attorney validation and legal authority',
        'Third-party claims and ownership disputes check'
      ]
    },
    {
      id: 'documents',
      title: 'Document Validation',
      description: 'Comprehensive verification of all legal documents and certificates required for property transaction.',
      icon: FileText,
      color: 'bg-green-500',
      checks: [
        'Sale deed authenticity and registration status',
        'Khata certificate and property tax records',
        'Encumbrance certificate (EC) for 30+ years',
        'Mother deed and original allotment documents',
        'NOC from society/builder and completion certificate'
      ]
    },
    {
      id: 'litigation',
      title: 'Litigation & Dispute Check',
      description: 'Thorough screening for ongoing and past legal disputes, court cases, and RERA complaints.',
      icon: Scale,
      color: 'bg-red-500',
      checks: [
        'High Court and Supreme Court case search',
        'Civil and criminal case history investigation',
        'RERA complaints and regulatory violations',
        'Society disputes and maintenance issues',
        'Pending litigation status and resolution timeline'
      ]
    },
    {
      id: 'encumbrance',
      title: 'Encumbrance & Liability Assessment',
      description: 'Detailed analysis of financial encumbrances, loans, and liabilities attached to the property.',
      icon: CreditCard,
      color: 'bg-yellow-500',
      checks: [
        'Existing home loans and mortgage verification',
        'Property tax dues and municipal liabilities',
        'Society maintenance dues and pending charges',
        'Legal liens and third-party claims assessment',
        'Clear title certification and liability clearance'
      ]
    },
    {
      id: 'zoning',
      title: 'Land Use & Zoning Compliance',
      description: 'Verification of land use permissions, zoning compliance, and master plan alignment.',
      icon: MapPin,
      color: 'bg-purple-500',
      checks: [
        'Master plan compliance and zoning verification',
        'Land use conversion and permission status',
        'BBMP/BDA approvals and layout sanctions',
        'Environmental clearance and NOC verification',
        'Compliance certificate and regulatory approval'
      ]
    },
    {
      id: 'project_approval',
      title: 'Project Approval Status',
      description: 'Complete verification of project approvals, licenses, and regulatory compliance status.',
      icon: Building,
      color: 'bg-indigo-500',
      checks: [
        'RERA registration and project approval status',
        'Building plan approval and construction permit',
        'Occupancy certificate (OC) and completion status',
        'Commencement certificate (CC) and timeline compliance',
        'Fire safety and structural stability certificates'
      ]
    }
  ];

  useEffect(() => {
    // fetchDashboardData();
    // fetchReports();
  }, []);

  const fetchDashboardData = async () => {
    try {
      // const response = await legalHealthAPI.getDashboardStats();
      // setDashboardStats(response.data);
    } catch (error) {
      console.error('Error fetching dashboard data:', error);
    }
  };

  const fetchReports = async () => {
    try {
      setLoading(true);
      // const response = await legalHealthAPI.getUserReports();
      // setReports(response.data);
    } catch (error) {
      console.error('Error fetching reports:', error);
      toast.error('Failed to fetch reports');
    } finally {
      setLoading(false);
    }
  };

  const handleCreateReport = async (formData) => {
    try {
      setLoading(true);
      
      // Simulate API call delay
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      // Create new report with mock data
      const newReport = {
        id: Date.now().toString(),
        report_number: `LHR-2024-${String(reports.length + 1).padStart(3, '0')}`,
        property_address: formData.property_address,
        property_type: formData.property_type,
        report_status: 'initiated',
        overall_score: 0,
        risk_level: 'unknown',
        created_at: new Date().toISOString()
      };
      
      // Add to reports list
      setReports(prev => [newReport, ...prev]);
      
      // Update dashboard stats
      setDashboardStats(prev => ({
        ...prev,
        total_reports: prev.total_reports + 1,
        in_progress_reports: prev.in_progress_reports + 1
      }));
      
      toast.success('Legal health report created successfully!');
      setShowCreateDialog(false);
    } catch (error) {
      console.error('Error creating report:', error);
      toast.error('Failed to create report');
    } finally {
      setLoading(false);
    }
  };

  const handleViewDetails = (report) => {
    setSelectedReport(report);
    toast.info(`Viewing details for ${report.report_number}`);
    // Will show detailed report view
  };

  const handleDownload = (report) => {
    toast.success(`Downloading report ${report.report_number}`);
    // Will trigger PDF download
  };

  const getRiskLevelColor = (riskLevel) => {
    switch (riskLevel) {
      case 'low': return 'bg-green-100 text-green-800';
      case 'medium': return 'bg-yellow-100 text-yellow-800';
      case 'high': return 'bg-orange-100 text-orange-800';
      case 'critical': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'completed': return 'bg-green-100 text-green-800';
      case 'in_progress': return 'bg-blue-100 text-blue-800';
      case 'initiated': return 'bg-yellow-100 text-yellow-800';
      case 'failed': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <>
      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-green-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-blue-900 via-blue-800 to-green-800 text-black">
          <div className="container mx-auto px-4 py-16">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6 }}
              className="text-center"
            >
              <h1 className="text-5xl font-bold mb-6 text-black">
                Legal Health & Title Status Report
              </h1>
              <p className="text-xl mb-8 max-w-3xl mx-auto text-black">
                Comprehensive legal safety verification through ownership verification, 
                documentation review, encumbrance checks, and land-use compliance analysis.
              </p>
              <div className="flex flex-wrap justify-center gap-4 mb-8">
                <Badge variant="secondary" className="text-lg px-4 py-2 bg-white text-black">
                  <Shield className="w-5 h-5 mr-2" />
                  Complete Legal Safety
                </Badge>
                <Badge variant="secondary" className="text-lg px-4 py-2 bg-white text-black">
                  <CheckCircle className="w-5 h-5 mr-2" />
                  Clear Title Verification
                </Badge>
                <Badge variant="secondary" className="text-lg px-4 py-2 bg-white text-black">
                  <MapPin className="w-5 h-5 mr-2" />
                  Regulatory Compliance
                </Badge>
              </div>
              
              <Dialog open={showCreateDialog} onOpenChange={setShowCreateDialog}>
                <DialogTrigger asChild>
                  <Button size="lg" className="bg-white text-blue-900 hover:bg-gray-100">
                    <FileText className="w-5 h-5 mr-2" />
                    Order Report Now - Starting from ₹9,999
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-md">
                  <DialogHeader>
                    <DialogTitle>Create Legal Health Report</DialogTitle>
                  </DialogHeader>
                  <CreateReportForm onSubmit={handleCreateReport} loading={loading} />
                </DialogContent>
              </Dialog>
            </motion.div>
          </div>
        </div>

        <div className="container mx-auto px-4 py-12">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 text-center">
              <Users className="w-12 h-12 text-[#0056D2] mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Ownership Verification</h3>
              <p className="text-sm text-gray-600">Title clarity & ownership tracking</p>
            </div>
            
            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 text-center">
              <FileText className="w-12 h-12 text-[#0056D2] mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Document Validation</h3>
              <p className="text-sm text-gray-600">Legal papers & certificate check</p>
            </div>
            
            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 text-center">
              <AlertTriangle className="w-12 h-12 text-[#0056D2] mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Litigation Check</h3>
              <p className="text-sm text-gray-600">Court cases & dispute screening</p>
            </div>
            
            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 text-center">
              <Shield className="w-12 h-12 text-[#0056D2] mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Compliance Status</h3>
              <p className="text-sm text-gray-600">RERA & regulatory approvals</p>
            </div>
          </div>
        </div>

        {/* Detailed Analysis Modules */}
        <div className="bg-white py-12">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="space-y-12">
              
              {/* 1. Ownership Verification */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                <div>
                  <div className="flex items-center mb-4">
                    <div className="bg-[#0056D2] text-white rounded-full w-8 h-8 flex items-center justify-center text-sm font-bold mr-3">1</div>
                    <h3 className="text-2xl font-bold text-gray-900">Ownership Verification</h3>
                  </div>
                  <p className="text-gray-600 mb-6">Complete verification of current and historical ownership with title clarity assessment.</p>
                  <ul className="space-y-3">
                    <li className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Current owner identity verification and authorization</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Chain of ownership tracking from original allotment</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Title deed authenticity and registration verification</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Power of attorney validation and legal authority</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Third-party claims and ownership disputes check</span>
                    </li>
                  </ul>
                </div>
                <div className="bg-gradient-to-br from-blue-50 to-indigo-50 p-8 rounded-2xl">
                  <img
                    src="https://images.unsplash.com/photo-1589829545856-d10d557cf95f?w=500&h=350&fit=crop"
                    alt="Ownership Verification Process"
                    className="w-full h-64 object-cover rounded-xl shadow-lg"
                  />
                </div>
              </div>

              {/* 2. Document Validation */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                <div className="bg-gradient-to-br from-green-50 to-emerald-50 p-8 rounded-2xl order-2 lg:order-1">
                  <img
                    src="https://images.unsplash.com/photo-1554224155-8d04cb21cd6c?w=500&h=350&fit=crop"
                    alt="Document Validation"
                    className="w-full h-64 object-cover rounded-xl shadow-lg"
                  />
                </div>
                <div className="order-1 lg:order-2">
                  <div className="flex items-center mb-4">
                    <div className="bg-[#0056D2] text-white rounded-full w-8 h-8 flex items-center justify-center text-sm font-bold mr-3">2</div>
                    <h3 className="text-2xl font-bold text-gray-900">Document Validation</h3>
                  </div>
                  <p className="text-gray-600 mb-6">Comprehensive verification of all legal documents and certificates required for property transaction.</p>
                  <ul className="space-y-3">
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Sale deed authenticity and registration status</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Khata certificate and property tax records</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Encumbrance certificate (EC) for 30+ years</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Mother deed and original allotment documents</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">NOC from society/builder and completion certificate</span>
                    </li>
                  </ul>
                </div>
              </div>

              {/* 3. Litigation & Dispute Check */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                <div>
                  <div className="flex items-center mb-4">
                    <div className="bg-[#0056D2] text-white rounded-full w-8 h-8 flex items-center justify-center text-sm font-bold mr-3">3</div>
                    <h3 className="text-2xl font-bold text-gray-900">Litigation & Dispute Check</h3>
                  </div>
                  <p className="text-gray-600 mb-6">Thorough screening for ongoing and past legal disputes, court cases, and RERA complaints.</p>
                  <ul className="space-y-3">
                    <li className="flex items-start gap-3">
                      <Search className="w-5 h-5 text-orange-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">High Court and Supreme Court case search</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <AlertTriangle className="w-5 h-5 text-red-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Civil and criminal case history investigation</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Building className="w-5 h-5 text-purple-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">RERA complaints and regulatory violations</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Users className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Society disputes and maintenance issues</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Eye className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Pending litigation status and resolution timeline</span>
                    </li>
                  </ul>
                </div>
                <div className="bg-gradient-to-br from-purple-50 to-pink-50 p-8 rounded-2xl">
                  <img
                    src="https://images.unsplash.com/photo-1505664194779-8beaceb93744?w=500&h=350&fit=crop"
                    alt="Legal Dispute Analysis"
                    className="w-full h-64 object-cover rounded-xl shadow-lg"
                  />
                </div>
              </div>

              {/* 4. Encumbrance & Liability Assessment */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                <div className="bg-gradient-to-br from-orange-50 to-red-50 p-8 rounded-2xl order-2 lg:order-1">
                  <img
                    src="https://images.unsplash.com/photo-1450101499163-c8848c66ca85?w=500&h=350&fit=crop"
                    alt="Financial Liability Check"
                    className="w-full h-64 object-cover rounded-xl shadow-lg"
                  />
                </div>
                <div className="order-1 lg:order-2">
                  <div className="flex items-center mb-4">
                    <div className="bg-[#0056D2] text-white rounded-full w-8 h-8 flex items-center justify-center text-sm font-bold mr-3">4</div>
                    <h3 className="text-2xl font-bold text-gray-900">Encumbrance & Liability Assessment</h3>
                  </div>
                  <p className="text-gray-600 mb-6">Detailed analysis of financial encumbrances, loans, and liabilities attached to the property.</p>
                  <ul className="space-y-3">
                    <li className="flex items-start gap-3">
                      <Lock className="w-5 h-5 text-red-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Existing home loans and mortgage verification</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-orange-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Property tax dues and municipal liabilities</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Building className="w-5 h-5 text-purple-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Society maintenance dues and pending charges</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Shield className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Legal liens and third-party claims assessment</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Clear title certification and liability clearance</span>
                    </li>
                  </ul>
                </div>
              </div>

              {/* 5. Land Use & Zoning Compliance */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                <div>
                  <div className="flex items-center mb-4">
                    <div className="bg-[#0056D2] text-white rounded-full w-8 h-8 flex items-center justify-center text-sm font-bold mr-3">5</div>
                    <h3 className="text-2xl font-bold text-gray-900">Land Use & Zoning Compliance</h3>
                  </div>
                  <p className="text-gray-600 mb-6">Verification of land use permissions, zoning compliance, and master plan alignment.</p>
                  <ul className="space-y-3">
                    <li className="flex items-start gap-3">
                      <MapPin className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Master plan compliance and zoning verification</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Building className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Land use conversion and permission status</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Shield className="w-5 h-5 text-purple-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">BBMP/BDA approvals and layout sanctions</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-orange-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Environmental clearance and NOC verification</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Award className="w-5 h-5 text-yellow-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Compliance certificate and regulatory approval</span>
                    </li>
                  </ul>
                </div>
                <div className="bg-gradient-to-br from-teal-50 to-cyan-50 p-8 rounded-2xl">
                  <img
                    src="https://images.unsplash.com/photo-1449824913935-59a10b8d2000?w=500&h=350&fit=crop"
                    alt="Zoning Compliance Check"
                    className="w-full h-64 object-cover rounded-xl shadow-lg"
                  />
                </div>
              </div>

              {/* 6. Project Approval Status */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                <div className="bg-gradient-to-br from-indigo-50 to-purple-50 p-8 rounded-2xl order-2 lg:order-1">
                  <img
                    src="https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?w=500&h=350&fit=crop"
                    alt="Project Approval Verification"
                    className="w-full h-64 object-cover rounded-xl shadow-lg"
                  />
                </div>
                <div className="order-1 lg:order-2">
                  <div className="flex items-center mb-4">
                    <div className="bg-[#0056D2] text-white rounded-full w-8 h-8 flex items-center justify-center text-sm font-bold mr-3">6</div>
                    <h3 className="text-2xl font-bold text-gray-900">Project Approval Status</h3>
                  </div>
                  <p className="text-gray-600 mb-6">Complete verification of project approvals, licenses, and regulatory compliance status.</p>
                  <ul className="space-y-3">
                    <li className="flex items-start gap-3">
                      <Award className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">RERA registration and project approval status</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Building className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Building plan approval and construction permit</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <Shield className="w-5 h-5 text-purple-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Occupancy certificate (OC) and completion status</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <FileText className="w-5 h-5 text-orange-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Commencement certificate (CC) and timeline compliance</span>
                    </li>
                    <li className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">Fire safety and structural stability certificates</span>
                    </li>
                  </ul>
                </div>
              </div>

              {/* Final Legal Health Certificate */}
              <div className="bg-[#0056D2] text-white p-12 rounded-2xl text-center">
                <div className="flex items-center justify-center mb-6">
                  <div className="bg-white text-[#0056D2] rounded-full w-12 h-12 flex items-center justify-center text-lg font-bold mr-4">✓</div>
                  <h3 className="text-3xl font-bold">Legal Health Certificate</h3>
                </div>
                <p className="text-xl text-blue-100 mb-8 max-w-3xl mx-auto">
                  Comprehensive legal safety certification with clear investment recommendation and risk assessment
                </p>
                <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-6">
                  <div className="bg-white/10 p-4 rounded-lg">
                    <Users className="w-8 h-8 mx-auto mb-2" />
                    <p className="text-sm font-medium">Title Clarity</p>
                  </div>
                  <div className="bg-white/10 p-4 rounded-lg">
                    <FileText className="w-8 h-8 mx-auto mb-2" />
                    <p className="text-sm font-medium">Document Status</p>
                  </div>
                  <div className="bg-white/10 p-4 rounded-lg">
                    <AlertTriangle className="w-8 h-8 mx-auto mb-2" />
                    <p className="text-sm font-medium">Litigation Free</p>
                  </div>
                  <div className="bg-white/10 p-4 rounded-lg">
                    <Lock className="w-8 h-8 mx-auto mb-2" />
                    <p className="text-sm font-medium">No Encumbrance</p>
                  </div>
                  <div className="bg-white/10 p-4 rounded-lg">
                    <MapPin className="w-8 h-8 mx-auto mb-2" />
                    <p className="text-sm font-medium">Zoning Compliant</p>
                  </div>
                  <div className="bg-white/10 p-4 rounded-lg">
                    <Shield className="w-8 h-8 mx-auto mb-2" />
                    <p className="text-sm font-medium">RERA Approved</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div className="container mx-auto px-4 py-12">
          <Tabs defaultValue="overview" className="space-y-8">
            <TabsList className="grid w-full grid-cols-4">
              <TabsTrigger value="overview">Overview</TabsTrigger>
              <TabsTrigger value="reports">My Reports ({reports.length})</TabsTrigger>
              <TabsTrigger value="verification">Verification Process</TabsTrigger>
              <TabsTrigger value="certificate">Certificate</TabsTrigger>
            </TabsList>

            {/* Overview Tab */}
            <TabsContent value="overview" className="space-y-8">
              {dashboardStats && (
                <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                  <Card>
                    <CardContent className="p-6 text-center">
                      <div className="text-3xl font-bold text-blue-600">{dashboardStats.total_reports}</div>
                      <p className="text-gray-600">Total Reports</p>
                    </CardContent>
                  </Card>
                  <Card>
                    <CardContent className="p-6 text-center">
                      <div className="text-3xl font-bold text-green-600">{dashboardStats.completed_reports}</div>
                      <p className="text-gray-600">Completed</p>
                    </CardContent>
                  </Card>
                  <Card>
                    <CardContent className="p-6 text-center">
                      <div className="text-3xl font-bold text-orange-600">{dashboardStats.in_progress_reports}</div>
                      <p className="text-gray-600">In Progress</p>
                    </CardContent>
                  </Card>
                  <Card>
                    <CardContent className="p-6 text-center">
                      <div className="text-3xl font-bold text-purple-600">7-10</div>
                      <p className="text-gray-600">Days Delivery</p>
                    </CardContent>
                  </Card>
                </div>
              )}
            </TabsContent>

            {/* My Reports Tab */}
            <TabsContent value="reports" className="space-y-8">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center justify-between">
                    <span>My Legal Health Reports</span>
                    <Button onClick={() => setShowCreateDialog(true)}>
                      <FileText className="w-4 h-4 mr-2" />
                      Create New Report
                    </Button>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {reports.map((report) => (
                      <Card key={report.id} className="hover:shadow-md transition-shadow">
                        <CardContent className="p-6">
                          <div className="flex items-start justify-between">
                            <div className="flex-1">
                              <div className="flex items-center gap-3 mb-2">
                                <h3 className="font-semibold text-lg">{report.report_number}</h3>
                                <Badge className={getStatusColor(report.report_status)}>
                                  {report.report_status.replace('_', ' ').toUpperCase()}
                                </Badge>
                                {report.risk_level !== 'unknown' && (
                                  <Badge className={getRiskLevelColor(report.risk_level)}>
                                    {report.risk_level.toUpperCase()} RISK
                                  </Badge>
                                )}
                              </div>
                              <p className="text-gray-600 mb-2">{report.property_address}</p>
                              <div className="flex items-center gap-4 text-sm text-gray-500">
                                <span className="flex items-center">
                                  <Building className="w-4 h-4 mr-1" />
                                  {report.property_type.charAt(0).toUpperCase() + report.property_type.slice(1)}
                                </span>
                                <span className="flex items-center">
                                  <Clock className="w-4 h-4 mr-1" />
                                  Created {new Date(report.created_at).toLocaleDateString()}
                                </span>
                                {report.overall_score > 0 && (
                                  <span className="flex items-center">
                                    <Star className="w-4 h-4 mr-1" />
                                    Score: {report.overall_score}/100
                                  </span>
                                )}
                              </div>
                              {report.recommendation && (
                                <p className="text-sm text-green-700 mt-2 font-medium">
                                  ✓ {report.recommendation}
                                </p>
                              )}
                            </div>
                            <div className="flex flex-col gap-2">
                              <Button variant="outline" size="sm" onClick={() => handleViewDetails(report)}>
                                View Details
                              </Button>
                              {report.report_status === 'completed' && (
                                <Button size="sm" onClick={() => handleDownload(report)}>
                                  <Download className="w-4 h-4 mr-2" />
                                  Download
                                </Button>
                              )}
                            </div>
                          </div>
                          {report.report_status === 'in_progress' && (
                            <div className="mt-4">
                              <div className="flex items-center justify-between text-sm mb-2">
                                <span>Verification Progress</span>
                                <span>40%</span>
                              </div>
                              <Progress value={40} className="h-2" />
                            </div>
                          )}
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </>
  );
};

// Create Report Form Component
const CreateReportForm = ({ onSubmit, loading }) => {
  const [formData, setFormData] = useState({
    property_address: '',
    property_type: '',
    property_id: null
  });

  const handleSubmit = (e) => {
    e.preventDefault();
    onSubmit(formData);
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="property_address">Property Address</Label>
        <Textarea
          id="property_address"
          placeholder="Enter complete property address..."
          value={formData.property_address}
          onChange={(e) => setFormData({ ...formData, property_address: e.target.value })}
          required
          className="mt-1"
        />
      </div>
      
      <div>
        <Label htmlFor="property_type">Property Type</Label>
        <Select 
          value={formData.property_type} 
          onValueChange={(value) => setFormData({ ...formData, property_type: value })}
        >
          <SelectTrigger className="mt-1">
            <SelectValue placeholder="Select property type" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="apartment">Apartment</SelectItem>
            <SelectItem value="villa">Villa</SelectItem>
            <SelectItem value="plot">Plot</SelectItem>
            <SelectItem value="commercial">Commercial</SelectItem>
            <SelectItem value="warehouse">Warehouse</SelectItem>
          </SelectContent>
        </Select>
      </div>
      
      <Button type="submit" className="w-full" disabled={loading}>
        {loading ? (
          <>
            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
            Creating Report...
          </>
        ) : (
          <>
            <FileText className="w-4 h-4 mr-2" />
            Create Report - ₹9,999
          </>
        )}
      </Button>
    </form>
  );
};

export { LegalHealth };