// LocalityIntelligence.tsx
import React, { useEffect, useMemo, useRef, useState } from "react";
import {
  Shield, MapPin, Building, TrendingUp, Users, FileText, Download, Search, BarChart3, Car, GraduationCap, Heart, ShoppingBag, TreePine,
  CheckCircle, AlertTriangle, Calendar, UserCheck, Folder, TrendingDown, Smile, Navigation, Waves, Scale, BookOpen
} from "lucide-react";
import { Button } from "../ui/button";
import { useNavigate } from "react-router-dom";
import { googleMapsService, PlaceDetails, NearbyPlace } from "../../services/googleMaps";
import { propertyApiService } from "../../services/propertyApi";
import { GoogleMap } from "../GoogleMap";
import { HeatmapLayer } from "../HeatmapLayer";

import {
  LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, BarChart, Bar, RadarChart, PolarGrid, PolarAngleAxis, PolarRadiusAxis, Radar, Legend,
} from "recharts";

/*
  NOTES:
  - Dependencies: react-leaflet, leaflet, recharts, (html2canvas, jspdf used via dynamic import)
  - Replace mock fetch functions (fetchLocalityReport, fetchNearbyLocalities) with real API calls
*/

// All data now comes from GROQ AI - no static data

// Types
type LocalityData = {
  name: string;
  center: { lat: number; lng: number };
  pricePerSqft: number;
  oneYearGrowth: number;
  fiveYearGrowth: number;
  rentalYield: number;
  avgRent: number;
  ribl: number;
  demand: string;
  overallScore: number;
  safetyIndex: number;
  connectivityScore: number;
  amenitiesScore: number;
  environmentScore: number;
  growthPotential: number;
  heatmapPoints: Array<{ lat: number; lng: number; intensity: number }>;
  priceTrend: Array<{ year: string; price: number }>;
  amenities: Array<{
    type: string;
    name: string;
    distKm: number;
    rating: number;
    travelTime: string;
  }>;
  safety: {
    crimeRatePerMonth: number;
    policeStationDistKm: number;
    nightSafety: number;
    streetLighting: string;
    cctvCoverage: string;
    emergencyResponse: string;
    crimeStats: string;
  };
  connectivity: {
    metroDistance: number;
    busStops: number;
    trafficScore: number;
    roadQuality: string;
    airportDistance: number;
    railwayDistance: number;
    peakTrafficTime: string;
  };
  lifestyle: {
    malls: string[];
    restaurants: string[];
    entertainment: string[];
    parks: string[];
  };
  environment: {
    aqi: number;
    noiseDb: number;
    greenCoverPct: number;
    waterQuality: string;
    airQualityTrend: string;
    pollutionSources: string;
  };
  upcomingDevelopments: Array<{
    project: string;
    timeline: string;
    impact: string;
  }>;
  demographics: {
    avgIncomeLakhs: number;
    renterRatioPct: number;
    ageMedian: number;
  };
  transitScore?: number;
  transitBreakdown?: {
    metro: number;
    bus: number;
    auto: number;
    frequency: number;
  };
  heatmapLayers?: {
    price: Array<{ lat: number; lng: number; value: number }>;
    demand: Array<{ lat: number; lng: number; value: number }>;
    rental: Array<{ lat: number; lng: number; value: number }>;
    crime: Array<{ lat: number; lng: number; value: number }>;
    flood: Array<{ lat: number; lng: number; value: number }>;
  };
};

// Dynamic locality suggestions powered by GROQ AI
async function getLocalitySuggestions(query: string): Promise<string[]> {
  if (!query || query.length < 2) return [];
  
  try {
    const suggestions = await groqService.getLocalitySuggestions(query);
    return suggestions;
  } catch (error) {
    console.error('Failed to get locality suggestions:', error);
    return []; // No fallback - pure GROQ only
  }
}

// Report type configurations
const REPORT_TYPES = {
  comprehensive: { label: "Comprehensive Analysis", focus: "all" },
  investment: { label: "Investment Focus", focus: "growth,rental,ribl" },
  family: { label: "Family Living", focus: "safety,schools,amenities" },
  commercial: {
    label: "Commercial Potential",
    focus: "connectivity,growth,commercial",
  },
  nri: { label: "NRI Investment", focus: "investment,legal,tax" },
  "first-time": {
    label: "First-time Buyer",
    focus: "affordability,basics,guidance",
  },
};

// -------- Enhanced API fetch functions with Google Maps integration --------
import { groqService } from '../../services/groqService';
import { cacheUtils } from '../../utils/localityCache';

// All data now comes from GROQ AI - no hardcoded location data

// All data generation now handled by GROQ AI - no hardcoded fallbacks

async function fetchLocalityReportByType(
  localityName: string,
  reportType: string
): Promise<LocalityData> {
  // Check cache first for consistent data
  
  // Check cache first for consistent data
  const cachedData = cacheUtils.getLocality(localityName, reportType);
  if (cachedData) {
    console.log(`📦 Using cached data for ${localityName} (${reportType})`);
    return cachedData;
  }
  
  try {
    // First try Groq AI for real-time analysis
    console.log('🤖 Fetching AI-powered locality data from Groq...');
    const groqData = await groqService.getLocalityData(localityName, reportType);
    
    // Get location data from Google Maps for coordinates
    const placeDetails = await googleMapsService.geocodeLocation(localityName);
    if (placeDetails) {
      groqData.center = {
        lat: placeDetails.geometry.location.lat,
        lng: placeDetails.geometry.location.lng
      };
    }
    
    // Enhance with real amenities from Google Maps
    if (placeDetails) {
      const amenityTypes = getAmenityTypesByReportType(reportType);
      const amenityPromises = amenityTypes.map(type => 
        googleMapsService.getNearbyPlaces(
          placeDetails.geometry.location.lat, 
          placeDetails.geometry.location.lng, 
          type.googleType, 
          type.radius
        )
      );
      
      const amenityResults = await Promise.all(amenityPromises);
      
      const realAmenities = amenityResults.flatMap((places, index) => 
        places.slice(0, amenityTypes[index].limit).map(place => ({
          type: amenityTypes[index].displayName,
          name: place.name,
          distKm: Number((place.distance || 0).toFixed(1)),
          rating: place.rating || 4.0,
          travelTime: `${Math.ceil((place.distance || 0) * 3)} min`
        }))
      );
      
      if (realAmenities.length > 0) {
        // Merge Groq amenities with Google Maps amenities
        groqData.amenities = [...(groqData.amenities || []), ...realAmenities];
      }
    }
    
    console.log('✅ Successfully fetched AI-powered data');
    
    // Cache the data for consistency
    cacheUtils.setLocality(localityName, groqData, reportType);
    
    return groqData;
    
  } catch (error) {
    console.error('❌ Groq API failed, falling back to property data:', error);
    
    // Fallback to existing property API approach
    try {
      const properties = await propertyApiService.getPropertiesByLocation(localityName);
      const placeDetails = await googleMapsService.geocodeLocation(localityName);
      
      let localityData: LocalityData;
      
      if (properties.length > 0) {
        localityData = propertyApiService.generateLocalityDataFromProperties(
          localityName, 
          properties, 
          placeDetails
        );
      } else {
        throw new Error('No property data available - using GROQ AI only');
      }
      
      const customizedData = customizeDataByReportType(localityData, reportType);
      
      // Cache the fallback data too
      cacheUtils.setLocality(localityName, customizedData, reportType);
      
      return customizedData;
      
    } catch (fallbackError) {
      console.error('❌ All data sources failed:', fallbackError);
      throw new Error('Unable to fetch locality data - please try again');
    }
  }
}

// Helper function to get amenity types based on report type
function getAmenityTypesByReportType(reportType: string) {
  const baseAmenities = [
    { googleType: 'school', displayName: 'School', radius: 3000, limit: 3 },
    { googleType: 'hospital', displayName: 'Hospital', radius: 3000, limit: 2 }
  ];
  
  switch (reportType) {
    case 'family':
      return [
        ...baseAmenities,
        { googleType: 'park', displayName: 'Park', radius: 2000, limit: 3 },
        { googleType: 'shopping_mall', displayName: 'Mall', radius: 5000, limit: 2 }
      ];
    case 'commercial':
      return [
        { googleType: 'shopping_mall', displayName: 'Mall', radius: 5000, limit: 4 },
        { googleType: 'restaurant', displayName: 'Restaurant', radius: 2000, limit: 5 },
        { googleType: 'bank', displayName: 'Bank', radius: 3000, limit: 3 }
      ];
    case 'investment':
    case 'nri':
      return [
        ...baseAmenities,
        { googleType: 'subway_station', displayName: 'Metro', radius: 5000, limit: 2 },
        { googleType: 'shopping_mall', displayName: 'Mall', radius: 3000, limit: 2 }
      ];
    default:
      return [
        ...baseAmenities,
        { googleType: 'shopping_mall', displayName: 'Mall', radius: 3000, limit: 2 },
        { googleType: 'restaurant', displayName: 'Restaurant', radius: 2000, limit: 3 }
      ];
  }
}

// Helper function to customize data based on report type
function customizeDataByReportType(data: LocalityData, reportType: string): LocalityData {
  const customizedData = { ...data };
  
  switch (reportType) {
    case 'investment':
      // Boost growth and rental metrics for investment focus
      customizedData.growthPotential = Math.min(100, data.growthPotential + 5);
      customizedData.ribl = Math.min(10, data.ribl + 0.3);
      break;
    case 'family':
      // Boost safety and amenities for family focus
      customizedData.safetyIndex = Math.min(100, data.safetyIndex + 8);
      customizedData.amenitiesScore = Math.min(100, data.amenitiesScore + 5);
      break;
    case 'commercial':
      // Boost connectivity and growth for commercial focus
      customizedData.connectivityScore = Math.min(100, data.connectivityScore + 7);
      customizedData.growthPotential = Math.min(100, data.growthPotential + 3);
      break;
    case 'nri':
      // Boost overall scores for NRI investment
      customizedData.overallScore = Math.min(100, data.overallScore + 5);
      customizedData.ribl = Math.min(10, data.ribl + 0.2);
      break;
    case 'first-time':
      // Moderate all scores for first-time buyers
      customizedData.pricePerSqft = Math.max(5000, data.pricePerSqft - 1000);
      break;
  }
  
  return customizedData;
}

// Keep the original function for backward compatibility
async function fetchLocalityReport(
  localityName: string
): Promise<LocalityData> {
  return fetchLocalityReportByType(localityName, 'comprehensive');
}



// Helper function to calculate distance between two coordinates
function calculateDistance(lat1: number, lng1: number, lat2: number, lng2: number): number {
  const R = 6371; // Earth's radius in kilometers
  const dLat = (lat2 - lat1) * Math.PI / 180;
  const dLng = (lng2 - lng1) * Math.PI / 180;
  const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
    Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
    Math.sin(dLng/2) * Math.sin(dLng/2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
  return R * c;
}

// Generate mock heatmap data around a center point with time variation
function generateHeatmapData(center: { lat: number; lng: number }, type: string, count: number = 50, timeOffset: number = 0) {
  const data = [];
  const radius = 0.02; // ~2km radius
  
  for (let i = 0; i < count; i++) {
    const angle = Math.random() * 2 * Math.PI;
    const distance = Math.random() * radius;
    const lat = center.lat + distance * Math.cos(angle);
    const lng = center.lng + distance * Math.sin(angle);
    
    let value = Math.random();
    // Adjust value based on type and time
    const timeVariation = Math.sin(timeOffset * 0.5) * 0.15; // ±15% variation over time
    switch(type) {
      case 'price':
        value = 0.3 + value * 0.7 + timeVariation; // Higher values
        break;
      case 'demand':
        value = 0.4 + value * 0.6 + timeVariation;
        break;
      case 'rental':
        value = 0.2 + value * 0.8 + timeVariation;
        break;
      case 'crime':
        value = value * 0.4 - timeVariation * 0.5; // Lower values, inverse time effect
        break;
      case 'flood':
        value = value * 0.3; // Lower values, no time variation
        break;
    }
    
    value = Math.max(0, Math.min(1, value)); // Clamp between 0 and 1
    data.push({ lat, lng, value });
  }
  
  return data;
}

// Filter heatmap data by intensity
function filterByIntensity(data: Array<{ lat: number; lng: number; value: number }>, filter: 'all' | 'high' | 'medium' | 'low') {
  if (filter === 'all') return data;
  
  return data.filter(point => {
    if (filter === 'high') return point.value >= 0.7;
    if (filter === 'medium') return point.value >= 0.4 && point.value < 0.7;
    if (filter === 'low') return point.value < 0.4;
    return true;
  });
}

// Export heatmap as image
async function exportHeatmapAsImage(elementRef: React.RefObject<HTMLDivElement>, filename: string) {
  if (!elementRef.current) return;
  
  try {
    const html2canvas = (await import('html2canvas')).default;
    const canvas = await html2canvas(elementRef.current, {
      backgroundColor: '#ffffff',
      scale: 2,
      logging: false,
      useCORS: true,
      ignoreElements: (element) => {
        return element.classList?.contains('pointer-events-none');
      },
      onclone: (clonedDoc) => {
        const elements = clonedDoc.querySelectorAll('*');
        elements.forEach((el: any) => {
          const style = el.style;
          if (style) {
            ['color', 'backgroundColor', 'borderColor'].forEach(prop => {
              const value = style[prop];
              if (value && value.includes('oklch')) {
                style[prop] = '#000000';
              }
            });
          }
        });
      }
    });
    
    const link = document.createElement('a');
    link.download = filename;
    link.href = canvas.toDataURL('image/png');
    link.click();
  } catch (error) {
    console.error('Failed to export heatmap:', error);
    alert('Failed to export heatmap. Please try again.');
  }
}



// -------------------------------------------------------------------

export function LocalityIntelligence() {
  const navigate = useNavigate();
  const [selectedLocation, setSelectedLocation] = useState("");
  const [reportType, setReportType] = useState("comprehensive");
  const [showInteractiveMap, setShowInteractiveMap] = useState(true);
  const [searchQuery, setSearchQuery] = useState("");
  const [showSuggestions, setShowSuggestions] = useState(false);
  const [suggestions, setSuggestions] = useState<string[]>([]);
  const [loadingSuggestions, setLoadingSuggestions] = useState(false);
  const [timeframe, setTimeframe] = useState("5Y");
  const [compareSelection, setCompareSelection] = useState<string[]>([]);
  const [sentimentTimeSeries, setSentimentTimeSeries] = useState<any[]>([]);
  const [lastRefresh, setLastRefresh] = useState(new Date());
  const [autoRefresh, setAutoRefresh] = useState(true);
  const [comparisonHistory, setComparisonHistory] = useState<string[]>([]);
  const [activeTab, setActiveTab] = useState(0);

  const [locality, setLocality] = useState<LocalityData | null>(null);
  const [nearby, setNearby] = useState<LocalityData[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [isExporting, setIsExporting] = useState(false);
  const [dataSource, setDataSource] = useState<'groq' | 'api' | 'generated' | 'mixed'>('generated');
  const [isFromCache, setIsFromCache] = useState(false);

  // New feature states
  const [bookings, setBookings] = useState<any[]>([]);
  const [documents, setDocuments] = useState<any[]>([]);
  const [showBookingModal, setShowBookingModal] = useState(false);
  const [showDocumentModal, setShowDocumentModal] = useState(false);
  const [uploadingFile, setUploadingFile] = useState(false);
  const [dragActive, setDragActive] = useState(false);

  const [liveStats, setLiveStats] = useState<any>(null);
  const [statsLoading, setStatsLoading] = useState(false);

  const [compareModalOpen, setCompareModalOpen] = useState(false);
  const reportRef = useRef<HTMLDivElement>(null);
  const [activeHeatmapLayer, setActiveHeatmapLayer] = useState<'price' | 'demand' | 'rental' | 'crime' | 'flood'>('price');
  const [heatmapData, setHeatmapData] = useState<any>(null);
  const [comparisonMode, setComparisonMode] = useState(false);
  const [secondaryLayer, setSecondaryLayer] = useState<'price' | 'demand' | 'rental' | 'crime' | 'flood'>('demand');
  const [timeSliderValue, setTimeSliderValue] = useState(0);
  const [intensityFilter, setIntensityFilter] = useState<'all' | 'high' | 'medium' | 'low'>('all');
  const [showReportTypeDropdown, setShowReportTypeDropdown] = useState(false);
  const heatmapRef = useRef<HTMLDivElement>(null);

  // Auto-refresh data every 5 minutes if enabled
  useEffect(() => {
    if (autoRefresh) {
      const interval = setInterval(() => {
        if (selectedLocation) {
          loadReport(selectedLocation);
          setLastRefresh(new Date());
        }
      }, 300000); // 5 minutes
      return () => clearInterval(interval);
    }
  }, [autoRefresh, selectedLocation]);

  useEffect(() => {
    // Only fetch live stats on mount, don't load any location data
    fetchLiveStats();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // Fetch live stats every 30 seconds
  useEffect(() => {
    const statsInterval = setInterval(fetchLiveStats, 30000);
    return () => clearInterval(statsInterval);
  }, []);

  // Load comparison history from localStorage
  useEffect(() => {
    const savedHistory = localStorage.getItem("comparison-history");
    if (savedHistory) {
      setComparisonHistory(JSON.parse(savedHistory));
    }
  }, []);

  async function loadReport(locationName: string, reportTypeFilter?: string) {
    setLoading(true);
    setError(null);
    setIsFromCache(false);
    try {
      const currentReportType = reportTypeFilter || reportType;
      
      // Check if data is coming from cache
      const cachedData = cacheUtils.getLocality(locationName, currentReportType);
      if (cachedData) {
        setIsFromCache(true);
      }
      
      // Fetch AI-powered data with Groq
      console.log('🚀 Loading AI-powered locality report...');
      const data = await fetchLocalityReportByType(locationName, currentReportType);
      setLocality(data);
      setDataSource('groq'); // Set data source to Groq
      
      // Fetch nearby localities from GROQ
      let near: LocalityData[] = [];
      console.log(`🏘️ Fetching nearby localities for: ${locationName}`);
      near = await groqService.getNearbyLocalities(locationName);
      console.log(`✅ GROQ returned ${near.length} nearby localities`);
      
      const filteredNearby = near.filter((n) => n.name !== data.name);
      console.log(`📍 Setting ${filteredNearby.length} nearby localities:`, filteredNearby.map(n => n.name));
      setNearby(filteredNearby);
      
      // Fetch additional AI insights
      try {
        const sentimentData = await groqService.getSentimentAnalysis(locationName);
        setSentimentTimeSeries(sentimentData.timeSeries || sentimentTimeSeries);
        
        // Update locality with additional AI data
        setLocality(prev => prev ? {
          ...prev,
          sentimentScore: sentimentData.sentimentScore,
          walkabilityScore: sentimentData.walkabilityScore,
          walkabilityBreakdown: sentimentData.walkabilityBreakdown,
          floodRisk: sentimentData.floodRisk
        } as any : null);
      } catch (sentimentError) {
        console.warn('⚠️ Failed to fetch sentiment analysis:', sentimentError);
      }
      
      // Fetch heatmap data
      try {
        console.log('🗺️ Fetching heatmap data...');
        const heatmapLayers = await groqService.getHeatmapData(locationName, data.center);
        setHeatmapData(heatmapLayers);
        console.log('✅ Heatmap data loaded');
      } catch (heatmapError) {
        console.warn('⚠️ Failed to fetch heatmap data:', heatmapError);
      }
      
    } catch (err) {
      console.error("❌ Failed to fetch locality", err);
      setError("Failed to load locality data");
    } finally {
      setLoading(false);
    }
  }

  async function handleGenerate() {
    if (!selectedLocation.trim()) {
      setError('Please select a locality first');
      return;
    }
    
    await loadReport(selectedLocation, reportType);
    setLastRefresh(new Date());
    // Add to comparison history
    if (!comparisonHistory.includes(selectedLocation)) {
      const newHistory = [selectedLocation, ...comparisonHistory.slice(0, 9)];
      setComparisonHistory(newHistory);
      localStorage.setItem("comparison-history", JSON.stringify(newHistory));
    }
  }

  async function handleForceRefresh() {
    if (!selectedLocation.trim()) {
      setError('Please select a locality first');
      return;
    }
    
    // Clear cache for this location and report type
    const { localityCache } = await import('../../utils/localityCache');
    const cacheKey = `locality_${selectedLocation.toLowerCase().replace(/[^a-z0-9]/g, '_')}_${reportType}`;
    localityCache.delete(cacheKey);
    console.log(`🗑️ Cleared cache for ${selectedLocation} (${reportType})`);
    
    // Force reload
    await loadReport(selectedLocation, reportType);
    setLastRefresh(new Date());
  }

  // Dynamic locality search with Google Maps integration
  const [locationSuggestions, setLocationSuggestions] = useState<PlaceDetails[]>([]);
  const [isSearching, setIsSearching] = useState(false);
  
  async function handleLocationSearch(query: string) {
    setSearchQuery(query);
    setSelectedLocation(query);
    setShowSuggestions(query.length > 0);
    
    if (query.length > 2) {
      setIsSearching(true);
      try {
        // Get real location suggestions from Google Maps
        const placeDetails = await googleMapsService.geocodeLocation(query);
        if (placeDetails) {
          setLocationSuggestions([placeDetails]);
        }
      } catch (error) {
        console.error('Location search error:', error);
      } finally {
        setIsSearching(false);
      }
    }
  }

  function selectSuggestion(locality: string) {
    setSelectedLocation(locality);
    setSearchQuery(locality);
    setShowSuggestions(false);
    setLocationSuggestions([]);
    loadReport(locality);
  }

  // Report type filtering
  function handleReportTypeChange(type: string) {
    setReportType(type);
    // Regenerate report with new focus if location is selected
    if (selectedLocation) {
      loadReport(selectedLocation, type);
    }
  }



  // Handle modal escape key
  useEffect(() => {
    const handleEscape = (e: KeyboardEvent) => {
      if (e.key === "Escape") {
        setCompareModalOpen(false);
        setShowBookingModal(false);
        setShowDocumentModal(false);
      }
    };
    window.addEventListener("keydown", handleEscape);
    return () => window.removeEventListener("keydown", handleEscape);
  }, []);

  // Performance calculations
  function calculateSROI(locality: LocalityData) {
    const socialImpact =
      (locality.safetyIndex +
        locality.environmentScore +
        locality.amenitiesScore) /
      3;
    const financialReturn =
      (locality.oneYearGrowth + locality.rentalYield) * 10;
    return ((socialImpact + financialReturn) / 2).toFixed(1);
  }

  function calculateCashFlow(
    locality: LocalityData,
    investment: number = 5000000
  ) {
    const monthlyRent = (investment * locality.rentalYield) / (12 * 100);
    const monthlyEMI = investment * 0.008; // Assuming 9.6% interest
    return {
      monthlyRent: Math.round(monthlyRent),
      monthlyEMI: Math.round(monthlyEMI),
      netCashFlow: Math.round(monthlyRent - monthlyEMI),
    };
  }

  function handleCompareToggle(name: string) {
    setCompareSelection((prev) => {
      if (prev.includes(name)) return prev.filter((p) => p !== name);
      if (prev.length >= 3) return prev; // max 3
      return [...prev, name];
    });
  }

  // Enhanced comparison with GROQ-powered analysis
  async function getComparisonData() {
    if (compareSelection.length === 0) return [];
    
    try {
      console.log(`🏆 Getting competitive analysis for: ${compareSelection.join(', ')}`);
      const competitiveAnalysis = await groqService.getCompetitiveAnalysis(compareSelection);
      
      const localityData = await Promise.all(
        compareSelection.map(name => groqService.getLocalityData(name, 'investment'))
      );
      
      return {
        localities: localityData,
        analysis: competitiveAnalysis
      };
    } catch (error) {
      console.error('❌ Error getting comparison data:', error);
      return { localities: [], analysis: null };
    }
  }

  async function exportPDF() {
    if (!locality) return;
    setIsExporting(true);
    try {
      const jspdfModule = await import("jspdf");
      const jsPDF = jspdfModule.default;
      const pdf = new jsPDF();

      // Add NAL logo
      const logoImg = new Image();
      logoImg.onload = () => {
        // Add blue header background
        pdf.setFillColor(52, 86, 209);
        pdf.rect(0, 0, 210, 60, "F");

        // Add logo on blue background
        pdf.addImage(logoImg, "PNG", 20, 15, 25, 25);

        // Add white text on blue background
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("LOCALITY INTELLIGENCE REPORT", 55, 28);

        pdf.setFontSize(14);
        pdf.setFont("helvetica", "normal");
        pdf.text("Professional Locality Assessment & Market Analysis", 55, 38);

        pdf.setFontSize(11);
        pdf.text(
          `${
            locality.name
          } Analysis - Generated: ${new Date().toLocaleDateString()}`,
          55,
          48
        );

        // Add watermark function
        const addWatermark = () => {
          pdf.setTextColor(200, 200, 200);
          pdf.setFontSize(50);
          pdf.setFont("helvetica", "bold");
          const pageWidth = pdf.internal.pageSize.width;
          const pageHeight = pdf.internal.pageSize.height;
          pdf.text("NAL INDIA", pageWidth / 2, pageHeight / 2, {
            align: "center",
            angle: 45,
          });
          pdf.setTextColor(0, 0, 0);
        };

        addWatermark();
        let yPos = 80;

        // Locality Summary
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("LOCALITY SUMMARY", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          `• Price per Sq.Ft: ₹${locality.pricePerSqft?.toLocaleString()}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(`• 1-Year Growth: ${locality.oneYearGrowth}%`, 20, yPos);
        yPos += 6;
        pdf.text(`• 5-Year Growth: ${locality.fiveYearGrowth}%`, 20, yPos);
        yPos += 6;
        pdf.text(`• Rental Yield: ${locality.rentalYield}%`, 20, yPos);
        yPos += 6;
        pdf.text(`• RIBL Score: ${locality.ribl}/10`, 20, yPos);
        yPos += 6;
        pdf.text(`• Overall Score: ${locality.overallScore}/100`, 20, yPos);
        yPos += 20;

        // Scores Breakdown
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("DETAILED SCORES", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(`• Safety Index: ${locality.safetyIndex}/100`, 20, yPos);
        yPos += 6;
        pdf.text(
          `• Connectivity Score: ${locality.connectivityScore}/100`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(`• Amenities Score: ${locality.amenitiesScore}/100`, 20, yPos);
        yPos += 6;
        pdf.text(
          `• Environment Score: ${locality.environmentScore}/100`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Growth Potential: ${locality.growthPotential}/100`,
          20,
          yPos
        );
        yPos += 20;

        // Add new page
        pdf.addPage();
        addWatermark();
        yPos = 30;

        // Safety & Security Details
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("SAFETY & SECURITY", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          `• Crime Rate: ${
            locality.safety?.crimeRatePerMonth || "N/A"
          } incidents/month`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Police Station Distance: ${
            locality.safety?.policeStationDistKm || "N/A"
          } km`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Night Safety Score: ${locality.safety?.nightSafety || "N/A"}/10`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• CCTV Coverage: ${locality.safety?.cctvCoverage || "N/A"}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Emergency Response: ${
            locality.safety?.emergencyResponse || "N/A"
          }`,
          20,
          yPos
        );
        yPos += 20;

        // Connectivity Details
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("CONNECTIVITY", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          `• Metro Distance: ${
            locality.connectivity?.metroDistance || "N/A"
          } km`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Bus Stops: ${locality.connectivity?.busStops || "N/A"} nearby`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Airport Distance: ${
            locality.connectivity?.airportDistance || "N/A"
          } km`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Road Quality: ${locality.connectivity?.roadQuality || "N/A"}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Peak Traffic Time: ${
            locality.connectivity?.peakTrafficTime || "N/A"
          }`,
          20,
          yPos
        );
        yPos += 20;

        // Add new page
        pdf.addPage();
        addWatermark();
        yPos = 30;

        // Top Amenities
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("TOP AMENITIES", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        (locality.amenities || []).slice(0, 8).forEach((amenity, index) => {
          pdf.text(`${index + 1}. ${amenity.name} (${amenity.type})`, 20, yPos);
          yPos += 6;
          pdf.text(
            `   Distance: ${amenity.distKm} km | Rating: ${amenity.rating}/5 | Travel: ${amenity.travelTime}`,
            25,
            yPos
          );
          yPos += 10;
        });

        yPos += 10;

        // Environment Details
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("ENVIRONMENT", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(
          `• Air Quality Index: ${locality.environment?.aqi || "N/A"}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Noise Level: ${locality.environment?.noiseDb || "N/A"} dB`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Green Cover: ${locality.environment?.greenCoverPct || "N/A"}%`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Water Quality: ${locality.environment?.waterQuality || "N/A"}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Air Quality Trend: ${
            locality.environment?.airQualityTrend || "N/A"
          }`,
          20,
          yPos
        );
        yPos += 20;

        // Investment Analysis
        const cashFlow = calculateCashFlow(locality);
        const sroi = calculateSROI(locality);

        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("INVESTMENT ANALYSIS", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text(`• SROI Score: ${sroi}`, 20, yPos);
        yPos += 6;
        pdf.text(
          `• Monthly Rental Income: ₹${cashFlow.monthlyRent.toLocaleString()}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          `• Net Cash Flow: ₹${cashFlow.netCashFlow.toLocaleString()}`,
          20,
          yPos
        );
        yPos += 6;
        pdf.text(`• Demand Level: ${locality.demand}`, 20, yPos);
        yPos += 6;
        pdf.text(
          `• Average Rent: ₹${locality.avgRent || "N/A"}K per month`,
          20,
          yPos
        );
        yPos += 20;

        // Add new page for upcoming developments
        pdf.addPage();
        addWatermark();
        yPos = 30;

        // Upcoming Developments
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("UPCOMING DEVELOPMENTS", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        (locality.upcomingDevelopments || []).forEach((dev, index) => {
          pdf.text(`${index + 1}. ${dev.project}`, 20, yPos);
          yPos += 6;
          pdf.text(
            `   Timeline: ${dev.timeline} | Impact: ${dev.impact}`,
            25,
            yPos
          );
          yPos += 10;
        });

        yPos += 10;

        // Investment Recommendations
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("INVESTMENT RECOMMENDATIONS", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        pdf.text("SHORT-TERM (1-2 years):", 20, yPos);
        yPos += 8;
        pdf.text(
          "• Focus on rental yield properties for immediate income",
          25,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Target 2-3 BHK apartments for better rental demand",
          25,
          yPos
        );
        yPos += 10;

        pdf.text("LONG-TERM (3-5 years):", 20, yPos);
        yPos += 8;
        pdf.text(
          "• Invest in upcoming infrastructure corridor properties",
          25,
          yPos
        );
        yPos += 6;
        pdf.text(
          "• Consider pre-launch projects for maximum appreciation",
          25,
          yPos
        );
        yPos += 15;

        // Risk Assessment
        pdf.setFontSize(16);
        pdf.setFont("helvetica", "bold");
        pdf.text("RISK ASSESSMENT", 20, yPos);
        yPos += 15;

        pdf.setFontSize(11);
        pdf.setFont("helvetica", "normal");
        const riskLevel =
          locality.overallScore >= 80
            ? "LOW"
            : locality.overallScore >= 60
            ? "MEDIUM"
            : "HIGH";
        pdf.text(`Overall Risk Level: ${riskLevel}`, 20, yPos);
        yPos += 8;
        pdf.text(
          "Market Stability: High with consistent growth patterns",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "Liquidity: Excellent due to high demand and transaction volume",
          20,
          yPos
        );
        yPos += 6;
        pdf.text(
          "Regulatory Environment: Stable with RERA compliance",
          20,
          yPos
        );
        yPos += 20;

        // Disclaimer
        pdf.setFontSize(12);
        pdf.setFont("helvetica", "bold");
        pdf.text("DISCLAIMER", 20, yPos);
        yPos += 10;

        pdf.setFontSize(9);
        pdf.setFont("helvetica", "normal");
        const disclaimerText =
          "This report is prepared based on available market data and analysis. It is intended for informational purposes only and should not be construed as investment advice. Market conditions are subject to change, and past performance does not guarantee future results. Investors are advised to conduct independent research and consult qualified financial and legal advisors.";
        const disclaimerLines = pdf.splitTextToSize(disclaimerText, 170);
        pdf.text(disclaimerLines, 20, yPos);

        // Footer
        const addFooter = () => {
          pdf.setFillColor(52, 86, 209);
          pdf.rect(0, 275, 210, 22, "F");
          pdf.setTextColor(255, 255, 255);
          pdf.setFontSize(10);
          pdf.setFont("helvetica", "bold");
          pdf.text("NAL INDIA - New Age Land", 20, 285);
          pdf.setFont("helvetica", "normal");
          pdf.text(
            "www.nglindia.com | Professional Real Estate Services",
            20,
            292
          );
        };

        const totalPages = (pdf as any).internal.getNumberOfPages();
        for (let i = 1; i <= totalPages; i++) {
          pdf.setPage(i);
          addFooter();
        }

        pdf.save(
          `${locality.name}-Locality-Analysis-${
            new Date().toISOString().split("T")[0]
          }.pdf`
        );
        setIsExporting(false);
      };

      logoImg.src = "/favicon.png";
    } catch (err) {
      console.error("PDF export failed", err);
      alert("Export failed — check console for details.");
      setIsExporting(false);
    }
  }

  function exportCSV() {
    if (!locality) return;
    const cashFlow = calculateCashFlow(locality);
    const sroi = calculateSROI(locality);

    const rows = [
      ["Metric", "Value", "Category"],
      ["Locality", locality.name, "Basic"],
      ["Price per sqft", locality.pricePerSqft, "Pricing"],
      ["1Y Growth %", locality.oneYearGrowth, "Growth"],
      ["5Y Growth %", locality.fiveYearGrowth, "Growth"],
      ["Rental Yield %", locality.rentalYield, "Rental"],
      ["RIBL Score", locality.ribl, "Investment"],
      ["Overall Score", locality.overallScore, "Scoring"],
      ["Safety Index", locality.safetyIndex, "Scoring"],
      ["Connectivity Score", locality.connectivityScore, "Scoring"],
      ["Amenities Score", locality.amenitiesScore, "Scoring"],
      ["Environment Score", locality.environmentScore, "Scoring"],
      ["Growth Potential", locality.growthPotential, "Scoring"],
      ["SROI Score", sroi, "Performance"],
      ["Monthly Rent Potential", cashFlow.monthlyRent, "Performance"],
      ["Net Cash Flow", cashFlow.netCashFlow, "Performance"],
      [
        "Metro Distance (km)",
        locality.connectivity?.metroDistance || "N/A",
        "Connectivity",
      ],
      [
        "Airport Distance (km)",
        locality.connectivity?.airportDistance || "N/A",
        "Connectivity",
      ],
      ["AQI", locality.environment?.aqi || "N/A", "Environment"],
      [
        "Crime Rate/Month",
        locality.safety?.crimeRatePerMonth || "N/A",
        "Safety",
      ],
      ["Last Updated", new Date().toISOString(), "Meta"],
    ];

    const csv = rows.map((r) => r.map((c) => `"${c}"`).join(",")).join("\n");
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `${locality.name}-comprehensive-data-${
      new Date().toISOString().split("T")[0]
    }.csv`;
    a.click();
    URL.revokeObjectURL(url);
  }

  // Export chart data
  function exportChartData() {
    if (!locality) return;
    const chartData = {
      priceTrend: locality.priceTrend,
      radarData: radarData,
      performanceMetrics: {
        sroi: calculateSROI(locality),
        cashFlow: calculateCashFlow(locality),
        scores: {
          overall: locality.overallScore,
          safety: locality.safetyIndex,
          connectivity: locality.connectivityScore,
          amenities: locality.amenitiesScore,
          environment: locality.environmentScore,
          growth: locality.growthPotential,
        },
      },
    };

    const blob = new Blob([JSON.stringify(chartData, null, 2)], {
      type: "application/json",
    });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `${locality.name}-chart-data.json`;
    a.click();
    URL.revokeObjectURL(url);
  }

  // Dynamic chart data based on report type and timeframe
  const radarData = useMemo(() => {
    if (!locality) return [];
    const reportFocus =
      REPORT_TYPES[reportType as keyof typeof REPORT_TYPES].focus;

    let data = [
      { subject: "Safety", A: locality.safetyIndex / 10, fullMark: 10 },
      {
        subject: "Connectivity",
        A: locality.connectivityScore / 10,
        fullMark: 10,
      },
      { subject: "Amenities", A: locality.amenitiesScore / 10, fullMark: 10 },
      {
        subject: "Environment",
        A: locality.environmentScore / 10,
        fullMark: 10,
      },
      { subject: "Growth", A: locality.growthPotential / 10, fullMark: 10 },
    ];

    // Filter based on report type focus
    if (reportFocus !== "all") {
      const focusAreas = reportFocus.split(",");
      if (focusAreas.includes("safety"))
        data = data.filter(
          (d) => d.subject === "Safety" || d.subject === "Environment"
        );
      if (focusAreas.includes("growth"))
        data = data.filter(
          (d) => d.subject === "Growth" || d.subject === "Connectivity"
        );
      if (focusAreas.includes("amenities"))
        data = data.filter(
          (d) => d.subject === "Amenities" || d.subject === "Safety"
        );
    }

    return data;
  }, [locality, reportType]);

  // Dynamic price trend based on timeframe
  const filteredPriceTrend = useMemo(() => {
    if (!locality?.priceTrend) return [];
    const currentYear = new Date().getFullYear();

    switch (timeframe) {
      case "1Y":
        return locality.priceTrend.slice(-2);
      case "3Y":
        return locality.priceTrend.slice(-4);
      case "5Y":
      default:
        return locality.priceTrend;
    }
  }, [locality, timeframe]);

  // Handle search query changes with debounced GROQ suggestions
  useEffect(() => {
    const timeoutId = setTimeout(async () => {
      if (searchQuery.length >= 2) {
        setLoadingSuggestions(true);
        try {
          const newSuggestions = await getLocalitySuggestions(searchQuery);
          setSuggestions(newSuggestions);
        } catch (error) {
          console.error('Error fetching suggestions:', error);
          setSuggestions([]);
        } finally {
          setLoadingSuggestions(false);
        }
      } else {
        setSuggestions([]);
      }
    }, 300);

    return () => clearTimeout(timeoutId);
  }, [searchQuery]);

  // Dynamic suggestions for autocomplete
  const filteredSuggestions = suggestions;



  // API functions for dynamic data using GROQ
  async function fetchLiveStats() {
    setStatsLoading(true);
    try {
      console.log('📈 Fetching live stats from GROQ...');
      const stats = await groqService.getLiveStats();
      setLiveStats(stats);
      console.log('✅ Live stats updated successfully');
    } catch (error) {
      console.error("❌ Failed to fetch live stats:", error);
      // No fallback data - keep stats empty if GROQ fails
      setLiveStats(null);
    } finally {
      setStatsLoading(false);
    }
  }

  // File upload handler
  const handleFileUpload = async (
    event: React.ChangeEvent<HTMLInputElement>
  ) => {
    const files = event.target.files;
    if (!files) return;

    setUploadingFile(true);

    try {
      const formData = new FormData();
      Array.from(files).forEach((file) => formData.append("files", file));

      const response = await fetch("/api/documents/upload", {
        method: "POST",
        body: formData,
      }).catch(() => null);

      if (response?.ok) {
        const uploadedDocs = await response.json();
        setDocuments((prev) => [...prev, ...uploadedDocs]);
      } else {
        const newDocs = Array.from(files).map((file, index) => ({
          id: `doc-${Date.now()}-${index}`,
          name: file.name,
          type: file.type,
          size: `${(file.size / 1024 / 1024).toFixed(2)} MB`,
          uploadDate: new Date().toLocaleDateString(),
          verified: Math.random() > 0.5,
        }));
        setDocuments((prev) => [...prev, ...newDocs]);
      }

      fetchLiveStats();
    } catch (error) {
      console.error("Upload failed:", error);
    } finally {
      setUploadingFile(false);
    }
  };

  if (error) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <AlertTriangle className="w-16 h-16 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-bold mb-2">Error Loading</h2>
          <p className="text-gray-600 mb-4 text-sm">{error}</p>
          <div className="flex gap-2 justify-center flex-wrap">
            <Button onClick={() => window.location.reload()} size="sm">
              Reload
            </Button>
            <Button variant="outline" onClick={() => setError(null)} size="sm">
              Retry
            </Button>
          </div>
        </div>
      </div>
    );
  }

  // Show empty state if no location is selected
  const showEmptyState = !selectedLocation || !locality;

  if (loading) {
    return (
      <section className="bg-gradient-to-br from-blue-50 to-indigo-100 py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="animate-pulse space-y-4">
            <div className="h-12 bg-gray-200 rounded"></div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="h-64 bg-gray-200 rounded"></div>
              <div className="h-64 bg-gray-200 rounded"></div>
            </div>
          </div>
        </div>
      </section>
    );
  }

  return (
    <>
      {/* TOP CONTROLS */}
      <section className="bg-gradient-to-br from-blue-50 to-indigo-100 py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="bg-white rounded-2xl p-6 shadow-lg mb-6">
            <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 items-end">
              <div className="lg:col-span-3">
                <label className="text-sm font-semibold text-gray-700 mb-3 block">
                  Locality
                </label>
                <div className="relative flex gap-2">
                  <div className="relative flex-1">
                    <Search className="absolute left-3 top-3 w-4 h-4 text-gray-400" />
                    <input
                      className="w-full pl-10 py-2 pr-3 border rounded-full"
                      style={{ borderRadius: '9999px', WebkitAppearance: 'none' }}
                      value={searchQuery.length > 0 ? searchQuery : selectedLocation}
                      onChange={(e) => handleLocationSearch(e.target.value)}
                      onFocus={() => setShowSuggestions(true)}
                      onBlur={() => setTimeout(() => setShowSuggestions(false), 200)}
                      placeholder="Search any location worldwide (powered by Google Maps)"
                    />
                  </div>

                  {showSuggestions && (locationSuggestions.length > 0 || filteredSuggestions.length > 0 || isSearching) && (
                    <div className="absolute top-full left-0 right-16 bg-white border border-t-0 rounded-b-lg shadow-lg z-50 max-h-60 overflow-y-auto">
                      {isSearching && (
                        <div className="px-4 py-2 text-center text-gray-500">
                          <div className="inline-flex items-center gap-2">
                            <div className="w-4 h-4 border-2 border-gray-300 border-t-blue-600 rounded-full animate-spin" />
                            Searching locations...
                          </div>
                        </div>
                      )}
                      
                      {/* Google Maps suggestions */}
                      {locationSuggestions.map((suggestion, idx) => (
                        <div
                          key={`gm-${idx}`}
                          className="px-4 py-2 hover:bg-blue-50 cursor-pointer border-b border-blue-100"
                          onClick={() => selectSuggestion(suggestion.formatted_address)}
                        >
                          <div className="flex items-center gap-2">
                            <MapPin className="w-4 h-4 text-blue-600" />
                            <div>
                              <div className="font-medium text-blue-800">
                                {suggestion.name}
                              </div>
                              <div className="text-sm text-blue-600">
                                {suggestion.formatted_address}
                              </div>
                            </div>
                          </div>
                        </div>
                      ))}
                      
                      {/* GROQ AI suggestions */}
                      {filteredSuggestions.map((suggestion, idx) => (
                        <div
                          key={`groq-${idx}`}
                          className="px-4 py-2 hover:bg-gray-50 cursor-pointer border-b last:border-b-0"
                          onClick={() => selectSuggestion(suggestion)}
                        >
                          <div className="font-medium">
                            {suggestion.split(",")[0]}
                          </div>
                          <div className="text-sm text-gray-500">
                            {suggestion.split(",").slice(1).join(",")}
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              </div>

              <div className="lg:col-span-2">
                <label className="text-sm font-semibold text-gray-700 mb-2 block">
                  Report Type
                </label>
                <select
                  value={reportType}
                  onChange={(e) => handleReportTypeChange(e.target.value)}
                  className="w-full py-2 px-3 border rounded-full bg-white appearance-none cursor-pointer"
                  style={{ 
                    borderRadius: '9999px', 
                    WebkitAppearance: 'none',
                    backgroundImage: 'url("data:image/svg+xml,%3csvg xmlns=\'http://www.w3.org/2000/svg\' fill=\'none\' viewBox=\'0 0 20 20\'%3e%3cpath stroke=\'%236b7280\' stroke-linecap=\'round\' stroke-linejoin=\'round\' stroke-width=\'1.5\' d=\'M6 8l4 4 4-4\'/%3e%3c/svg%3e")',
                    backgroundPosition: 'right 0.5rem center',
                    backgroundRepeat: 'no-repeat',
                    backgroundSize: '1.5em 1.5em',
                    paddingRight: '2.5rem'
                  }}
                >
                  {Object.entries(REPORT_TYPES).map(([key, config]) => (
                    <option key={key} value={key}>
                      {config.label}
                    </option>
                  ))}
                </select>
              </div>

              <div className="lg:col-span-7 flex flex-wrap gap-2 justify-end">
                <button
                  onClick={() => handleGenerate()}
                  className="h-9 px-4 py-2 inline-flex items-center gap-2 rounded-full border-2 border-gray-300 bg-gray-100 hover:bg-gray-200 active:scale-95 transition-all font-medium text-sm"
                >
                  <BarChart3 className="w-4 h-4" />
                  Generate
                </button>
                <button
                  onClick={() => handleForceRefresh()}
                  className="h-9 px-4 py-2 inline-flex items-center gap-2 rounded-full border-2 border-gray-300 bg-gray-100 hover:bg-gray-200 active:scale-95 transition-all font-medium text-sm"
                  title="Clear cache and fetch fresh data"
                >
                  <TrendingUp className="w-4 h-4" />
                  Refresh
                </button>
                <button
                  onClick={() => setShowInteractiveMap((s) => !s)}
                  className="h-9 px-4 py-2 inline-flex items-center gap-2 rounded-full border-2 border-gray-300 bg-gray-100 hover:bg-gray-200 active:scale-95 transition-all font-medium text-sm"
                >
                  <MapPin className="w-4 h-4" />
                  {showInteractiveMap ? "Hide" : "Show"} Map
                </button>
                <button
                  onClick={() => setShowBookingModal(true)}
                  className="h-9 px-4 py-2 inline-flex items-center gap-2 rounded-full border-2 border-gray-300 bg-gray-100 hover:bg-gray-200 active:scale-95 transition-all font-medium text-sm"
                >
                  <Calendar className="w-4 h-4" /> Book Visit
                </button>
                <button
                  onClick={() => setShowDocumentModal(true)}
                  className="h-9 px-4 py-2 inline-flex items-center gap-2 rounded-full border-2 border-gray-300 bg-gray-100 hover:bg-gray-200 active:scale-95 transition-all font-medium text-sm"
                >
                  <FileText className="w-4 h-4" /> Documents
                </button>
                <button
                  onClick={() => exportPDF()}
                  className="h-9 px-4 py-2 inline-flex items-center gap-2 rounded-full border-2 border-gray-300 bg-gray-100 hover:bg-gray-200 disabled:opacity-50 disabled:cursor-not-allowed active:scale-95 transition-all font-medium text-sm"
                  disabled={isExporting}
                  aria-busy={isExporting}
                >
                  <Download className="w-4 h-4" />
                  {isExporting ? "Exporting..." : "PDF"}
                </button>
              </div>
            </div>
          </div>

          {/* Tab Navigation - Only show when location is selected */}
          {!showEmptyState && (
            <div className="bg-white rounded-2xl p-4 shadow mb-6">
              <div className="flex flex-wrap gap-2">
                {["Overview & Analytics", "Demographics", "Livability Metrics", "Government Policies", "Live Locality Analysis", "Scorecard Summary", "Live Market Stats", "Competitive Analysis", "Heatmap Analysis"].map((tab, index) => (
                  <button 
                    key={index} 
                    onClick={() => setActiveTab(index)} 
                    className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                      activeTab === index 
                        ? "bg-blue-500 text-white" 
                        : "bg-gray-200 text-gray-800 hover:bg-gray-300"
                    }`}
                  >
                    {tab}
                  </button>
                ))}
              </div>
            </div>
          )}

          {/* Empty State - Show when no location selected */}
          {showEmptyState && (
            <div className="bg-white rounded-2xl p-12 shadow text-center">
              <div className="max-w-md mx-auto">
                <MapPin className="w-20 h-20 text-gray-300 mx-auto mb-6" />
                <h3 className="text-2xl font-bold text-gray-800 mb-3">
                  Select a Location to Get Started
                </h3>
                <p className="text-gray-600 mb-6">
                  Search for any location worldwide using the search box above to view comprehensive locality intelligence, market analysis, and detailed insights.
                </p>
                <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                  <p className="text-sm text-blue-800">
                    💡 <strong>Tip:</strong> Type a locality name (e.g., "Whitefield, Bangalore" or "Manhattan, New York") and select from suggestions to generate your report.
                  </p>
                </div>
              </div>
            </div>
          )}

          {/* Primary report area - Only show when location is selected */}
          {!showEmptyState && (
          <div ref={reportRef} className="space-y-6">
            <div className="bg-white rounded-2xl p-6 shadow">
              <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                <div>
                  <div className="flex items-center gap-3">
                    <h3 className="text-xl font-bold truncate" title={locality?.name}>{locality?.name || "—"}</h3>
                    <div className="flex items-center gap-2">
                      <div className={`flex items-center gap-1 px-2 py-1 text-xs rounded-full ${
                        dataSource === 'groq' 
                          ? 'bg-purple-100 text-purple-800' 
                          : 'bg-green-100 text-green-800'
                      }`}>
                        <div className={`w-2 h-2 rounded-full animate-pulse ${
                          dataSource === 'groq' 
                            ? 'bg-purple-500' 
                            : 'bg-green-500'
                        }`} />
                        {dataSource === 'groq' ? 'AI-Powered by Groq' : 'Google Maps Synced'}
                      </div>
                      {isFromCache && (
                        <div className="flex items-center gap-1 px-2 py-1 text-xs rounded-full bg-blue-100 text-blue-800">
                          <div className="w-2 h-2 rounded-full bg-blue-500" />
                          Cached Data
                        </div>
                      )}
                    </div>
                  </div>
                  <p className="text-sm text-gray-600">
                    {isFromCache 
                      ? 'Data loaded from cache for consistent results. Use "Refresh" to get fresh data.' 
                      : dataSource === 'groq' 
                        ? 'AI-powered locality intelligence by Groq LLM (Cached for consistency)' 
                        : 'Real-time locality intelligence powered by Google Maps API'}
                  </p>
                </div>
                <div className="flex gap-4 items-center">
                  <div className="text-center">
                    <div className="text-sm text-gray-500">Price / sqft</div>
                    <div className="text-lg font-bold">
                      ₹{locality?.pricePerSqft ?? "—"}
                    </div>
                  </div>
                  <div className="text-center">
                    <div className="text-sm text-gray-500">1Y Growth</div>
                    <div className="text-lg font-bold">
                      {locality?.oneYearGrowth ?? "—"}%
                    </div>
                  </div>
                  <div className="text-center">
                    <div className="text-sm text-gray-500">Rental Yield</div>
                    <div className="text-lg font-bold">
                      {locality?.rentalYield ?? "—"}%
                    </div>
                  </div>
                  <div className="text-center">
                    <div className="text-sm text-gray-500">RIBL Score</div>
                    <div className="text-lg font-bold">
                      {locality?.ribl ?? "—"}/10
                    </div>
                  </div>
                  <div className="text-center">
                    <div className="text-sm text-gray-500">Last Sync</div>
                    <div className="text-xs font-medium text-green-600">
                      {lastRefresh.toLocaleTimeString()}
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {activeTab === 0 && (
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 lg:gap-6">
              <div className="bg-white rounded-2xl p-4 lg:p-6 shadow space-y-4 lg:space-y-6">
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 lg:gap-4">
                  <div style={{ height: 240 }}>
                    <h4 className="font-semibold mb-2">
                      Price Trend (last 5 years)
                    </h4>
                    {locality?.priceTrend ? (
                      <ResponsiveContainer width="100%" height={200}>
                        <LineChart data={locality.priceTrend}>
                          <CartesianGrid strokeDasharray="3 3" />
                          <XAxis dataKey="year" />
                          <YAxis />
                          <Tooltip />
                          <Line
                            type="monotone"
                            dataKey="price"
                            stroke="#1e3a8a"
                            strokeWidth={2}
                            dot
                            isAnimationActive={true}
                            animationDuration={800}
                          />
                        </LineChart>
                      </ResponsiveContainer>
                    ) : (
                      <div className="h-[200px] flex items-center justify-center text-gray-500">
                        {loading ? "Loading..." : "No data available"}
                      </div>
                    )}
                  </div>

                  <div style={{ height: 240 }}>
                    <h4 className="font-semibold mb-2">
                      Rent vs Price (sample)
                    </h4>
                    {locality?.priceTrend ? (
                      <ResponsiveContainer width="100%" height={200}>
                        <BarChart
                          data={locality.priceTrend.map((p, i) => ({
                            name: p.year,
                            price: p.price,
                            rent: (locality.avgRent || 30) * (i + 1),
                          }))}
                        >
                          <CartesianGrid strokeDasharray="3 3" />
                          <XAxis dataKey="name" />
                          <YAxis />
                          <Tooltip />
                          <Bar
                            dataKey="price"
                            fill="#2563eb"
                            isAnimationActive={true}
                            animationDuration={800}
                          />
                          <Bar
                            dataKey="rent"
                            fill="#10b981"
                            isAnimationActive={true}
                            animationDuration={800}
                          />
                        </BarChart>
                      </ResponsiveContainer>
                    ) : (
                      <div className="h-[200px] flex items-center justify-center text-gray-500">
                        {loading ? "Loading..." : "No data available"}
                      </div>
                    )}
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="p-4 rounded-lg border">
                    <h5 className="font-semibold mb-2">Top Amenities</h5>
                    <ul className="text-sm space-y-2">
                      {(Array.isArray(locality?.amenities) ? locality.amenities.slice(0, 5) : []).map(
                        (a, idx) => (
                          <li key={idx}>
                            <span className="font-semibold">{a.name}</span>{" "}
                            <span className="text-gray-500">
                              — {a.type}, {a.distKm} km
                            </span>
                          </li>
                        )
                      )}
                      {(!locality?.amenities ||
                        !Array.isArray(locality.amenities) ||
                        locality.amenities.length === 0) && (
                        <li className="text-gray-500">No amenities data</li>
                      )}
                    </ul>
                  </div>

                  <div className="p-4 rounded-lg border">
                    <h5 className="font-semibold mb-2">Safety Snapshot</h5>
                    <div className="text-sm">
                      <div>
                        Crime rate (avg / month):{" "}
                        <b>{locality?.safety?.crimeRatePerMonth ?? "—"}</b>
                      </div>
                      <div>
                        Police station distance:{" "}
                        <b>{locality?.safety?.policeStationDistKm ?? "—"} km</b>
                      </div>
                      <div>
                        Night Safety:{" "}
                        <b>{locality?.safety?.nightSafety ?? "—"}/10</b>
                      </div>
                    </div>
                  </div>

                  <div className="p-4 rounded-lg border">
                    <h5 className="font-semibold mb-2">Environment</h5>
                    <div className="text-sm">
                      <div>
                        AQI: <b>{locality?.environment?.aqi ?? "—"}</b>
                      </div>
                      <div>
                        Noise (dB):{" "}
                        <b>{locality?.environment?.noiseDb ?? "—"}</b>
                      </div>
                      <div>
                        Green Cover:{" "}
                        <b>{locality?.environment?.greenCoverPct ?? "—"}%</b>
                      </div>
                    </div>
                  </div>
                </div>

                <div>
                  <h4 className="font-semibold mb-3">Composite Scores</h4>
                  <div style={{ width: "100%", height: 260 }}>
                    {locality ? (
                      <ResponsiveContainer width="100%" height={260}>
                        <RadarChart
                          cx="50%"
                          cy="50%"
                          outerRadius="80%"
                          data={radarData}
                        >
                          <PolarGrid />
                          <PolarAngleAxis dataKey="subject" />
                          <PolarRadiusAxis angle={30} domain={[0, 10]} />
                          <Radar
                            name={locality.name}
                            dataKey="A"
                            stroke="#7c3aed"
                            fill="#7c3aed"
                            fillOpacity={0.4}
                            isAnimationActive={true}
                            animationDuration={800}
                          />
                          <Legend />
                        </RadarChart>
                      </ResponsiveContainer>
                    ) : (
                      <div className="text-gray-500">No data</div>
                    )}
                  </div>
                </div>
              </div>

              {/* MAP + INSIGHTS column */}
              <div className="bg-white rounded-2xl p-4 lg:p-6 shadow space-y-4 lg:space-y-6">
                <div>
                  <h4 className="font-semibold mb-2">Interactive Map</h4>
                  {showInteractiveMap && locality ? (
                    <div style={{ height: 300, position: "relative", zIndex: 10 }}>
                      {window.google ? (
                        <GoogleMap
                          center={locality.center}
                          height="300px"
                        />
                      ) : (
                        <div className="h-[300px] w-full rounded-lg bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center">
                          <div className="text-center p-6">
                            <MapPin className="w-12 h-12 text-blue-500 mx-auto mb-3" />
                            <h5 className="font-semibold text-gray-700 mb-2">Map View</h5>
                            <p className="text-sm text-gray-600 mb-2">{locality.name}</p>
                            <p className="text-xs text-gray-500">Coordinates: {locality.center.lat.toFixed(4)}, {locality.center.lng.toFixed(4)}</p>
                            <div className="mt-4 text-xs text-gray-400">
                              Google Maps API not available
                            </div>
                          </div>
                        </div>
                      )}
                    </div>
                  ) : (
                    <div className="h-64 w-full rounded-lg bg-gray-50 flex items-center justify-center text-gray-500">
                      {loading
                        ? "Loading..."
                        : "Map is hidden. Click 'Show Map' to view."}
                    </div>
                  )}


                </div>

                <div>
                  <h4 className="font-semibold mb-3">Key Insights</h4>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                      <span className="text-sm font-medium">
                        Investment Score
                      </span>
                      <span className="text-lg font-bold text-blue-600">
                        {locality?.ribl || "8.5"}/10
                      </span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                      <span className="text-sm font-medium">
                        Sentiment Score
                      </span>
                      <span className="text-lg font-bold text-green-600">
                        {(locality as any)?.sentimentScore || "78"}/100
                      </span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-orange-50 rounded-lg">
                      <span className="text-sm font-medium">Walkability</span>
                      <span className="text-lg font-bold text-orange-600">
                        {(locality as any)?.walkabilityScore || "72"}/100
                      </span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-cyan-50 rounded-lg">
                      <span className="text-sm font-medium">Flood Risk</span>
                      <span
                        className={`text-sm font-bold ${
                          (locality as any)?.floodRisk?.riskLevel === "Low" ||
                          (locality as any)?.floodRisk?.riskLevel === "Very Low"
                            ? "text-green-600"
                            : (locality as any)?.floodRisk?.riskLevel ===
                              "Medium"
                            ? "text-yellow-600"
                            : "text-red-600"
                        }`}
                      >
                        {(locality as any)?.floodRisk?.riskLevel || "Low"}
                      </span>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            )}

            {activeTab === 1 && (
            <div className="bg-white rounded-2xl p-8 shadow">
              <div className="flex items-center gap-3 mb-6">
                <Users className="w-6 h-6 text-indigo-600" />
                <h3 className="text-xl font-bold">Geo-Demographical Intelligence</h3>
              </div>
              
              {/* Key Metrics */}
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-8">
                <div className="p-4 bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl">
                  <div className="text-sm text-blue-600 mb-1">Avg Income</div>
                  <div className="text-2xl font-bold text-blue-900">₹{(locality as any)?.demographics?.avgIncomeLakhs || 12}L</div>
                </div>
                <div className="p-4 bg-gradient-to-br from-green-50 to-green-100 rounded-xl">
                  <div className="text-sm text-green-600 mb-1">Median Age</div>
                  <div className="text-2xl font-bold text-green-900">{(locality as any)?.demographics?.ageMedian || 32} yrs</div>
                </div>
                <div className="p-4 bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl">
                  <div className="text-sm text-purple-600 mb-1">Renter Ratio</div>
                  <div className="text-2xl font-bold text-purple-900">{(locality as any)?.demographics?.renterRatioPct || 35}%</div>
                </div>
                <div className="p-4 bg-gradient-to-br from-orange-50 to-orange-100 rounded-xl">
                  <div className="text-sm text-orange-600 mb-1">Population Density</div>
                  <div className="text-2xl font-bold text-orange-900">8.5K/km²</div>
                </div>
              </div>

              {/* Charts Section */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
                {/* Income Distribution Chart */}
                <div className="p-6 border rounded-xl">
                  <h4 className="font-semibold mb-4">Income Distribution</h4>
                  <ResponsiveContainer width="100%" height={200}>
                    <BarChart data={[
                      { bracket: '<5L', count: 15 },
                      { bracket: '5-10L', count: 25 },
                      { bracket: '10-20L', count: 35 },
                      { bracket: '20-50L', count: 20 },
                      { bracket: '>50L', count: 5 }
                    ]}>
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="bracket" />
                      <YAxis />
                      <Tooltip />
                      <Bar dataKey="count" fill="#3b82f6" />
                    </BarChart>
                  </ResponsiveContainer>
                </div>

                {/* Age Distribution Chart */}
                <div className="p-6 border rounded-xl">
                  <h4 className="font-semibold mb-4">Age Distribution</h4>
                  <ResponsiveContainer width="100%" height={200}>
                    <BarChart data={[
                      { age: '0-18', count: 20 },
                      { age: '19-30', count: 30 },
                      { age: '31-45', count: 35 },
                      { age: '46-60', count: 12 },
                      { age: '60+', count: 3 }
                    ]}>
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="age" />
                      <YAxis />
                      <Tooltip />
                      <Bar dataKey="count" fill="#10b981" />
                    </BarChart>
                  </ResponsiveContainer>
                </div>
              </div>

              {/* Additional Demographics */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="p-6 border rounded-xl">
                  <h4 className="font-semibold mb-4">Housing Profile</h4>
                  <div className="space-y-3">
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Renters</span>
                      <span className="font-semibold">{(locality as any)?.demographics?.renterRatioPct || 35}%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Owners</span>
                      <span className="font-semibold">{100 - ((locality as any)?.demographics?.renterRatioPct || 35)}%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Avg Family Size</span>
                      <span className="font-semibold">3.8</span>
                    </div>
                  </div>
                </div>
                <div className="p-6 border rounded-xl">
                  <h4 className="font-semibold mb-4">Education Level</h4>
                  <div className="space-y-3">
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Graduate+</span>
                      <span className="font-semibold">65%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Post-Graduate</span>
                      <span className="font-semibold">25%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Professional</span>
                      <span className="font-semibold">10%</span>
                    </div>
                  </div>
                </div>
                <div className="p-6 border rounded-xl">
                  <h4 className="font-semibold mb-4">Occupation Mix</h4>
                  <div className="space-y-3">
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">IT/Tech</span>
                      <span className="font-semibold">45%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Business</span>
                      <span className="font-semibold">30%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Others</span>
                      <span className="font-semibold">25%</span>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            )}

            {activeTab === 2 && (
            <div className="space-y-8">
              {/* Enhanced Demographics Panel */}
              <div className="bg-white rounded-2xl p-8 shadow-lg mb-8">
                <div className="flex items-center gap-3 mb-6">
                  <Users className="w-8 h-8 text-indigo-600" />
                  <div>
                    <h3 className="text-xl font-bold text-gray-900">Demographics Profile</h3>
                    <p className="text-sm text-gray-600">Population characteristics and socio-economic indicators</p>
                  </div>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  <div className="p-4 bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl">
                    <div className="text-sm text-blue-600 mb-1">Average Income</div>
                    <div className="text-2xl font-bold text-blue-900">₹{(locality as any)?.demographics?.avgIncomeLakhs || 12}L</div>
                    <div className="text-xs text-blue-700 mt-1">Per household annually</div>
                  </div>
                  <div className="p-4 bg-gradient-to-br from-green-50 to-green-100 rounded-xl">
                    <div className="text-sm text-green-600 mb-1">Median Age</div>
                    <div className="text-2xl font-bold text-green-900">{(locality as any)?.demographics?.ageMedian || 32} yrs</div>
                    <div className="text-xs text-green-700 mt-1">Working age population</div>
                  </div>
                  <div className="p-4 bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl">
                    <div className="text-sm text-purple-600 mb-1">Renter Ratio</div>
                    <div className="text-2xl font-bold text-purple-900">{(locality as any)?.demographics?.renterRatioPct || 35}%</div>
                    <div className="text-xs text-purple-700 mt-1">Rental vs ownership</div>
                  </div>
                </div>
              </div>

              {/* Enhanced Sentiment Analysis */}
              <div className="bg-white rounded-2xl p-8 shadow-lg mb-8">
                <div className="flex items-center gap-3 mb-6">
                  <Smile className="w-8 h-8 text-green-600" />
                  <div>
                    <h3 className="text-xl font-bold text-gray-900">Sentiment Analysis</h3>
                    <p className="text-sm text-gray-600">Community perception and satisfaction metrics</p>
                  </div>
                </div>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                  <div className="text-center">
                    <div className="relative inline-flex items-center justify-center w-32 h-32 mb-4">
                      <div className="w-32 h-32 rounded-full border-8 border-gray-200 relative">
                        <div 
                          className="absolute inset-0 rounded-full border-8 border-green-500 transition-all duration-1000"
                          style={{
                            background: `conic-gradient(#10b981 0deg ${((locality as any)?.sentimentScore || 78) * 3.6}deg, transparent ${((locality as any)?.sentimentScore || 78) * 3.6}deg 360deg)`,
                            mask: 'radial-gradient(circle at center, transparent 60%, black 60%)',
                            WebkitMask: 'radial-gradient(circle at center, transparent 60%, black 60%)'
                          }}
                        />
                      </div>
                      <div className="absolute inset-0 flex items-center justify-center">
                        <div className="text-center">
                          <div className="text-2xl font-bold text-green-600">{(locality as any)?.sentimentScore || 78}</div>
                          <div className="text-xs text-gray-500">Score</div>
                        </div>
                      </div>
                    </div>
                    <div className="text-lg font-semibold text-gray-900 mb-2">
                      {((locality as any)?.sentimentScore || 78) >= 80 ? "Very Positive" : ((locality as any)?.sentimentScore || 78) >= 60 ? "Positive" : "Neutral"}
                    </div>
                    <div className="text-sm text-gray-600">Overall community sentiment</div>
                  </div>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                      <span className="text-sm font-medium text-green-800">Resident Satisfaction</span>
                      <span className="text-sm font-bold text-green-600">85%</span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                      <span className="text-sm font-medium text-blue-800">Infrastructure Rating</span>
                      <span className="text-sm font-bold text-blue-600">4.2/5</span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-purple-50 rounded-lg">
                      <span className="text-sm font-medium text-purple-800">Community Engagement</span>
                      <span className="text-sm font-bold text-purple-600">High</span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-orange-50 rounded-lg">
                      <span className="text-sm font-medium text-orange-800">Future Outlook</span>
                      <span className="text-sm font-bold text-orange-600">Optimistic</span>
                    </div>
                  </div>
                </div>
                <div className="mt-6 p-4 bg-gray-50 rounded-lg">
                  <div className="text-xs text-gray-500 mb-2">Data Sources</div>
                  <div className="flex flex-wrap gap-2">
                    <span className="px-2 py-1 bg-white rounded text-xs text-gray-600">Social Media</span>
                    <span className="px-2 py-1 bg-white rounded text-xs text-gray-600">Resident Reviews</span>
                    <span className="px-2 py-1 bg-white rounded text-xs text-gray-600">Community Surveys</span>
                    <span className="px-2 py-1 bg-white rounded text-xs text-gray-600">Local Forums</span>
                  </div>
                </div>
              </div>

              {/* Enhanced Walkability Analysis */}
              <div className="bg-white rounded-2xl p-8 shadow-lg mb-8">
                <div className="flex items-center gap-3 mb-6">
                  <Navigation className="w-8 h-8 text-blue-600" />
                  <div>
                    <h3 className="text-xl font-bold text-gray-900">Walkability Index</h3>
                    <p className="text-sm text-gray-600">Pedestrian infrastructure and accessibility assessment</p>
                  </div>
                </div>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                  <div className="text-center">
                    <div className="relative inline-flex items-center justify-center w-32 h-32 mb-4">
                      <div className="w-32 h-32 rounded-full border-8 border-gray-200 relative">
                        <div 
                          className="absolute inset-0 rounded-full border-8 border-blue-500 transition-all duration-1000"
                          style={{
                            background: `conic-gradient(#3b82f6 0deg ${((locality as any)?.walkabilityScore || 72) * 3.6}deg, transparent ${((locality as any)?.walkabilityScore || 72) * 3.6}deg 360deg)`,
                            mask: 'radial-gradient(circle at center, transparent 60%, black 60%)',
                            WebkitMask: 'radial-gradient(circle at center, transparent 60%, black 60%)'
                          }}
                        />
                      </div>
                      <div className="absolute inset-0 flex items-center justify-center">
                        <div className="text-center">
                          <div className="text-2xl font-bold text-blue-600">{(locality as any)?.walkabilityScore || 72}</div>
                          <div className="text-xs text-gray-500">Score</div>
                        </div>
                      </div>
                    </div>
                    <div className="text-lg font-semibold text-gray-900 mb-2">
                      {((locality as any)?.walkabilityScore || 72) >= 80 ? "Excellent" : ((locality as any)?.walkabilityScore || 72) >= 60 ? "Good" : "Fair"}
                    </div>
                    <div className="text-sm text-gray-600">Walkability rating</div>
                  </div>
                  <div className="space-y-4">
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">🚶 Pedestrian Infrastructure</span>
                        <span className="font-semibold">{(locality as any)?.walkabilityBreakdown?.pedestrianInfra || 75}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className="bg-blue-600 h-2 rounded-full" style={{ width: `${(locality as any)?.walkabilityBreakdown?.pedestrianInfra || 75}%` }} />
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">🛤️ Sidewalk Quality</span>
                        <span className="font-semibold">{(locality as any)?.walkabilityBreakdown?.sidewalkQuality || 68}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className="bg-green-600 h-2 rounded-full" style={{ width: `${(locality as any)?.walkabilityBreakdown?.sidewalkQuality || 68}%` }} />
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">🚦 Safe Crossings</span>
                        <span className="font-semibold">{(locality as any)?.walkabilityBreakdown?.crossings || 82}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className="bg-purple-600 h-2 rounded-full" style={{ width: `${(locality as any)?.walkabilityBreakdown?.crossings || 82}%` }} />
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">💡 Street Lighting</span>
                        <span className="font-semibold">{(locality as any)?.walkabilityBreakdown?.lighting || 79}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className="bg-orange-600 h-2 rounded-full" style={{ width: `${(locality as any)?.walkabilityBreakdown?.lighting || 79}%` }} />
                      </div>
                    </div>
                  </div>
                </div>
                <div className="mt-6 grid grid-cols-2 md:grid-cols-4 gap-4">
                  <div className="text-center p-3 bg-blue-50 rounded-lg">
                    <div className="text-lg font-bold text-blue-600">2.3km</div>
                    <div className="text-xs text-blue-700">Avg walk to amenities</div>
                  </div>
                  <div className="text-center p-3 bg-green-50 rounded-lg">
                    <div className="text-lg font-bold text-green-600">15min</div>
                    <div className="text-xs text-green-700">Walk to transit</div>
                  </div>
                  <div className="text-center p-3 bg-purple-50 rounded-lg">
                    <div className="text-lg font-bold text-purple-600">85%</div>
                    <div className="text-xs text-purple-700">Accessible routes</div>
                  </div>
                  <div className="text-center p-3 bg-orange-50 rounded-lg">
                    <div className="text-lg font-bold text-orange-600">4.2/5</div>
                    <div className="text-xs text-orange-700">Safety rating</div>
                  </div>
                </div>
              </div>

              {/* Enhanced Transit Analysis */}
              <div className="bg-white rounded-2xl p-8 shadow-lg mb-8">
                <div className="flex items-center gap-3 mb-6">
                  <Car className="w-8 h-8 text-purple-600" />
                  <div>
                    <h3 className="text-xl font-bold text-gray-900">Transit Connectivity</h3>
                    <p className="text-sm text-gray-600">Public transportation accessibility and network analysis</p>
                  </div>
                  <div className={`ml-auto px-4 py-2 rounded-full text-sm font-semibold animate-pulse ${
                    (locality?.transitScore || 75) >= 80
                      ? "bg-green-100 text-green-800 border-2 border-green-300"
                      : (locality?.transitScore || 75) >= 60
                      ? "bg-yellow-100 text-yellow-800 border-2 border-yellow-300"
                      : "bg-red-100 text-red-800 border-2 border-red-300"
                  }`}>
                    {(locality?.transitScore || 75) >= 80
                      ? "🌟 Excellent"
                      : (locality?.transitScore || 75) >= 60
                      ? "👍 Good"
                      : "⚠️ Fair"}
                  </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                  {/* Overall Score */}
                  <div className="text-center">
                    <div className="relative inline-flex items-center justify-center w-32 h-32 mb-4">
                      <div className="w-32 h-32 rounded-full border-8 border-gray-200 relative">
                        <div 
                          className="absolute inset-0 rounded-full border-8 border-purple-500 transition-all duration-1000"
                          style={{
                            background: `conic-gradient(#8b5cf6 0deg ${(locality?.transitScore || 75) * 3.6}deg, transparent ${(locality?.transitScore || 75) * 3.6}deg 360deg)`,
                            mask: 'radial-gradient(circle at center, transparent 60%, black 60%)',
                            WebkitMask: 'radial-gradient(circle at center, transparent 60%, black 60%)'
                          }}
                        />
                      </div>
                      <div className="absolute inset-0 flex items-center justify-center">
                        <div className="text-center">
                          <div className="text-2xl font-bold text-purple-600">{locality?.transitScore || 75}</div>
                          <div className="text-xs text-gray-500">Score</div>
                        </div>
                      </div>
                    </div>
                    <div className="text-lg font-semibold text-gray-900 mb-2">Transit Score</div>
                    <div className="text-sm text-gray-600">Overall connectivity rating</div>
                  </div>

                  {/* Transit Breakdown */}
                  <div className="space-y-4">
                    <h4 className="font-semibold text-gray-900 mb-4">Network Coverage</h4>
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">🚇 Metro Access</span>
                        <span className="font-semibold">{locality?.transitBreakdown?.metro || 78}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-3">
                        <div className="bg-purple-600 h-3 rounded-full transition-all" style={{ width: `${locality?.transitBreakdown?.metro || 78}%` }} />
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">🚌 Bus Network</span>
                        <span className="font-semibold">{locality?.transitBreakdown?.bus || 85}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-3">
                        <div className="bg-blue-600 h-3 rounded-full transition-all" style={{ width: `${locality?.transitBreakdown?.bus || 85}%` }} />
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">🚕 Auto/Taxi</span>
                        <span className="font-semibold">{locality?.transitBreakdown?.auto || 92}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-3">
                        <div className="bg-green-600 h-3 rounded-full transition-all" style={{ width: `${locality?.transitBreakdown?.auto || 92}%` }} />
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-2">
                        <span className="text-gray-600">⏱️ Frequency</span>
                        <span className="font-semibold">{locality?.transitBreakdown?.frequency || 68}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-3">
                        <div className="bg-orange-600 h-3 rounded-full transition-all" style={{ width: `${locality?.transitBreakdown?.frequency || 68}%` }} />
                      </div>
                    </div>
                  </div>

                  {/* Key Metrics */}
                  <div className="space-y-4">
                    <h4 className="font-semibold text-gray-900 mb-4">Key Metrics</h4>
                    <div className="space-y-3">
                      <div className="p-3 bg-blue-50 rounded-lg">
                        <div className="text-sm font-medium text-blue-900">Nearest Metro</div>
                        <div className="text-lg font-bold text-blue-600">{locality?.connectivity?.metroDistance || 1.2} km</div>
                        <div className="text-xs text-blue-700">12 min walk</div>
                      </div>
                      <div className="p-3 bg-green-50 rounded-lg">
                        <div className="text-sm font-medium text-green-900">Bus Stops</div>
                        <div className="text-lg font-bold text-green-600">{locality?.connectivity?.busStops || 8}</div>
                        <div className="text-xs text-green-700">Within 500m radius</div>
                      </div>
                      <div className="p-3 bg-purple-50 rounded-lg">
                        <div className="text-sm font-medium text-purple-900">Peak Frequency</div>
                        <div className="text-lg font-bold text-purple-600">5-8 min</div>
                        <div className="text-xs text-purple-700">Average wait time</div>
                      </div>
                      <div className="p-3 bg-orange-50 rounded-lg">
                        <div className="text-sm font-medium text-orange-900">Accessibility</div>
                        <div className="text-lg font-bold text-orange-600">85%</div>
                        <div className="text-xs text-orange-700">Barrier-free access</div>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="mt-8 grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="p-4 bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg">
                    <div className="text-sm font-medium text-blue-900 mb-1">Daily Commute Cost</div>
                    <div className="text-xl font-bold text-blue-600">₹120-180</div>
                    <div className="text-xs text-blue-700">Average round trip</div>
                  </div>
                  <div className="p-4 bg-gradient-to-r from-green-50 to-green-100 rounded-lg">
                    <div className="text-sm font-medium text-green-900 mb-1">Network Coverage</div>
                    <div className="text-xl font-bold text-green-600">92%</div>
                    <div className="text-xs text-green-700">Area accessibility</div>
                  </div>
                  <div className="p-4 bg-gradient-to-r from-purple-50 to-purple-100 rounded-lg">
                    <div className="text-sm font-medium text-purple-900 mb-1">Service Hours</div>
                    <div className="text-xl font-bold text-purple-600">5AM-11PM</div>
                    <div className="text-xs text-purple-700">Daily operation</div>
                  </div>
                </div>
              </div>

              {/* Enhanced Flood Risk Assessment */}
              <div className="bg-white rounded-2xl p-8 shadow-lg">
                <div className="flex items-center gap-3 mb-6">
                  <Waves className="w-8 h-8 text-cyan-600" />
                  <div>
                    <h3 className="text-xl font-bold text-gray-900">Flood Risk Assessment</h3>
                    <p className="text-sm text-gray-600">Natural disaster preparedness and risk evaluation</p>
                  </div>
                </div>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                  <div className="text-center">
                    <div className={`inline-flex items-center justify-center w-32 h-32 rounded-full mb-4 ${
                      (locality as any)?.floodRisk?.riskLevel === "Low" || (locality as any)?.floodRisk?.riskLevel === "Very Low"
                        ? "bg-green-100"
                        : (locality as any)?.floodRisk?.riskLevel === "Medium"
                        ? "bg-yellow-100"
                        : "bg-red-100"
                    }`}>
                      <Waves className={`w-16 h-16 ${
                        (locality as any)?.floodRisk?.riskLevel === "Low" || (locality as any)?.floodRisk?.riskLevel === "Very Low"
                          ? "text-green-600"
                          : (locality as any)?.floodRisk?.riskLevel === "Medium"
                          ? "text-yellow-600"
                          : "text-red-600"
                      }`} />
                    </div>
                    <div className={`text-2xl font-bold mb-2 ${
                      (locality as any)?.floodRisk?.riskLevel === "Low" || (locality as any)?.floodRisk?.riskLevel === "Very Low"
                        ? "text-green-600"
                        : (locality as any)?.floodRisk?.riskLevel === "Medium"
                        ? "text-yellow-600"
                        : "text-red-600"
                    }`}>
                      {(locality as any)?.floodRisk?.riskLevel || "Low"}
                    </div>
                    <div className="text-sm text-gray-600">Risk Level</div>
                  </div>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                      <span className="text-sm font-medium text-blue-800">Flood Zone</span>
                      <span className="text-sm font-bold text-blue-600">{(locality as any)?.floodRisk?.floodZone || "Zone X"}</span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                      <span className="text-sm font-medium text-green-800">Drainage System</span>
                      <span className="text-sm font-bold text-green-600">Good</span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-purple-50 rounded-lg">
                      <span className="text-sm font-medium text-purple-800">Historical Events</span>
                      <span className="text-sm font-bold text-purple-600">None (10 yrs)</span>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-orange-50 rounded-lg">
                      <span className="text-sm font-medium text-orange-800">Emergency Response</span>
                      <span className="text-sm font-bold text-orange-600">Excellent</span>
                    </div>
                  </div>
                </div>
                <div className="mt-6 p-4 bg-gray-50 rounded-lg">
                  <div className="text-xs text-gray-500 mb-2">Risk Factors</div>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <div className="text-center">
                      <div className="text-sm font-bold text-gray-700">2.5m</div>
                      <div className="text-xs text-gray-600">Elevation above sea level</div>
                    </div>
                    <div className="text-center">
                      <div className="text-sm font-bold text-gray-700">850mm</div>
                      <div className="text-xs text-gray-600">Annual rainfall</div>
                    </div>
                    <div className="text-center">
                      <div className="text-sm font-bold text-gray-700">1.2km</div>
                      <div className="text-xs text-gray-600">Distance to water body</div>
                    </div>
                    <div className="text-center">
                      <div className="text-sm font-bold text-gray-700">95%</div>
                      <div className="text-xs text-gray-600">Drainage efficiency</div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            )}

            {activeTab === 3 && (
            <div className="bg-white rounded-2xl p-8 shadow mb-6">
              <div className="flex items-center gap-3 mb-6">
                <Scale className="w-6 h-6 text-purple-600" />
                <h3 className="text-xl font-bold">
                  Government Guidelines & Policies
                </h3>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <div className="p-4 border rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <CheckCircle className="w-4 h-4 text-green-600" />
                    <span className="font-semibold text-sm">RERA Status</span>
                  </div>
                  <p className="text-sm text-gray-600">
                    {(locality as any)?.policies?.reraApproved
                      ? "RERA Approved"
                      : "Pending Approval"}
                  </p>
                </div>
                <div className="p-4 border rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <BookOpen className="w-4 h-4 text-blue-600" />
                    <span className="font-semibold text-sm">Zoning</span>
                  </div>
                  <p className="text-sm text-gray-600">
                    {(locality as any)?.policies?.zoning || "Mixed Use"}
                  </p>
                </div>
                <div className="p-4 border rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <Building className="w-4 h-4 text-orange-600" />
                    <span className="font-semibold text-sm">FSI Limit</span>
                  </div>
                  <p className="text-sm text-gray-600">
                    {(locality as any)?.policies?.fsiLimit || "N/A"}
                  </p>
                </div>
                <div className="p-4 border rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <Car className="w-4 h-4 text-gray-600" />
                    <span className="font-semibold text-sm">Parking Norms</span>
                  </div>
                  <p className="text-sm text-gray-600">
                    {(locality as any)?.policies?.parkingNorms || "Standard"}
                  </p>
                </div>
              </div>
            </div>
            )}

            {activeTab === 4 && (
            <div className="bg-white rounded-2xl p-8 shadow">
              <div className="text-center mb-8">
                <h3 className="text-2xl font-bold mb-4">
                  Live Locality Analysis - {locality?.name || "Select Location"}
                </h3>
                <p className="text-gray-600 max-w-4xl mx-auto">
                  Real-time data analysis across 10 key categories for{" "}
                  {locality?.name || "your selected area"}
                </p>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {/* Safety & Security */}
                <div className="p-6 border rounded-xl hover:shadow-md transition-all hover:scale-105">
                  <div className="flex items-center gap-3 mb-4">
                    <div
                      className={`p-2 rounded-lg ${
                        (locality?.safetyIndex || 0) >= 80
                          ? "bg-green-100"
                          : (locality?.safetyIndex || 0) >= 60
                          ? "bg-yellow-100"
                          : "bg-red-100"
                      }`}
                    >
                      <Shield
                        className={`w-5 h-5 ${
                          (locality?.safetyIndex || 0) >= 80
                            ? "text-green-600"
                            : (locality?.safetyIndex || 0) >= 60
                            ? "text-yellow-600"
                            : "text-red-600"
                        }`}
                      />
                    </div>
                    <h4 className="font-semibold text-lg">Safety & Security</h4>
                    <div
                      className={`ml-auto px-2 py-1 rounded-full text-xs font-bold ${
                        (locality?.safetyIndex || 0) >= 80
                          ? "bg-green-100 text-green-800"
                          : (locality?.safetyIndex || 0) >= 60
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-red-100 text-red-800"
                      }`}
                    >
                      {locality?.safetyIndex || 0}/100
                    </div>
                  </div>
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span>Crime Rate</span>
                      <span className="font-medium">
                        {locality?.safety?.crimeRatePerMonth || 0}/month
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Police Station</span>
                      <span className="font-medium">
                        {locality?.safety?.policeStationDistKm || 0} km
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>CCTV Coverage</span>
                      <span className="font-medium">
                        {locality?.safety?.cctvCoverage || "0%"}
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Emergency Response</span>
                      <span className="font-medium">
                        {locality?.safety?.emergencyResponse || "N/A"}
                      </span>
                    </div>
                  </div>
                </div>

                {/* Connectivity */}
                <div className="p-6 border rounded-xl hover:shadow-md transition-all hover:scale-105">
                  <div className="flex items-center gap-3 mb-4">
                    <div
                      className={`p-2 rounded-lg ${
                        (locality?.connectivityScore || 0) >= 80
                          ? "bg-green-100"
                          : (locality?.connectivityScore || 0) >= 60
                          ? "bg-yellow-100"
                          : "bg-red-100"
                      }`}
                    >
                      <Car
                        className={`w-5 h-5 ${
                          (locality?.connectivityScore || 0) >= 80
                            ? "text-green-600"
                            : (locality?.connectivityScore || 0) >= 60
                            ? "text-yellow-600"
                            : "text-red-600"
                        }`}
                      />
                    </div>
                    <h4 className="font-semibold text-lg">Connectivity</h4>
                    <div
                      className={`ml-auto px-2 py-1 rounded-full text-xs font-bold ${
                        (locality?.connectivityScore || 0) >= 80
                          ? "bg-green-100 text-green-800"
                          : (locality?.connectivityScore || 0) >= 60
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-red-100 text-red-800"
                      }`}
                    >
                      {locality?.connectivityScore || 0}/100
                    </div>
                  </div>
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span>Metro Distance</span>
                      <span className="font-medium">
                        {locality?.connectivity?.metroDistance || 0} km
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Bus Stops</span>
                      <span className="font-medium">
                        {locality?.connectivity?.busStops || 0} nearby
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Airport Distance</span>
                      <span className="font-medium">
                        {locality?.connectivity?.airportDistance || 0} km
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Road Quality</span>
                      <span className="font-medium">
                        {locality?.connectivity?.roadQuality || "N/A"}
                      </span>
                    </div>
                  </div>
                </div>

                {/* Amenities */}
                <div className="p-6 border rounded-xl hover:shadow-md transition-all hover:scale-105">
                  <div className="flex items-center gap-3 mb-4">
                    <div
                      className={`p-2 rounded-lg ${
                        (locality?.amenitiesScore || 0) >= 80
                          ? "bg-green-100"
                          : (locality?.amenitiesScore || 0) >= 60
                          ? "bg-yellow-100"
                          : "bg-red-100"
                      }`}
                    >
                      <Building
                        className={`w-5 h-5 ${
                          (locality?.amenitiesScore || 0) >= 80
                            ? "text-green-600"
                            : (locality?.amenitiesScore || 0) >= 60
                            ? "text-yellow-600"
                            : "text-red-600"
                        }`}
                      />
                    </div>
                    <h4 className="font-semibold text-lg">Amenities</h4>
                    <div
                      className={`ml-auto px-2 py-1 rounded-full text-xs font-bold ${
                        (locality?.amenitiesScore || 0) >= 80
                          ? "bg-green-100 text-green-800"
                          : (locality?.amenitiesScore || 0) >= 60
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-red-100 text-red-800"
                      }`}
                    >
                      {locality?.amenitiesScore || 0}/100
                    </div>
                  </div>
                  <div className="space-y-2">
                    {(Array.isArray(locality?.amenities) ? locality.amenities.slice(0, 4) : []).map(
                      (amenity, idx) => (
                        <div key={idx} className="flex justify-between text-sm">
                          <span>{amenity.type}</span>
                          <span className="font-medium">
                            {amenity.distKm} km
                          </span>
                        </div>
                      )
                    )}
                    {(!locality?.amenities ||
                      !Array.isArray(locality.amenities) ||
                      locality.amenities.length === 0) && (
                      <div className="text-sm text-gray-500">
                        No amenities data available
                      </div>
                    )}
                  </div>
                </div>

                {/* Environment */}
                <div className="p-6 border rounded-xl hover:shadow-md transition-all hover:scale-105">
                  <div className="flex items-center gap-3 mb-4">
                    <div
                      className={`p-2 rounded-lg ${
                        (locality?.environmentScore || 0) >= 80
                          ? "bg-green-100"
                          : (locality?.environmentScore || 0) >= 60
                          ? "bg-yellow-100"
                          : "bg-red-100"
                      }`}
                    >
                      <TreePine
                        className={`w-5 h-5 ${
                          (locality?.environmentScore || 0) >= 80
                            ? "text-green-600"
                            : (locality?.environmentScore || 0) >= 60
                            ? "text-yellow-600"
                            : "text-red-600"
                        }`}
                      />
                    </div>
                    <h4 className="font-semibold text-lg">Environment</h4>
                    <div
                      className={`ml-auto px-2 py-1 rounded-full text-xs font-bold ${
                        (locality?.environmentScore || 0) >= 80
                          ? "bg-green-100 text-green-800"
                          : (locality?.environmentScore || 0) >= 60
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-red-100 text-red-800"
                      }`}
                    >
                      {locality?.environmentScore || 0}/100
                    </div>
                  </div>
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span>Air Quality (AQI)</span>
                      <span className="font-medium">
                        {locality?.environment?.aqi || 0}
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Noise Level</span>
                      <span className="font-medium">
                        {locality?.environment?.noiseDb || 0} dB
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Green Cover</span>
                      <span className="font-medium">
                        {locality?.environment?.greenCoverPct || 0}%
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Water Quality</span>
                      <span className="font-medium">
                        {locality?.environment?.waterQuality || "N/A"}
                      </span>
                    </div>
                  </div>
                </div>

                {/* Lifestyle */}
                <div className="p-6 border rounded-xl hover:shadow-md transition-all hover:scale-105">
                  <div className="flex items-center gap-3 mb-4">
                    <div className="p-2 bg-purple-100 rounded-lg">
                      <ShoppingBag className="w-5 h-5 text-purple-600" />
                    </div>
                    <h4 className="font-semibold text-lg">Lifestyle</h4>
                  </div>
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span>Malls</span>
                      <span className="font-medium">
                        {locality?.lifestyle?.malls?.length || 0}
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Restaurants</span>
                      <span className="font-medium">
                        {locality?.lifestyle?.restaurants?.length || 0}
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Entertainment</span>
                      <span className="font-medium">
                        {locality?.lifestyle?.entertainment?.length || 0}
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Parks</span>
                      <span className="font-medium">
                        {locality?.lifestyle?.parks?.length || 0}
                      </span>
                    </div>
                  </div>
                </div>

                {/* Growth Potential */}
                <div className="p-6 border rounded-xl hover:shadow-md transition-all hover:scale-105">
                  <div className="flex items-center gap-3 mb-4">
                    <div
                      className={`p-2 rounded-lg ${
                        (locality?.growthPotential || 0) >= 80
                          ? "bg-green-100"
                          : (locality?.growthPotential || 0) >= 60
                          ? "bg-yellow-100"
                          : "bg-red-100"
                      }`}
                    >
                      <TrendingUp
                        className={`w-5 h-5 ${
                          (locality?.growthPotential || 0) >= 80
                            ? "text-green-600"
                            : (locality?.growthPotential || 0) >= 60
                            ? "text-yellow-600"
                            : "text-red-600"
                        }`}
                      />
                    </div>
                    <h4 className="font-semibold text-lg">Growth Potential</h4>
                    <div
                      className={`ml-auto px-2 py-1 rounded-full text-xs font-bold ${
                        (locality?.growthPotential || 0) >= 80
                          ? "bg-green-100 text-green-800"
                          : (locality?.growthPotential || 0) >= 60
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-red-100 text-red-800"
                      }`}
                    >
                      {locality?.growthPotential || 0}/100
                    </div>
                  </div>
                  <div className="space-y-2">
                    <div className="flex justify-between text-sm">
                      <span>1Y Growth</span>
                      <span className="font-medium">
                        {locality?.oneYearGrowth || 0}%
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>5Y Growth</span>
                      <span className="font-medium">
                        {locality?.fiveYearGrowth || 0}%
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Upcoming Projects</span>
                      <span className="font-medium">
                        {locality?.upcomingDevelopments?.length || 0}
                      </span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Investment Grade</span>
                      <span className="font-medium">
                        {locality?.ribl || 0}/10
                      </span>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            )}

            {activeTab === 5 && (
            <div className="bg-white rounded-2xl p-8 shadow">
              <div className="text-center mb-8">
                <h3 className="text-2xl font-bold mb-2">
                   Locality Intelligence Report
                </h3>
                <p className="text-gray-600">
                  See what insights you'll receive in your comprehensive
                  locality analysis
                </p>
              </div>

              {/* Locality Scorecard */}
              <div className="mb-8">
                <h4 className="text-xl font-semibold mb-6">
                  Locality Scorecard
                </h4>
                <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-4">
                  <div className="text-center p-4 bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl">
                    <div className="text-3xl font-bold text-blue-600">
                      {locality?.overallScore || 78}
                    </div>
                    <div className="text-sm font-medium text-blue-800">
                      Overall Score
                    </div>
                  </div>
                  <div className="text-center p-4 bg-gradient-to-br from-green-50 to-green-100 rounded-xl">
                    <div className="text-3xl font-bold text-green-600">
                      {locality?.safetyIndex || 82}
                    </div>
                    <div className="text-sm font-medium text-green-800">
                      Safety Index
                    </div>
                  </div>
                  <div className="text-center p-4 bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl">
                    <div className="text-3xl font-bold text-purple-600">
                      {locality?.connectivityScore || 75}
                    </div>
                    <div className="text-sm font-medium text-purple-800">
                      Connectivity
                    </div>
                  </div>
                  <div className="text-center p-4 bg-gradient-to-br from-orange-50 to-orange-100 rounded-xl">
                    <div className="text-3xl font-bold text-orange-600">
                      {locality?.amenitiesScore || 85}
                    </div>
                    <div className="text-sm font-medium text-orange-800">
                      Amenities
                    </div>
                  </div>
                  <div className="text-center p-4 bg-gradient-to-br from-teal-50 to-teal-100 rounded-xl">
                    <div className="text-3xl font-bold text-teal-600">
                      {locality?.environmentScore || 70}
                    </div>
                    <div className="text-sm font-medium text-teal-800">
                      Environment
                    </div>
                  </div>
                  <div className="text-center p-4 bg-gradient-to-br from-indigo-50 to-indigo-100 rounded-xl">
                    <div className="text-3xl font-bold text-indigo-600">
                      {locality?.growthPotential || 88}
                    </div>
                    <div className="text-sm font-medium text-indigo-800">
                      Growth Potential
                    </div>
                  </div>
                </div>
              </div>

              {/* Key Insights */}
              <div className="mb-8">
                <h4 className="text-xl font-semibold mb-6">Key Insights</h4>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="p-6 bg-green-50 rounded-xl border border-green-200">
                    <div className="flex items-center gap-3 mb-3">
                      <Shield className="w-6 h-6 text-green-600" />
                      <h5 className="font-semibold text-green-800">
                        Excellent Safety Record
                      </h5>
                    </div>
                    <p className="text-green-700">
                      {locality?.safety?.crimeStats ||
                        "Low crime rate with 24/7 security presence"}
                    </p>
                  </div>
                  <div className="p-6 bg-blue-50 rounded-xl border border-blue-200">
                    <div className="flex items-center gap-3 mb-3">
                      <Car className="w-6 h-6 text-blue-600" />
                      <h5 className="font-semibold text-blue-800">
                        Strong Infrastructure
                      </h5>
                    </div>
                    <p className="text-blue-700">
                      Metro connectivity within{" "}
                      {locality?.connectivity?.metroDistance || 1.2}km,{" "}
                      {locality?.connectivity?.roadQuality?.toLowerCase() ||
                        "good"}{" "}
                      road network
                    </p>
                  </div>
                  <div className="p-6 bg-purple-50 rounded-xl border border-purple-200">
                    <div className="flex items-center gap-3 mb-3">
                      <Building className="w-6 h-6 text-purple-600" />
                      <h5 className="font-semibold text-purple-800">
                        Premium Amenities
                      </h5>
                    </div>
                    <p className="text-purple-700">
                      Top-rated schools and hospitals nearby
                    </p>
                  </div>
                  <div className="p-6 bg-orange-50 rounded-xl border border-orange-200">
                    <div className="flex items-center gap-3 mb-3">
                      <TrendingUp className="w-6 h-6 text-orange-600" />
                      <h5 className="font-semibold text-orange-800">
                        High Growth Potential
                      </h5>
                    </div>
                    <p className="text-orange-700">
                      Major infrastructure projects planned
                    </p>
                  </div>
                </div>
              </div>
            </div>
            )}

            {activeTab === 6 && (
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4 lg:gap-6 mb-6">
              {/* Urgent Sales Trends */}
              <div className="bg-white rounded-2xl p-6 shadow">
                <div className="flex items-center gap-3 mb-4">
                  <TrendingDown className="w-6 h-6 text-red-600" />
                  <h4 className="font-semibold text-lg">Urgent Sales</h4>
                  {statsLoading && (
                    <div className="w-4 h-4 border-2 border-gray-300 border-t-red-600 rounded-full animate-spin" />
                  )}
                </div>
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">This Month</span>
                    <span className="font-semibold text-red-600">
                      {statsLoading || !liveStats
                        ? "..."
                        : `${liveStats.urgentSales?.thisMonth || 0} properties`}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">Avg Discount</span>
                    <span className="font-semibold">
                      {statsLoading || !liveStats
                        ? "..."
                        : `${liveStats.urgentSales?.avgDiscount || 0}%`}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">
                      Quick Sale Time
                    </span>
                    <span className="font-semibold">
                      {statsLoading || !liveStats
                        ? "..."
                        : liveStats.urgentSales?.quickSaleTime || 'N/A'}
                    </span>
                  </div>
                </div>
              </div>

              {/* Agent Activity */}
              <div className="bg-white rounded-2xl p-6 shadow">
                <div className="flex items-center gap-3 mb-4">
                  <UserCheck className="w-6 h-6 text-blue-600" />
                  <h4 className="font-semibold text-lg">Agent Activity</h4>
                  {statsLoading && (
                    <div className="w-4 h-4 border-2 border-gray-300 border-t-blue-600 rounded-full animate-spin" />
                  )}
                </div>
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">Active Agents</span>
                    <span className="font-semibold">
                      {statsLoading || !liveStats
                        ? "..."
                        : liveStats.agentActivity?.activeAgents || 0}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">
                      Listings Added
                    </span>
                    <span className="font-semibold text-green-600">
                      {statsLoading || !liveStats
                        ? "..."
                        : `+${liveStats.agentActivity?.listingsAdded || 0} today`}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">Site Visits</span>
                    <span className="font-semibold">
                      {statsLoading || !liveStats
                        ? "..."
                        : `${liveStats.agentActivity?.siteVisits || 0} this week`}
                    </span>
                  </div>
                </div>
              </div>

              {/* Document Repository */}
              <div className="bg-white rounded-2xl p-6 shadow">
                <div className="flex items-center gap-3 mb-4">
                  <Folder className="w-6 h-6 text-purple-600" />
                  <h4 className="font-semibold text-lg">Documents</h4>
                  {statsLoading && (
                    <div className="w-4 h-4 border-2 border-gray-300 border-t-purple-600 rounded-full animate-spin" />
                  )}
                </div>
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">Total Files</span>
                    <span className="font-semibold">
                      {statsLoading || !liveStats
                        ? "..."
                        : (
                            liveStats.documents?.totalFiles || 0
                          ).toLocaleString()}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">Verified Docs</span>
                    <span className="font-semibold text-green-600">
                      {statsLoading || !liveStats
                        ? "..."
                        : `${liveStats.documents?.verifiedDocs || 0}%`}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">
                      Recent Uploads
                    </span>
                    <span className="font-semibold">
                      {statsLoading || !liveStats
                        ? "..."
                        : `${liveStats.documents?.recentUploads || 0} today`}
                    </span>
                  </div>
                </div>
              </div>
            </div>
            )}

            {activeTab === 8 && (
            <div className="bg-white rounded-2xl p-8 shadow">
              <div className="text-center mb-8">
                <h3 className="text-2xl font-bold mb-4">
                  Geo-Heatmap Analysis - {locality?.name || "Select Location"}
                </h3>
                <p className="text-gray-600 max-w-4xl mx-auto">
                  Interactive heatmap visualization for price, demand, rental, crime, and flood risk analysis
                </p>
              </div>

              {/* Advanced Controls */}
              <div className="mb-6 space-y-4">
                {/* Layer Selection */}
                <div className="flex flex-wrap gap-3 justify-center">
                  {[
                    { id: 'price', label: 'Price Heatmap', color: 'blue', icon: '💰' },
                    { id: 'demand', label: 'Demand Heatmap', color: 'green', icon: '📈' },
                    { id: 'rental', label: 'Rental Yield', color: 'purple', icon: '🏠' },
                    { id: 'crime', label: 'Crime Rate', color: 'red', icon: '🚨' },
                    { id: 'flood', label: 'Flood Risk', color: 'cyan', icon: '🌊' }
                  ].map((layer) => (
                    <button
                      key={layer.id}
                      onClick={() => setActiveHeatmapLayer(layer.id as any)}
                      className={`px-4 py-3 rounded-lg font-medium transition-all transform hover:scale-105 ${
                        activeHeatmapLayer === layer.id
                          ? layer.color === 'blue' ? 'bg-blue-500 text-white shadow-lg border-2 border-blue-300 animate-pulse'
                          : layer.color === 'green' ? 'bg-green-500 text-white shadow-lg border-2 border-green-300 animate-pulse'
                          : layer.color === 'purple' ? 'bg-purple-500 text-black shadow-lg border-2 border-purple-300 animate-pulse'
                          : layer.color === 'red' ? 'bg-red-500 text-white shadow-lg border-2 border-red-300 animate-pulse'
                          : 'bg-cyan-500 text-black shadow-lg border-2 border-cyan-300 animate-pulse'
                          : 'bg-gray-200 text-gray-700 hover:bg-gray-300 border-2 border-transparent'
                      }`}
                    >
                      <span className="mr-2 text-lg">{layer.icon}</span>
                      {layer.label}
                    </button>
                  ))}
                </div>

                {/* Control Panel */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 p-4 bg-gray-50 rounded-xl">
                  <div className="flex items-center gap-3">
                    <input
                      type="checkbox"
                      id="comparison-mode"
                      checked={comparisonMode}
                      onChange={(e) => setComparisonMode(e.target.checked)}
                      className="w-4 h-4 rounded"
                    />
                    <label htmlFor="comparison-mode" className="text-sm font-medium cursor-pointer">
                      🔄 Comparison Mode
                    </label>
                  </div>

                  {comparisonMode && (
                    <div className="flex items-center gap-2">
                      <label className="text-sm font-medium">vs</label>
                      <select
                        value={secondaryLayer}
                        onChange={(e) => setSecondaryLayer(e.target.value as any)}
                        className="text-sm border rounded px-2 py-1"
                      >
                        {[
                          { id: 'price', label: 'Price' },
                          { id: 'demand', label: 'Demand' },
                          { id: 'rental', label: 'Rental' },
                          { id: 'crime', label: 'Crime' },
                          { id: 'flood', label: 'Flood' }
                        ].filter(l => l.id !== activeHeatmapLayer).map(layer => (
                          <option key={layer.id} value={layer.id}>{layer.label}</option>
                        ))}
                      </select>
                    </div>
                  )}

                  <div className="flex items-center gap-2">
                    <label className="text-sm font-medium">🎯 Filter:</label>
                    <select
                      value={intensityFilter}
                      onChange={(e) => setIntensityFilter(e.target.value as any)}
                      className="text-sm border rounded px-2 py-1"
                    >
                      <option value="all">All Zones</option>
                      <option value="high">High (≥70%)</option>
                      <option value="medium">Medium (40-70%)</option>
                      <option value="low">Low (&lt;40%)</option>
                    </select>
                  </div>
                </div>
              </div>

              {/* Heatmap Visualization */}
              <div ref={heatmapRef} className="bg-gray-50 rounded-xl p-6 mb-6 relative">
                {locality && window.google && window.google.maps && window.google.maps.visualization ? (
                  <div className="relative">
                    <HeatmapLayer
                      center={locality.center}
                      data={filterByIntensity(
                        (heatmapData && heatmapData[activeHeatmapLayer]) || generateHeatmapData(locality.center, activeHeatmapLayer, 80, 0),
                        intensityFilter
                      )}
                      layerType={activeHeatmapLayer}
                    gradient={
                      activeHeatmapLayer === 'price' ? [
                        'rgba(0, 255, 255, 0)',
                        'rgba(0, 191, 255, 1)',
                        'rgba(0, 127, 255, 1)',
                        'rgba(0, 63, 255, 1)',
                        'rgba(0, 0, 255, 1)',
                        'rgba(63, 0, 191, 1)',
                        'rgba(127, 0, 127, 1)',
                        'rgba(191, 0, 63, 1)',
                        'rgba(255, 0, 0, 1)'
                      ] : activeHeatmapLayer === 'demand' ? [
                        'rgba(0, 255, 0, 0)',
                        'rgba(0, 255, 0, 1)',
                        'rgba(127, 255, 0, 1)',
                        'rgba(255, 255, 0, 1)',
                        'rgba(255, 127, 0, 1)',
                        'rgba(255, 0, 0, 1)'
                      ] : activeHeatmapLayer === 'rental' ? [
                        'rgba(255, 0, 255, 0)',
                        'rgba(191, 0, 255, 1)',
                        'rgba(127, 0, 255, 1)',
                        'rgba(63, 0, 255, 1)',
                        'rgba(0, 0, 255, 1)'
                      ] : activeHeatmapLayer === 'crime' ? [
                        'rgba(255, 255, 0, 0)',
                        'rgba(255, 191, 0, 1)',
                        'rgba(255, 127, 0, 1)',
                        'rgba(255, 63, 0, 1)',
                        'rgba(255, 0, 0, 1)'
                      ] : [
                        'rgba(0, 255, 255, 0)',
                        'rgba(0, 191, 255, 1)',
                        'rgba(0, 127, 255, 1)',
                        'rgba(0, 63, 255, 1)',
                        'rgba(0, 0, 255, 1)'
                      ]
                    }
                      radius={25}
                      opacity={comparisonMode ? 0.5 : 0.7}
                      height="500px"
                    />
                    {comparisonMode && (
                      <div className="absolute inset-0 pointer-events-none">
                        <HeatmapLayer
                          center={locality.center}
                          data={filterByIntensity(
                            (heatmapData && heatmapData[secondaryLayer]) || generateHeatmapData(locality.center, secondaryLayer, 80, 0),
                            intensityFilter
                          )}
                          layerType={secondaryLayer}
                          gradient={
                            secondaryLayer === 'price' ? [
                              'rgba(0, 255, 255, 0)',
                              'rgba(0, 191, 255, 0.5)',
                              'rgba(0, 127, 255, 0.5)',
                              'rgba(0, 63, 255, 0.5)',
                              'rgba(0, 0, 255, 0.5)',
                              'rgba(63, 0, 191, 0.5)',
                              'rgba(127, 0, 127, 0.5)',
                              'rgba(191, 0, 63, 0.5)',
                              'rgba(255, 0, 0, 0.5)'
                            ] : secondaryLayer === 'demand' ? [
                              'rgba(0, 255, 0, 0)',
                              'rgba(0, 255, 0, 0.5)',
                              'rgba(127, 255, 0, 0.5)',
                              'rgba(255, 255, 0, 0.5)',
                              'rgba(255, 127, 0, 0.5)',
                              'rgba(255, 0, 0, 0.5)'
                            ] : secondaryLayer === 'rental' ? [
                              'rgba(255, 0, 255, 0)',
                              'rgba(191, 0, 255, 0.5)',
                              'rgba(127, 0, 255, 0.5)',
                              'rgba(63, 0, 255, 0.5)',
                              'rgba(0, 0, 255, 0.5)'
                            ] : secondaryLayer === 'crime' ? [
                              'rgba(255, 255, 0, 0)',
                              'rgba(255, 191, 0, 0.5)',
                              'rgba(255, 127, 0, 0.5)',
                              'rgba(255, 63, 0, 0.5)',
                              'rgba(255, 0, 0, 0.5)'
                            ] : [
                              'rgba(0, 255, 255, 0)',
                              'rgba(0, 191, 255, 0.5)',
                              'rgba(0, 127, 255, 0.5)',
                              'rgba(0, 63, 255, 0.5)',
                              'rgba(0, 0, 255, 0.5)'
                            ]
                          }
                          radius={25}
                          opacity={0.4}
                          height="500px"
                        />
                      </div>
                    )}
                    {comparisonMode && (
                      <div className="absolute top-4 right-4 bg-white rounded-lg shadow-xl p-3 border-2 border-blue-500">
                        <div className="text-xs font-semibold text-gray-700 mb-2">🔄 Comparing Layers</div>
                        <div className="flex items-center gap-2 text-xs">
                          <div className="flex items-center gap-1">
                            <div className="w-3 h-3 bg-blue-500 rounded" />
                            <span>{activeHeatmapLayer}</span>
                          </div>
                          <span className="text-gray-400">vs</span>
                          <div className="flex items-center gap-1">
                            <div className="w-3 h-3 bg-green-500 rounded opacity-50" />
                            <span>{secondaryLayer}</span>
                          </div>
                        </div>
                      </div>
                    )}
                    {intensityFilter !== 'all' && (
                      <div className="absolute top-4 left-4 bg-white rounded-lg shadow-xl px-3 py-2 border-2 border-orange-500">
                        <div className="text-xs font-semibold text-orange-700">
                          🎯 Showing: {intensityFilter.toUpperCase()} intensity zones
                        </div>
                      </div>
                    )}
                  </div>
                ) : locality ? (
                  <div className="aspect-video bg-gradient-to-br from-blue-50 to-purple-100 rounded-lg shadow-inner flex items-center justify-center relative overflow-hidden">
                    {/* Simulated Heatmap Background */}
                    <div className="absolute inset-0 opacity-30">
                      <div className="grid grid-cols-8 grid-rows-6 h-full w-full gap-1 p-4">
                        {Array.from({length: 48}, (_, i) => {
                          const intensity = Math.sin(i * 0.5) * 0.5 + 0.5;
                          const colors = {
                            price: `rgba(59, 130, 246, ${intensity})`,
                            demand: `rgba(34, 197, 94, ${intensity})`,
                            rental: `rgba(147, 51, 234, ${intensity})`,
                            crime: `rgba(239, 68, 68, ${intensity})`,
                            flood: `rgba(6, 182, 212, ${intensity})`
                          };
                          return (
                            <div 
                              key={i} 
                              className="rounded-sm transition-all duration-1000" 
                              style={{ 
                                backgroundColor: colors[activeHeatmapLayer as keyof typeof colors],
                                animation: `pulse ${2 + (i % 3)}s infinite`
                              }} 
                            />
                          );
                        })}
                      </div>
                    </div>
                    <div className="text-center p-8 relative z-10">
                      <div className="text-6xl mb-4">🗺️</div>
                      <h4 className="text-xl font-bold text-gray-700 mb-2">Heatmap Visualization</h4>
                      <p className="text-gray-600 mb-4">Layer: {activeHeatmapLayer.toUpperCase()}</p>
                      <div className="inline-flex items-center gap-2 px-4 py-2 bg-white/90 backdrop-blur-sm text-gray-800 rounded-lg text-sm border">
                        <div className="w-3 h-3 bg-blue-500 rounded-full animate-pulse" />
                        Simulated heatmap data (Google Maps API required for live data)
                      </div>
                      <p className="text-xs text-gray-600 mt-4 bg-white/80 backdrop-blur-sm rounded px-3 py-1 inline-block">
                        📊 Showing sample {activeHeatmapLayer} distribution patterns
                      </p>
                    </div>
                  </div>
                ) : (
                  <div className="aspect-video bg-white rounded-lg shadow-inner flex items-center justify-center">
                    <div className="text-center text-gray-500">
                      <MapPin className="w-16 h-16 mx-auto mb-4 text-gray-300" />
                      <p>Select a locality to view heatmap analysis</p>
                    </div>
                  </div>
                )}
              </div>

              {/* Heatmap Legend & Stats */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                <div className="p-4 bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl">
                  <div className="flex items-center gap-2 mb-2">
                    <div className="w-3 h-3 bg-blue-500 rounded-full animate-pulse" />
                    <h5 className="font-semibold text-blue-900">Price Range</h5>
                  </div>
                  <p className="text-2xl font-bold text-blue-700">₹{locality?.pricePerSqft?.toLocaleString() || '12,500'}</p>
                  <p className="text-xs text-blue-600">per sq.ft average</p>
                </div>

                <div className="p-4 bg-gradient-to-br from-green-50 to-green-100 rounded-xl">
                  <div className="flex items-center gap-2 mb-2">
                    <div className="w-3 h-3 bg-green-500 rounded-full animate-pulse" />
                    <h5 className="font-semibold text-green-900">Demand Level</h5>
                  </div>
                  <p className="text-2xl font-bold text-green-700">{locality?.demand || 'High'}</p>
                  <p className="text-xs text-green-600">market demand</p>
                </div>

                <div className="p-4 bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl">
                  <div className="flex items-center gap-2 mb-2">
                    <div className="w-3 h-3 bg-purple-500 rounded-full animate-pulse" />
                    <h5 className="font-semibold text-purple-900">Rental Yield</h5>
                  </div>
                  <p className="text-2xl font-bold text-purple-700">{locality?.rentalYield || 3.2}%</p>
                  <p className="text-xs text-purple-600">annual return</p>
                </div>

                <div className="p-4 bg-gradient-to-br from-red-50 to-red-100 rounded-xl">
                  <div className="flex items-center gap-2 mb-2">
                    <div className="w-3 h-3 bg-red-500 rounded-full animate-pulse" />
                    <h5 className="font-semibold text-red-900">Crime Rate</h5>
                  </div>
                  <p className="text-2xl font-bold text-red-700">{locality?.safety?.crimeRatePerMonth || 4}</p>
                  <p className="text-xs text-red-600">incidents/month</p>
                </div>

                <div className="p-4 bg-gradient-to-br from-cyan-50 to-cyan-100 rounded-xl">
                  <div className="flex items-center gap-2 mb-2">
                    <div className="w-3 h-3 bg-cyan-500 rounded-full animate-pulse" />
                    <h5 className="font-semibold text-cyan-900">Flood Risk</h5>
                  </div>
                  <p className="text-2xl font-bold text-cyan-700">{(locality as any)?.floodRisk?.riskLevel || 'Low'}</p>
                  <p className="text-xs text-cyan-600">risk level</p>
                </div>
              </div>

              {(comparisonMode || intensityFilter !== 'all') && (
                <div className="mb-6 p-4 bg-gradient-to-r from-purple-50 to-blue-50 rounded-xl border-2 border-purple-200">
                  <h5 className="font-semibold text-purple-900 mb-3">✨ Active Features</h5>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                    {comparisonMode && (
                      <div className="flex items-center gap-2 text-purple-700">
                        <div className="w-2 h-2 bg-purple-500 rounded-full" />
                        <span>Comparing {activeHeatmapLayer} vs {secondaryLayer}</span>
                      </div>
                    )}
                    {intensityFilter !== 'all' && (
                      <div className="flex items-center gap-2 text-orange-700">
                        <div className="w-2 h-2 bg-orange-500 rounded-full" />
                        <span>Filtered: {intensityFilter} intensity</span>
                      </div>
                    )}
                  </div>
                </div>
              )}

              {/* Heatmap Insights */}
              <div className="mt-6 grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="p-6 bg-blue-50 rounded-xl border border-blue-200">
                  <h5 className="font-semibold text-blue-900 mb-3">💡 Price Insights</h5>
                  <ul className="space-y-2 text-sm text-blue-800">
                    <li>• Highest prices concentrated in central areas</li>
                    <li>• {locality?.oneYearGrowth || 0}% growth in last year</li>
                    <li>• Premium zones near metro stations</li>
                  </ul>
                </div>

                <div className="p-6 bg-green-50 rounded-xl border border-green-200">
                  <h5 className="font-semibold text-green-900 mb-3">📊 Demand Patterns</h5>
                  <ul className="space-y-2 text-sm text-green-800">
                    <li>• {locality?.demand || 'High'} demand in this locality</li>
                    <li>• Strong buyer interest near amenities</li>
                    <li>• Consistent rental demand</li>
                    {comparisonMode && (
                      <li className="text-purple-700 font-medium">• Comparing {activeHeatmapLayer} vs {secondaryLayer} patterns</li>
                    )}
                  </ul>
                </div>
              </div>
            </div>
            )}

            {activeTab === 7 && (
            <div className="bg-white rounded-2xl p-8 shadow">
              <div className="flex items-center justify-between mb-6">
                <h4 className="text-xl font-semibold">
                  Competitive Analysis - Nearby Localities
                </h4>
                <div className="flex gap-2">
                  <button
                    onClick={() => setCompareModalOpen(true)}
                    className="inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-md text-sm font-medium transition-all disabled:pointer-events-none disabled:opacity-50 outline-none focus-visible:border-ring focus-visible:ring-ring/50 focus-visible:ring-[3px] h-9 px-4 py-2 bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
                    disabled={compareSelection.length === 0}
                  >
                    Compare Selected ({compareSelection.length})
                  </button>
                </div>
              </div>

              {!locality ? (
                <div className="col-span-full text-center py-12">
                  <MapPin className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                  <p className="text-gray-500 text-sm">
                    Select a locality to compare
                  </p>
                </div>
              ) : loading ? (
                <div className="col-span-full text-center py-12">
                  <div className="w-12 h-12 border-4 border-gray-300 border-t-blue-600 rounded-full animate-spin mx-auto mb-3" />
                  <p className="text-gray-500 text-sm">Loading nearby localities...</p>
                </div>
              ) : nearby.length === 0 ? (
                <div className="col-span-full text-center py-12">
                  <AlertTriangle className="w-12 h-12 text-yellow-500 mx-auto mb-3" />
                  <p className="text-gray-700 font-medium mb-2">No nearby localities found</p>
                  <p className="text-gray-500 text-sm mb-4">
                    We couldn't find nearby localities for {locality.name}. This might be due to limited data availability.
                  </p>
                  <Button onClick={() => handleGenerate()} variant="outline" size="sm">
                    Retry
                  </Button>
                </div>
              ) : (
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4 lg:gap-6">
                  {(
                    [locality, ...nearby].filter((item): item is LocalityData =>
                      Boolean(item)
                    ) || []
                  ).map((item, idx) => (
                    <div
                      key={idx}
                      className={`p-6 rounded-xl border-2 transition-all hover:shadow-lg ${
                        item.name === locality?.name
                          ? "border-blue-500 bg-blue-50 hover:bg-blue-100"
                          : "border-gray-200 hover:border-gray-300 hover:bg-gray-50"
                      }`}
                    >
                      <div className="flex items-center justify-between mb-4">
                        <h5
                          className="font-bold text-lg truncate"
                          title={item.name}
                        >
                          {item.name}
                        </h5>
                        {item.name === locality?.name && (
                          <span className="px-2 py-1 bg-blue-600 text-white text-xs rounded-full">
                            Current
                          </span>
                        )}
                      </div>

                      <div className="space-y-3">
                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">
                            Price per sqft
                          </span>
                          <span className="font-semibold text-lg">
                            ₹{item.pricePerSqft?.toLocaleString()}
                          </span>
                        </div>

                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">
                            1Y Growth
                          </span>
                          <span
                            className={`font-semibold ${
                              (item.oneYearGrowth || 0) >= 8
                                ? "text-green-600"
                                : (item.oneYearGrowth || 0) >= 5
                                ? "text-yellow-600"
                                : "text-red-600"
                            }`}
                          >
                            {item.oneYearGrowth}%
                          </span>
                        </div>

                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">
                            RIBL Score
                          </span>
                          <div className="flex items-center gap-2">
                            <span className="font-semibold">
                              {item.ribl}/10
                            </span>
                            <div
                              className={`w-2 h-2 rounded-full ${
                                (item.ribl || 0) >= 8
                                  ? "bg-green-500"
                                  : (item.ribl || 0) >= 6
                                  ? "bg-yellow-500"
                                  : "bg-red-500"
                              }`}
                            />
                          </div>
                        </div>

                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">Demand</span>
                          <span
                            className={`px-2 py-1 rounded-full text-xs font-medium ${
                              item.demand === "Very High"
                                ? "bg-red-100 text-red-800"
                                : item.demand === "High"
                                ? "bg-orange-100 text-orange-800"
                                : "bg-gray-100 text-gray-800"
                            }`}
                          >
                            {item.demand}
                          </span>
                        </div>

                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">
                            Rental Yield
                          </span>
                          <span className="font-semibold">
                            {item.rentalYield}%
                          </span>
                        </div>
                      </div>

                      <div className="mt-4 pt-4 border-t">
                        <div className="flex items-center justify-between">
                          <label className="inline-flex items-center gap-2 cursor-pointer">
                            <input
                              type="checkbox"
                              checked={compareSelection.includes(item.name)}
                              onChange={() => handleCompareToggle(item.name)}
                              className="rounded border-gray-300"
                            />
                            <span className="text-sm font-medium">Compare</span>
                          </label>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => {
                              setSelectedLocation(item.name);
                              handleGenerate();
                            }}
                          >
                            View Details
                          </Button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}

              {compareSelection.length > 0 && (
                <div
                  className="mt-6 p-4 bg-blue-50 rounded-lg"
                  role="status"
                  aria-live="polite"
                >
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium text-blue-800">
                      {compareSelection.length} of 3 localities selected
                    </span>
                    <Button
                      onClick={() => setCompareModalOpen(true)}
                      size="sm"
                      aria-label="Open comparison modal"
                    >
                      Compare
                    </Button>
                  </div>
                </div>
              )}
            </div>
            )}
          </div>
          )}

          {/* Modals */}
          {compareModalOpen && (
            <div
              className="fixed inset-0 z-50 flex items-center justify-center p-4"
              role="dialog"
              aria-modal="true"
              aria-labelledby="compare-modal-title"
            >
              <div
                className="absolute inset-0 bg-black/40"
                onClick={() => setCompareModalOpen(false)}
                role="button"
                tabIndex={0}
                onKeyDown={(e) =>
                  e.key === "Escape" && setCompareModalOpen(false)
                }
                aria-label="Close modal"
              />
              <div className="relative bg-white rounded-xl p-6 w-full max-w-3xl z-50 shadow-lg animate-in fade-in zoom-in duration-300">
                <div id="compare-modal-title" className="sr-only">
                  Compare Localities
                </div>
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-lg font-bold">Compare Localities</h3>
                  <button
                    onClick={() => setCompareModalOpen(false)}
                    className="text-gray-500 hover:text-gray-700 font-bold text-xl"
                    aria-label="Close modal"
                  >
                    ×
                  </button>
                </div>
                <div className="mb-4">
                  <p className="text-sm text-gray-600">
                    Pick up to 3 localities to compare (powered by GROQ AI)
                  </p>
                </div>
                <div>
                  <h4 className="font-semibold mb-2">Summary</h4>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    {compareSelection.length === 0 ? (
                      <div className="text-gray-500">
                        No localities selected
                      </div>
                    ) : (
                      compareSelection.map((name) => {
                        const item = [locality, ...nearby].find(l => l?.name === name);
                        
                        if (!item) return (
                          <div key={name} className="p-3 border rounded-lg">
                            <div className="font-bold">{name}</div>
                            <div className="text-sm text-gray-500">Loading...</div>
                          </div>
                        );
                        
                        return (
                          <div key={name} className="p-3 border rounded-lg">
                            <div className="font-bold">{item.name}</div>
                            <div className="text-sm text-gray-600">
                              ₹{item.pricePerSqft} / sqft
                            </div>
                            <div className="text-sm">
                              1Y Growth: {item.oneYearGrowth}%
                            </div>
                            <div className="text-sm">RIBL: {item.ribl}</div>
                            <div className="mt-2">
                              <Button
                                onClick={() => {
                                  setSelectedLocation(item.name);
                                  handleGenerate();
                                  setCompareModalOpen(false);
                                }}
                                className="px-3 py-1 text-sm"
                              >
                                Load
                              </Button>
                            </div>
                          </div>
                        );
                      })
                    )}
                  </div>
                </div>
              </div>
            </div>
          )}

          {/* Booking Modal */}
          {showBookingModal && (
            <div
              className="fixed inset-0 z-50 flex items-center justify-center p-4"
              role="dialog"
              aria-modal="true"
            >
              <div
                className="absolute inset-0 bg-black/40"
                onClick={() => setShowBookingModal(false)}
                role="button"
                tabIndex={0}
                onKeyDown={(e) =>
                  e.key === "Escape" && setShowBookingModal(false)
                }
                aria-label="Close modal"
              />
              <div className="relative bg-white rounded-xl p-6 w-full max-w-md z-10 shadow-lg animate-in fade-in zoom-in duration-300">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-lg font-bold">Book Site Visit</h3>
                  <button
                    onClick={() => setShowBookingModal(false)}
                    className="text-gray-500"
                  >
                    Close
                  </button>
                </div>
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium mb-2">
                      Property
                    </label>
                    <input
                      className="w-full p-2 border rounded"
                      value={locality?.name || ""}
                      readOnly
                    />
                  </div>
                  <div>
                    <label
                      className="block text-sm font-medium mb-2"
                      htmlFor="booking-date"
                    >
                      Preferred Date
                    </label>
                    <input
                      id="booking-date"
                      type="date"
                      className="w-full p-2 border rounded"
                      min={new Date().toISOString().split("T")[0]}
                      required
                      aria-required="true"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-2">
                      Time Slot
                    </label>
                    <select className="w-full p-2 border rounded">
                      <option>10:00 AM - 11:00 AM</option>
                      <option>2:00 PM - 3:00 PM</option>
                      <option>4:00 PM - 5:00 PM</option>
                    </select>
                  </div>
                  <div>
                    <label
                      className="block text-sm font-medium mb-2"
                      htmlFor="booking-phone"
                    >
                      Contact Number
                    </label>
                    <input
                      id="booking-phone"
                      type="tel"
                      className="w-full p-2 border rounded"
                      placeholder="+91 9876543210"
                      pattern="[0-9+\-\s]{10,}"
                      required
                      aria-required="true"
                    />
                  </div>
                  <Button
                    className="w-full"
                    onClick={() => {
                      setBookings([
                        ...bookings,
                        { locality: locality?.name, date: new Date() },
                      ]);
                      setShowBookingModal(false);
                    }}
                  >
                    Confirm Booking
                  </Button>
                </div>
              </div>
            </div>
          )}

          {/* Document Modal */}
          {showDocumentModal && (
            <div
              className="fixed inset-0 z-50 flex items-center justify-center p-4"
              role="dialog"
              aria-modal="true"
            >
              <div
                className="absolute inset-0 bg-black/40"
                onClick={() => setShowDocumentModal(false)}
                role="button"
                tabIndex={0}
                onKeyDown={(e) =>
                  e.key === "Escape" && setShowDocumentModal(false)
                }
                aria-label="Close modal"
              />
              <div className="relative bg-white rounded-xl p-6 w-full max-w-2xl z-10 shadow-lg animate-in fade-in zoom-in duration-300">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-lg font-bold">Document Repository</h3>
                  <button
                    onClick={() => setShowDocumentModal(false)}
                    className="text-gray-500"
                  >
                    Close
                  </button>
                </div>
                <div className="space-y-4">
                  <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                    {[
                      "Property Papers",
                      "NOC Certificate",
                      "Approval Plans",
                      "Tax Receipts",
                      "Legal Documents",
                      "Insurance Papers",
                    ].map((doc, idx) => (
                      <div
                        key={idx}
                        className="p-4 border rounded-lg hover:bg-gray-50 cursor-pointer"
                      >
                        <FileText className="w-8 h-8 text-blue-600 mb-2" />
                        <h4 className="font-medium text-sm">{doc}</h4>
                        <p className="text-xs text-gray-500">PDF • 2.3 MB</p>
                      </div>
                    ))}
                  </div>
                  <div
                    className={`border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
                      dragActive
                        ? "border-blue-500 bg-blue-50"
                        : "border-gray-300 hover:border-blue-400 hover:bg-blue-50"
                    }`}
                    onDragOver={(e) => {
                      e.preventDefault();
                      setDragActive(true);
                    }}
                    onDragLeave={() => setDragActive(false)}
                    onDrop={(e) => {
                      e.preventDefault();
                      setDragActive(false);
                      const files = e.dataTransfer.files;
                      if (files.length > 0) {
                        const input = document.getElementById(
                          "file-upload-modal"
                        ) as HTMLInputElement;
                        if (input) {
                          input.files = files;
                          handleFileUpload({ target: { files } } as any);
                        }
                      }
                    }}
                  >
                    <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                    <p className="text-gray-600">
                      Drag & drop files here or click to upload
                    </p>
                    <input
                      type="file"
                      multiple
                      accept=".pdf,.doc,.docx,.jpg,.jpeg,.png,.gif"
                      onChange={handleFileUpload}
                      className="hidden"
                      id="file-upload-modal"
                      aria-label="Upload documents"
                      disabled={uploadingFile}
                    />
                    <button
                      data-slot="button"
                      className="inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-md text-sm font-medium [&_svg]:pointer-events-none [&_svg:not([class*='size-'])]:size-4 shrink-0 [&_svg]:shrink-0 outline-none focus-visible:border-ring focus-visible:ring-ring/50 focus-visible:ring-[3px] aria-invalid:ring-destructive/20 dark:aria-invalid:ring-destructive/40 aria-invalid:border-destructive border bg-background text-foreground hover:bg-accent hover:text-accent-foreground dark:bg-input/30 dark:border-input dark:hover:bg-input/50 h-9 px-4 py-2 has-[>svg]:px-3 mt-2 cursor-pointer disabled:opacity-50 disabled:cursor-not-allowed disabled:pointer-events-none active:scale-95 transition-transform"
                      aria-busy={uploadingFile}
                      aria-label="Choose files to upload"
                      disabled={uploadingFile}
                      type="button"
                      onClick={() => document.getElementById('file-upload-modal')?.click()}
                    >
                      {uploadingFile ? "Uploading..." : "Choose Files"}
                    </button>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>
      </section>
    </>
  );
}