import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, Area, ComposedChart } from 'recharts';
import { ForecastResponse } from '@/api/forecast';

interface PriceForecastChartProps {
  data: ForecastResponse;
}

export const PriceForecastChart: React.FC<PriceForecastChartProps> = ({ data }) => {
  // Combine historical and forecast data
  const chartData = [
    // Historical data
    ...data.historical_trend.data.map(point => ({
      date: new Date(point.ds).toLocaleDateString('en-US', { month: 'short', year: '2-digit' }),
      historical: Math.round(point.y),
      forecast: null,
      lower: null,
      upper: null,
      type: 'historical'
    })),
    // Forecast data
    ...data.forecast_data.map(point => ({
      date: new Date(point.ds).toLocaleDateString('en-US', { month: 'short', year: '2-digit' }),
      historical: null,
      forecast: Math.round(point.yhat),
      lower: Math.round(point.yhat_lower),
      upper: Math.round(point.yhat_upper),
      type: 'forecast'
    }))
  ];

  const CustomTooltip = ({ active, payload, label }: any) => {
    if (active && payload && payload.length) {
      const data = payload[0].payload;
      return (
        <div className="bg-white p-3 border rounded-lg shadow-lg">
          <p className="font-medium">{label}</p>
          {data.historical && (
            <p className="text-blue-600">
              Historical: ₹{data.historical.toLocaleString()}
            </p>
          )}
          {data.forecast && (
            <>
              <p className="text-green-600">
                Forecast: ₹{data.forecast.toLocaleString()}
              </p>
              <p className="text-gray-500 text-sm">
                Range: ₹{data.lower.toLocaleString()} - ₹{data.upper.toLocaleString()}
              </p>
            </>
          )}
        </div>
      );
    }
    return null;
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          Price Trend & Forecast
          <span className="text-sm font-normal text-gray-500">
            ({data.locality.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())})
          </span>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="h-80">
          <ResponsiveContainer width="100%" height="100%">
            <ComposedChart data={chartData} margin={{ top: 5, right: 30, left: 20, bottom: 5 }}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis 
                dataKey="date" 
                stroke="#666"
                fontSize={12}
                tick={{ fontSize: 12 }}
              />
              <YAxis 
                stroke="#666"
                fontSize={12}
                tick={{ fontSize: 12 }}
                tickFormatter={(value) => `₹${(value / 1000).toFixed(0)}K`}
              />
              <Tooltip content={<CustomTooltip />} />
              
              {/* Historical line */}
              <Line
                type="monotone"
                dataKey="historical"
                stroke="#3b82f6"
                strokeWidth={2}
                dot={false}
                connectNulls={false}
                name="Historical"
              />
              
              {/* Forecast line */}
              <Line
                type="monotone"
                dataKey="forecast"
                stroke="#10b981"
                strokeWidth={2}
                strokeDasharray="5 5"
                dot={false}
                connectNulls={false}
                name="Forecast"
              />
              
              {/* Confidence interval area */}
              <Area
                type="monotone"
                dataKey="upper"
                stroke="none"
                fill="#10b981"
                fillOpacity={0.1}
              />
              <Area
                type="monotone"
                dataKey="lower"
                stroke="none"
                fill="#10b981"
                fillOpacity={0.1}
              />
            </ComposedChart>
          </ResponsiveContainer>
        </div>
        
        {/* Legend */}
        <div className="flex items-center justify-center gap-6 mt-4 text-sm">
          <div className="flex items-center gap-2">
            <div className="w-4 h-0.5 bg-blue-500"></div>
            <span>Historical Data</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-4 h-0.5 bg-green-500 border-dashed border-t-2 border-green-500"></div>
            <span>AI Forecast</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-4 h-2 bg-green-100 border border-green-200"></div>
            <span>Confidence Range</span>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};