import React, { useState } from 'react';
import { api } from '../../api/client';

interface PropertyInputs {
  city: string;
  locality: string;
  propertyType: string;
  bhkConfig: string;
  builtUpArea: number;
  purchasePrice: number;
  furnishingStatus: string;
  propertyAge: string;
}

interface RentalYieldResult {
  monthlyRent: number;
  annualRent: number;
  rentalYield: number;
  roi: number;
  marketComparison: {
    averageYield: number;
    performance: string;
  };
  projections: {
    year1: number;
    year3: number;
    year5: number;
  };
}

export function RentalYieldDebug() {
  const [inputs, setInputs] = useState<PropertyInputs>({
    city: 'Bangalore',
    locality: 'Rajajinagar',
    propertyType: 'apartment',
    bhkConfig: '2',
    builtUpArea: 1200,
    purchasePrice: 8000000,
    furnishingStatus: 'semi-furnished',
    propertyAge: '0-2'
  });

  const [result, setResult] = useState<RentalYieldResult | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const calculateRentalYield = async () => {
    setLoading(true);
    setError(null);
    
    try {
      const response = await api.post('/rental-yield/calculate', inputs);
      console.log('API Response:', response);
      const data = response.data?.data || response.data;
      
      setResult({
        monthlyRent: data.monthlyRent,
        annualRent: data.annualRent,
        rentalYield: data.rentalYield,
        roi: data.roi,
        marketComparison: data.marketComparison,
        projections: data.projections
      });
    } catch (err: any) {
      console.error('Calculation error:', err);
      setError(err.response?.data?.detail || err.message || 'Failed to calculate rental yield');
    } finally {
      setLoading(false);
    }
  };

  const handleInputChange = (field: keyof PropertyInputs, value: string | number) => {
    setInputs(prev => ({ ...prev, [field]: value }));
  };

  const downloadReport = () => {
    if (!result) return;

    // Create PDF content
    const reportHTML = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Rental Yield Analysis Report</title>
  <style>
    body { font-family: Arial, sans-serif; padding: 40px; color: #333; }
    .header { text-align: center; margin-bottom: 30px; border-bottom: 3px solid #0056D2; padding-bottom: 20px; }
    .logo { width: 80px; height: 80px; margin-bottom: 10px; }
    h1 { color: #0056D2; margin: 10px 0; }
    .section { margin: 25px 0; }
    .section-title { background: #0056D2; color: white; padding: 10px; font-size: 16px; font-weight: bold; margin-bottom: 15px; }
    .row { display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #eee; }
    .label { font-weight: 600; color: #555; }
    .value { color: #000; font-weight: 500; }
    .highlight { background: #f0f9ff; padding: 15px; border-left: 4px solid #0056D2; margin: 10px 0; }
    .footer { text-align: center; margin-top: 40px; padding-top: 20px; border-top: 2px solid #eee; color: #666; font-size: 12px; }
  </style>
</head>
<body>
  <div class="header">
    <img src="/favicon.png" class="logo" alt="Logo">
    <h1>RENTAL YIELD ANALYSIS REPORT</h1>
    <p style="color: #666;">Generated on ${new Date().toLocaleString()}</p>
  </div>

  <div class="section">
    <div class="section-title">Property Details</div>
    <div class="row"><span class="label">City:</span><span class="value">${inputs.city}</span></div>
    <div class="row"><span class="label">Locality:</span><span class="value">${inputs.locality}</span></div>
    <div class="row"><span class="label">Property Type:</span><span class="value">${inputs.propertyType}</span></div>
    <div class="row"><span class="label">BHK Configuration:</span><span class="value">${inputs.bhkConfig} BHK</span></div>
    <div class="row"><span class="label">Built-up Area:</span><span class="value">${inputs.builtUpArea} sq ft</span></div>
    <div class="row"><span class="label">Purchase Price:</span><span class="value">₹${inputs.purchasePrice.toLocaleString()}</span></div>
    <div class="row"><span class="label">Furnishing Status:</span><span class="value">${inputs.furnishingStatus}</span></div>
    <div class="row"><span class="label">Property Age:</span><span class="value">${inputs.propertyAge} years</span></div>
  </div>

  <div class="section">
    <div class="section-title">Rental Analysis</div>
    <div class="highlight">
      <div class="row"><span class="label">Monthly Rent:</span><span class="value" style="color: #0056D2; font-size: 18px;">₹${result.monthlyRent.toLocaleString()}</span></div>
    </div>
    <div class="row"><span class="label">Annual Rental Income:</span><span class="value">₹${result.annualRent.toLocaleString()}</span></div>
    <div class="row"><span class="label">Rental Yield:</span><span class="value" style="color: #16a34a; font-weight: bold;">${result.rentalYield.toFixed(2)}%</span></div>
    <div class="row"><span class="label">Return on Investment (ROI):</span><span class="value">${result.roi.toFixed(2)}%</span></div>
  </div>

  <div class="section">
    <div class="section-title">Market Comparison</div>
    <div class="row"><span class="label">Market Average Yield:</span><span class="value">${result.marketComparison.averageYield}%</span></div>
    <div class="row"><span class="label">Your Performance:</span><span class="value" style="color: ${result.marketComparison.performance === 'Above Average' ? '#16a34a' : result.marketComparison.performance === 'Average' ? '#ca8a04' : '#dc2626'}; font-weight: bold;">${result.marketComparison.performance}</span></div>
  </div>

  <div class="section">
    <div class="section-title">Yield Projections (3% Annual Growth)</div>
    <div class="row"><span class="label">Year 1:</span><span class="value">${result.projections.year1.toFixed(2)}%</span></div>
    <div class="row"><span class="label">Year 3:</span><span class="value">${result.projections.year3.toFixed(2)}%</span></div>
    <div class="row"><span class="label">Year 5:</span><span class="value">${result.projections.year5.toFixed(2)}%</span></div>
  </div>

  <div class="footer">
    <p>This report is generated based on market data and projections. Actual results may vary.</p>
    <p>© ${new Date().getFullYear()} - All Rights Reserved</p>
  </div>
</body>
</html>
    `;

    // Create a new window and print to PDF
    const printWindow = window.open('', '_blank');
    if (printWindow) {
      printWindow.document.write(reportHTML);
      printWindow.document.close();
      printWindow.onload = () => {
        printWindow.print();
      };
    }
  };

  return (
    <div className="bg-gray-50 py-16">
      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Input Form */}
          <div className="bg-white rounded-lg shadow-lg p-6">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Property Details</h2>
            
            <div className="space-y-4">
              {/* Location */}
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">City</label>
                  <select
                    value={inputs.city}
                    onChange={(e) => handleInputChange('city', e.target.value)}
                    className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                  >
                    <option value="Bangalore">Bangalore</option>
                    <option value="Mumbai">Mumbai</option>
                    <option value="Delhi">Delhi</option>
                    <option value="Chennai">Chennai</option>
                    <option value="Hyderabad">Hyderabad</option>
                    <option value="Pune">Pune</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Locality</label>
                  <input
                    type="text"
                    value={inputs.locality}
                    onChange={(e) => handleInputChange('locality', e.target.value)}
                    placeholder="Rajajinagar"
                    className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                  />
                </div>
              </div>

              {/* Property Type */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
                <select
                  value={inputs.propertyType}
                  onChange={(e) => handleInputChange('propertyType', e.target.value)}
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                >
                  <option value="apartment">Apartment</option>
                  <option value="independent-house">Independent House</option>
                  <option value="villa">Villa</option>
                </select>
              </div>

              {/* BHK and Area */}
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">BHK Configuration</label>
                  <select
                    value={inputs.bhkConfig}
                    onChange={(e) => handleInputChange('bhkConfig', e.target.value)}
                    className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                  >
                    <option value="1">1 BHK</option>
                    <option value="2">2 BHK</option>
                    <option value="3">3 BHK</option>
                    <option value="4">4 BHK</option>
                    <option value="5+">5+ BHK</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Built-up Area (sq ft)</label>
                  <input
                    type="number"
                    value={inputs.builtUpArea}
                    onChange={(e) => handleInputChange('builtUpArea', parseInt(e.target.value))}
                    className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                  />
                </div>
              </div>

              {/* Purchase Price */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Property Purchase Price (₹)</label>
                <input
                  type="number"
                  value={inputs.purchasePrice}
                  onChange={(e) => handleInputChange('purchasePrice', parseInt(e.target.value))}
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                />
              </div>

              {/* Furnishing and Age */}
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Furnishing Status</label>
                  <select
                    value={inputs.furnishingStatus}
                    onChange={(e) => handleInputChange('furnishingStatus', e.target.value)}
                    className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                  >
                    <option value="unfurnished">Unfurnished</option>
                    <option value="semi-furnished">Semi-furnished</option>
                    <option value="fully-furnished">Fully Furnished</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Property Age</label>
                  <select
                    value={inputs.propertyAge}
                    onChange={(e) => handleInputChange('propertyAge', e.target.value)}
                    className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none"
                  >
                    <option value="0-2">0-2 years</option>
                    <option value="3-5">3-5 years</option>
                    <option value="6-10">6-10 years</option>
                    <option value="11-15">11-15 years</option>
                    <option value="16+">16+ years</option>
                  </select>
                </div>
              </div>

              <div className="mt-6">
                <button
                  onClick={calculateRentalYield}
                  disabled={loading}
                  className="w-full bg-[#0056D2] text-white py-4 px-6 rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed font-semibold text-lg shadow-md transition-all duration-200"
                >
                  {loading ? 'Calculating...' : 'Calculate Rental Yield'}
                </button>
              </div>
            </div>
          </div>

          {/* Results */}
          <div className="bg-white rounded-lg shadow-lg p-6">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Rental Yield Analysis</h2>
            
            {error && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                <p className="text-red-700">{error}</p>
              </div>
            )}

            {result ? (
              <div className="space-y-6">
                {/* Key Metrics */}
                <div className="grid grid-cols-2 gap-4">
                  <div className="bg-blue-50 p-4 rounded-lg">
                    <h3 className="text-sm font-medium text-gray-600">Monthly Rent</h3>
                    <p className="text-2xl font-bold text-blue-600">₹{result.monthlyRent.toLocaleString()}</p>
                  </div>
                  <div className="bg-green-50 p-4 rounded-lg">
                    <h3 className="text-sm font-medium text-gray-600">Rental Yield</h3>
                    <p className="text-2xl font-bold text-green-600">{result.rentalYield.toFixed(2)}%</p>
                  </div>
                </div>

                {/* Annual Metrics */}
                <div className="bg-gray-50 p-4 rounded-lg">
                  <h3 className="text-sm font-medium text-gray-600 mb-2">Annual Rental Income</h3>
                  <p className="text-xl font-semibold">₹{result.annualRent.toLocaleString()}</p>
                </div>

                {/* Market Comparison */}
                <div className="border-t pt-4">
                  <h3 className="font-medium mb-3">Market Comparison</h3>
                  <div className="flex justify-between items-center">
                    <span className="text-gray-600">Market Average:</span>
                    <span className="font-medium">{result.marketComparison.averageYield}%</span>
                  </div>
                  <div className="flex justify-between items-center mt-2">
                    <span className="text-gray-600">Your Performance:</span>
                    <span className={`font-medium ${
                      result.marketComparison.performance === 'Above Average' ? 'text-green-600' :
                      result.marketComparison.performance === 'Average' ? 'text-yellow-600' : 'text-red-600'
                    }`}>
                      {result.marketComparison.performance}
                    </span>
                  </div>
                </div>

                {/* Projections */}
                <div className="border-t pt-4">
                  <h3 className="font-medium mb-3">Yield Projections</h3>
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Year 1:</span>
                      <span className="font-medium">{result.projections.year1.toFixed(2)}%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Year 3:</span>
                      <span className="font-medium">{result.projections.year3.toFixed(2)}%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Year 5:</span>
                      <span className="font-medium">{result.projections.year5.toFixed(2)}%</span>
                    </div>
                  </div>
                  <button
                    onClick={downloadReport}
                    className="w-full mt-4 bg-[#0056D2] text-white py-3 px-4 rounded-lg hover:bg-blue-700 font-medium transition-all duration-200 flex items-center justify-center gap-2"
                  >
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    Download PDF Report
                  </button>
                </div>
              </div>
            ) : (
              <div className="text-center text-gray-500 py-12">
                <p>Enter property details and click "Calculate Rental Yield" to see analysis</p>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}