import React, { useState, useEffect } from 'react';
import { User, Phone, Mail, Calendar, CheckCircle, Building, IndianRupee, Loader2 } from 'lucide-react';
import { getAgentProfile, type AgentProfile } from '../../api/agentProfile';

const AgentProfile: React.FC = () => {
  const [profile, setProfile] = useState<AgentProfile | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchAgentData = async () => {
      try {
        setLoading(true);
        const profileData = await getAgentProfile();
        setProfile(profileData);
      } catch (err: any) {
        console.error('Error fetching agent data:', err);
        setError(err.message || 'Failed to load agent data');
      } finally {
        setLoading(false);
      }
    };

    fetchAgentData();
  }, []);

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <Loader2 className="w-8 h-8 animate-spin text-[#2B256D]" />
        <span className="ml-2 text-gray-600">Loading profile...</span>
      </div>
    );
  }

  if (error || !profile) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-lg p-6">
        <h3 className="text-red-800 font-semibold mb-2">Error Loading Profile</h3>
        <p className="text-red-600">{error || 'Profile not found'}</p>
      </div>
    );
  }
  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-6">Agent Profile</h2>
        
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Profile Picture and Basic Info */}
          <div className="flex flex-col items-center lg:items-start space-y-4">
            <div className="w-32 h-32 rounded-full border-4 border-[#2B256D]/20 bg-[#2B256D] text-white flex items-center justify-center text-5xl font-bold">
              {profile.profile_image_url ? (
                <img 
                  src={profile.profile_image_url} 
                  alt={profile.full_name}
                  className="w-full h-full rounded-full object-cover"
                />
              ) : (
                profile.full_name?.charAt(0) || 'A'
              )}
            </div>
            <div className="text-center lg:text-left">
              <h3 className="text-xl font-semibold text-gray-900">{profile.full_name}</h3>
              <div className="flex items-center space-x-2 mt-2">
                <CheckCircle className={`w-5 h-5 ${profile.status === 'active' ? 'text-green-500' : 'text-yellow-500'}`} />
                <span className={`text-sm font-medium ${profile.status === 'active' ? 'text-green-600' : 'text-yellow-600'}`}>
                  {profile.status === 'active' ? 'Verified' : profile.status === 'pending_verification' ? 'Pending Verification' : 'Inactive'}
                </span>
              </div>
            </div>
          </div>
          
          {/* Contact Information */}
          <div className="flex-1 space-y-6">
            <div>
              <h4 className="text-lg font-semibold text-gray-900 mb-4">Contact Information</h4>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Mail className="w-5 h-5 text-[#2B256D]" />
                  <div>
                    <p className="text-sm text-gray-500">Email</p>
                    <p className="font-medium text-gray-900">{profile.email}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Phone className="w-5 h-5 text-green-600" />
                  <div>
                    <p className="text-sm text-gray-500">Phone</p>
                    <p className="font-medium text-gray-900">{profile.phone_number}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Calendar className="w-5 h-5 text-purple-600" />
                  <div>
                    <p className="text-sm text-gray-500">Join Date</p>
                    <p className="font-medium text-gray-900">{new Date(profile.created_at).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' })}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <User className="w-5 h-5 text-orange-600" />
                  <div>
                    <p className="text-sm text-gray-500">Agent ID</p>
                    <p className="font-medium text-gray-900">#{profile.id}</p>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Professional Information */}
            <div>
              <h4 className="text-lg font-semibold text-gray-900 mb-4">Professional Information</h4>
              <div className="grid md:grid-cols-2 gap-4">
                {profile.agency_name && (
                  <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                    <Building className="w-5 h-5 text-blue-600" />
                    <div>
                      <p className="text-sm text-gray-500">Agency</p>
                      <p className="font-medium text-gray-900">{profile.agency_name}</p>
                    </div>
                  </div>
                )}
                {profile.license_number && (
                  <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                    <CheckCircle className="w-5 h-5 text-green-600" />
                    <div>
                      <p className="text-sm text-gray-500">License Number</p>
                      <p className="font-medium text-gray-900">{profile.license_number}</p>
                    </div>
                  </div>
                )}
                {profile.experience_years && (
                  <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                    <Calendar className="w-5 h-5 text-purple-600" />
                    <div>
                      <p className="text-sm text-gray-500">Experience</p>
                      <p className="font-medium text-gray-900">{profile.experience_years} years</p>
                    </div>
                  </div>
                )}
                {profile.commission_rate && (
                  <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                    <IndianRupee className="w-5 h-5 text-green-600" />
                    <div>
                      <p className="text-sm text-gray-500">Commission Rate</p>
                      <p className="font-medium text-gray-900">{profile.commission_rate}%</p>
                    </div>
                  </div>
                )}
              </div>
            </div>
            
            {/* Performance Stats */}
            <div>
              <h4 className="text-lg font-semibold text-gray-900 mb-4">Performance Overview</h4>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="bg-gradient-to-r from-[#2B256D]/10 to-[#2B256D]/20 p-6 rounded-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-[#2B256D] text-sm font-medium">Total Properties</p>
                      <p className="text-3xl font-bold text-[#2B256D]">{profile.total_properties}</p>
                    </div>
                    <div className="w-12 h-12 bg-[#2B256D]/20 rounded-full flex items-center justify-center">
                      <Building className="w-6 h-6 text-[#2B256D]" />
                    </div>
                  </div>
                </div>
                <div className="bg-gradient-to-r from-green-50 to-green-100 p-6 rounded-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-green-600 text-sm font-medium">Total Earnings</p>
                      <p className="text-3xl font-bold text-green-900">₹{profile.total_earnings.toLocaleString()}</p>
                    </div>
                    <div className="w-12 h-12 bg-green-200 rounded-full flex items-center justify-center">
                      <IndianRupee className="w-6 h-6 text-green-600" />
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      {/* Status Card */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Account Status</h3>
        <div className={`p-4 rounded-lg border-2 ${
          profile.status === 'active' 
            ? 'border-green-200 bg-green-50' 
            : profile.status === 'pending_verification'
            ? 'border-yellow-200 bg-yellow-50'
            : 'border-red-200 bg-red-50'
        }`}>
          <div className="flex items-center space-x-3">
            <CheckCircle className={`w-5 h-5 ${
              profile.status === 'active' 
                ? 'text-green-500' 
                : profile.status === 'pending_verification'
                ? 'text-yellow-500'
                : 'text-red-500'
            }`} />
            <div>
              <p className={`font-medium ${
                profile.status === 'active' 
                  ? 'text-green-800' 
                  : profile.status === 'pending_verification'
                  ? 'text-yellow-800'
                  : 'text-red-800'
              }`}>
                {profile.status === 'active' ? 'Account Active' : 
                 profile.status === 'pending_verification' ? 'Pending Verification' : 'Account Inactive'}
              </p>
              <p className={`text-sm ${
                profile.status === 'active' 
                  ? 'text-green-600' 
                  : profile.status === 'pending_verification'
                  ? 'text-yellow-600'
                  : 'text-red-600'
              }`}>
                {profile.status === 'active' 
                  ? 'Your account is verified and you have full access to all features.' 
                  : profile.status === 'pending_verification'
                  ? 'Your account is under review. Some features may be limited.'
                  : 'Your account is inactive. Please contact support.'}
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* Business Details */}
      {(profile.office_address || profile.languages_spoken?.length || profile.preferred_contact_method) && (
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4">Business Details</h3>
          <div className="space-y-4">
            {profile.office_address && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-1">Office Address</p>
                <p className="font-medium text-gray-900">{profile.office_address}</p>
              </div>
            )}
            {profile.languages_spoken?.length > 0 && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-2">Languages Spoken</p>
                <div className="flex flex-wrap gap-2">
                  {profile.languages_spoken.map((lang, index) => (
                    <span key={index} className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">
                      {lang}
                    </span>
                  ))}
                </div>
              </div>
            )}
            {profile.preferred_contact_method && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-1">Preferred Contact Method</p>
                <p className="font-medium text-gray-900">{profile.preferred_contact_method}</p>
              </div>
            )}
          </div>
        </div>
      )}
      
      {/* Specialization & Service Areas */}
      {(profile.specialization?.length > 0 || profile.service_areas?.length > 0) && (
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4">Expertise</h3>
          <div className="space-y-4">
            {profile.specialization?.length > 0 && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-2">Specialization</p>
                <div className="flex flex-wrap gap-2">
                  {profile.specialization.map((spec, index) => (
                    <span key={index} className="px-3 py-1 bg-[#2B256D] text-white text-sm rounded-full">
                      {spec}
                    </span>
                  ))}
                </div>
              </div>
            )}
            {profile.service_areas?.length > 0 && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-2">Service Areas</p>
                <div className="flex flex-wrap gap-2">
                  {profile.service_areas.map((area, index) => (
                    <span key={index} className="px-3 py-1 bg-green-100 text-green-800 text-sm rounded-full">
                      {area}
                    </span>
                  ))}
                </div>
              </div>
            )}
          </div>
        </div>
      )}
      
      {/* Bio Section */}
      {profile.bio && (
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4">About Me</h3>
          <p className="text-gray-700 leading-relaxed">{profile.bio}</p>
        </div>
      )}
      
      {/* Additional Information */}
      {(profile.education || profile.certifications?.length || profile.website_url) && (
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4">Additional Information</h3>
          <div className="space-y-4">
            {profile.education && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-1">Education</p>
                <p className="font-medium text-gray-900">{profile.education}</p>
              </div>
            )}
            {profile.certifications?.length > 0 && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-2">Certifications</p>
                <div className="flex flex-wrap gap-2">
                  {profile.certifications.map((cert, index) => (
                    <span key={index} className="px-3 py-1 bg-yellow-100 text-yellow-800 text-sm rounded-full">
                      {cert}
                    </span>
                  ))}
                </div>
              </div>
            )}
            {profile.website_url && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-1">Website</p>
                <a href={profile.website_url} target="_blank" rel="noopener noreferrer" className="font-medium text-blue-600 hover:text-blue-800">
                  {profile.website_url}
                </a>
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
};

export default AgentProfile;