import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { User, Building, Award, MapPin, Upload, Check, X } from 'lucide-react';
import { completeAgentProfile, getAgentProfile, getAgentProfileCompletionStatus, type AgentProfileCreate } from '../../api/agentProfile';

interface AgentProfileData extends AgentProfileCreate {
  specialization: string[];
  service_areas: string[];
  social_media_links: Record<string, string>;
}

const AgentProfileCompletion: React.FC = () => {
  const navigate = useNavigate();
  const [currentStep, setCurrentStep] = useState(1);
  const [loading, setLoading] = useState(false); // For form submission only
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');
  const [completionStatus, setCompletionStatus] = useState<any>(null);

  const [profileData, setProfileData] = useState<AgentProfileData>({
    full_name: '',
    email: '',
    phone_number: '+91',
    license_number: '',
    agency_name: '',
    experience_years: 0,
    specialization: [],
    service_areas: [],
    bio: '',
    profile_image_url: '',
    website_url: '',
    social_media_links: {},
    office_address: '',
    languages_spoken: [],
    certifications: [],
    education: '',
    commission_rate: 0,
    availability_hours: '',
    preferred_contact_method: '',
    business_registration_number: ''
  });

  const specializationOptions = ['Residential', 'Commercial', 'Luxury', 'Industrial', 'Land', 'Rental'];
  const serviceAreaOptions = ['Bangalore', 'Mumbai', 'Delhi', 'Chennai', 'Hyderabad', 'Pune', 'Kolkata'];
  const languageOptions = ['English', 'Hindi', 'Kannada', 'Tamil', 'Telugu', 'Marathi', 'Bengali', 'Gujarati'];
  const certificationOptions = ['RERA Certified', 'Real Estate License', 'Property Management', 'Investment Advisory', 'Legal Compliance'];
  const contactMethodOptions = ['Phone', 'Email', 'WhatsApp', 'Video Call'];

  useEffect(() => {
    loadExistingProfile();
  }, []);

  const loadExistingProfile = async () => {
    try {
      const [profile, status] = await Promise.all([
        getAgentProfile().catch(() => null),
        getAgentProfileCompletionStatus().catch(() => null)
      ]);

      console.log('Loaded profile:', profile);
      console.log('Completion status:', status);

      if (profile) {
        const updatedData = {
          full_name: profile?.full_name || '',
          email: profile?.email || '',
          phone_number: profile?.phone_number || '+91',
          license_number: profile?.license_number || '',
          agency_name: profile?.agency_name || '',
          experience_years: profile?.experience_years || 0,
          specialization: Array.isArray(profile?.specialization) ? profile?.specialization : [],
          service_areas: Array.isArray(profile?.service_areas) ? profile?.service_areas : [],
          bio: profile?.bio || '',
          profile_image_url: profile?.profile_image_url || '',
          website_url: profile?.website_url || '',
          social_media_links: profile?.social_media_links || {},
          office_address: profile?.office_address || '',
          languages_spoken: Array.isArray(profile?.languages_spoken) ? profile?.languages_spoken : [],
          certifications: Array.isArray(profile?.certifications) ? profile?.certifications : [],
          education: profile?.education || '',
          commission_rate: profile?.commission_rate || 0,
          availability_hours: profile?.availability_hours || '',
          preferred_contact_method: profile?.preferred_contact_method || '',
          business_registration_number: profile?.business_registration_number || ''
        };
        console.log('Setting profile data:', updatedData);
        setProfileData(updatedData);
        // Force re-render to ensure form fields update
        setTimeout(() => setProfileData(prev => ({...prev})), 100);
      }

      if (status) {
        setCompletionStatus(status);
      }
    } catch (err) {
      console.error('Error loading profile:', err);
    }
  };

  const handleInputChange = (field: string, value: any) => {
    setProfileData(prev => ({ ...prev, [field]: value }));
    setError('');
  };

  const handleArrayToggle = (field: 'specialization' | 'service_areas' | 'languages_spoken' | 'certifications', value: string) => {
    setProfileData(prev => {
      const currentArray = prev[field] || [];
      return {
        ...prev,
        [field]: currentArray.includes(value)
          ? currentArray.filter(item => item !== value)
          : [...currentArray, value]
      };
    });
  };

  const handleSocialMediaChange = (platform: string, url: string) => {
    setProfileData(prev => ({
      ...prev,
      social_media_links: { ...prev.social_media_links, [platform]: url }
    }));
  };

  const validateStep = (step: number): boolean => {
    switch (step) {
      case 1:
        return !!(profileData.full_name && profileData.email && profileData.phone_number);
      case 2:
        return !!(profileData.agency_name && profileData.experience_years !== undefined && profileData.specialization.length > 0);
      case 3:
        return !!(profileData.office_address && (profileData.languages_spoken?.length || 0) > 0);
      default:
        return true;
    }
  };

  const handleNext = () => {
    if (validateStep(currentStep)) {
      setCurrentStep(prev => prev + 1);
    } else {
      setError('Please fill all required fields');
    }
  };

  const handleSubmit = async () => {
    setLoading(true);
    setError('');

    try {
      const payload = { profile_data: profileData };
      await completeAgentProfile(payload);
      setSuccess('Profile completed successfully!');
      sessionStorage.setItem('profileJustCompleted', 'true');
      localStorage.setItem('agentProfileCompleted', 'true');
      setTimeout(() => navigate('/agent/dashboard', { replace: true }), 1500);
    } catch (err: any) {
      setError(err.message || 'Failed to complete profile');
    } finally {
      setLoading(false);
    }
  };

  const steps = [
    { number: 1, title: 'Basic Info', icon: User },
    { number: 2, title: 'Professional', icon: Award },
    { number: 3, title: 'Business Details', icon: Building },
    { number: 4, title: 'Profile & Media', icon: MapPin }
  ];

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4">
        <div className="bg-white rounded-2xl shadow-xl overflow-hidden">
          {/* Header */}
          <div className="px-8 py-6 border-b border-gray-200">
            <h1 className="text-2xl font-bold text-gray-900">Complete Your Agent Profile</h1>
            <p className="text-gray-600 mt-1">Build your professional presence</p>
            
            {/* Progress */}
            <div className="flex items-center justify-between mt-6">
              {steps.map((step, index) => (
                <React.Fragment key={step.number}>
                  <div className="flex items-center">
                    <div className={`w-10 h-10 rounded-full flex items-center justify-center ${
                      currentStep >= step.number ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-500'
                    }`}>
                      <step.icon className="w-5 h-5" />
                    </div>
                    <span className={`ml-2 text-sm font-medium ${
                      currentStep >= step.number ? 'text-blue-600' : 'text-gray-500'
                    }`}>
                      {step.title}
                    </span>
                  </div>
                  {index < steps.length - 1 && (
                    <div className={`flex-1 h-1 mx-4 ${
                      currentStep > step.number ? 'bg-blue-600' : 'bg-gray-200'
                    }`} />
                  )}
                </React.Fragment>
              ))}
            </div>
          </div>

          {/* Content */}
          <div className="p-8">
            {error && (
              <div className="mb-6 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                {error}
              </div>
            )}

            {success && (
              <div className="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg flex items-center">
                <Check className="w-4 h-4 mr-2" />
                {success}
              </div>
            )}

            {/* Step 1: Professional Info */}
            {currentStep === 1 && (
              <div className="space-y-6">
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Full Name *
                    </label>
                    <input
                      type="text"
                      value={profileData.full_name}
                      onChange={(e) => handleInputChange('full_name', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Email *
                    </label>
                    <input
                      type="email"
                      value={profileData.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                </div>

                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Phone Number *
                    </label>
                    <input
                      type="tel"
                      value={profileData.phone_number}
                      onChange={(e) => handleInputChange('phone_number', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      License Number
                    </label>
                    <input
                      type="text"
                      value={profileData.license_number}
                      onChange={(e) => handleInputChange('license_number', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                </div>
              </div>
            )}

            {/* Step 2: Experience */}
            {currentStep === 2 && (
              <div className="space-y-6">
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Agency Name *
                    </label>
                    <input
                      type="text"
                      value={profileData.agency_name}
                      onChange={(e) => handleInputChange('agency_name', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Experience (Years) *
                    </label>
                    <select
                      value={profileData.experience_years}
                      onChange={(e) => handleInputChange('experience_years', parseInt(e.target.value))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    >
                      <option value={0}>0-1 years</option>
                      <option value={2}>2-5 years</option>
                      <option value={6}>6-10 years</option>
                      <option value={11}>11-20 years</option>
                      <option value={21}>20+ years</option>
                    </select>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Specialization *
                  </label>
                  <div className="grid grid-cols-3 gap-3">
                    {specializationOptions.map(option => (
                      <label key={option} className="flex items-center space-x-2 cursor-pointer">
                        <input
                          type="checkbox"
                          checked={profileData.specialization.includes(option)}
                          onChange={() => handleArrayToggle('specialization', option)}
                          className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <span className="text-sm text-gray-700">{option}</span>
                      </label>
                    ))}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Service Areas
                  </label>
                  <div className="grid grid-cols-3 gap-3">
                    {serviceAreaOptions.map(area => (
                      <label key={area} className="flex items-center space-x-2 cursor-pointer">
                        <input
                          type="checkbox"
                          checked={profileData.service_areas.includes(area)}
                          onChange={() => handleArrayToggle('service_areas', area)}
                          className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <span className="text-sm text-gray-700">{area}</span>
                      </label>
                    ))}
                  </div>
                </div>
              </div>
            )}

            {/* Step 3: Business Details */}
            {currentStep === 3 && (
              <div className="space-y-6">

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Office Address *
                  </label>
                  <textarea
                    value={profileData.office_address}
                    onChange={(e) => handleInputChange('office_address', e.target.value)}
                    rows={3}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="Enter your office address..."
                    required
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Languages Spoken *
                  </label>
                  <div className="grid grid-cols-3 gap-3">
                    {languageOptions.map(language => (
                      <label key={language} className="flex items-center space-x-2 cursor-pointer">
                        <input
                          type="checkbox"
                          checked={profileData.languages_spoken?.includes(language) || false}
                          onChange={() => handleArrayToggle('languages_spoken', language)}
                          className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <span className="text-sm text-gray-700">{language}</span>
                      </label>
                    ))}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Certifications
                  </label>
                  <div className="grid grid-cols-2 gap-3">
                    {certificationOptions.map(cert => (
                      <label key={cert} className="flex items-center space-x-2 cursor-pointer">
                        <input
                          type="checkbox"
                          checked={profileData.certifications?.includes(cert) || false}
                          onChange={() => handleArrayToggle('certifications', cert)}
                          className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <span className="text-sm text-gray-700">{cert}</span>
                      </label>
                    ))}
                  </div>
                </div>

                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Education Background
                    </label>
                    <input
                      type="text"
                      value={profileData.education}
                      onChange={(e) => handleInputChange('education', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      placeholder="e.g., MBA in Real Estate, Bachelor's in Business"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Commission Rate (%)
                    </label>
                    <input
                      type="number"
                      step="0.1"
                      min="0"
                      max="10"
                      value={profileData.commission_rate || ''}
                      onChange={(e) => handleInputChange('commission_rate', e.target.value ? parseFloat(e.target.value) : 0)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      placeholder="2.5"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Availability Hours
                    </label>
                    <input
                      type="text"
                      value={profileData.availability_hours}
                      onChange={(e) => handleInputChange('availability_hours', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      placeholder="Mon-Fri 9AM-6PM, Sat 10AM-4PM"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Preferred Contact Method
                    </label>
                    <select
                      value={profileData.preferred_contact_method}
                      onChange={(e) => handleInputChange('preferred_contact_method', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    >
                      <option value="">Select method</option>
                      {contactMethodOptions.map(method => (
                        <option key={method} value={method}>{method}</option>
                      ))}
                    </select>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Business Registration Number
                  </label>
                  <input
                    type="text"
                    value={profileData.business_registration_number}
                    onChange={(e) => handleInputChange('business_registration_number', e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="Enter your business registration number"
                  />
                </div>
              </div>
            )}

            {/* Step 4: Profile & Media */}
            {currentStep === 4 && (
              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Professional Bio
                  </label>
                  <textarea
                    value={profileData.bio}
                    onChange={(e) => handleInputChange('bio', e.target.value)}
                    rows={4}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="Tell clients about your expertise and experience..."
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Profile Image URL
                  </label>
                  <input
                    type="url"
                    value={profileData.profile_image_url}
                    onChange={(e) => handleInputChange('profile_image_url', e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="https://example.com/your-photo.jpg"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Website URL
                  </label>
                  <input
                    type="url"
                    value={profileData.website_url}
                    onChange={(e) => handleInputChange('website_url', e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="https://your-website.com"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Social Media Links
                  </label>
                  <div className="space-y-3">
                    {['linkedin', 'facebook', 'instagram'].map(platform => (
                      <div key={platform} className="flex items-center space-x-3">
                        <span className="w-20 text-sm text-gray-600 capitalize">{platform}:</span>
                        <input
                          type="url"
                          value={profileData.social_media_links[platform] || ''}
                          onChange={(e) => handleSocialMediaChange(platform, e.target.value)}
                          className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                          placeholder={`https://${platform}.com/yourprofile`}
                        />
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            )}
          </div>

          {/* Footer */}
          <div className="px-8 py-6 border-t border-gray-200 bg-gray-50">
            <div className="flex justify-between">
              <button
                onClick={() => setCurrentStep(prev => prev - 1)}
                disabled={currentStep === 1}
                className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Back
              </button>
              
              {currentStep < 4 ? (
                <button
                  onClick={handleNext}
                  className="px-6 py-2 bg-blue-600 text-black rounded-lg hover:bg-blue-700"
                >
                  Next
                </button>
              ) : (
                <button
                  onClick={handleSubmit}
                  disabled={loading}
                  className="px-6 py-2 bg-blue-600 text-black rounded-lg hover:bg-blue-700 disabled:opacity-50 flex items-center"
                >
                  {loading && <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2" />}
                  {loading ? 'Completing...' : 'Complete Profile'}
                </button>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default AgentProfileCompletion;