import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { getAgentProfile, getAgentProfileCompletionStatus } from '../../api/agentProfile';

interface AgentProfileGuardProps {
  children: React.ReactNode;
}

const AgentProfileGuard: React.FC<AgentProfileGuardProps> = ({ children }) => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(true);
  const [shouldShowProfileCompletion, setShouldShowProfileCompletion] = useState(false);

  useEffect(() => {
    checkProfileStatus();
  }, []);

  const checkProfileStatus = async () => {
    try {
      // Check if user is authenticated
      const agentToken = localStorage.getItem('agentToken');
      if (!agentToken) {
        navigate('/agent', { replace: true });
        return;
      }

      const [profile, status] = await Promise.all([
        getAgentProfile().catch(() => null),
        getAgentProfileCompletionStatus().catch(() => null)
      ]);

      // If profile is already completed, redirect to dashboard
      if (status && status.completion_percentage >= 100) {
        navigate('/agent/dashboard', { replace: true });
        return;
      }

      // If profile exists and has essential data, also redirect to dashboard
      if (profile && profile.full_name && profile.email && profile.agency_name) {
        navigate('/agent/dashboard', { replace: true });
        return;
      }

      // If we reach here, profile completion is needed
      setShouldShowProfileCompletion(true);
    } catch (error) {
      console.error('Error checking profile status:', error);
      // On error, allow profile completion to show
      setShouldShowProfileCompletion(true);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="w-8 h-8 border-4 border-blue-600 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-gray-600">Checking profile status...</p>
        </div>
      </div>
    );
  }

  if (!shouldShowProfileCompletion) {
    return null; // Will redirect via useEffect
  }

  return <>{children}</>;
};

export default AgentProfileGuard;