import React, { useState, useEffect } from 'react';
import { Eye, Heart, Calendar, TrendingUp, Users, Activity, Clock, MapPin, BarChart3, X } from 'lucide-react';

interface PropertySummary {
  property_id: number;
  title: string;
  location: string;
  property_type: string;
  price: number;
  status: string;
  metrics: {
    views: number;
    saves: number;
    visits: number;
    wishlist: number;
    unique_users: number;
    engagement_score: number;
  };
}

interface EngagedUser {
  user_id: string;
  email?: string;
  phone?: string;
  name: string;
  activity: {
    views: number;
    saves: number;
    visits: number;
    wishlist: number;
  };
  engagement_level: 'high' | 'medium' | 'low';
  engagement_score: number;
  segment: string;
  last_interaction: string;
  first_interaction: string;
}

const getAuthHeaders = () => {
  const token = localStorage.getItem('agentToken') || localStorage.getItem('access_token');
  return {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json'
  };
};

const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export const AgentPropertyAnalytics: React.FC = () => {
  const [portfolio, setPortfolio] = useState<any>(null);
  const [selectedProperty, setSelectedProperty] = useState<number | null>(null);
  const [propertyDetails, setPropertyDetails] = useState<any>(null);
  const [engagedUsers, setEngagedUsers] = useState<EngagedUser[]>([]);
  const [insights, setInsights] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [dateRange, setDateRange] = useState<7 | 30 | 90>(30);
  const [showDetailsModal, setShowDetailsModal] = useState(false);

  useEffect(() => {
    loadPortfolioData();
  }, [dateRange]);

  const loadPortfolioData = async () => {
    try {
      setLoading(true);
      
      // Use the same API as manage properties page for agents
      const agentId = localStorage.getItem('agentId');
      const propertiesUrl = agentId 
        ? `${API_BASE}/properties/?listed_by_id=${agentId}`
        : `${API_BASE}/properties/`;
      
      const response = await fetch(propertiesUrl);
      if (!response.ok) {
        throw new Error('Failed to fetch properties');
      }
      
      const payload = await response.json();
      const allProperties = Array.isArray(payload) ? payload : (payload.properties ?? []);
      
      const activeProperties = allProperties.filter(
        (p: any) => p.status === "active"
      );
      
      console.log('Total properties from API:', allProperties.length);
      console.log('Active properties:', activeProperties.length);
      
      // Fetch analytics for each property
      const propertiesWithAnalytics = await Promise.all(
        activeProperties.map(async (prop: any) => {
          try {
            // Get behavior analytics
            const analyticsRes = await fetch(`${API_BASE}/behavior/seller/property-analytics/${prop.id}?days=${dateRange}`, { 
              headers: getAuthHeaders() 
            });
            
            let analytics: any = { metrics: {} };
            
            if (analyticsRes.ok) {
              analytics = await analyticsRes.json();
            }
            
            return {
              property_id: prop.id,
              title: prop.title,
              location: prop.city,
              property_type: prop.property_type || 'property',
              price: parseFloat(prop.sale_price || prop.monthly_rent || '0'),
              status: prop.status,
              metrics: {
                views: analytics.metrics?.total_views || 0,
                saves: analytics.metrics?.total_saves || 0,
                visits: analytics.metrics?.total_visits || 0,
                wishlist: analytics.metrics?.total_wishlist || 0,
                unique_users: analytics.metrics?.unique_users || 0,
                engagement_score: analytics.metrics?.engagement_score || 0
              }
            };
          } catch (error) {
            console.log(`Failed to get analytics for property ${prop.id}:`, error);
            return {
              property_id: prop.id,
              title: prop.title,
              location: prop.city,
              property_type: prop.property_type || 'property',
              price: parseFloat(prop.sale_price || prop.monthly_rent || '0'),
              status: prop.status,
              metrics: {
                views: 0,
                saves: 0,
                visits: 0,
                wishlist: 0,
                unique_users: 0,
                engagement_score: 0
              }
            };
          }
        })
      );
      
      // Get unique users across all properties using behavior analytics API
      let totalUniqueUsers = 0;
      try {
        const uniqueUsersRes = await fetch(`${API_BASE}/behavior/seller/my-properties-analytics?days=${dateRange}`, { 
          headers: getAuthHeaders() 
        });
        if (uniqueUsersRes.ok) {
          const uniqueUsersData = await uniqueUsersRes.json();
          console.log('Unique users API response:', uniqueUsersData);
          totalUniqueUsers = uniqueUsersData.summary?.unique_users || 0;
        } else {
          console.log('Unique users API failed:', uniqueUsersRes.status);
        }
      } catch (error) {
        console.log('Failed to get unique users count:', error);
      }
      
      // If no unique users from API, calculate from individual properties
      if (totalUniqueUsers === 0) {
        const allUniqueUsers = propertiesWithAnalytics.reduce((sum, prop) => sum + prop.metrics.unique_users, 0);
        totalUniqueUsers = allUniqueUsers;
        console.log('Using fallback unique users calculation:', totalUniqueUsers);
      }
      
      // Calculate summary from individual property analytics
      const summary = propertiesWithAnalytics.reduce(
        (acc, prop) => ({
          total_views: acc.total_views + prop.metrics.views,
          total_saves: acc.total_saves + prop.metrics.saves,
          total_visits: acc.total_visits + prop.metrics.visits,
          total_wishlist: acc.total_wishlist + prop.metrics.wishlist,
          unique_users: totalUniqueUsers // Use API-calculated unique users to avoid double counting
        }),
        { total_views: 0, total_saves: 0, total_visits: 0, total_wishlist: 0, unique_users: 0 }
      );
      
      const portfolioData = {
        total_properties: propertiesWithAnalytics.length,
        properties: propertiesWithAnalytics,
        summary
      };
      
      console.log('Final portfolio data:', portfolioData);
      setPortfolio(portfolioData);
    } catch (error) {
      console.error('Error loading portfolio:', error);
    } finally {
      setLoading(false);
    }
  };

  const loadPropertyDetails = async (propertyId: number) => {
    try {
      setSelectedProperty(propertyId);
      setShowDetailsModal(true);
      
      setPropertyDetails(null);
      setEngagedUsers([]);
      setInsights(null);
      
      const [analyticsRes, engagementRes, insightsRes] = await Promise.all([
        fetch(`${API_BASE}/behavior/seller/property-analytics/${propertyId}?days=${dateRange}`, { headers: getAuthHeaders() }),
        fetch(`${API_BASE}/behavior/seller/property-engagement/${propertyId}?limit=50`, { headers: getAuthHeaders() }),
        fetch(`${API_BASE}/behavior/seller/property-insights/${propertyId}`, { headers: getAuthHeaders() })
      ]);
      
      const [analytics, engagement, insightsData] = await Promise.all([
        analyticsRes.ok ? analyticsRes.json() : { metrics: {} },
        engagementRes.ok ? engagementRes.json() : { users: [] },
        insightsRes.ok ? insightsRes.json() : {}
      ]);
      
      console.log('Analytics data:', analytics);
      console.log('Engagement data:', engagement);
      console.log('Insights data:', insightsData);
      
      setPropertyDetails(analytics);
      setEngagedUsers(engagement.users || []);
      setInsights(insightsData);
    } catch (error) {
      console.error('Error loading property details:', error);
      const property = portfolio?.properties?.find((p: any) => p.property_id === propertyId);
      if (property) {
        setPropertyDetails({
          property_id: propertyId,
          metrics: {
            total_views: property.metrics.views,
            total_saves: property.metrics.saves,
            total_visits: property.metrics.visits,
            total_wishlist: property.metrics.wishlist,
            unique_users: property.metrics.unique_users,
            engagement_score: property.metrics.engagement_score
          },
          conversion_rates: {
            view_to_save: property.metrics.saves > 0 ? (property.metrics.saves / Math.max(property.metrics.views, 1)) * 100 : 0,
            save_to_visit: property.metrics.visits > 0 ? (property.metrics.visits / Math.max(property.metrics.saves, 1)) * 100 : 0
          }
        });
        setEngagedUsers([]);
        setInsights({ peak_hours: [] });
      }
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="text-gray-500">Loading analytics...</div>
      </div>
    );
  }

  if (!portfolio) {
    return (
      <div className="bg-white rounded-lg shadow p-8 text-center">
        <Activity className="w-16 h-16 text-gray-300 mx-auto mb-4" />
        <p className="text-gray-500 font-medium">No properties found</p>
        <p className="text-sm text-gray-400 mt-2">Add properties to see analytics</p>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Property Analytics</h2>
          <p className="text-sm text-gray-500 mt-1">Track engagement and leads for your properties</p>
        </div>
        <select
          value={dateRange}
          onChange={(e) => setDateRange(Number(e.target.value) as 7 | 30 | 90)}
          className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
        >
          <option value={7}>Last 7 days</option>
          <option value={30}>Last 30 days</option>
          <option value={90}>Last 90 days</option>
        </select>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg shadow-sm p-6 border border-blue-200">
          <div className="flex items-center gap-3 mb-2">
            <div className="bg-blue-500 rounded-lg p-2">
              <Eye className="w-5 h-5 text-white" />
            </div>
            <div className="text-3xl font-bold text-blue-900">{portfolio?.summary?.total_views || 0}</div>
          </div>
          <div className="text-sm text-blue-700 font-medium">Total Views</div>
        </div>

        <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg shadow-sm p-6 border border-green-200">
          <div className="flex items-center gap-3 mb-2">
            <div className="bg-blue-500 rounded-lg p-2">
              <Heart className="w-5 h-5 text-white" />
            </div>
            <div className="text-3xl font-bold text-green-900">{portfolio?.summary?.total_saves || 0}</div>
          </div>
          <div className="text-sm text-green-700 font-medium">Total Saves</div>
        </div>

        <div className="bg-gradient-to-br from-purple-50 to-purple-100 rounded-lg shadow-sm p-6 border border-purple-200">
          <div className="flex items-center gap-3 mb-2">
            <div className="bg-blue-500 rounded-lg p-2">
              <Calendar className="w-5 h-5 text-white" />
            </div>
            <div className="text-3xl font-bold text-purple-900">{portfolio?.summary?.total_visits || 0}</div>
          </div>
          <div className="text-sm text-purple-700 font-medium">Visit Requests</div>
        </div>

        <div className="bg-gradient-to-br from-orange-50 to-orange-100 rounded-lg shadow-sm p-6 border border-orange-200">
          <div className="flex items-center gap-3 mb-2">
            <div className="bg-blue-500 rounded-lg p-2">
              <Users className="w-5 h-5 text-white" />
            </div>
            <div className="text-3xl font-bold text-orange-900">{portfolio?.summary?.unique_users || 0}</div>
          </div>
          <div className="text-sm text-orange-700 font-medium">Unique Users</div>
        </div>

        <div className="bg-gradient-to-br from-pink-50 to-pink-100 rounded-lg shadow-sm p-6 border border-pink-200">
          <div className="flex items-center gap-3 mb-2">
            <div className="bg-blue-500 rounded-lg p-2">
              <TrendingUp className="w-5 h-5 text-white" />
            </div>
            <div className="text-3xl font-bold text-pink-900">{portfolio?.total_properties || 0}</div>
          </div>
          <div className="text-sm text-pink-700 font-medium">Properties</div>
        </div>
      </div>

      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="font-semibold text-lg text-gray-900">Property Performance</h3>
        </div>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Property</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Views</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Saves</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Visits</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Users</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Engagement</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {portfolio?.properties?.map((property: PropertySummary) => (
                <tr key={property.property_id} className="hover:bg-gray-50">
                  <td className="px-6 py-4">
                    <div>
                      <div className="font-medium text-gray-900">{property.title}</div>
                      <div className="text-sm text-gray-500 flex items-center gap-1">
                        <MapPin className="w-3 h-3" />
                        {property.location}
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4 text-sm text-gray-900">{property.metrics.views}</td>
                  <td className="px-6 py-4 text-sm text-gray-900">{property.metrics.saves}</td>
                  <td className="px-6 py-4 text-sm text-gray-900">{property.metrics.visits}</td>
                  <td className="px-6 py-4 text-sm text-gray-900">{property.metrics.unique_users}</td>
                  <td className="px-6 py-4">
                    <div className="flex items-center gap-2">
                      <div className="flex-1 bg-gray-200 rounded-full h-2">
                        <div
                          className="bg-blue-500 h-2 rounded-full"
                          style={{ width: `${Math.min(property.metrics.engagement_score, 100)}%` }}
                        />
                      </div>
                      <span className="text-sm font-medium text-gray-700">
                        {property.metrics.engagement_score.toFixed(0)}
                      </span>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <button
                      onClick={() => loadPropertyDetails(property.property_id)}
                      className="text-blue-600 hover:text-blue-800 font-medium text-sm"
                    >
                      View Details
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {showDetailsModal && (
        <div className="fixed inset-0 bg-white bg-opacity-95 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg shadow-xl max-w-6xl w-full max-h-[90vh] overflow-y-auto">
            <div className="sticky top-0 bg-white border-b border-gray-200 p-6 flex items-center justify-between">
              <h3 className="text-xl font-bold text-gray-900">Property Analytics Details</h3>
              <button
                onClick={() => setShowDetailsModal(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <X className="w-6 h-6" />
              </button>
            </div>

            <div className="p-6 space-y-6">
              {!propertyDetails ? (
                <div className="flex items-center justify-center py-8">
                  <div className="text-gray-500">Loading property details...</div>
                </div>
              ) : (
                <>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <div className="bg-blue-50 rounded-lg p-4">
                      <div className="text-2xl font-bold text-blue-900">{propertyDetails?.metrics?.total_views || 0}</div>
                      <div className="text-sm text-blue-700">Views</div>
                    </div>
                    <div className="bg-green-50 rounded-lg p-4">
                      <div className="text-2xl font-bold text-green-900">{propertyDetails?.metrics?.total_saves || 0}</div>
                      <div className="text-sm text-green-700">Saves</div>
                    </div>
                    <div className="bg-purple-50 rounded-lg p-4">
                      <div className="text-2xl font-bold text-purple-900">{propertyDetails?.metrics?.total_visits || 0}</div>
                      <div className="text-sm text-purple-700">Visits</div>
                    </div>
                    <div className="bg-orange-50 rounded-lg p-4">
                      <div className="text-2xl font-bold text-orange-900">{propertyDetails?.metrics?.unique_users || 0}</div>
                      <div className="text-sm text-orange-700">Unique Users</div>
                    </div>
                  </div>

                  <div className="bg-gray-50 rounded-lg p-4">
                    <h4 className="font-semibold text-gray-900 mb-3">Conversion Rates</h4>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <div className="text-lg font-bold text-gray-900">
                          {propertyDetails?.conversion_rates?.view_to_save?.toFixed(1) || '0.0'}%
                        </div>
                        <div className="text-sm text-gray-600">View to Save</div>
                      </div>
                      <div>
                        <div className="text-lg font-bold text-gray-900">
                          {propertyDetails?.conversion_rates?.save_to_visit?.toFixed(1) || '0.0'}%
                        </div>
                        <div className="text-sm text-gray-600">Save to Visit</div>
                      </div>
                    </div>
                  </div>

                  {insights && insights.peak_hours && insights.peak_hours.length > 0 && (
                    <div className="bg-white border border-gray-200 rounded-lg p-4">
                      <h4 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                        <Clock className="w-5 h-5" />
                        Peak Activity Hours
                      </h4>
                      <div className="flex gap-2 flex-wrap">
                        {insights.peak_hours.map((hour: any) => (
                          <div key={hour.hour} className="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm">
                            {hour.hour}:00 ({hour.activity} activities)
                          </div>
                        ))}
                      </div>
                    </div>
                  )}

                  <div className="bg-white border border-gray-200 rounded-lg p-4">
                    <h4 className="font-semibold text-gray-900 mb-4 flex items-center gap-2">
                      <Users className="w-5 h-5" />
                      Interested Users ({engagedUsers?.length || 0})
                    </h4>
                    <div className="space-y-3 max-h-96 overflow-y-auto">
                      {engagedUsers && engagedUsers.length > 0 ? engagedUsers.map((user) => (
                        <div key={user.user_id} className="border border-gray-200 rounded-lg p-4 hover:bg-gray-50">
                          <div className="flex items-start justify-between">
                            <div className="flex-1">
                              <div className="font-medium text-gray-900">{user.name || 'Anonymous User'}</div>
                              {user.email && <div className="text-sm text-gray-600">{user.email}</div>}
                              {user.phone && <div className="text-sm text-gray-600">{user.phone}</div>}
                            </div>
                            <div className="flex flex-col items-end gap-1">
                              <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                                user.engagement_level === 'high' ? 'bg-green-100 text-green-800' :
                                user.engagement_level === 'medium' ? 'bg-yellow-100 text-yellow-800' :
                                'bg-gray-100 text-gray-800'
                              }`}>
                                {user.engagement_level?.toUpperCase() || 'LOW'}
                              </span>
                              <span className="text-xs text-gray-500 capitalize">{user.segment?.replace('_', ' ') || 'new user'}</span>
                            </div>
                          </div>
                          <div className="mt-3 grid grid-cols-4 gap-2 text-center">
                            <div className="bg-blue-50 rounded p-2">
                              <div className="text-sm font-bold text-blue-900">{user.activity?.views || 0}</div>
                              <div className="text-xs text-blue-700">Views</div>
                            </div>
                            <div className="bg-green-50 rounded p-2">
                              <div className="text-sm font-bold text-green-900">{user.activity?.saves || 0}</div>
                              <div className="text-xs text-green-700">Saves</div>
                            </div>
                            <div className="bg-purple-50 rounded p-2">
                              <div className="text-sm font-bold text-purple-900">{user.activity?.visits || 0}</div>
                              <div className="text-xs text-purple-700">Visits</div>
                            </div>
                            <div className="bg-pink-50 rounded p-2">
                              <div className="text-sm font-bold text-pink-900">{user.activity?.wishlist || 0}</div>
                              <div className="text-xs text-pink-700">Wishlist</div>
                            </div>
                          </div>
                          {user.last_interaction && (
                            <div className="mt-2 text-xs text-gray-500">
                              Last seen: {new Date(user.last_interaction).toLocaleString()}
                            </div>
                          )}
                        </div>
                      )) : (
                        <div className="text-center py-8 text-gray-400">
                          No engaged users yet
                        </div>
                      )}
                    </div>
                  </div>
                </>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};