import React, { useState, useEffect } from "react";
import { testAuthentication } from "../../utils/authTest";
import { getAbsorptionRate } from "../../api/analytics";
import { Card, CardContent, CardHeader, CardTitle } from "../ui/card";
import { Button } from "../ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../ui/select";
import {
  Eye,
  Users,
  TrendingUp,
  Clock,
  Download,
  BarChart3,
  IndianRupee,
  Home,
  Calendar,
  Target,
  Activity,
  MapPin,
  Building,
  Phone,
  Mail,
  MousePointer,
  Percent,
  CheckCircle,
  RefreshCw,
} from "lucide-react";

interface Property {
  id: string;
  title: string;
  sale_price?: string;
  monthly_rent?: string;
  views?: number;
  inquiries?: number;
  created_at: string;
  city: string;
  locality: string;
  listing_type: string;
  status?: string;
}

interface AnalyticsData {
  totalProperties: number;
  totalViews: number;
  totalInquiries: number;
  avgPrice: number;
  conversionRate: number;
  topPerformers: Property[];
  soldProperties: number;
  absorptionRate: number;
}

export default function AnalyticsMarketInsights() {
  const [timePeriodFilter, setTimePeriodFilter] = useState<string>("30d");
  const [propertyFilter, setPropertyFilter] = useState<string>("all");
  const [properties, setProperties] = useState<Property[]>([]);
  const [analytics, setAnalytics] = useState<AnalyticsData | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchAnalyticsData();
  }, [timePeriodFilter, propertyFilter]);

  useEffect(() => {
    const interval = setInterval(() => {
      fetchAnalyticsData();
    }, 60000);
    return () => clearInterval(interval);
  }, [timePeriodFilter, propertyFilter]);

  const fetchAnalyticsData = async () => {
    try {
      setLoading(true);
      
      // Test authentication first
      const authTest = testAuthentication();
      if (!authTest.success) {
        console.error('❌ Authentication test failed:', authTest.error);
        setLoading(false);
        return;
      }
      console.log('✅ Authentication test passed');
      
      const agentId = localStorage.getItem('agentId');
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const seller = JSON.parse(localStorage.getItem('seller') || '{}');
      const userId = agentId || user.user_id || seller.user_id || seller.id || user.id;
      
      if (!userId) {
        console.warn('No user ID found');
        setLoading(false);
        return;
      }

      // Get auth token
      const agentToken = localStorage.getItem('agentToken');
      const token = localStorage.getItem('token');
      const accessToken = localStorage.getItem('access_token');
      const authToken = agentToken || token || accessToken;
      
      console.log('🔐 Auth Debug:', {
        agentToken: agentToken ? 'exists' : 'missing',
        token: token ? 'exists' : 'missing', 
        accessToken: accessToken ? 'exists' : 'missing',
        selectedToken: authToken ? 'found' : 'none',
        userId: userId
      });
      
      const headers: HeadersInit = {
        'Content-Type': 'application/json'
      };
      
      if (authToken) {
        headers['Authorization'] = `Bearer ${authToken}`;
      }

      // Use the properties API with listed_by_id filter
      const propertiesResponse = await fetch(
        `http://localhost:8090/api/properties?listed_by_id=${userId}`,
        { headers }
      );
      
      if (!propertiesResponse.ok) {
        throw new Error(`HTTP error! status: ${propertiesResponse.status}`);
      }

      const propertiesData = await propertiesResponse.json();
      console.log('📊 Analytics - Raw response:', propertiesData);
      
      const properties = propertiesData.properties || propertiesData.results || propertiesData || [];
      console.log('📊 Analytics - Agent properties for user', userId, ':', Array.isArray(properties) ? properties.length : 0);

      setProperties(Array.isArray(properties) ? properties : []);

      const propertiesArray = Array.isArray(properties) ? properties : [];
      const totalProperties = propertiesArray.length;
      const totalViews = propertiesArray.reduce(
        (sum: number, p: Property) => sum + (p.views || 0),
        0
      );
      const totalInquiries = 0;
      const avgPrice =
        totalProperties > 0
          ? propertiesArray.reduce((sum: number, p: Property) => {
              const price = parseFloat(p.sale_price || p.monthly_rent || "0");
              return sum + price;
            }, 0) / totalProperties
          : 0;
      const conversionRate =
        totalViews > 0 ? (totalInquiries / totalViews) * 100 : 0;
      // Fetch absorption rate from backend API
      let absorptionRateData = {
        active_properties: 0,
        sold_properties: 0,
        total_inventory: 0,
        absorption_rate: 0
      };
      
      try {
        absorptionRateData = await getAbsorptionRate(userId, timePeriodFilter, 'agent');
        console.log('✅ Absorption rate data from backend:', absorptionRateData);
      } catch (error) {
        console.error('❌ Error fetching absorption rate from backend:', error);
        throw error; // Re-throw to trigger error state
      }
      
      const activeProperties = absorptionRateData.active_properties;
      const soldProperties = absorptionRateData.sold_properties;
      const absorptionRate = absorptionRateData.absorption_rate;
      const topPerformers = propertiesArray
        .sort((a: Property, b: Property) => (b.views || 0) - (a.views || 0))
        .slice(0, 3);

      setAnalytics({
        totalProperties: totalProperties,
        totalViews,
        totalInquiries,
        avgPrice,
        conversionRate,
        topPerformers,
        soldProperties,
        absorptionRate,
      });
    } catch (error) {
      console.error("Error fetching analytics:", error);
      setAnalytics({
        totalProperties: 0,
        totalViews: 0,
        totalInquiries: 0,
        avgPrice: 0,
        conversionRate: 0,
        topPerformers: [],
        soldProperties: 0,
        absorptionRate: 0,
      });
    } finally {
      setLoading(false);
    }
  };

  const marketData = {
    averageMarketPrice: analytics?.avgPrice || 0,
    priceGrowth:
      analytics && analytics.totalProperties > 0
        ? ((analytics.avgPrice / 750000 - 1) * 100).toFixed(1)
        : "0",
    daysOnMarket:
      analytics && analytics.totalProperties > 0
        ? Math.ceil(30 - analytics.conversionRate * 0.3)
        : 30,
    marketActivity:
      analytics && analytics.totalViews > 50
        ? "High"
        : analytics && analytics.totalViews > 20
        ? "Medium"
        : "Low",
    competitorListings: Math.max(analytics?.totalProperties || 0, 1) * 4,
    seasonalTrend:
      new Date().getMonth() >= 2 && new Date().getMonth() <= 4
        ? "Spring peak season"
        : "Regular season",
  };

  if (loading) {
    return (
      <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">
          Analytics & Market Insights
        </h2>
      </div>
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center gap-4">
          <Select value={timePeriodFilter} onValueChange={setTimePeriodFilter}>
            <SelectTrigger className="w-48">
              <SelectValue placeholder="Last 30 Days" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="7d">Last 7 Days</SelectItem>
              <SelectItem value="30d">Last 30 Days</SelectItem>
              <SelectItem value="3m">Last 3 Months</SelectItem>
              <SelectItem value="1y">Last Year</SelectItem>
            </SelectContent>
          </Select>
          <Select value={propertyFilter} onValueChange={setPropertyFilter}>
            <SelectTrigger className="w-48">
              <SelectValue placeholder="All Properties" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Properties</SelectItem>
              {properties.map((property) => (
                <SelectItem key={property.id} value={property.id}>
                  {property.title}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={fetchAnalyticsData}
            disabled={loading}
          >
            <RefreshCw
              className={`mr-2 h-4 w-4 ${loading ? "animate-spin" : ""}`}
            />
            Refresh
          </Button>
          <Button variant="secondary">
            <Download className="mr-2 h-4 w-4" />
            Export Report
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 mb-8">
        <Card className="border-l-4 border-l-blue-500">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-gray-600 font-medium">
                  Active Listings
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {properties.filter(p => p.status === "active" || p.status === "Available").length || 0}
                </p>
              </div>
              <div className="w-12 h-12 bg-blue-100 rounded-xl flex items-center justify-center">
                <Home className="w-6 h-6 text-blue-600" />
              </div>
            </div>
            <div className="flex items-center text-sm">
              <span className="text-blue-600 font-semibold">
                {properties.filter(p => p.status === "active" || p.status === "Available").length || 0}
              </span>
              <span className="text-gray-500 ml-2">total active</span>
            </div>
          </CardContent>
        </Card>

        <Card className="border-l-4 border-l-blue-500">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-gray-600 font-medium">
                  Sold Properties
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {analytics?.soldProperties || 0}
                </p>
              </div>
              <div className="w-12 h-12 bg-blue-100 rounded-xl flex items-center justify-center">
                <TrendingUp className="w-6 h-6 text-blue-600" />
              </div>
            </div>
            <div className="flex items-center text-sm">
              <span className="text-blue-600 font-semibold">
                {analytics?.absorptionRate.toFixed(1) || 0}%
              </span>
              <span className="text-gray-500 ml-2">absorption rate</span>
            </div>
          </CardContent>
        </Card>

        <Card className="border-l-4 border-l-green-500">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-gray-600 font-medium">Total Views</p>
                <p className="text-3xl font-bold text-gray-900">
                  {analytics?.totalViews.toLocaleString() || 0}
                </p>
              </div>
              <div className="w-12 h-12 bg-green-100 rounded-xl flex items-center justify-center">
                <Eye className="w-6 h-6 text-green-600" />
              </div>
            </div>
            <div className="flex items-center text-sm">
              <span className="text-green-600 font-semibold">
                {(
                  (analytics?.totalViews || 0) /
                  Math.max(analytics?.totalProperties || 1, 1)
                ).toFixed(0)}
              </span>
              <span className="text-gray-500 ml-2">avg per property</span>
            </div>
          </CardContent>
        </Card>

        <Card className="border-l-4 border-l-orange-500">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-gray-600 font-medium">
                  Total Inquiries
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {analytics?.totalInquiries || 0}
                </p>
              </div>
              <div className="w-12 h-12 bg-orange-100 rounded-xl flex items-center justify-center">
                <Phone className="w-6 h-6 text-orange-600" />
              </div>
            </div>
            <div className="flex items-center text-sm">
              <span className="text-orange-600 font-semibold">
                {analytics?.totalInquiries || 0}
              </span>
              <span className="text-gray-500 ml-2">total received</span>
            </div>
          </CardContent>
        </Card>

        <Card className="border-l-4 border-l-purple-500">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-gray-600 font-medium">
                  Conversion Rate
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {analytics?.conversionRate.toFixed(1) || 0}%
                </p>
              </div>
              <div className="w-12 h-12 bg-purple-100 rounded-xl flex items-center justify-center">
                <Percent className="w-6 h-6 text-purple-600" />
              </div>
            </div>
            <div className="flex items-center text-sm">
              <span className="text-purple-600 font-semibold">
                {analytics?.conversionRate.toFixed(1) || 0}%
              </span>
              <span className="text-gray-500 ml-2">inquiry rate</span>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        <Card className="shadow-lg">
          <CardHeader className="bg-gradient-to-r from-green-50 to-emerald-50">
            <CardTitle className="flex items-center gap-2 text-gray-800">
              <TrendingUp className="h-5 w-5 text-green-600" />
              Market Insights
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="space-y-4">
              <div className="flex items-center justify-between p-4 bg-gradient-to-r from-green-50 to-green-100 rounded-xl border-l-4 border-green-500 shadow-sm">
                <div>
                  <p className="font-semibold text-green-900">
                    Market Price Trend
                  </p>
                  <p className="text-sm text-green-700 font-medium">
                    ₹{(marketData.averageMarketPrice / 100000).toFixed(1)}L your
                    avg. price
                  </p>
                </div>
                <div className="text-right">
                  <p className="text-2xl font-bold text-green-800">
                    {Number(marketData.priceGrowth) > 0 ? "+" : ""}
                    {marketData.priceGrowth}%
                  </p>
                  <p className="text-xs text-green-600 font-medium">
                    vs market avg
                  </p>
                </div>
              </div>
              <div className="flex items-center justify-between p-4 bg-gradient-to-r from-blue-50 to-blue-100 rounded-xl border-l-4 border-blue-500 shadow-sm">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-blue-200 rounded-full flex items-center justify-center">
                    <Clock className="h-5 w-5 text-blue-700" />
                  </div>
                  <div>
                    <p className="font-semibold text-blue-900">
                      Days on Market
                    </p>
                    <p className="text-sm text-blue-700 font-medium">
                      Area average selling time
                    </p>
                  </div>
                </div>
                <p className="text-2xl font-bold text-blue-800">
                  {marketData.daysOnMarket}
                </p>
              </div>
              <div className="flex items-center justify-between p-4 bg-gradient-to-r from-orange-50 to-orange-100 rounded-xl border-l-4 border-orange-500 shadow-sm">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-orange-200 rounded-full flex items-center justify-center">
                    <Activity className="h-5 w-5 text-orange-700" />
                  </div>
                  <div>
                    <p className="font-semibold text-orange-900">
                      Market Activity
                    </p>
                    <p className="text-sm text-orange-700 font-medium">
                      {marketData.seasonalTrend}
                    </p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-xl font-bold text-orange-800">
                    {marketData.marketActivity}
                  </p>
                  <div className="flex items-center gap-1 mt-1">
                    <div className="w-2 h-2 bg-orange-500 rounded-full animate-pulse"></div>
                    <span className="text-xs text-orange-600">Live</span>
                  </div>
                </div>
              </div>
              <div className="flex items-center justify-between p-4 bg-gradient-to-r from-purple-50 to-purple-100 rounded-xl border-l-4 border-purple-500 shadow-sm">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-purple-200 rounded-full flex items-center justify-center">
                    <Percent className="h-5 w-5 text-purple-700" />
                  </div>
                  <div>
                    <p className="font-semibold text-purple-900">
                      Absorption Rate
                    </p>
                    <p className="text-sm text-purple-700 font-medium">
                      Percentage of inventory sold
                    </p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-xl font-bold text-purple-800">
                    {analytics?.absorptionRate.toFixed(1) || 0}%
                  </p>
                  <div className="flex items-center gap-1 mt-1">
                    <div className="w-2 h-2 bg-purple-500 rounded-full animate-pulse"></div>
                    <span className="text-xs text-purple-600">Live</span>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="shadow-lg">
          <CardHeader className="bg-gradient-to-r from-blue-50 to-indigo-50">
            <CardTitle className="flex items-center gap-2 text-gray-800">
              <Building className="h-5 w-5 text-blue-600" />
              Top Performing Properties
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="space-y-4">
              {analytics?.topPerformers &&
              analytics.topPerformers.length > 0 ? (
                analytics.topPerformers.map((property, index) => {
                  const inquiryCount = 0;
                  const conversionRate = property.views
                    ? ((inquiryCount / property.views) * 100).toFixed(1)
                    : "0";
                  const price = Number(
                    property.sale_price || property.monthly_rent || "0"
                  );

                  return (
                    <div
                      key={property.id}
                      className="flex items-center justify-between p-4 bg-gradient-to-r from-gray-50 to-gray-100 rounded-xl border border-gray-200 hover:shadow-md transition-shadow"
                    >
                      <div className="flex items-center gap-4">
                        <div
                          className={`w-10 h-10 rounded-xl flex items-center justify-center text-white font-bold ${
                            index === 0
                              ? "bg-yellow-500"
                              : index === 1
                              ? "bg-gray-400"
                              : "bg-orange-500"
                          }`}
                        >
                          #{index + 1}
                        </div>
                        <div>
                          <p className="font-semibold text-gray-900 truncate max-w-48">
                            {property.title}
                          </p>
                          <div className="flex items-center gap-4 text-sm text-gray-600">
                            <span className="flex items-center gap-1">
                              <Eye className="w-3 h-3" />
                              {property.views || 0} views
                            </span>
                            <span className="flex items-center gap-1">
                              <Phone className="w-3 h-3" />
                              {inquiryCount} inquiries
                            </span>
                          </div>
                        </div>
                      </div>
                      <div className="text-right">
                        <p className="text-lg font-bold text-green-600">
                          {conversionRate}%
                        </p>
                        <p className="text-xs text-gray-500">conversion</p>
                        <p className="text-sm font-medium text-gray-700">
                          ₹{(price / 100000).toFixed(1)}L
                        </p>
                      </div>
                    </div>
                  );
                })
              ) : (
                <div className="text-center py-8 text-gray-500">
                  <Building className="w-12 h-12 mx-auto mb-2 text-gray-400" />
                  <p>No property data available</p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card className="shadow-lg">
          <CardHeader className="bg-gradient-to-r from-purple-50 to-pink-50">
            <CardTitle className="flex items-center gap-2 text-gray-800">
              <Target className="h-5 w-5 text-purple-600" />
              Competitive Analysis
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="space-y-4">
              <div className="flex items-center justify-between p-4 bg-gradient-to-r from-purple-50 to-purple-100 rounded-xl border-l-4 border-purple-500 shadow-sm">
                <div>
                  <p className="font-semibold text-purple-900">
                    Total Listings
                  </p>
                  <p className="text-sm text-purple-700 font-medium">
                    All active properties
                  </p>
                </div>
                <div className="text-right">
                  <p className="text-2xl font-bold text-purple-800">
                    {analytics?.totalProperties || 0}
                  </p>
                  <p className="text-xs text-purple-600">in database</p>
                </div>
              </div>
              <div className="flex items-center justify-between p-4 bg-gradient-to-r from-red-50 to-red-100 rounded-xl border-l-4 border-red-500 shadow-sm">
                <div>
                  <p className="font-semibold text-red-900">Average Price</p>
                  <p className="text-sm text-red-700 font-medium">
                    Your properties average
                  </p>
                </div>
                <div className="text-right">
                  <p className="text-xl font-bold text-red-800">
                    ₹{((analytics?.avgPrice || 0) / 100000).toFixed(1)}L
                  </p>
                  <p className="text-xs text-red-600">per property</p>
                </div>
              </div>
              <div className="flex items-center justify-between p-4 bg-gradient-to-r from-teal-50 to-teal-100 rounded-xl border-l-4 border-teal-500 shadow-sm">
                <div>
                  <p className="font-semibold text-teal-900">
                    Performance Rate
                  </p>
                  <p className="text-sm text-teal-700 font-medium">
                    Views to inquiry ratio
                  </p>
                </div>
                <div className="text-right">
                  <p className="text-2xl font-bold text-teal-800">
                    {analytics?.conversionRate.toFixed(1) || 0}%
                  </p>
                  <div className="w-16 bg-teal-200 rounded-full h-2 mt-1">
                    <div
                      className="bg-teal-600 h-2 rounded-full"
                      style={{
                        width: `${Math.min(
                          analytics?.conversionRate || 0,
                          100
                        )}%`,
                      }}
                    ></div>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="shadow-lg">
          <CardHeader className="bg-gradient-to-r from-indigo-50 to-blue-50">
            <CardTitle className="flex items-center gap-2 text-gray-800">
              <BarChart3 className="h-5 w-5 text-indigo-600" />
              Market Trends & Insights
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="space-y-6">
              <div className="grid grid-cols-2 gap-4">
                <div className="bg-gradient-to-br from-blue-50 to-blue-100 p-4 rounded-xl">
                  <div className="flex items-center gap-2 mb-2">
                    <TrendingUp className="w-4 h-4 text-blue-600" />
                    <span className="text-sm font-semibold text-blue-900">
                      Price Growth
                    </span>
                  </div>
                  <p className="text-2xl font-bold text-blue-800">
                    {Number(marketData.priceGrowth) > 0 ? "+" : ""}
                    {marketData.priceGrowth}%
                  </p>
                  <p className="text-xs text-blue-600">Year over year</p>
                </div>
                <div className="bg-gradient-to-br from-green-50 to-green-100 p-4 rounded-xl">
                  <div className="flex items-center gap-2 mb-2">
                    <Home className="w-4 h-4 text-green-600" />
                    <span className="text-sm font-semibold text-green-900">
                      Inventory
                    </span>
                  </div>
                  <p className="text-2xl font-bold text-green-800">
                    {analytics && analytics.totalProperties > 0
                      ? (30 / Math.max(analytics.totalProperties, 1)).toFixed(1)
                      : "0"}
                  </p>
                  <p className="text-xs text-green-600">months supply</p>
                </div>
              </div>
              <div className="bg-gradient-to-r from-gray-50 to-gray-100 p-4 rounded-xl">
                <h4 className="font-semibold text-gray-900 mb-3">
                  Key Market Insights
                </h4>
                <div className="space-y-2 text-sm">
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                    <span className="text-gray-700">
                      Avg listing price: ₹
                      {((analytics?.avgPrice || 0) / 100000).toFixed(1)}L
                    </span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                    <span className="text-gray-700">
                      Est. time to sell: {marketData.daysOnMarket} days
                    </span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-orange-500 rounded-full"></div>
                    <span className="text-gray-700">
                      Season: {marketData.seasonalTrend}
                    </span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-purple-500 rounded-full"></div>
                    <span className="text-gray-700">
                      Market activity: {marketData.marketActivity}
                    </span>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}