import React, { useState, useEffect } from 'react';
import { Users, Clock, CheckCircle, XCircle, Eye, Percent, Send } from 'lucide-react';
import { collaborationApi, Collaboration } from '../../api/collaborations';
import { toast } from 'sonner';

const CollaborationInvitations: React.FC = () => {
  const [receivedInvitations, setReceivedInvitations] = useState<Collaboration[]>([]);
  const [sentInvitations, setSentInvitations] = useState<Collaboration[]>([]);
  const [loading, setLoading] = useState(true);
  const [actionLoading, setActionLoading] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<'received' | 'sent'>('received');

  const fetchInvitations = async () => {
    try {
      console.log('=== FETCHING INVITATIONS ===');
      
      // Get logged-in user's email from localStorage (check multiple possible keys)
      const userEmail = localStorage.getItem('agentEmail') || 
                       localStorage.getItem('sellerEmail') || 
                       localStorage.getItem('email');
      
      // Fallback: try parsing user object from localStorage
      if (!userEmail) {
        const userStr = localStorage.getItem('user') || localStorage.getItem('userData') || '{}';
        console.log('localStorage user string:', userStr);
        
        try {
          const user = JSON.parse(userStr);
          console.log('Parsed user object:', user);
          const fallbackEmail = user.email || user.user_email;
          if (fallbackEmail) {
            console.log('Found email in user object:', fallbackEmail);
          }
        } catch (e) {
          console.log('Could not parse user object from localStorage');
        }
      }
      
      console.log('User email:', userEmail);
      
      if (!userEmail) {
        console.log('No email found in localStorage - user not logged in');
        setReceivedInvitations([]);
        setSentInvitations([]);
        setLoading(false);
        return;
      }
      
      console.log('Calling API...');
      const response = await collaborationApi.getMyInvitations();
      console.log('API Response:', response);
      
      const allInvitations = response.collaborations || [];
      console.log('All invitations from API:', allInvitations);
      
      // Filter invitations for current user's email (case insensitive) - only pending ones
      const myReceivedInvitations = allInvitations.filter(
        (invitation: any) => {
          const match = invitation.secondary_agent_email && 
                       invitation.secondary_agent_email.toLowerCase() === userEmail.toLowerCase() &&
                       invitation.status === 'pending';
          console.log(`Checking received: ${invitation.secondary_agent_email} === ${userEmail} && status=${invitation.status} ? ${match}`);
          return match;
        }
      );
      
      // Filter invitations sent by current user - check both primary_agent_name (email) and compare
      const mySentInvitations = allInvitations.filter(
        (invitation: any) => {
          const primaryEmail = (invitation.primary_agent_name || '').toLowerCase();
          const userEmailLower = userEmail.toLowerCase();
          const match = primaryEmail === userEmailLower;
          console.log(`Checking sent: primary_agent_name='${invitation.primary_agent_name}' vs userEmail='${userEmail}' ? ${match}`);
          console.log(`  - Status: ${invitation.status}`);
          console.log(`  - Secondary: ${invitation.secondary_agent_email}`);
          return match;
        }
      );
      
      console.log('Filtered received invitations:', myReceivedInvitations);
      console.log('Filtered sent invitations:', mySentInvitations);
      console.log('Total invitations from backend:', allInvitations.length);
      console.log('User email for comparison:', userEmail);
      console.log('=== END FETCHING ===');
      
      setReceivedInvitations(myReceivedInvitations);
      setSentInvitations(mySentInvitations);
    } catch (error) {
      console.error('Error fetching invitations:', error);
      setReceivedInvitations([]);
      setSentInvitations([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchInvitations();
  }, []);

  const handleAccept = async (collaborationId: string) => {
    setActionLoading(collaborationId);
    try {
      await collaborationApi.acceptInvitation(collaborationId);
      toast.success('Collaboration accepted successfully!');
      fetchInvitations(); // Refresh the list
    } catch (error) {
      toast.error(error instanceof Error ? error.message : 'Failed to accept invitation');
    } finally {
      setActionLoading(null);
    }
  };

  const handleDecline = async (collaborationId: string) => {
    setActionLoading(collaborationId);
    try {
      await collaborationApi.declineInvitation(collaborationId);
      toast.success('Collaboration declined');
      fetchInvitations(); // Refresh the list
    } catch (error) {
      toast.error(error instanceof Error ? error.message : 'Failed to decline invitation');
    } finally {
      setActionLoading(null);
    }
  };

  if (loading) {
    return (
      <div className="bg-white rounded-lg shadow-sm p-6">
        <div className="animate-pulse">
          <div className="h-4 bg-gray-200 rounded w-1/4 mb-4"></div>
          <div className="space-y-3">
            <div className="h-20 bg-gray-200 rounded"></div>
            <div className="h-20 bg-gray-200 rounded"></div>
          </div>
        </div>
      </div>
    );
  }

  // Check if user is logged in
  const userEmail = localStorage.getItem('agentEmail') || localStorage.getItem('sellerEmail');
  if (!userEmail) {
    return (
      <div className="bg-white rounded-lg shadow-sm p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-gray-900">Collaboration Invitations</h3>
        </div>
        <div className="text-center py-8">
          <Users className="w-12 h-12 text-gray-400 mx-auto mb-3" />
          <p className="text-gray-500 mb-2">Please log in to view collaboration invitations</p>
          <p className="text-sm text-gray-400">You need to be logged in as an agent to access this feature</p>
        </div>
      </div>
    );
  }

  if (receivedInvitations.length === 0 && sentInvitations.length === 0) {
    return (
      <div className="bg-white rounded-lg shadow-sm p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-gray-900">Collaboration Invitations</h3>
        </div>
        <div className="text-center py-8">
          <Users className="w-12 h-12 text-gray-400 mx-auto mb-3" />
          <p className="text-gray-500">No collaboration invitations</p>
        </div>
      </div>
    );
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'active':
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case 'declined':
        return <XCircle className="w-4 h-4 text-red-600" />;
      default:
        return <Clock className="w-4 h-4 text-yellow-600" />;
    }
  };

  const getStatusText = (status: string, email: string) => {
    switch (status) {
      case 'active':
        return `Accepted by ${email}`;
      case 'declined':
        return `Declined by ${email}`;
      default:
        return `Sent to ${email} - Pending`;
    }
  };

  const currentInvitations = activeTab === 'received' ? receivedInvitations : sentInvitations;

  return (
    <div className="bg-white rounded-lg shadow-sm p-6">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-semibold text-gray-900">Collaboration Invitations</h3>
        {receivedInvitations.length > 0 && (
          <span className="bg-red-500 text-white text-xs px-2 py-1 rounded-full">
            {receivedInvitations.length}
          </span>
        )}
      </div>
      
      {/* Tabs */}
      <div className="flex space-x-1 mb-4 bg-gray-100 p-1 rounded-lg">
        <button
          onClick={() => setActiveTab('received')}
          className={`flex-1 px-3 py-2 text-sm font-medium rounded-md transition-colors ${
            activeTab === 'received'
              ? 'bg-white text-blue-600 shadow-sm'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          Received ({receivedInvitations.length})
        </button>
        <button
          onClick={() => setActiveTab('sent')}
          className={`flex-1 px-3 py-2 text-sm font-medium rounded-md transition-colors ${
            activeTab === 'sent'
              ? 'bg-white text-blue-600 shadow-sm'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          Sent ({sentInvitations.length})
        </button>
      </div>
      
      <div className="space-y-4">
        {currentInvitations.map((invitation) => (
          <div
            key={invitation.id}
            className="border border-blue-200 bg-blue-50 rounded-lg p-4"
          >
            <div className="flex items-start justify-between">
              <div className="flex-1">
                <div className="flex items-center mb-2">
                  {activeTab === 'received' ? (
                    <>
                      <Users className="w-4 h-4 text-blue-600 mr-2" />
                      <p className="text-sm font-medium text-gray-900">
                        <span className="text-blue-600">{invitation.primary_agent_name}</span> invited you to collaborate
                      </p>
                    </>
                  ) : (
                    <>
                      <Send className="w-4 h-4 text-blue-600 mr-2" />
                      <p className="text-sm font-medium text-gray-900">
                        Collaboration invitation sent
                      </p>
                    </>
                  )}
                </div>
                
                <p className="text-sm text-gray-700 mb-2">
                  <strong>Property:</strong> {invitation.property_title || 'Unknown Property'}
                </p>
                
                {activeTab === 'received' ? (
                  <div className="flex items-center space-x-4 text-xs text-gray-600 mb-3">
                    <div className="flex items-center">
                      <Percent className="w-3 h-3 mr-1" />
                      <span>Commission: {invitation.commission_split_secondary || 0}%</span>
                    </div>
                    <div className="flex items-center">
                      <Clock className="w-3 h-3 mr-1" />
                      <span>{new Date(invitation.created_at).toLocaleDateString()}</span>
                    </div>
                  </div>
                ) : (
                  <div className="flex items-center space-x-4 text-xs text-gray-600 mb-3">
                    <div className="flex items-center">
                      {getStatusIcon(invitation.status)}
                      <span className="ml-1">{getStatusText(invitation.status, invitation.secondary_agent_email || 'Unknown')}</span>
                    </div>
                    <div className="flex items-center">
                      <Clock className="w-3 h-3 mr-1" />
                      <span>{new Date(invitation.created_at).toLocaleDateString()}</span>
                    </div>
                  </div>
                )}

                {activeTab === 'received' && (
                  <>
                    {/* Permissions */}
                    <div className="text-xs text-gray-600 mb-3">
                      <strong>Permissions:</strong>
                      <div className="flex flex-wrap gap-2 mt-1">
                        {invitation.can_respond_inquiries && (
                          <span className="bg-green-100 text-green-800 px-2 py-1 rounded">
                            Respond to inquiries
                          </span>
                        )}
                        {invitation.can_schedule_visits && (
                          <span className="bg-green-100 text-green-800 px-2 py-1 rounded">
                            Schedule visits
                          </span>
                        )}
                        {invitation.can_edit_listing && (
                          <span className="bg-green-100 text-green-800 px-2 py-1 rounded">
                            Edit listing
                          </span>
                        )}
                        {invitation.can_view_analytics && (
                          <span className="bg-green-100 text-green-800 px-2 py-1 rounded">
                            View analytics
                          </span>
                        )}
                      </div>
                    </div>
                  </>
                )}

                {invitation.invitation_message && (
                  <div className="text-xs text-gray-600 mb-3">
                    <strong>Message:</strong> {invitation.invitation_message}
                  </div>
                )}
              </div>
              
              {activeTab === 'received' && invitation.status === 'pending' && (
                <div className="flex gap-2 ml-4">
                  <button
                    onClick={() => handleAccept(invitation.id)}
                    disabled={actionLoading === invitation.id}
                    className="flex items-center px-3 py-1 bg-green-600 text-black text-xs rounded hover:bg-green-700 transition-colors disabled:opacity-50"
                  >
                    <CheckCircle className="w-3 h-3 mr-1" />
                    {actionLoading === invitation.id ? 'Processing...' : 'Accept'}
                  </button>
                  <button
                    onClick={() => handleDecline(invitation.id)}
                    disabled={actionLoading === invitation.id}
                    className="flex items-center px-3 py-1 bg-gray-300 text-gray-700 text-xs rounded hover:bg-gray-400 transition-colors disabled:opacity-50"
                  >
                    <XCircle className="w-3 h-3 mr-1" />
                    Decline
                  </button>
                </div>
              )}
            </div>
          </div>
        ))}
      </div>
    </div>
  );
};

export default CollaborationInvitations;