import React, { useState } from 'react';
import { X, Users, Send } from 'lucide-react';
import { collaborationApi } from '../../api/collaborations';

interface CollaborationModalProps {
  open: boolean;
  onClose: () => void;
  propertyId: string | null;
}

const CollaborationModal: React.FC<CollaborationModalProps> = ({ open, onClose, propertyId }) => {
  const [inviteEmail, setInviteEmail] = useState('');
  const [message, setMessage] = useState('');
  const [loading, setLoading] = useState(false);

  if (!open) return null;

  const handleSendInvitation = async () => {
    if (!inviteEmail || !propertyId) {
      alert('Please enter an email address');
      return;
    }
    
    // Get logged-in user's email
    const senderEmail = localStorage.getItem('agentEmail') || 
                       localStorage.getItem('sellerEmail') || 
                       localStorage.getItem('email');
    
    if (!senderEmail) {
      alert('Please login first to send collaboration invitations');
      return;
    }
    
    // Prevent self-invitation
    if (inviteEmail.toLowerCase() === senderEmail.toLowerCase()) {
      alert('You cannot send a collaboration invitation to yourself');
      return;
    }
    
    setLoading(true);
    try {
      console.log('Sending collaboration invitation...');
      console.log('Property ID:', propertyId);
      console.log('Sender Email:', senderEmail);
      console.log('Invite Email:', inviteEmail);
      
      // Check if user is authenticated
      const token = localStorage.getItem('agentToken') || 
                   localStorage.getItem('sellerToken') || 
                   localStorage.getItem('access_token') || 
                   localStorage.getItem('token');
      if (!token) {
        alert('Please login first to send collaboration invitations');
        setLoading(false);
        return;
      }
      
      const response = await collaborationApi.inviteAgent(parseInt(propertyId), {
        agent_email: inviteEmail,
        sender_email: senderEmail,
        commission_split_secondary: 50,
        can_edit_listing: true,
        can_respond_inquiries: true,
        can_schedule_visits: true,
        can_view_analytics: true,
        invitation_message: message
      });
      
      console.log('Invitation sent successfully:', response);
      
      if (response.data && response.data.stored_in_db) {
        alert(`Collaboration invitation sent successfully to ${response.data.agent_email}!`);
      } else {
        alert(`Warning: ${response.data?.message || 'Unknown error'}`);
      }
      
      onClose();
      setInviteEmail('');
      setMessage('');
    } catch (error: any) {
      console.error('Failed to send invitation:', error);
      
      let errorMessage = 'Failed to send invitation';
      if (error.message) {
        errorMessage = error.message;
      } else if (typeof error === 'string') {
        errorMessage = error;
      }
      
      alert(`Error: ${errorMessage}`);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center space-x-2">
            <Users className="w-5 h-5 text-blue-600" />
            <h3 className="text-lg font-semibold">Invite Collaborator</h3>
          </div>
          <button onClick={onClose} className="text-gray-400 hover:text-gray-600">
            <X className="w-5 h-5" />
          </button>
        </div>

        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Agent Email
            </label>
            <input
              type="email"
              value={inviteEmail}
              onChange={(e) => setInviteEmail(e.target.value)}
              placeholder="Enter agent's email"
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-1 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Message (Optional)
            </label>
            <textarea
              value={message}
              onChange={(e) => setMessage(e.target.value)}
              placeholder="Add a personal message..."
              rows={3}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-1 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>

          <div className="flex space-x-3 pt-4">
            <button
              onClick={onClose}
              className="flex-1 px-4 py-2 text-gray-700 border border-gray-300 rounded-md hover:bg-gray-50"
            >
              Cancel
            </button>
            <button
              onClick={handleSendInvitation}
              disabled={!inviteEmail || loading}
              className="flex-1 flex items-center justify-center space-x-2 px-4 py-2 bg-white text-blue-600 border border-blue-600 rounded-md hover:bg-blue-50 disabled:opacity-50"
            >
              <Send className="w-4 h-4" />
              <span>{loading ? 'Sending...' : 'Send Invite'}</span>
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default CollaborationModal;