import React, { useState, useMemo } from 'react';

// Type definitions
interface Commission {
  id: string;
  propertyTitle: string;
  propertyId: string;
  clientName: string;
  clientId: string;
  amount: number;
  percentage: number;
  propertyValue: number;
  transactionType: string;
  date: string;
  dueDate: string;
  paidDate: string | null;
  status: string;
  priority: string;
  paymentMethod: string;
  invoiceNumber: string;
  taxDeducted: number;
  netAmount: number;
  location: string;
  propertyType: string;
  dealClosureTime: number;
  leadSource: string;
  notes: string;
}
import { 
  TrendingUp, Calendar, CheckCircle, Clock, User, AlertTriangle,
  Download, Filter, Search, Eye, BarChart3, Target, Zap, Bell, CreditCard,
  FileText, Phone, Mail, MapPin, Building, Calendar as CalendarIcon,
  TrendingDown, RefreshCw, Settings, Star, Award, ChevronDown, ChevronUp
} from 'lucide-react';

const CommissionSummary: React.FC = () => {
  const [filterStatus, setFilterStatus] = useState('All');
  const [filterType, setFilterType] = useState('All');
  const [filterPriority, setFilterPriority] = useState('All');
  const [searchTerm, setSearchTerm] = useState('');
  const [sortBy, setSortBy] = useState('date');
  const [sortOrder, setSortOrder] = useState('desc');
  const [showAnalytics, setShowAnalytics] = useState(true);
  const [selectedCommission, setSelectedCommission] = useState<Commission | null>(null);
  const [commissionData, setCommissionData] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const getUserId = () => {
    // Check if user is authenticated first
    const accessToken = localStorage.getItem('access_token');
    if (!accessToken) {
      return null; // Return null for unauthenticated users
    }
    
    // Try to get from user object first (primary source)
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      if (user.user_id && !user.user_id.startsWith('guest_')) {
        return user.user_id;
      }
    } catch (e) {
      console.warn('Failed to parse user data:', e);
    }
    
    // Fallback to direct user_id storage
    const userId = localStorage.getItem('user_id');
    if (userId && !userId.startsWith('guest_')) {
      return userId;
    }
    
    return null; // Return null for guest users or invalid data
  };

  const fetchCommissionData = async () => {
    try {
      setLoading(true);
      const userId = getUserId();
      
      console.log('🔍 Debug info:');
      console.log('- User ID:', userId);
      console.log('- Access Token:', localStorage.getItem('access_token') ? 'Present' : 'Missing');
      console.log('- User Object:', localStorage.getItem('user'));
      
      // If no valid user ID (guest user or unauthenticated), use test user ID
      let finalUserId = userId;
      if (!userId) {
        console.log('⚠️ No valid user ID found, using test user ID');
        finalUserId = 'b55871df-6864-426a-9bbb-71ccc75e9acc'; // Test user with commission data
      }
      
      const token = localStorage.getItem('access_token');
      const headers = {
        'Content-Type': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` })
      };
      
      console.log('🔄 Fetching commission data for user:', userId);
      const response = await fetch(`http://localhost:8090/api/agent-dashboard/commissions/summary?user_id=${userId}`, { headers });
      console.log('📊 Commission response status:', response.status);
      if (!response.ok) {
        const errorText = await response.text();
        console.error('❌ Commission API error:', errorText);
        throw new Error(`Failed to fetch commission data: ${response.status}`);
      }
      const data = await response.json();
      console.log('✅ Commission data received:', data);
      
      // Also fetch property assignments for this agent (optional)
      try {
        const assignmentsResponse = await fetch(`http://localhost:8090/api/agent-property-management/assignments?user_id=${userId}`, { headers });
        if (assignmentsResponse.ok) {
          const assignmentsData = await assignmentsResponse.json();
          data.property_assignments = assignmentsData.assignments || [];
        }
      } catch (assignmentError) {
        console.log('Property assignments not available:', assignmentError);
        data.property_assignments = [];
      }
      setCommissionData(data);
      setError(null);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to load commission data');
    } finally {
      setLoading(false);
    }
  };

  React.useEffect(() => {
    fetchCommissionData();
  }, []);

  const commissions = commissionData?.commissions || [];
  const stats = {
    totalEarned: commissionData?.total_earned || 0,
    thisMonth: commissionData?.this_month || 0,
    conversionRate: commissionData?.conversion_rate || 0,
    avgClosureTime: commissionData?.avg_closure_time || 0,
    achievementRate: commissionData?.achievement_rate || 0
  };
  const reminders = commissionData?.payment_reminders || [];

  const filteredCommissions = useMemo(() => {
    return commissions.filter((commission: any) => {
      const matchesStatus = filterStatus === 'All' || commission.status === filterStatus;
      const matchesType = filterType === 'All' || commission.transaction_type === filterType;
      const matchesPriority = filterPriority === 'All' || commission.priority === filterPriority;
      const matchesSearch = searchTerm === '' || 
        commission.property_title?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        commission.client_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        commission.property_location?.toLowerCase().includes(searchTerm.toLowerCase());
      
      return matchesStatus && matchesType && matchesPriority && matchesSearch;
    }).sort((a: any, b: any) => {
      const aVal = a[sortBy === 'date' ? 'earned_date' : sortBy === 'amount' ? 'commission_amount' : sortBy];
      const bVal = b[sortBy === 'date' ? 'earned_date' : sortBy === 'amount' ? 'commission_amount' : sortBy];
      const multiplier = sortOrder === 'asc' ? 1 : -1;
      
      if (typeof aVal === 'number' && typeof bVal === 'number') {
        return (aVal - bVal) * multiplier;
      }
      return String(aVal).localeCompare(String(bVal)) * multiplier;
    });
  }, [commissions, filterStatus, filterType, filterPriority, searchTerm, sortBy, sortOrder]);

  const totalCommission = filteredCommissions.reduce((sum: number, commission: any) => sum + (commission.commission_amount || 0), 0);
  const paidCommissions = filteredCommissions.filter((c: any) => c.status === 'paid');
  const pendingCommissions = filteredCommissions.filter((c: any) => c.status === 'pending' || c.status === 'processing');
  const overdueCommissions = filteredCommissions.filter((c: any) => c.status === 'overdue');

  const growth = {
    value: commissionData?.growth_percentage || 0,
    isPositive: (commissionData?.growth_percentage || 0) >= 0
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-[#2B256D]"></div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-lg p-4">
        <p className="text-red-800">Error: {error}</p>
        <button onClick={fetchCommissionData} className="mt-2 bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700">Retry</button>
      </div>
    );
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Paid':
        return 'bg-green-100 text-green-800 border-green-200';
      case 'Pending':
        return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'Processing':
        return 'bg-blue-100 text-blue-800 border-blue-200';
      case 'Overdue':
        return 'bg-red-100 text-red-800 border-red-200';
      default:
        return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'Paid':
        return <CheckCircle className="w-4 h-4" />;
      case 'Pending':
        return <Clock className="w-4 h-4" />;
      case 'Processing':
        return <RefreshCw className="w-4 h-4 animate-spin" />;
      case 'Overdue':
        return <AlertTriangle className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'Urgent':
        return 'text-red-600';
      case 'High':
        return 'text-orange-600';
      case 'Medium':
        return 'text-yellow-600';
      case 'Low':
        return 'text-green-600';
      default:
        return 'text-gray-600';
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0
    }).format(amount);
  };

  return (
    <div className="space-y-6">
      {/* Enhanced Header with Quick Actions */}
      <div className="bg-white rounded-xl shadow-lg p-6 border border-gray-200">
        <div className="flex items-center justify-between mb-4">
          <div>
            <h2 className="text-3xl font-bold flex items-center gap-3 text-black">
              <span className="w-8 h-8 text-black text-3xl font-bold flex items-center justify-center">₹</span>
              Commission Dashboard
            </h2>
            <p className="text-gray-700 text-lg">Your complete earnings management center</p>
          </div>
          <div className="flex items-center gap-3">
            <button className="bg-[#2B256D] hover:bg-[#4A3F8C] text-white px-4 py-2 rounded-lg transition-colors flex items-center gap-2">
              <Download className="w-4 h-4" />
              Export
            </button>
            <button className="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded-lg transition-colors flex items-center gap-2">
              <Settings className="w-4 h-4" />
              Settings
            </button>
          </div>
        </div>
        
        {/* Commission Quick Actions */}
        <div className="bg-gray-50 rounded-lg p-4 mb-4">
          <h3 className="text-lg font-semibold text-black mb-3 flex items-center gap-2">
            <span className="text-black font-bold">₹</span>
            Commission Quick Actions
          </h3>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
            <button className="text-black hover:bg-gray-100 px-3 py-2 rounded-lg transition-colors text-sm font-medium">
              View All Commissions
            </button>
            <button className="text-black hover:bg-gray-100 px-3 py-2 rounded-lg transition-colors text-sm font-medium">
              Send Reminders
            </button>
            <button className="text-black hover:bg-gray-100 px-3 py-2 rounded-lg transition-colors text-sm font-medium">
              Generate Report
            </button>
            <button className="text-black hover:bg-gray-100 px-3 py-2 rounded-lg transition-colors text-sm font-medium">
              Analytics in the agent dashboard
            </button>
          </div>
        </div>
        
        {/* Quick Stats Row */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
          <div className="bg-gray-50 rounded-lg p-3 border border-gray-200">
            <div className="text-2xl font-bold text-black">{formatCurrency(stats.totalEarned)}</div>
            <div className="text-sm text-gray-700 font-medium">Total Earned</div>
          </div>
          <div className="bg-gray-50 rounded-lg p-3 border border-gray-200">
            <div className="text-2xl font-bold text-black">{formatCurrency(stats.thisMonth)}</div>
            <div className="text-sm text-gray-700 font-medium flex items-center gap-1">
              This Month
              {growth.isPositive ? 
                <TrendingUp className="w-3 h-3 text-green-600" /> : 
                <TrendingDown className="w-3 h-3 text-red-600" />
              }
            </div>
          </div>
          <div className="bg-gray-50 rounded-lg p-3 border border-gray-200">
            <div className="text-2xl font-bold text-black">{Math.round(stats.conversionRate)}%</div>
            <div className="text-sm text-gray-700 font-medium">Conversion Rate</div>
          </div>
          <div className="bg-gray-50 rounded-lg p-3 border border-gray-200">
            <div className="text-2xl font-bold text-black">{stats.avgClosureTime}d</div>
            <div className="text-sm text-gray-700 font-medium">Avg Closure</div>
          </div>
        </div>
      </div>

      {/* Advanced Filters & Search */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <div className="flex flex-wrap items-center gap-4 mb-4">
          <div className="flex items-center gap-2">
            <Search className="w-4 h-4 text-gray-400" />
            <input
              type="text"
              placeholder="Search properties, clients, locations..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D] w-64"
            />
          </div>
          
          <select
            value={filterStatus}
            onChange={(e) => setFilterStatus(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
          >
            <option value="All">All Status</option>
            <option value="Paid">Paid</option>
            <option value="Pending">Pending</option>
            <option value="Processing">Processing</option>
            <option value="Overdue">Overdue</option>
          </select>

          <select
            value={filterType}
            onChange={(e) => setFilterType(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
          >
            <option value="All">All Types</option>
            <option value="Sale">Sale</option>
            <option value="Rent">Rent</option>
            <option value="Lease">Lease</option>
          </select>

          <select
            value={filterPriority}
            onChange={(e) => setFilterPriority(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
          >
            <option value="All">All Priority</option>
            <option value="Urgent">Urgent</option>
            <option value="High">High</option>
            <option value="Medium">Medium</option>
            <option value="Low">Low</option>
          </select>

          <select
            value={sortBy}
            onChange={(e) => setSortBy(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
          >
            <option value="date">Sort by Date</option>
            <option value="amount">Sort by Amount</option>
            <option value="status">Sort by Status</option>
            <option value="priority">Sort by Priority</option>
          </select>

          <button
            onClick={() => setSortOrder(sortOrder === 'asc' ? 'desc' : 'asc')}
            className="px-3 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 flex items-center gap-1"
          >
            {sortOrder === 'asc' ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
            {sortOrder === 'asc' ? 'Asc' : 'Desc'}
          </button>
        </div>

        <div className="flex items-center justify-between">
          <div className="text-sm text-gray-600">
            Showing {filteredCommissions.length} of {commissions.length} commissions
          </div>
          <button
            onClick={() => setShowAnalytics(!showAnalytics)}
            className="flex items-center gap-2 text-[#2B256D] hover:text-[#4A3F8C] transition-colors"
          >
            <BarChart3 className="w-4 h-4" />
            {showAnalytics ? 'Hide' : 'Show'} Analytics
          </button>
        </div>
      </div>

      {/* Enhanced Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-[#2B256D]">
          <div className="flex items-center justify-between mb-2">
            <div className="w-12 h-12 bg-[#2B256D]/20 rounded-full flex items-center justify-center">
              <span className="text-2xl font-bold text-[#2B256D]">₹</span>
            </div>
            <span className="text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded-full">Total</span>
          </div>
          <p className="text-gray-600 text-sm font-medium">Total Commission</p>
          <p className="text-3xl font-bold text-[#2B256D] mb-1">{formatCurrency(stats.totalEarned)}</p>
          <p className="text-xs text-gray-500">{commissions.length} transactions</p>
        </div>

        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-green-500">
          <div className="flex items-center justify-between mb-2">
            <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
              <CheckCircle className="w-6 h-6 text-green-600" />
            </div>
            <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full">Received</span>
          </div>
          <p className="text-gray-600 text-sm font-medium">Paid Commission</p>
          <p className="text-3xl font-bold text-green-600 mb-1">
            {formatCurrency(paidCommissions.reduce((sum: number, c: any) => sum + (c.commission_amount || 0), 0))}
          </p>
          <p className="text-xs text-gray-500">{paidCommissions.length} payments</p>
        </div>

        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-yellow-500">
          <div className="flex items-center justify-between mb-2">
            <div className="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center">
              <Clock className="w-6 h-6 text-yellow-600" />
            </div>
            <span className="text-xs bg-yellow-100 text-yellow-800 px-2 py-1 rounded-full">Pending</span>
          </div>
          <p className="text-gray-600 text-sm font-medium">Pending Commission</p>
          <p className="text-3xl font-bold text-yellow-600 mb-1">
            {formatCurrency(pendingCommissions.reduce((sum: number, c: any) => sum + (c.commission_amount || 0), 0))}
          </p>
          <p className="text-xs text-gray-500">{pendingCommissions.length} pending</p>
        </div>

        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-red-500">
          <div className="flex items-center justify-between mb-2">
            <div className="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center">
              <AlertTriangle className="w-6 h-6 text-red-600" />
            </div>
            <span className="text-xs bg-red-100 text-red-800 px-2 py-1 rounded-full">Urgent</span>
          </div>
          <p className="text-gray-600 text-sm font-medium">Overdue Commission</p>
          <p className="text-3xl font-bold text-red-600 mb-1">
            {formatCurrency(overdueCommissions.reduce((sum: number, c: any) => sum + (c.commission_amount || 0), 0))}
          </p>
          <p className="text-xs text-gray-500">{overdueCommissions.length} overdue</p>
        </div>
      </div>

      {/* Analytics Section */}
      {showAnalytics && (
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Performance Metrics */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <Target className="w-5 h-5 text-[#2B256D]" />
              Performance Metrics
            </h3>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <span className="text-sm text-gray-600">Monthly Target Progress</span>
                <span className="text-sm font-medium">{Math.round(stats.achievementRate)}%</span>
              </div>
              <div className="w-full bg-gray-200 rounded-full h-2">
                <div 
                  className="bg-[#2B256D] h-2 rounded-full transition-all duration-500" 
                  style={{ width: `${Math.min(stats.achievementRate, 100)}%` }}
                ></div>
              </div>
              
              <div className="grid grid-cols-2 gap-4 mt-4">
                <div className="text-center p-3 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold text-[#2B256D]">{formatCurrency(commissionData?.avg_deal_size || 0)}</div>
                  <div className="text-xs text-gray-600">Avg Deal Size</div>
                </div>
                <div className="text-center p-3 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold text-[#2B256D]">{Math.round(stats.achievementRate)}%</div>
                  <div className="text-xs text-gray-600">Achievement Rate</div>
                </div>
              </div>
            </div>
          </div>

          {/* Payment Reminders */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <Bell className="w-5 h-5 text-orange-500" />
              Payment Reminders
            </h3>
            <div className="space-y-3">
              {reminders.map((reminder: any) => (
                <div key={reminder.id} className="flex items-center justify-between p-3 bg-orange-50 rounded-lg border border-orange-200">
                  <div>
                    <div className="font-medium text-gray-900">{reminder.client_name}</div>
                    <div className="text-sm text-gray-600">{formatCurrency(reminder.amount)}</div>
                  </div>
                  <div className="text-right">
                    <div className={`text-sm font-medium ${getPriorityColor(reminder.priority)}`}>
                      {reminder.priority}
                    </div>
                    <div className="text-xs text-gray-500">
                      {reminder.days_overdue > 0 ? `${reminder.days_overdue} days overdue` : 'Due soon'}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* Enhanced Commission Table */}
      <div className="bg-white rounded-xl shadow-sm overflow-hidden">
        <div className="px-6 py-4 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
            <FileText className="w-5 h-5 text-[#2B256D]" />
            Commission Records
          </h3>
        </div>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Property Details
                </th>
                <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Client & Type
                </th>
                <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Financial Details
                </th>
                <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Timeline
                </th>
                <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status & Priority
                </th>
                <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {filteredCommissions.map((commission: any) => (
                <tr key={commission.id} className="hover:bg-gray-50 transition-colors">
                  <td className="px-6 py-4">
                    <div className="flex items-start space-x-3">
                      <div className="w-10 h-10 bg-[#2B256D]/10 rounded-lg flex items-center justify-center">
                        <Building className="w-5 h-5 text-[#2B256D]" />
                      </div>
                      <div>
                        <div className="text-sm font-medium text-gray-900">{commission.property_title}</div>
                        <div className="text-xs text-gray-500 flex items-center gap-1">
                          <MapPin className="w-3 h-3" />
                          {commission.property_location}
                        </div>
                        <div className="text-xs text-gray-500">{commission.property_type}</div>
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <div className="flex items-start space-x-3">
                      <div className="w-8 h-8 rounded-full bg-[#2B256D] text-white flex items-center justify-center">
                        <User className="w-4 h-4" />
                      </div>
                      <div>
                        <div className="text-sm font-medium text-gray-900">{commission.client_name}</div>
                        <div className="text-xs text-gray-500">{commission.transaction_type}</div>
                        <div className="text-xs text-blue-600">{commission.lead_source}</div>
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <div>
                      <div className="text-sm font-bold text-gray-900">{formatCurrency(commission.commission_amount)}</div>
                      <div className="text-xs text-gray-500">{commission.commission_rate}% of {formatCurrency(commission.property_value)}</div>
                      <div className="text-xs text-green-600">Net: {formatCurrency(commission.net_amount)}</div>
                      <div className="text-xs text-gray-400">Tax: {formatCurrency(commission.tax_deducted)}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <div>
                      <div className="text-sm text-gray-900 flex items-center gap-1">
                        <CalendarIcon className="w-3 h-3" />
                        {commission.earned_date}
                      </div>
                      <div className="text-xs text-gray-500">Due: {commission.due_date}</div>
                      {commission.paid_date && (
                        <div className="text-xs text-green-600">Paid: {commission.paid_date}</div>
                      )}
                      <div className="text-xs text-blue-600">{commission.deal_closure_time} days to close</div>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <div className="space-y-2">
                      <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(commission.status)}`}>
                        {getStatusIcon(commission.status)}
                        <span className="ml-1">{commission.status}</span>
                      </span>
                      <div className={`text-xs font-medium ${getPriorityColor(commission.priority)}`}>
                        {commission.priority} Priority
                      </div>
                      {commission.payment_method && commission.payment_method !== 'Pending' && (
                        <div className="text-xs text-gray-500 flex items-center gap-1">
                          <CreditCard className="w-3 h-3" />
                          {commission.payment_method}
                        </div>
                      )}
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <div className="flex flex-col space-y-1">
                      <button 
                        onClick={() => setSelectedCommission(commission as Commission)}
                        className="text-xs bg-[#2B256D] text-white px-2 py-1 rounded hover:bg-[#4A3F8C] transition-colors flex items-center gap-1"
                      >
                        <Eye className="w-3 h-3" />
                        View
                      </button>
                      {commission.status === 'pending' && (
                        <button className="text-xs bg-orange-500 text-white px-2 py-1 rounded hover:bg-orange-600 transition-colors flex items-center gap-1">
                          <Bell className="w-3 h-3" />
                          Remind
                        </button>
                      )}
                      <button className="text-xs bg-gray-500 text-white px-2 py-1 rounded hover:bg-gray-600 transition-colors flex items-center gap-1">
                        <Download className="w-3 h-3" />
                        Invoice
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Enhanced Empty State */}
      {filteredCommissions.length === 0 && (
        <div className="bg-white rounded-xl shadow-sm p-12 text-center">
          <div className="w-20 h-20 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <TrendingUp className="w-10 h-10 text-gray-400" />
          </div>
          <h3 className="text-xl font-semibold text-gray-900 mb-2">No commissions found</h3>
          <p className="text-gray-600 mb-4">
            {filterStatus !== 'All' || filterType !== 'All' || filterPriority !== 'All' || searchTerm
              ? 'Try adjusting your filters or search terms'
              : 'No commission records available yet. Start by adding properties and closing deals!'
            }
          </p>
          {(filterStatus !== 'All' || filterType !== 'All' || filterPriority !== 'All' || searchTerm) && (
            <button 
              onClick={() => {
                setFilterStatus('All');
                setFilterType('All');
                setFilterPriority('All');
                setSearchTerm('');
              }}
              className="bg-[#2B256D] text-white px-4 py-2 rounded-lg hover:bg-[#4A3F8C] transition-colors"
            >
              Clear All Filters
            </button>
          )}
        </div>
      )}

      {/* Commission Detail Modal */}
      {selectedCommission && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <div className="flex items-center justify-between">
                <h3 className="text-xl font-semibold text-gray-900">Commission Details</h3>
                <button 
                  onClick={() => setSelectedCommission(null)}
                  className="text-gray-400 hover:text-gray-600 transition-colors"
                >
                  ✕
                </button>
              </div>
            </div>
            <div className="p-6 space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">Property Information</h4>
                  <div className="space-y-2 text-sm">
                    <div><span className="text-gray-600">Title:</span> {selectedCommission?.propertyTitle}</div>
                    <div><span className="text-gray-600">Location:</span> {selectedCommission?.location}</div>
                    <div><span className="text-gray-600">Type:</span> {selectedCommission?.propertyType}</div>
                    <div><span className="text-gray-600">Value:</span> {selectedCommission?.propertyValue ? formatCurrency(selectedCommission.propertyValue) : 'N/A'}</div>
                  </div>
                </div>
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">Client Information</h4>
                  <div className="space-y-2 text-sm">
                    <div><span className="text-gray-600">Name:</span> {selectedCommission?.clientName}</div>
                    <div><span className="text-gray-600">Transaction:</span> {selectedCommission?.transactionType}</div>
                    <div><span className="text-gray-600">Lead Source:</span> {selectedCommission?.leadSource}</div>
                    <div><span className="text-gray-600">Closure Time:</span> {selectedCommission?.dealClosureTime} days</div>
                  </div>
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold text-gray-900 mb-3">Financial Breakdown</h4>
                <div className="bg-gray-50 rounded-lg p-4 space-y-2 text-sm">
                  <div className="flex justify-between"><span>Commission Rate:</span> <span>{selectedCommission?.percentage}%</span></div>
                  <div className="flex justify-between"><span>Gross Amount:</span> <span>{selectedCommission?.amount ? formatCurrency(selectedCommission.amount) : 'N/A'}</span></div>
                  <div className="flex justify-between"><span>Tax Deducted:</span> <span className="text-red-600">-{selectedCommission?.taxDeducted ? formatCurrency(selectedCommission.taxDeducted) : 'N/A'}</span></div>
                  <div className="flex justify-between font-semibold border-t pt-2"><span>Net Amount:</span> <span className="text-green-600">{selectedCommission?.netAmount ? formatCurrency(selectedCommission.netAmount) : 'N/A'}</span></div>
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold text-gray-900 mb-3">Payment Information</h4>
                <div className="space-y-2 text-sm">
                  <div><span className="text-gray-600">Status:</span> 
                    <span className={`ml-2 px-2 py-1 rounded-full text-xs ${selectedCommission?.status ? getStatusColor(selectedCommission.status) : ''}`}>
                      {selectedCommission?.status}
                    </span>
                  </div>
                  <div><span className="text-gray-600">Invoice:</span> {selectedCommission?.invoiceNumber}</div>
                  <div><span className="text-gray-600">Due Date:</span> {selectedCommission?.dueDate}</div>
                  {selectedCommission?.paidDate && (
                    <div><span className="text-gray-600">Paid Date:</span> {selectedCommission.paidDate}</div>
                  )}
                  <div><span className="text-gray-600">Payment Method:</span> {selectedCommission?.paymentMethod}</div>
                </div>
              </div>
              
              {selectedCommission?.notes && (
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">Notes</h4>
                  <p className="text-sm text-gray-600 bg-gray-50 rounded-lg p-3">{selectedCommission.notes}</p>
                </div>
              )}
            </div>
            <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
              <button className="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition-colors">
                Download Invoice
              </button>
              {selectedCommission?.status === 'Pending' && (
                <button className="px-4 py-2 bg-orange-500 text-white rounded-lg hover:bg-orange-600 transition-colors">
                  Send Reminder
                </button>
              )}
              <button 
                onClick={() => setSelectedCommission(null)}
                className="px-4 py-2 bg-[#2B256D] text-white rounded-lg hover:bg-[#4A3F8C] transition-colors"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default CommissionSummary;