import React, { useState } from 'react';
import { Send, Search, MessageSquare, Phone, User, Clock } from 'lucide-react';
import { mockMessages } from './mockData';

const CommunicationPanel: React.FC = () => {
  const [selectedClient, setSelectedClient] = useState<string>('Amit Sharma');
  const [newMessage, setNewMessage] = useState('');
  const [searchTerm, setSearchTerm] = useState('');

  // Group messages by client
  const messagesByClient = mockMessages.reduce((acc, message) => {
    if (!acc[message.clientName]) {
      acc[message.clientName] = [];
    }
    acc[message.clientName].push(message);
    return acc;
  }, {} as Record<string, typeof mockMessages>);

  const clientList = Object.keys(messagesByClient);
  const filteredClients = clientList.filter(client =>
    client.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const currentMessages = messagesByClient[selectedClient] || [];

  const handleSendMessage = (e: React.FormEvent) => {
    e.preventDefault();
    if (!newMessage.trim()) return;
    
    console.log('Sending message:', newMessage);
    setNewMessage('');
  };

  const formatTime = (timestamp: string) => {
    return new Date(timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
  };

  return (
    <div className="bg-white rounded-xl shadow-sm overflow-hidden h-[600px] flex">
      {/* Client List Sidebar */}
      <div className="w-80 border-r border-gray-200 flex flex-col flex-shrink-0">
        <div className="p-6 border-b border-gray-100">
          <h2 className="text-lg font-semibold text-gray-900 mb-4">Messages</h2>
          <input
            type="text"
            placeholder="Search clients..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="w-full px-4 py-3 border border-gray-200 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D] text-sm bg-gray-50"
          />
        </div>
        
        <div className="flex-1 overflow-y-auto">
          {filteredClients.map((clientName) => {
            const lastMessage = messagesByClient[clientName][messagesByClient[clientName].length - 1];
            const unreadCount = messagesByClient[clientName].filter(msg => !msg.read && !msg.isAgent).length;
            
            return (
              <div
                key={clientName}
                onClick={() => setSelectedClient(clientName)}
                className={`p-4 cursor-pointer border-b border-gray-50 hover:bg-gray-50 transition-colors ${
                  selectedClient === clientName ? 'bg-[#2B256D]/5 border-l-4 border-l-[#2B256D]' : ''
                }`}
              >
                <div className="flex items-center justify-between mb-1">
                  <div className="flex items-center">
                    <div className="w-10 h-10 bg-gray-300 rounded-full flex items-center justify-center mr-3">
                      <User className="w-5 h-5 text-gray-600" />
                    </div>
                    <div>
                      <h3 className="font-medium text-gray-900 text-sm">{clientName}</h3>
                      <p className="text-xs text-gray-500">
                        Property: {lastMessage.propertyId}
                      </p>
                    </div>
                  </div>
                  {unreadCount > 0 && (
                    <span className="bg-blue-600 text-white text-xs rounded-full w-5 h-5 flex items-center justify-center">
                      {unreadCount}
                    </span>
                  )}
                </div>
                <p className="text-sm text-gray-600 truncate">
                  {lastMessage.isAgent ? 'You: ' : ''}{lastMessage.message}
                </p>
                <p className="text-xs text-gray-400 mt-1">
                  {formatTime(lastMessage.timestamp)}
                </p>
              </div>
            );
          })}
        </div>
      </div>

      {/* Chat Area */}
      <div className="flex-1 flex flex-col">
        {selectedClient ? (
          <>
            {/* Chat Header */}
            <div className="p-4 border-b border-gray-200 bg-gray-50">
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <div className="w-10 h-10 bg-gray-300 rounded-full flex items-center justify-center mr-3">
                    <User className="w-5 h-5 text-gray-600" />
                  </div>
                  <div>
                    <h3 className="font-medium text-gray-900">{selectedClient}</h3>
                    <p className="text-sm text-gray-500">Online</p>
                  </div>
                </div>
                <div className="flex items-center space-x-2">
                  <button className="p-2 text-gray-600 hover:text-gray-900 hover:bg-gray-200 rounded-full">
                    <Phone className="w-5 h-5" />
                  </button>
                </div>
              </div>
            </div>

            {/* Messages */}
            <div className="flex-1 overflow-y-auto p-6 space-y-4 bg-gray-50">
              {currentMessages.map((message) => (
                <div
                  key={message.id}
                  className={`flex ${message.isAgent ? 'justify-end' : 'justify-start'}`}
                >
                  <div
                    className={`max-w-xs lg:max-w-md px-4 py-3 rounded-2xl shadow-sm ${
                      message.isAgent
                        ? 'bg-[#2B256D] text-white'
                        : 'bg-white text-gray-900 border border-gray-200'
                    }`}
                  >
                    <p className="text-sm leading-relaxed">{message.message}</p>
                    <p
                      className={`text-xs mt-2 ${
                        message.isAgent ? 'text-white/70' : 'text-gray-500'
                      }`}
                    >
                      {formatTime(message.timestamp)}
                    </p>
                  </div>
                </div>
              ))}
            </div>

            {/* Message Input */}
            <div className="p-4 border-t border-gray-200">
              <form onSubmit={handleSendMessage} className="flex items-center space-x-2">
                <input
                  type="text"
                  value={newMessage}
                  onChange={(e) => setNewMessage(e.target.value)}
                  placeholder="Type your message..."
                  className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
                <button
                  type="submit"
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-colors"
                >
                  <Send className="w-4 h-4" />
                </button>
              </form>
            </div>
          </>
        ) : (
          <div className="flex-1 flex items-center justify-center">
            <div className="text-center">
              <MessageSquare className="w-16 h-16 text-gray-300 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Select a conversation</h3>
              <p className="text-gray-500">Choose a client from the list to start messaging</p>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default CommunicationPanel;