import React, { useState, useEffect } from "react";
import {
  TrendingUp,
  TrendingDown,
  Building,
  Users,
  Calendar,
  MessageCircle,
  ChevronDown,
  ChevronRight,
  Eye,
  EyeOff,
  Home,
  Star,
  Clock,
  FileText,
  Activity,
} from "lucide-react";
import {
  mockAgent,
  mockVisits,
  mockMessages,
  mockCommissions,
  mockNotifications,
} from "./mockData";
import { useProperty } from "./PropertyContext";
import { useNavigate } from "react-router-dom";
import { reportService } from "../../services/reportService";
import { toast } from "sonner";
import SystemHealthDashboard from '../../system-health/components/SystemHealthDashboard';

interface Lead {
  id: string;
  property_id: number;
  name: string;
  email: string;
  phone: string;
  message?: string;
  budget?: string;
  preferred_contact: string;
  lead_type: string;
  status: string;
  score?: number;
  ai_category?: string;
  created_at: string;
  updated_at: string;
  property_title?: string;
}

const Dashboard: React.FC = () => {
  const { properties } = useProperty();
  const navigate = useNavigate();
  const [collapsedSections, setCollapsedSections] = useState({
    stats: false,
    properties: false,
    activities: false,
    quickActions: false,
    systemHealth: false,
  });
  const [leads, setLeads] = useState<Lead[]>([]);
  const [leadsLoading, setLeadsLoading] = useState(true);
  const [reportLoading, setReportLoading] = useState(false);

  useEffect(() => {
    fetchLeads();
    // Auto-refresh leads every 30 seconds
    const interval = setInterval(fetchLeads, 30000);
    return () => clearInterval(interval);
  }, []);

  const fetchLeads = async () => {
    try {
      setLeadsLoading(true);
      const userStr = localStorage.getItem("user");
      if (!userStr) {
        console.log("No user found in localStorage");
        setLeads([]);
        return;
      }
      const user = JSON.parse(userStr);
      console.log("Dashboard - Current user object:", user);
      // Try multiple user ID fields and fallback to known agent ID
      let userId = user.id || user.user_id || user.userId || user.user?.id;
      
      // If no user ID found, use test agent ID
      if (!userId) {
        console.warn("Dashboard - No user ID found, using fallback agent ID");
        userId = "6e7676f0-fbb6-4bd8-ad88-f0599ce853f2"; // Known agent ID from database
      }
      
      console.log("Dashboard - Using user ID:", userId);

      const API_BASE = import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const token = localStorage.getItem("access_token");
      const apiUrl = `${API_BASE}/leads-management/leads/by-user-type?user_type=agent&user_id=${userId}`;
      console.log("Dashboard - API URL:", apiUrl);

      const response = await fetch(apiUrl, {
        headers: {
          "Content-Type": "application/json",
          ...(token && { Authorization: `Bearer ${token}` }),
        },
      });

      console.log("Dashboard - Response status:", response.status);
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Dashboard - API Error:", errorText);
        throw new Error(`HTTP ${response.status}: ${errorText}`);
      }

      const data = await response.json();
      console.log("Dashboard - Leads data received:", data);
      const leadsData = Array.isArray(data) ? data : data.leads || [];
      console.log("Dashboard - Processed leads data:", leadsData);
      setLeads(leadsData);
    } catch (error) {
      console.error("Dashboard - Error fetching leads:", error);
      setLeads([]);
    } finally {
      setLeadsLoading(false);
    }
  };

  const handleGenerateReport = async () => {
    try {
      setReportLoading(true);
      const filename = await reportService.downloadDashboardReport();
      console.log('Report downloaded:', filename);
      toast.success(`Report downloaded: ${filename}`, {
        description: 'Your dashboard report has been generated successfully.'
      });
    } catch (error) {
      console.error('Failed to generate report:', error);
      const toast = document.createElement('div');
      toast.textContent = error.message || 'Failed to generate report';
      toast.style.cssText = 'position:fixed;top:20px;right:20px;background:#ef4444;color:white;padding:16px 24px;border-radius:8px;box-shadow:0 4px 6px rgba(0,0,0,0.1);z-index:9999;font-weight:500';
      document.body.appendChild(toast);
      setTimeout(() => toast.remove(), 4000);
    } finally {
      setReportLoading(false);
    }
  };

  const getUserEmail = () => {
    return localStorage.getItem("agentEmail") || "User";
  };

  const toggleSection = (section: keyof typeof collapsedSections) => {
    setCollapsedSections((prev) => ({
      ...prev,
      [section]: !prev[section],
    }));
  };

  // Calculate property counts based on status
  const availableProperties = properties.filter(
    (p) => p.status === "Available"
  ).length;
  const soldProperties = properties.filter((p) => p.status === "Sold").length;
  const pendingProperties = properties.filter(
    (p) => p.status === "Pending"
  ).length;

  return (
    <div className="space-y-3">
      {/* Compact Welcome Header */}
      <div className="bg-white rounded-lg shadow-sm p-4 border border-gray-100">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div className="w-12 h-12 bg-gradient-to-r from-[#2B256D] to-[#4A3F8C] rounded-full flex items-center justify-center">
              <span className="text-lg">👋</span>
            </div>
            <div>
              <h1 className="text-2xl font-bold text-black">
                Welcome back, {getUserEmail()}!
              </h1>
              <p className="text-gray-600 text-base">
                Here's what's happening with your properties today
              </p>
            </div>
          </div>

          <div className="flex items-center space-x-4">
            {/* Generate Report Button */}
            <button
              onClick={handleGenerateReport}
              disabled={reportLoading}
              style={{ 
                opacity: reportLoading ? 0.6 : 1, 
                backgroundColor: '#3b82f6', 
                color: 'white', 
                pointerEvents: reportLoading ? 'none' : 'auto',
                border: 'none',
                display: 'flex',
                alignItems: 'center',
                gap: '8px',
                padding: '8px 16px',
                borderRadius: '8px'
              }}
            >
              <FileText className="w-4 h-4" style={{ color: 'white' }} />
              <span style={{ color: 'white', fontSize: '14px', fontWeight: '500' }}>
                {reportLoading ? 'Generating...' : 'Generate Report'}
              </span>
            </button>

            {/* Total Leads */}
            <div className="bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg px-4 py-3 text-center w-[140px] border border-blue-200">
              <div className="flex items-center justify-center space-x-1 mb-1">
                <Users className="w-5 h-5 text-blue-600" />
                <span className="text-2xl font-bold text-blue-900">
                  {leadsLoading ? "..." : leads.length}
                </span>
              </div>
              <p className="text-sm font-semibold text-blue-700">
                Total Leads
              </p>
            </div>

            {/* New Leads */}
            <div className="bg-gradient-to-r from-green-50 to-green-100 rounded-lg px-4 py-3 text-center w-[140px] border border-green-200">
              <div className="flex items-center justify-center space-x-1 mb-1">
                <Clock className="w-5 h-5 text-green-600" />
                <span className="text-2xl font-bold text-green-900">
                  {leadsLoading ? "..." : leads.filter(l => l.status === "new").length}
                </span>
              </div>
              <p className="text-sm font-semibold text-green-700">
                New Leads
              </p>
            </div>

            {/* Properties Count */}
            <div className="bg-gray-50 rounded-lg px-4 py-3 text-center w-[140px] border border-gray-200">
              <div className="flex items-center justify-center space-x-1 mb-1">
                <Home className="w-5 h-5 text-black" />
                <span className="text-2xl font-bold text-black">
                  {properties.length}
                </span>
              </div>
              <p className="text-sm font-semibold text-gray-700">Properties</p>
            </div>
          </div>
        </div>
      </div>

      {/* Compact Stats Grid */}
      <div className="bg-white rounded-lg shadow-sm p-4">
        <div className="flex items-center justify-between mb-3">
          <h2 className="text-xl font-semibold text-gray-900">
            Performance Statistics
          </h2>
          <button
            onClick={() => toggleSection("stats")}
            className="flex items-center space-x-1 text-[#2B256D] hover:text-[#4A3F8C] transition-colors"
          >
            {collapsedSections.stats ? (
              <EyeOff className="w-4 h-4" />
            ) : (
              <Eye className="w-4 h-4" />
            )}
            <span className="text-xs font-medium">
              {collapsedSections.stats ? "Show" : "Hide"}
            </span>
          </button>
        </div>
        {!collapsedSections.stats && (
          <div className="grid grid-cols-2 lg:grid-cols-4 gap-3">
            <div className="bg-gradient-to-r from-[#2B256D]/10 to-[#2B256D]/20 p-4 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-[#2B256D] text-xs font-medium">
                    Total Properties
                  </p>
                  <p className="text-2xl font-bold text-[#2B256D]">
                    {properties.length}
                  </p>
                  <p className="text-[#2B256D] text-xs">
                    Active listings
                  </p>
                </div>
                <div className="w-8 h-8 bg-[#2B256D]/20 rounded-full flex items-center justify-center">
                  <Building className="w-4 h-4 text-[#2B256D]" />
                </div>
              </div>
            </div>

            <div className="bg-gradient-to-r from-green-50 to-green-100 p-4 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-green-600 text-xs font-medium">
                    Available
                  </p>
                  <p className="text-2xl font-bold text-green-900">
                    {availableProperties}
                  </p>
                  <p className="text-green-600 text-xs">
                    Properties
                  </p>
                </div>
                <div className="w-8 h-8 bg-green-200 rounded-full flex items-center justify-center">
                  <Building className="w-4 h-4 text-green-600" />
                </div>
              </div>
            </div>

            <div className="bg-gradient-to-r from-blue-50 to-blue-100 p-4 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-blue-600 text-xs font-medium">Sold</p>
                  <p className="text-2xl font-bold text-blue-900">
                    {soldProperties}
                  </p>
                  <p className="text-blue-600 text-xs">
                    Properties
                  </p>
                </div>
                <div className="w-8 h-8 bg-blue-200 rounded-full flex items-center justify-center">
                  <Building className="w-4 h-4 text-blue-600" />
                </div>
              </div>
            </div>

            <div className="bg-gradient-to-r from-yellow-50 to-yellow-100 p-4 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-yellow-600 text-xs font-medium">Pending</p>
                  <p className="text-2xl font-bold text-yellow-900">
                    {pendingProperties}
                  </p>
                  <p className="text-yellow-600 text-xs">
                    Properties
                  </p>
                </div>
                <div className="w-8 h-8 bg-yellow-200 rounded-full flex items-center justify-center">
                  <Building className="w-4 h-4 text-yellow-600" />
                </div>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* System Health Monitoring */}
      <div className="bg-white rounded-lg shadow-sm p-4">
        <div className="flex items-center justify-between mb-3">
          <h2 className="text-xl font-semibold text-gray-900 flex items-center space-x-2">
            <Activity className="w-5 h-5 text-[#2B256D]" />
            <span>System Health</span>
          </h2>
          <button
            onClick={() => toggleSection("systemHealth")}
            className="flex items-center space-x-1 text-[#2B256D] hover:text-[#4A3F8C] transition-colors"
          >
            {collapsedSections.systemHealth ? (
              <ChevronRight className="w-4 h-4" />
            ) : (
              <ChevronDown className="w-4 h-4" />
            )}
            <span className="text-xs font-medium">
              {collapsedSections.systemHealth ? "Expand" : "Collapse"}
            </span>
          </button>
        </div>
        {!collapsedSections.systemHealth && (
          <div className="mt-4">
            <SystemHealthDashboard />
          </div>
        )}
      </div>

      {/* Compact Recent Activities */}
      <div className="bg-white rounded-lg shadow-sm p-4">
        <div className="flex items-center justify-between mb-3">
          <h2 className="text-xl font-semibold text-gray-900">
            Recent Activities
          </h2>
          <button
            onClick={() => toggleSection("activities")}
            className="flex items-center space-x-1 text-[#2B256D] hover:text-[#4A3F8C] transition-colors"
          >
            {collapsedSections.activities ? (
              <ChevronRight className="w-4 h-4" />
            ) : (
              <ChevronDown className="w-4 h-4" />
            )}
            <span className="text-xs font-medium">
              {collapsedSections.activities ? "Expand" : "Collapse"}
            </span>
          </button>
        </div>
        {!collapsedSections.activities && (
          <div className="space-y-2">
            {properties.length > 3 && (
              <div className="flex items-center space-x-2 p-2 bg-gray-50 rounded-lg">
                <div className="w-2 h-2 bg-[#2B256D] rounded-full"></div>
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-900">
                    New property added
                  </p>
                  <p className="text-xs text-gray-600">
                    {properties[0]?.title}
                  </p>
                </div>
                <span className="text-xs text-gray-500">Just now</span>
              </div>
            )}
            <div className="flex items-center space-x-2 p-2 bg-gray-50 rounded-lg">
              <div className="w-2 h-2 bg-green-500 rounded-full"></div>
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-900">
                  Property sold
                </p>
                <p className="text-xs text-gray-600">
                  Commercial Plot in Cyber City
                </p>
              </div>
              <span className="text-xs text-gray-500">1 day ago</span>
            </div>
            <div className="flex items-center space-x-2 p-2 bg-gray-50 rounded-lg">
              <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-900">
                  Visit scheduled
                </p>
                <p className="text-xs text-gray-600">
                  Independent Villa in Whitefield
                </p>
              </div>
              <span className="text-xs text-gray-500">2 days ago</span>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default Dashboard;