import React, { useState, useEffect } from "react";
import {
  Search,
  Phone,
  Archive,
  ChevronLeft,
  ChevronRight,
} from "lucide-react";

interface Lead {
  id: string;
  name: string;
  email: string;
  phone: string;
  buyer_name?: string;
  buyer_email?: string;
  buyer_phone?: string;
  property_id: number;
  property_title?: string;
  message?: string;
  budget?: string;
  preferred_contact: string;
  lead_type: string;
  status: string;
  score?: number;
  ai_category?: string;
  created_at: string;
  updated_at: string;
}

const InquiryManagement: React.FC = () => {
  const [activeTab, setActiveTab] = useState("all");
  const [leads, setLeads] = useState<Lead[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [filterType, setFilterType] = useState("all");
  const [sortOrder, setSortOrder] = useState("recent");
  const [currentPage, setCurrentPage] = useState(1);
  const leadsPerPage = 10;

  useEffect(() => {
    fetchLeads();
    const interval = setInterval(fetchLeads, 30000);
    return () => clearInterval(interval);
  }, []);

  const fetchLeads = async () => {
    try {
      setLoading(true);
      let userStr =
        localStorage.getItem("agent") ||
        localStorage.getItem("user") ||
        localStorage.getItem("seller") ||
        localStorage.getItem("currentUser");
      let userId = null;

      if (userStr) {
        try {
          const user = JSON.parse(userStr);
          userId = user.id || user.user_id || user.userId || user.user?.id;
        } catch (e) {
          console.error("Error parsing user data:", e);
        }
      }

      if (!userId) {
        userId =
          localStorage.getItem("agentId") ||
          localStorage.getItem("user_id") ||
          localStorage.getItem("sellerId");
      }

      if (!userId) {
        userId = "6e7676f0-fbb6-4bd8-ad88-f0599ce853f2";
      }

      const API_BASE =
        import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const token =
        localStorage.getItem("access_token") ||
        localStorage.getItem("token") ||
        localStorage.getItem("agentToken") ||
        localStorage.getItem("sellerToken");
      const apiUrl = `${API_BASE}/leads-management/leads/by-user-type?user_type=agent&user_id=${userId}`;

      const response = await fetch(apiUrl, {
        headers: {
          "Content-Type": "application/json",
          ...(token && { Authorization: `Bearer ${token}` }),
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }

      const data = await response.json();
      const leadsData = Array.isArray(data) ? data : data.leads || [];
      setLeads(leadsData);
    } catch (error) {
      console.error("Error fetching leads:", error);
      setLeads([]);
    } finally {
      setLoading(false);
    }
  };

  const getTimeLabel = (dateString: string) => {
    const now = new Date();
    const date = new Date(dateString);
    const diffMs = now.getTime() - date.getTime();
    const diffMins = Math.floor(diffMs / (1000 * 60));
    const diffHours = Math.floor(diffMs / (1000 * 60 * 60));
    const diffDays = Math.floor(diffMs / (1000 * 60 * 60 * 24));

    if (diffMins < 1) return "Just now";
    if (diffMins < 60) return `${diffMins} mins ago`;
    if (diffHours < 24) return `${diffHours} hours ago`;
    if (diffDays === 1) return "Yesterday";
    if (diffDays < 7) return `${diffDays} days ago`;
    return date.toLocaleDateString();
  };

  const isHotLead = (lead: Lead) => {
    return (
      lead.ai_category?.toLowerCase() === "hot" ||
      (lead.score && lead.score >= 75) ||
      lead.budget?.toLowerCase().includes("crore")
    );
  };

  const filteredLeads = leads.filter((lead) => {
    if (activeTab === "new" && lead.status !== "new") return false;
    if (
      activeTab === "contacted" &&
      lead.status !== "contacted" &&
      lead.status !== "prospect"
    )
      return false;
    if (activeTab === "archived" && lead.status !== "archived") return false;
    if (activeTab === "prospects" && lead.status !== "prospect") return false;
    if (
      searchTerm &&
      !lead.name.toLowerCase().includes(searchTerm.toLowerCase()) &&
      !lead.message?.toLowerCase().includes(searchTerm.toLowerCase())
    )
      return false;
    if (filterType === "hot" && !isHotLead(lead)) return false;
    return true;
  });

  const sortedLeads = [...filteredLeads].sort((a, b) => {
    const dateA = new Date(a.created_at).getTime();
    const dateB = new Date(b.created_at).getTime();
    return sortOrder === "recent" ? dateB - dateA : dateA - dateB;
  });

  const totalPages = Math.ceil(sortedLeads.length / leadsPerPage);
  const startIndex = (currentPage - 1) * leadsPerPage;
  const paginatedLeads = sortedLeads.slice(
    startIndex,
    startIndex + leadsPerPage
  );

  const tabs = [
    { id: "all", label: "All Leads", count: leads.length },
    {
      id: "new",
      label: "New",
      count: leads.filter((l) => l.status === "new").length,
    },
    {
      id: "contacted",
      label: "Contacted",
      count: leads.filter((l) => l.status === "contacted").length,
    },
    {
      id: "prospects",
      label: "Prospects",
      count: leads.filter((l) => l.status === "prospect").length,
    },
    {
      id: "archived",
      label: "Archived",
      count: leads.filter((l) => l.status === "archived").length,
    },
  ];

  const updateLeadStatus = async (leadId: string, status: string) => {
    try {
      const API_BASE =
        import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const token =
        localStorage.getItem("access_token") ||
        localStorage.getItem("token") ||
        localStorage.getItem("agentToken") ||
        localStorage.getItem("sellerToken");

      const response = await fetch(
        `${API_BASE}/leads-management/leads/${leadId}/status?status=${status}&user_id=agent`,
        {
          method: "PUT",
          headers: {
            "Content-Type": "application/json",
            ...(token && { Authorization: `Bearer ${token}` }),
          },
        }
      );

      if (response.ok) {
        setLeads((prevLeads) =>
          prevLeads.map((lead) =>
            lead.id === leadId ? { ...lead, status } : lead
          )
        );
      }
    } catch (error) {
      console.error("Error updating lead status:", error);
    }
  };

  const handleCall = async (lead: Lead) => {
    if (lead.status === "archived") {
      alert("Cannot contact archived leads");
      return;
    }

    const phone = lead.phone || lead.buyer_phone;
    if (phone) {
      window.open(`tel:${phone}`, "_self");

      try {
        const API_BASE =
          import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
        const token =
          localStorage.getItem("access_token") || localStorage.getItem("token");
        const userStr =
          localStorage.getItem("user") || localStorage.getItem("agent");
        let userId = "6e7676f0-fbb6-4bd8-ad88-f0599ce853f2";

        if (userStr) {
          try {
            const user = JSON.parse(userStr);
            userId = user.id || user.user_id || user.userId || userId;
          } catch (e) {
            console.error("Error parsing user:", e);
          }
        }

        const response = await fetch(
          `${API_BASE}/leads-management/leads/${lead.id}/contact?user_id=${userId}&contact_method=call`,
          {
            method: "PUT",
            headers: {
              "Content-Type": "application/json",
              ...(token && { Authorization: `Bearer ${token}` }),
            },
          }
        );

        if (response.ok) {
          setLeads((prevLeads) =>
            prevLeads.map((l) =>
              l.id === lead.id ? { ...l, status: "contacted" } : l
            )
          );
        }
      } catch (error) {
        console.error("Error updating lead status:", error);
      }
    }
  };

  const handleWhatsApp = async (lead: Lead) => {
    if (lead.status === "archived") {
      alert("Cannot contact archived leads");
      return;
    }

    const phone = lead.phone || lead.buyer_phone;
    if (phone) {
      const cleanPhone = phone.replace(/[^+\d]/g, "");
      window.open(`https://wa.me/${cleanPhone}`, "_blank");

      try {
        const API_BASE =
          import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
        const token =
          localStorage.getItem("access_token") || localStorage.getItem("token");
        const userStr =
          localStorage.getItem("user") || localStorage.getItem("agent");
        let userId = "6e7676f0-fbb6-4bd8-ad88-f0599ce853f2";

        if (userStr) {
          try {
            const user = JSON.parse(userStr);
            userId = user.id || user.user_id || user.userId || userId;
          } catch (e) {
            console.error("Error parsing user:", e);
          }
        }

        const response = await fetch(
          `${API_BASE}/leads-management/leads/${lead.id}/contact?user_id=${userId}&contact_method=whatsapp`,
          {
            method: "PUT",
            headers: {
              "Content-Type": "application/json",
              ...(token && { Authorization: `Bearer ${token}` }),
            },
          }
        );

        if (response.ok) {
          setLeads((prevLeads) =>
            prevLeads.map((l) =>
              l.id === lead.id ? { ...l, status: "contacted" } : l
            )
          );
        }
      } catch (error) {
        console.error("Error updating lead status:", error);
      }
    }
  };

  const handleArchive = (lead: Lead) => {
    updateLeadStatus(lead.id, "archived");
  };

  return (
    <div className="space-y-6 p-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Leads & Enquiries</h1>
      </div>

      <div className="border-b border-gray-200">
        <div className="flex gap-6">
          {tabs.map((tab) => (
            <button
              key={tab.id}
              onClick={() => setActiveTab(tab.id)}
              className={`pb-3 px-1 font-medium text-sm border-b-2 transition-colors ${
                activeTab === tab.id
                  ? "border-blue-600 text-blue-600"
                  : "border-transparent text-gray-600 hover:text-gray-900"
              }`}
            >
              {tab.label}
              {tab.count > 0 && (
                <span
                  className={`ml-2 px-2 py-1 text-xs rounded-full ${
                    activeTab === tab.id
                      ? "bg-blue-100 text-blue-600"
                      : "bg-gray-100 text-gray-600"
                  }`}
                >
                  {tab.count}
                </span>
              )}
            </button>
          ))}
        </div>
      </div>

      <div className="flex flex-col sm:flex-row gap-4 items-start sm:items-center justify-between">
        <div className="relative flex-1 max-w-md">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
          <input
            type="text"
            placeholder="Search Leads..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          />
        </div>

        <div className="flex gap-4 items-center">
          <div className="flex items-center gap-2">
            <span className="text-sm text-gray-600">Filter:</span>
            <select
              value={filterType}
              onChange={(e) => setFilterType(e.target.value)}
              className="appearance-none bg-white border border-gray-300 rounded-lg px-3 py-2 pr-8 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="all">All</option>
              <option value="sale">For Sale</option>
              <option value="rent">For Rent</option>
              <option value="hot">Hot Leads</option>
            </select>
          </div>

          <div className="flex items-center gap-2">
            <span className="text-sm text-gray-600">Sort:</span>
            <select
              value={sortOrder}
              onChange={(e) => setSortOrder(e.target.value)}
              className="appearance-none bg-white border border-gray-300 rounded-lg px-3 py-2 pr-8 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="recent">Recent</option>
              <option value="oldest">Oldest</option>
            </select>
          </div>
        </div>
      </div>

      <div className="bg-white rounded-lg shadow border border-gray-200">
        {loading ? (
          <div className="flex justify-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
          </div>
        ) : paginatedLeads.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-gray-500 text-lg">No leads found</p>
            <p className="text-gray-400 text-sm mt-2">
              {searchTerm
                ? "Try adjusting your search terms"
                : "New leads will appear here"}
            </p>
          </div>
        ) : (
          <div className="divide-y divide-gray-200">
            {paginatedLeads.map((lead) => (
              <div
                key={lead.id}
                className="p-6 hover:bg-gray-50 transition-colors"
              >
                <div className="flex items-start justify-between">
                  <div className="flex items-start gap-4 flex-1">
                    <div className="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center">
                      <span className="text-lg font-medium text-gray-600">
                        {(lead.name || lead.buyer_name || "No Name")
                          .charAt(0)
                          .toUpperCase()}
                      </span>
                    </div>

                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2 mb-1">
                        <h3 className="text-lg font-medium text-gray-900">
                          {lead.name || lead.buyer_name || "No Name"}
                        </h3>
                        {isHotLead(lead) && (
                          <span className="px-2 py-1 text-xs font-medium bg-red-100 text-red-800 rounded-full">
                            Hot Lead
                          </span>
                        )}
                      </div>

                      <p className="text-gray-600 mb-2 line-clamp-2">
                        {lead.message || "No message provided"}
                      </p>

                      <div className="flex items-center gap-4 text-sm text-gray-500">
                        <span>{getTimeLabel(lead.created_at)}</span>
                        {lead.property_title && (
                          <>
                            <span>•</span>
                            <span>{lead.property_title}</span>
                          </>
                        )}
                        {lead.budget && (
                          <>
                            <span>•</span>
                            <span>Budget: {lead.budget}</span>
                          </>
                        )}
                      </div>

                      <div className="mt-3 grid grid-cols-2 gap-4 text-sm">
                        <div>
                          <span className="font-medium text-gray-700">
                            Email:
                          </span>
                          <span className="ml-2 text-gray-600">
                            {lead.email || lead.buyer_email || "Not provided"}
                          </span>
                        </div>
                        <div>
                          <span className="font-medium text-gray-700">
                            Phone:
                          </span>
                          <span className="ml-2 text-gray-600">
                            {lead.phone || lead.buyer_phone || "Not provided"}
                          </span>
                        </div>
                        {lead.score && (
                          <div>
                            <span className="font-medium text-gray-700">
                              Score:
                            </span>
                            <span className="ml-2 text-gray-600">
                              {lead.score}/100
                            </span>
                          </div>
                        )}
                        <div>
                          <span className="font-medium text-gray-700">
                            Status:
                          </span>
                          <span
                            className={`ml-2 px-2 py-1 text-xs rounded-full ${
                              lead.status === "new"
                                ? "bg-green-100 text-green-800"
                                : lead.status === "contacted"
                                ? "bg-blue-100 text-blue-800"
                                : lead.status === "prospect"
                                ? "bg-yellow-100 text-yellow-800"
                                : lead.status === "qualified"
                                ? "bg-yellow-100 text-yellow-800"
                                : "bg-gray-100 text-gray-800"
                            }`}
                          >
                            {lead.status.charAt(0).toUpperCase() +
                              lead.status.slice(1)}
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>

                  <div className="flex items-center gap-2 ml-4">
                    {(lead.phone || lead.buyer_phone) && (
                      <>
                        <button
                          onClick={() => handleWhatsApp(lead)}
                          className="flex items-center gap-2 px-3 py-2 text-sm font-medium text-white bg-green-500 border border-green-500 rounded-lg hover:bg-green-600 transition-colors"
                        >
                          <svg
                            className="w-4 h-4"
                            viewBox="0 0 24 24"
                            fill="currentColor"
                          >
                            <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.885 3.787" />
                          </svg>
                          WhatsApp
                        </button>
                        <button
                          onClick={() => handleCall(lead)}
                          className="flex items-center gap-2 px-3 py-2 text-sm font-medium text-white bg-blue-500 border border-blue-500 rounded-lg hover:bg-blue-600 transition-colors"
                        >
                          <Phone className="w-4 h-4" />
                          Call
                        </button>
                      </>
                    )}

                    {lead.status !== "prospect" &&
                      lead.status !== "archived" && (
                        <button
                          onClick={() => updateLeadStatus(lead.id, "prospect")}
                          className="flex items-center gap-2 px-3 py-2 text-sm font-medium text-yellow-700 bg-yellow-50 border border-yellow-200 rounded-lg hover:bg-yellow-100 transition-colors"
                        >
                          Mark Prospect
                        </button>
                      )}

                    {lead.status === "prospect" && (
                      <button
                        onClick={() => updateLeadStatus(lead.id, "contacted")}
                        className="flex items-center gap-2 px-3 py-2 text-sm font-medium text-gray-700 bg-gray-50 border border-gray-200 rounded-lg hover:bg-gray-100 transition-colors"
                      >
                        Remove Prospect
                      </button>
                    )}

                    <button
                      onClick={() => handleArchive(lead)}
                      className="flex items-center gap-2 px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                    >
                      <Archive className="w-4 h-4" />
                      Archive
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}

        {totalPages > 1 && (
          <div className="flex items-center justify-between px-6 py-4 border-t border-gray-200">
            <div className="text-sm text-gray-700">
              Showing {startIndex + 1} to{" "}
              {Math.min(startIndex + leadsPerPage, sortedLeads.length)} of{" "}
              {sortedLeads.length} results
            </div>

            <div className="flex items-center gap-2">
              <button
                onClick={() => setCurrentPage((prev) => Math.max(prev - 1, 1))}
                disabled={currentPage === 1}
                className="flex items-center gap-1 px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
              >
                <ChevronLeft className="w-4 h-4" />
                Prev
              </button>

              <span className="px-3 py-2 text-sm font-medium text-gray-900 bg-blue-50 border border-blue-200 rounded-lg">
                {currentPage}
              </span>

              <button
                onClick={() =>
                  setCurrentPage((prev) => Math.min(prev + 1, totalPages))
                }
                disabled={currentPage === totalPages}
                className="flex items-center gap-1 px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
              >
                Next
                <ChevronRight className="w-4 h-4" />
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default InquiryManagement;
