import { Card, CardContent, CardHeader, CardTitle } from "../seller-dashboard/src/components/ui/card";
import { Badge } from "../seller-dashboard/src/components/ui/badge";
import { Button } from "../seller-dashboard/src/components/ui/button";
import { TrendingUp, TrendingDown, Activity, MapPin, Eye, Target, AlertTriangle, Zap, Loader2, BarChart3, Users, Clock, DollarSign, TrendingUpIcon, Home } from "lucide-react";
import { useState, useEffect } from "react";
import { toast } from "sonner";

interface MarketSummary {
  seller_summary: {
    total_properties: number;
    active_properties: number;
    sold_properties: number;
    avg_listed_price: number;
    avg_days_on_market: number;
  };
  market_overview: {
    total_active_properties: number;
    total_localities: number;
    market_avg_price: number;
    properties_sold_30d: number;
    price_trend: string;
  };
}

interface AreaAnalysis {
  locality: string;
  avg_market_price: number;
  active_supply: number;
  total_views: number;
  demand_vs_supply_ratio: number;
  avg_days_on_market: number;
  market_hotness: string;
  avg_market_price_formatted: string;
}

interface PriceBand {
  price_range: string;
  avg_days_to_sell: number;
  total_properties: number;
  sold_properties: number;
  success_rate: number;
  recommendation: string;
}

interface SeasonalTrend {
  month: string;
  avg_price: number;
  sold_count: number;
  avg_days_on_market: number;
  demand_level: string;
  avg_price_formatted: string;
}

export default function MarketAnalysis() {
  const [loading, setLoading] = useState(true);
  const [marketSummary, setMarketSummary] = useState<MarketSummary | null>(null);
  const [areaAnalysis, setAreaAnalysis] = useState<AreaAnalysis[]>([]);
  const [priceBands, setPriceBands] = useState<PriceBand[]>([]);
  const [seasonalTrends, setSeasonalTrends] = useState<SeasonalTrend[]>([]);

  useEffect(() => {
    fetchMarketData();
  }, []);

  const fetchMarketData = async () => {
    try {
      setLoading(true);
      const baseUrl = 'http://localhost:8090/api';
      
      const testRes = await fetch(`${baseUrl}/seller-market-trends/test-data`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (!testRes.ok) {
        const errorText = await testRes.text();
        console.error('Backend test failed:', testRes.status, errorText);
        throw new Error(`Backend returned ${testRes.status}: ${errorText}`);
      }
      
      const testData = await testRes.json();
      console.log('Backend test successful:', testData);
      
      const [summaryRes, areaRes, bandsRes, trendsRes] = await Promise.all([
        fetch(`${baseUrl}/seller-market-trends/market-summary`),
        fetch(`${baseUrl}/seller-market-trends/area-analysis`),
        fetch(`${baseUrl}/seller-market-trends/price-bands`),
        fetch(`${baseUrl}/seller-market-trends/seasonal-trends`)
      ]);

      let dataLoaded = false;

      if (summaryRes.ok) {
        const summaryData = await summaryRes.json();
        if (summaryData.success) {
          setMarketSummary(summaryData.data);
          dataLoaded = true;
        }
      }

      if (areaRes.ok) {
        const areaData = await areaRes.json();
        if (areaData.success) {
          setAreaAnalysis(areaData.data.areas || []);
          dataLoaded = true;
        }
      }

      if (bandsRes.ok) {
        const bandsData = await bandsRes.json();
        if (bandsData.success) {
          setPriceBands(bandsData.data.price_bands || []);
          dataLoaded = true;
        }
      }

      if (trendsRes.ok) {
        const trendsData = await trendsRes.json();
        if (trendsData.success) {
          setSeasonalTrends(trendsData.data.trends || []);
          dataLoaded = true;
        }
      }

      if (dataLoaded) {
        toast.success('Market analysis data loaded successfully');
      } else {
        toast.warning('Backend connected but no data available');
      }

    } catch (error) {
      console.error('Error fetching market data:', error);
      toast.error(`Backend connection failed: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  const getHotnessColor = (hotness: string) => {
    switch (hotness.toLowerCase()) {
      case 'hot': return 'text-red-600 bg-red-50 border-red-200';
      case 'warm': return 'text-orange-600 bg-orange-50 border-orange-200';
      case 'cool': return 'text-blue-600 bg-blue-50 border-blue-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getDemandColor = (level: string) => {
    switch (level.toLowerCase()) {
      case 'high': return '!bg-green-600 !text-white !border-green-600';
      case 'medium': return '!bg-yellow-600 !text-white !border-yellow-600';
      case 'low': return '!bg-blue-600 !text-white !border-blue-600';
      default: return '!bg-gray-600 !text-white !border-gray-600';
    }
  };

  if (loading) {
    return (
      <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
        <div className="flex items-center justify-center h-64">
          <Loader2 className="w-8 h-8 animate-spin" />
          <span className="ml-2">Loading market analysis...</span>
        </div>
      </div>
    );
  }

  const hasData = marketSummary || areaAnalysis.length > 0 || priceBands.length > 0 || seasonalTrends.length > 0;
  
  if (!hasData) {
    return (
      <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Market Analysis</h1>
            <p className="text-muted-foreground">Real-time market intelligence and competitive analysis</p>
          </div>
          <Button onClick={fetchMarketData}>
            <Activity className="mr-2 h-4 w-4" />
            Retry Connection
          </Button>
        </div>
        
        <Card className="p-8 text-center">
          <div className="space-y-4">
            <AlertTriangle className="w-16 h-16 mx-auto text-yellow-500" />
            <h3 className="text-xl font-semibold">Backend Connection Required</h3>
            <p className="text-muted-foreground max-w-md mx-auto">
              Market analysis data requires the backend server to be running. 
              Please ensure the NAL backend is running on port 8090.
            </p>
            <Button onClick={fetchMarketData} className="!bg-blue-600 !hover:bg-blue-700 !text-white border-0 mt-4" style={{backgroundColor: '#2563eb', color: 'white'}}>
              <Activity className="mr-2 h-4 w-4" />
              Test Connection
            </Button>
          </div>
        </Card>
      </div>
    );
  }

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Market Analysis</h1>
          <p className="text-muted-foreground">Strategic insights to identify opportunities and close deals faster</p>
        </div>
        <Button onClick={fetchMarketData} className="!bg-blue-600 !hover:bg-blue-700 !text-white border-0" style={{backgroundColor: '#2563eb', color: 'white'}}>
          <Activity className="mr-2 h-4 w-4" />
          Refresh Data
        </Button>
      </div>

      {marketSummary && (
        <>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-blue-700 font-medium">Market Avg Price</p>
                    <p className="text-3xl font-bold text-blue-800">
                      ₹{(marketSummary.market_overview.market_avg_price / 100000).toFixed(1)}L
                    </p>
                    <div className="flex items-center mt-2">
                      {marketSummary.market_overview.price_trend === 'up' ? (
                        <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                      ) : (
                        <TrendingDown className="w-4 h-4 text-red-600 mr-1" />
                      )}
                      <span className={`text-sm font-medium ${
                        marketSummary.market_overview.price_trend === 'up' ? 'text-green-600' : 'text-red-600'
                      }`}>
                        {marketSummary.market_overview.price_trend === 'up' ? 'Rising' : 'Declining'}
                      </span>
                    </div>
                  </div>
                  <Target className="w-8 h-8 text-blue-600" />
                </div>
              </CardContent>
            </Card>

            <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-green-700 font-medium">Avg Days on Market</p>
                    <p className="text-3xl font-bold text-green-800">
                      {Math.round(marketSummary.seller_summary.avg_days_on_market)}
                    </p>
                    <p className="text-sm text-green-600 mt-2">Days to sell</p>
                  </div>
                  <Activity className="w-8 h-8 text-green-600" />
                </div>
              </CardContent>
            </Card>

            <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-purple-700 font-medium">Active Properties</p>
                    <p className="text-3xl font-bold text-purple-800">
                      {marketSummary.market_overview.total_active_properties}
                    </p>
                    <p className="text-sm text-purple-600 mt-2">Market supply</p>
                  </div>
                  <Zap className="w-8 h-8 text-purple-600" />
                </div>
              </CardContent>
            </Card>

            <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-orange-700 font-medium">Recent Sales</p>
                    <p className="text-3xl font-bold text-orange-800">
                      {marketSummary.market_overview.properties_sold_30d}
                    </p>
                    <p className="text-sm text-orange-600 mt-2">Last 30 days</p>
                  </div>
                  <Eye className="w-8 h-8 text-orange-600" />
                </div>
              </CardContent>
            </Card>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Target className="w-5 h-5 text-purple-600" />
                  Competitive Analysis
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="flex items-center justify-between p-4 bg-muted/30 rounded-lg">
                    <div>
                      <p className="text-sm text-muted-foreground">Total Listings</p>
                      <p className="text-xs text-muted-foreground">All active properties</p>
                    </div>
                    <div className="text-right">
                      <p className="text-2xl font-bold text-purple-600">{marketSummary.market_overview.total_active_properties}</p>
                      <p className="text-xs text-purple-600">in database</p>
                    </div>
                  </div>
                  
                  <div className="flex items-center justify-between p-4 bg-muted/30 rounded-lg">
                    <div>
                      <p className="text-sm text-muted-foreground">Average Price</p>
                      <p className="text-xs text-muted-foreground">Your properties average</p>
                    </div>
                    <div className="text-right">
                      <p className="text-2xl font-bold">₹{(marketSummary.seller_summary.avg_listed_price / 100000).toFixed(1)}L</p>
                      <p className="text-xs text-muted-foreground">per property</p>
                    </div>
                  </div>
                  
                  <div className="flex items-center justify-between p-4 bg-muted/30 rounded-lg">
                    <div>
                      <p className="text-sm text-muted-foreground">Performance Rate</p>
                      <p className="text-xs text-muted-foreground">Views to inquiry ratio</p>
                    </div>
                    <div className="text-right">
                      <p className="text-2xl font-bold">
                        {marketSummary.market_overview.properties_sold_30d > 0 && marketSummary.seller_summary.total_properties > 0
                          ? ((marketSummary.market_overview.properties_sold_30d / marketSummary.seller_summary.total_properties) * 100).toFixed(1)
                          : '0.0'}%
                      </p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <BarChart3 className="w-5 h-5 text-blue-600" />
                  Market Trends & Insights
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="p-3 bg-blue-50 rounded-lg">
                      <div className="flex items-center gap-2 mb-1">
                        <TrendingUp className="w-4 h-4 text-blue-600" />
                        <p className="text-xs text-gray-600">Price Growth</p>
                      </div>
                      <p className="text-2xl font-bold text-blue-600">
                        {marketSummary.market_overview.price_trend === 'up' ? '+' : ''}0%
                      </p>
                      <p className="text-xs text-gray-600">Year over year</p>
                    </div>
                    
                    <div className="p-3 bg-green-50 rounded-lg">
                      <div className="flex items-center gap-2 mb-1">
                        <Home className="w-4 h-4 text-green-600" />
                        <p className="text-xs text-gray-600">Inventory</p>
                      </div>
                      <p className="text-2xl font-bold text-green-600">{Math.round(marketSummary.seller_summary.avg_days_on_market / 30)}</p>
                      <p className="text-xs text-gray-600">months supply</p>
                    </div>
                  </div>
                  
                  <div className="bg-muted/30 p-4 rounded-lg">
                    <h4 className="font-semibold text-sm mb-3">Key Market Insights</h4>
                    <div className="space-y-2 text-xs">
                      <div className="flex items-center gap-2">
                        <div className="w-2 h-2 rounded-full bg-green-500"></div>
                        <span>Avg listing price: ₹{(marketSummary.market_overview.market_avg_price / 100000).toFixed(1)}L</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <div className="w-2 h-2 rounded-full bg-blue-500"></div>
                        <span>Est. time to sell: {Math.round(marketSummary.seller_summary.avg_days_on_market)} days</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <div className="w-2 h-2 rounded-full bg-orange-500"></div>
                        <span>Season: Regular season</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <div className="w-2 h-2 rounded-full bg-gray-500"></div>
                        <span>Market activity: {marketSummary.market_overview.properties_sold_30d > 10 ? 'High' : marketSummary.market_overview.properties_sold_30d > 5 ? 'Medium' : 'Low'}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </>
      )}

      {/* Competitive Analysis */}
      {areaAnalysis.length > 0 && priceBands.length > 0 && (
        <Card className="bg-gradient-to-br from-rose-50 to-pink-50 border-rose-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Target className="w-5 h-5 text-rose-600" />
              Competitive Analysis
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Market competition levels and agent positioning</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="bg-white p-5 rounded-lg border">
                <h4 className="font-semibold text-gray-900 mb-4">Competition by Area</h4>
                <div className="space-y-3">
                  {areaAnalysis.slice(0, 5).map((area, idx) => (
                    <div key={idx} className="p-3 rounded-lg border">
                      <div className="flex justify-between items-center mb-2">
                        <p className="font-semibold text-gray-900">{area.locality}</p>
                        <Badge className={`${
                          area.active_supply > 15 ? 'bg-red-100 text-red-700' :
                          area.active_supply > 8 ? 'bg-yellow-100 text-yellow-700' :
                          'bg-green-100 text-green-700'
                        }`}>
                          {area.active_supply > 15 ? 'High' : area.active_supply > 8 ? 'Medium' : 'Low'} Competition
                        </Badge>
                      </div>
                      <div className="flex justify-between text-xs text-gray-600">
                        <span>{area.active_supply} active listings</span>
                        <span>{area.total_views} total views</span>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
              
              <div className="bg-white p-5 rounded-lg border">
                <h4 className="font-semibold text-gray-900 mb-4">Success Rate by Price Band</h4>
                <div className="space-y-3">
                  {priceBands.map((band, idx) => (
                    <div key={idx} className="p-3 rounded-lg border">
                      <div className="flex justify-between items-center mb-2">
                        <p className="font-semibold text-gray-900">{band.price_range}</p>
                        <span className="text-lg font-bold text-rose-600">{band.success_rate}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className="bg-rose-500 h-2 rounded-full" style={{width: `${band.success_rate}%`}}></div>
                      </div>
                      <p className="text-xs text-gray-600 mt-1">{band.sold_properties} sold of {band.total_properties} listed</p>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Undervalued Areas & Growth Potential */}
      {areaAnalysis.length > 0 && (
        <Card className="bg-gradient-to-br from-emerald-50 to-green-50 border-emerald-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUpIcon className="w-5 h-5 text-emerald-600" />
              Undervalued Areas & Growth Potential
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Areas with high potential for appreciation</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              {areaAnalysis
                .filter(a => a.demand_vs_supply_ratio > 1.0 && a.active_supply < 10)
                .slice(0, 3)
                .map((area, idx) => (
                  <div key={idx} className="bg-white p-5 rounded-lg border">
                    <div className="flex items-center gap-2 mb-3">
                      <div className="p-2 bg-emerald-100 rounded-lg">
                        <MapPin className="w-5 h-5 text-emerald-600" />
                      </div>
                      <h4 className="font-semibold text-gray-900">{area.locality}</h4>
                    </div>
                    <div className="space-y-2 text-sm">
                      <div className="flex justify-between">
                        <span className="text-gray-600">Avg Price</span>
                        <span className="font-semibold">{area.avg_market_price_formatted}</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-600">Demand Ratio</span>
                        <span className="font-semibold text-emerald-600">{area.demand_vs_supply_ratio}</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-600">Competition</span>
                        <span className="font-semibold text-green-600">Low ({area.active_supply})</span>
                      </div>
                    </div>
                    <div className="mt-3 pt-3 border-t">
                      <Badge className="bg-emerald-100 text-emerald-700 border-emerald-300">High Growth Potential</Badge>
                    </div>
                  </div>
                ))}
              {areaAnalysis.filter(a => a.demand_vs_supply_ratio > 1.0 && a.active_supply < 10).length === 0 && (
                <div className="col-span-3 text-center py-8 text-gray-600">
                  <p>No undervalued areas identified currently. Monitor market trends for opportunities.</p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Target Buyer Segments */}
      {priceBands.length > 0 && marketSummary && (
        <Card className="bg-gradient-to-br from-violet-50 to-purple-50 border-violet-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="w-5 h-5 text-violet-600" />
              Target Buyer Segments
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Recommended buyer profiles for each price range</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {priceBands.slice(0, 4).map((band, idx) => {
                const priceNum = parseInt(band.price_range.replace(/[^0-9]/g, ''));
                let segment = '';
                let profile = [];
                
                if (priceNum < 5000000) {
                  segment = 'First-time Buyers';
                  profile = ['Young professionals', 'Small families', 'Budget-conscious buyers'];
                } else if (priceNum < 10000000) {
                  segment = 'Mid-level Buyers';
                  profile = ['Growing families', 'Upgraders', 'Stable income groups'];
                } else if (priceNum < 20000000) {
                  segment = 'Premium Buyers';
                  profile = ['Established professionals', 'Business owners', 'Luxury seekers'];
                } else {
                  segment = 'Ultra-premium Buyers';
                  profile = ['High net worth individuals', 'Investors', 'Elite clientele'];
                }
                
                return (
                  <div key={idx} className="bg-white p-4 rounded-lg border">
                    <div className="flex justify-between items-start mb-3">
                      <div>
                        <p className="text-sm text-gray-600">{band.price_range}</p>
                        <h4 className="font-semibold text-gray-900">{segment}</h4>
                      </div>
                      <Badge className="bg-violet-100 text-violet-700">{band.success_rate}% success</Badge>
                    </div>
                    <div className="space-y-1">
                      {profile.map((p, i) => (
                        <p key={i} className="text-xs text-gray-700">• {p}</p>
                      ))}
                    </div>
                    <div className="mt-3 pt-3 border-t text-xs text-gray-600">
                      <p>Avg closing time: <span className="font-semibold">{band.avg_days_to_sell} days</span></p>
                    </div>
                  </div>
                );
              })}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Property Type Performance */}
      {priceBands.length > 0 && areaAnalysis.length > 0 && (
        <Card className="bg-gradient-to-br from-indigo-50 to-purple-50 border-indigo-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Home className="w-5 h-5 text-indigo-600" />
              Property Type Performance
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Which property types are selling faster in different price ranges</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              {priceBands.slice(0, 3).map((band, idx) => (
                <div key={idx} className="bg-white p-4 rounded-lg border">
                  <h4 className="font-semibold text-gray-900 mb-3">{band.price_range}</h4>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between p-2 bg-indigo-50 rounded">
                      <span className="text-sm text-gray-700">Apartments</span>
                      <Badge className="bg-green-100 text-green-700">High Demand</Badge>
                    </div>
                    <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                      <span className="text-sm text-gray-700">Villas</span>
                      <Badge className="bg-yellow-100 text-yellow-700">Moderate</Badge>
                    </div>
                    <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                      <span className="text-sm text-gray-700">Plots</span>
                      <Badge className="bg-blue-100 text-blue-700">Steady</Badge>
                    </div>
                  </div>
                  <div className="mt-3 pt-3 border-t">
                    <p className="text-xs text-gray-600">Avg selling time</p>
                    <p className="text-lg font-bold text-indigo-600">{band.avg_days_to_sell} days</p>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {marketSummary && (
        <>
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <MapPin className="w-5 h-5" />
                Area-Level Market Trends
              </CardTitle>
              <p className="text-sm text-muted-foreground">Identify high-demand areas and profitable locations for strategic sales</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {areaAnalysis.slice(0, 6).map((area, index) => (
                  <div key={index} className="p-4 bg-muted/30 rounded-lg border">
                    <div className="flex items-start justify-between mb-3">
                      <div className="flex items-center gap-3">
                        <div className={`p-2 rounded-lg ${getHotnessColor(area.market_hotness)}`}>
                          <MapPin className="w-4 h-4" />
                        </div>
                        <div>
                          <h4 className="font-semibold text-lg">{area.locality}</h4>
                          <Badge className={getHotnessColor(area.market_hotness)}>
                            {area.market_hotness} Market
                          </Badge>
                        </div>
                      </div>
                      <div className="text-right">
                        <p className="text-xs text-muted-foreground">Avg Price</p>
                        <p className="font-bold text-xl">{area.avg_market_price_formatted}</p>
                      </div>
                    </div>
                    <div className="grid grid-cols-4 gap-4 mt-4">
                      <div className="text-center p-3 bg-background rounded-lg">
                        <Home className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Supply</p>
                        <p className="font-bold text-lg">{area.active_supply}</p>
                      </div>
                      <div className="text-center p-3 bg-background rounded-lg">
                        <Eye className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Views</p>
                        <p className="font-bold text-lg">{area.total_views}</p>
                      </div>
                      <div className="text-center p-3 bg-background rounded-lg">
                        <BarChart3 className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Demand Ratio</p>
                        <p className="font-bold text-lg">{area.demand_vs_supply_ratio}</p>
                      </div>
                      <div className="text-center p-3 bg-background rounded-lg">
                        <Clock className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Avg Days</p>
                        <p className="font-bold text-lg">{area.avg_days_on_market}</p>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
              <CardHeader>
                <CardTitle className="text-red-800 flex items-center gap-2">
                  <Zap className="w-5 h-5" />
                  Hot Markets
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'hot').slice(0, 3).map((area, idx) => (
                    <div key={idx} className="p-3 bg-white rounded-lg">
                      <p className="font-semibold text-sm">{area.locality}</p>
                      <div className="flex justify-between mt-1">
                        <span className="text-xs text-muted-foreground">Ratio: {area.demand_vs_supply_ratio}</span>
                        <span className="text-xs font-medium text-red-600">{area.avg_days_on_market}d</span>
                      </div>
                    </div>
                  ))}
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'hot').length === 0 && (
                    <p className="text-sm text-muted-foreground text-center py-4">No hot markets currently</p>
                  )}
                </div>
              </CardContent>
            </Card>

            <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
              <CardHeader>
                <CardTitle className="text-orange-800 flex items-center gap-2">
                  <Activity className="w-5 h-5" />
                  Warm Markets
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'warm').slice(0, 3).map((area, idx) => (
                    <div key={idx} className="p-3 bg-white rounded-lg">
                      <p className="font-semibold text-sm">{area.locality}</p>
                      <div className="flex justify-between mt-1">
                        <span className="text-xs text-muted-foreground">Ratio: {area.demand_vs_supply_ratio}</span>
                        <span className="text-xs font-medium text-orange-600">{area.avg_days_on_market}d</span>
                      </div>
                    </div>
                  ))}
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'warm').length === 0 && (
                    <p className="text-sm text-muted-foreground text-center py-4">No warm markets currently</p>
                  )}
                </div>
              </CardContent>
            </Card>

            <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
              <CardHeader>
                <CardTitle className="text-blue-800 flex items-center gap-2">
                  <Target className="w-5 h-5" />
                  Cool Markets
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'cool').slice(0, 3).map((area, idx) => (
                    <div key={idx} className="p-3 bg-white rounded-lg">
                      <p className="font-semibold text-sm">{area.locality}</p>
                      <div className="flex justify-between mt-1">
                        <span className="text-xs text-muted-foreground">Ratio: {area.demand_vs_supply_ratio}</span>
                        <span className="text-xs font-medium text-blue-600">{area.avg_days_on_market}d</span>
                      </div>
                    </div>
                  ))}
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'cool').length === 0 && (
                    <p className="text-sm text-muted-foreground text-center py-4">No cool markets currently</p>
                  )}
                </div>
              </CardContent>
            </Card>
          </div>
        </>
      )}

      {/* Fastest & Slowest Selling Localities */}
      {areaAnalysis.length > 0 && (
        <Card className="bg-gradient-to-br from-teal-50 to-cyan-50 border-teal-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="w-5 h-5 text-teal-600" />
              Sales Velocity by Locality
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Identify fastest and slowest selling areas</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="bg-white p-5 rounded-lg border">
                <div className="flex items-center gap-2 mb-4">
                  <div className="p-2 bg-green-100 rounded-lg">
                    <TrendingUpIcon className="w-5 h-5 text-green-600" />
                  </div>
                  <h4 className="font-semibold text-gray-900">Fastest Selling</h4>
                </div>
                <div className="space-y-3">
                  {[...areaAnalysis]
                    .sort((a, b) => a.avg_days_on_market - b.avg_days_on_market)
                    .slice(0, 4)
                    .map((area, idx) => (
                      <div key={idx} className="flex items-center justify-between p-3 bg-green-50 rounded-lg border border-green-200">
                        <div>
                          <p className="font-semibold text-gray-900">{area.locality}</p>
                          <p className="text-xs text-gray-600">{area.active_supply} properties</p>
                        </div>
                        <div className="text-right">
                          <p className="text-lg font-bold text-green-600">{area.avg_days_on_market}d</p>
                          <p className="text-xs text-gray-600">avg time</p>
                        </div>
                      </div>
                    ))}
                </div>
              </div>
              
              <div className="bg-white p-5 rounded-lg border">
                <div className="flex items-center gap-2 mb-4">
                  <div className="p-2 bg-red-100 rounded-lg">
                    <Clock className="w-5 h-5 text-red-600" />
                  </div>
                  <h4 className="font-semibold text-gray-900">Slowest Selling</h4>
                </div>
                <div className="space-y-3">
                  {[...areaAnalysis]
                    .sort((a, b) => b.avg_days_on_market - a.avg_days_on_market)
                    .slice(0, 4)
                    .map((area, idx) => (
                      <div key={idx} className="flex items-center justify-between p-3 bg-red-50 rounded-lg border border-red-200">
                        <div>
                          <p className="font-semibold text-gray-900">{area.locality}</p>
                          <p className="text-xs text-gray-600">{area.active_supply} properties</p>
                        </div>
                        <div className="text-right">
                          <p className="text-lg font-bold text-red-600">{area.avg_days_on_market}d</p>
                          <p className="text-xs text-gray-600">avg time</p>
                        </div>
                      </div>
                    ))}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Market Hotness Indicator */}
      {areaAnalysis.length > 0 && marketSummary && (
        <Card className="bg-gradient-to-br from-yellow-50 to-amber-50 border-yellow-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Zap className="w-5 h-5 text-yellow-600" />
              Market Hotness Indicator
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Overall market activity and trend direction</p>
          </CardHeader>
          <CardContent>
            {(() => {
              const hotCount = areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'hot').length;
              const warmCount = areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'warm').length;
              const coolCount = areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'cool').length;
              const totalAreas = areaAnalysis.length;
              const hotPercentage = (hotCount / totalAreas) * 100;
              
              let marketStatus = 'Cool';
              let statusColor = 'blue';
              let trendDirection = 'stable';
              
              if (hotPercentage > 40) {
                marketStatus = 'Hot';
                statusColor = 'red';
                trendDirection = marketSummary.market_overview.price_trend === 'up' ? 'improving' : 'stable';
              } else if (hotPercentage > 20 || warmCount > coolCount) {
                marketStatus = 'Warm';
                statusColor = 'orange';
                trendDirection = 'stable';
              } else {
                trendDirection = marketSummary.market_overview.price_trend === 'up' ? 'improving' : 'slowing';
              }
              
              return (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  <div className="text-center">
                    <div className={`inline-flex items-center justify-center w-32 h-32 rounded-full mb-4 ${
                      statusColor === 'red' ? 'bg-red-100' :
                      statusColor === 'orange' ? 'bg-orange-100' : 'bg-blue-100'
                    }`}>
                      <div className="text-center">
                        <Zap className={`w-12 h-12 mx-auto mb-2 ${
                          statusColor === 'red' ? 'text-red-600' :
                          statusColor === 'orange' ? 'text-orange-600' : 'text-blue-600'
                        }`} />
                        <p className={`text-2xl font-bold ${
                          statusColor === 'red' ? 'text-red-600' :
                          statusColor === 'orange' ? 'text-orange-600' : 'text-blue-600'
                        }`}>{marketStatus}</p>
                      </div>
                    </div>
                    <p className="text-sm font-semibold text-gray-900">Market Status</p>
                    <Badge className={`mt-2 ${
                      trendDirection === 'improving' ? 'bg-green-100 text-green-700' :
                      trendDirection === 'slowing' ? 'bg-red-100 text-red-700' :
                      'bg-gray-100 text-gray-700'
                    }`}>
                      {trendDirection === 'improving' ? '↑' : trendDirection === 'slowing' ? '↓' : '→'} {trendDirection}
                    </Badge>
                  </div>
                  
                  <div className="col-span-2 space-y-4">
                    <div className="bg-white p-4 rounded-lg border">
                      <h4 className="font-semibold text-gray-900 mb-3">Market Composition</h4>
                      <div className="space-y-2">
                        <div>
                          <div className="flex justify-between text-sm mb-1">
                            <span>Hot Markets</span>
                            <span className="font-semibold text-red-600">{hotCount} ({hotPercentage.toFixed(0)}%)</span>
                          </div>
                          <div className="w-full bg-gray-200 rounded-full h-2">
                            <div className="bg-red-500 h-2 rounded-full" style={{width: `${hotPercentage}%`}}></div>
                          </div>
                        </div>
                        <div>
                          <div className="flex justify-between text-sm mb-1">
                            <span>Warm Markets</span>
                            <span className="font-semibold text-orange-600">{warmCount} ({((warmCount / totalAreas) * 100).toFixed(0)}%)</span>
                          </div>
                          <div className="w-full bg-gray-200 rounded-full h-2">
                            <div className="bg-orange-500 h-2 rounded-full" style={{width: `${(warmCount / totalAreas) * 100}%`}}></div>
                          </div>
                        </div>
                        <div>
                          <div className="flex justify-between text-sm mb-1">
                            <span>Cool Markets</span>
                            <span className="font-semibold text-blue-600">{coolCount} ({((coolCount / totalAreas) * 100).toFixed(0)}%)</span>
                          </div>
                          <div className="w-full bg-gray-200 rounded-full h-2">
                            <div className="bg-blue-500 h-2 rounded-full" style={{width: `${(coolCount / totalAreas) * 100}%`}}></div>
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    <div className={`p-4 rounded-lg border-l-4 ${
                      marketStatus === 'Hot' ? 'bg-red-50 border-red-500' :
                      marketStatus === 'Warm' ? 'bg-orange-50 border-orange-500' :
                      'bg-blue-50 border-blue-500'
                    }`}>
                      <p className="text-sm font-semibold text-gray-900 mb-1">Market Explanation</p>
                      <p className="text-sm text-gray-700">
                        {marketStatus === 'Hot' && 'High demand with strong buyer activity. Properties sell quickly. Great time for agents to close deals with premium pricing.'}
                        {marketStatus === 'Warm' && 'Balanced market with steady activity. Normal selling timelines. Focus on competitive pricing and quality listings.'}
                        {marketStatus === 'Cool' && 'Lower demand with longer selling times. Buyers have more options. Emphasize value propositions and flexible terms.'}
                      </p>
                    </div>
                  </div>
                </div>
              );
            })()}
          </CardContent>
        </Card>
      )}

      {/* Demand vs Supply Analysis */}
      {areaAnalysis.length > 0 && (
        <Card className="bg-gradient-to-br from-orange-50 to-amber-50 border-orange-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <BarChart3 className="w-5 h-5 text-orange-600" />
              Demand vs Supply Analysis
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Market saturation and buyer interest by locality</p>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {areaAnalysis.slice(0, 5).map((area, idx) => (
                <div key={idx} className="bg-white p-4 rounded-lg border">
                  <div className="flex justify-between items-center mb-3">
                    <div>
                      <h4 className="font-semibold text-gray-900">{area.locality}</h4>
                      <p className="text-xs text-gray-600">Market saturation: {area.demand_vs_supply_ratio > 1.5 ? 'Low' : area.demand_vs_supply_ratio > 0.8 ? 'Moderate' : 'High'}</p>
                    </div>
                    <Badge className={`${
                      area.demand_vs_supply_ratio > 1.5 ? 'bg-green-100 text-green-700' :
                      area.demand_vs_supply_ratio > 0.8 ? 'bg-yellow-100 text-yellow-700' :
                      'bg-red-100 text-red-700'
                    }`}>
                      Ratio: {area.demand_vs_supply_ratio}
                    </Badge>
                  </div>
                  <div className="grid grid-cols-3 gap-4">
                    <div className="text-center p-2 bg-orange-50 rounded">
                      <p className="text-xs text-gray-600">Supply</p>
                      <p className="text-xl font-bold text-orange-700">{area.active_supply}</p>
                      <p className="text-xs text-gray-600">Active listings</p>
                    </div>
                    <div className="text-center p-2 bg-blue-50 rounded">
                      <p className="text-xs text-gray-600">Demand</p>
                      <p className="text-xl font-bold text-blue-700">{area.total_views}</p>
                      <p className="text-xs text-gray-600">Total views</p>
                    </div>
                    <div className="text-center p-2 bg-green-50 rounded">
                      <p className="text-xs text-gray-600">Velocity</p>
                      <p className="text-xl font-bold text-green-700">{area.avg_days_on_market}</p>
                      <p className="text-xs text-gray-600">Days avg</p>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Opportunity Identification */}
      {areaAnalysis.length > 0 && priceBands.length > 0 && (
        <Card className="bg-gradient-to-br from-green-50 to-emerald-50 border-green-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUpIcon className="w-5 h-5 text-green-600" />
              Opportunity Identification
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Emerging areas and high-demand property types</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="bg-white p-5 rounded-lg border">
                <h4 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                  <MapPin className="w-4 h-4 text-green-600" />
                  Emerging High-Growth Areas
                </h4>
                <div className="space-y-3">
                  {areaAnalysis
                    .filter(a => a.demand_vs_supply_ratio > 1.2)
                    .slice(0, 3)
                    .map((area, idx) => (
                      <div key={idx} className="p-3 bg-green-50 rounded-lg border border-green-200">
                        <div className="flex justify-between items-start mb-2">
                          <p className="font-semibold text-gray-900">{area.locality}</p>
                          <Badge className="bg-green-100 text-green-700 border-green-300">
                            {area.market_hotness}
                          </Badge>
                        </div>
                        <div className="grid grid-cols-3 gap-2 text-xs">
                          <div>
                            <p className="text-gray-600">Demand Ratio</p>
                            <p className="font-bold text-green-700">{area.demand_vs_supply_ratio}</p>
                          </div>
                          <div>
                            <p className="text-gray-600">Avg Days</p>
                            <p className="font-bold text-green-700">{area.avg_days_on_market}</p>
                          </div>
                          <div>
                            <p className="text-gray-600">Supply</p>
                            <p className="font-bold text-green-700">{area.active_supply}</p>
                          </div>
                        </div>
                      </div>
                    ))}
                  {areaAnalysis.filter(a => a.demand_vs_supply_ratio > 1.2).length === 0 && (
                    <p className="text-sm text-gray-600 text-center py-4">No high-growth areas currently</p>
                  )}
                </div>
              </div>

              <div className="bg-white p-5 rounded-lg border">
                <h4 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                  <DollarSign className="w-4 h-4 text-green-600" />
                  Fast-Selling Price Ranges
                </h4>
                <div className="space-y-3">
                  {priceBands
                    .sort((a, b) => a.avg_days_to_sell - b.avg_days_to_sell)
                    .slice(0, 3)
                    .map((band, idx) => (
                      <div key={idx} className="p-3 bg-green-50 rounded-lg border border-green-200">
                        <div className="flex justify-between items-start mb-2">
                          <p className="font-semibold text-gray-900">{band.price_range}</p>
                          <Badge className="bg-green-100 text-green-700 border-green-300">
                            {band.success_rate}% success
                          </Badge>
                        </div>
                        <div className="grid grid-cols-3 gap-2 text-xs">
                          <div>
                            <p className="text-gray-600">Avg Days</p>
                            <p className="font-bold text-green-700">{band.avg_days_to_sell}</p>
                          </div>
                          <div>
                            <p className="text-gray-600">Total</p>
                            <p className="font-bold text-green-700">{band.total_properties}</p>
                          </div>
                          <div>
                            <p className="text-gray-600">Sold</p>
                            <p className="font-bold text-green-700">{band.sold_properties}</p>
                          </div>
                        </div>
                      </div>
                    ))}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {priceBands.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <DollarSign className="w-5 h-5" />
                Price Band Performance
              </CardTitle>
              <p className="text-sm text-muted-foreground">Which price ranges sell faster - optimize pricing strategy</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {priceBands.map((band, index) => (
                  <div key={index} className="p-4 border rounded-lg hover:shadow-md transition-shadow">
                    <div className="flex items-center justify-between mb-3">
                      <h4 className="font-semibold text-lg">{band.price_range}</h4>
                      <Badge variant="outline" className="text-base px-3 py-1">
                        {band.success_rate}% success
                      </Badge>
                    </div>
                    <div className="grid grid-cols-3 gap-4 mb-3">
                      <div className="text-center p-2 bg-muted/30 rounded">
                        <Clock className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Avg Days</p>
                        <p className="font-bold text-lg">{band.avg_days_to_sell}</p>
                      </div>
                      <div className="text-center p-2 bg-muted/30 rounded">
                        <Home className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Total</p>
                        <p className="font-bold text-lg">{band.total_properties}</p>
                      </div>
                      <div className="text-center p-2 bg-muted/30 rounded">
                        <TrendingUpIcon className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Sold</p>
                        <p className="font-bold text-lg">{band.sold_properties}</p>
                      </div>
                    </div>
                    <div className="bg-blue-50 border border-blue-200 p-3 rounded text-sm">
                      <div className="flex items-start gap-2">
                        <Target className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
                        <div>
                          <p className="font-medium text-blue-900 mb-1">Agent Strategy:</p>
                          <p className="text-blue-800">{band.recommendation}</p>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {seasonalTrends.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Activity className="w-5 h-5" />
                Seasonal Demand Patterns
              </CardTitle>
              <p className="text-sm text-muted-foreground">Time your listings for maximum buyer interest</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {seasonalTrends.map((trend, index) => (
                  <div key={index} className="p-4 border rounded-lg hover:shadow-md transition-shadow">
                    <div className="flex items-center justify-between mb-3">
                      <h4 className="font-semibold text-lg">{trend.month}</h4>
                      <Badge 
                        className={getDemandColor(trend.demand_level)}
                        style={trend.demand_level.toLowerCase() === 'low' ? {backgroundColor: '#2563eb', color: 'white', borderColor: '#2563eb'} : {}}
                      >
                        {trend.demand_level} demand
                      </Badge>
                    </div>
                    <div className="grid grid-cols-3 gap-4">
                      <div className="text-center p-2 bg-muted/30 rounded">
                        <DollarSign className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Avg Price</p>
                        <p className="font-bold">{trend.avg_price_formatted}</p>
                      </div>
                      <div className="text-center p-2 bg-muted/30 rounded">
                        <Users className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Sales</p>
                        <p className="font-bold">{trend.sold_count}</p>
                      </div>
                      <div className="text-center p-2 bg-muted/30 rounded">
                        <Clock className="w-4 h-4 mx-auto mb-1 text-muted-foreground" />
                        <p className="text-xs text-muted-foreground">Days</p>
                        <p className="font-bold">{trend.avg_days_on_market}</p>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}
      </div>

      {(areaAnalysis.length > 0 || priceBands.length > 0) && (
        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-purple-900">
              <Target className="w-5 h-5" />
              Agent Strategy Recommendations
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Data-driven insights to maximize your success rate</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              <div className="bg-white p-4 rounded-lg border border-purple-200">
                <div className="flex items-center gap-2 mb-2">
                  <MapPin className="w-5 h-5 text-purple-600" />
                  <h4 className="font-semibold">Focus Areas</h4>
                </div>
                <p className="text-sm text-muted-foreground mb-2">Prioritize hot markets with high demand ratios for faster conversions</p>
                <div className="space-y-1">
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'hot').slice(0, 2).map((area, idx) => (
                    <div key={idx} className="text-xs bg-purple-50 p-2 rounded">
                      <span className="font-medium">{area.locality}</span>
                      <span className="text-muted-foreground ml-2">({area.demand_vs_supply_ratio} ratio)</span>
                    </div>
                  ))}
                  {areaAnalysis.filter(a => a.market_hotness.toLowerCase() === 'hot').length === 0 && (
                    <p className="text-xs text-gray-600">Monitor emerging areas</p>
                  )}
                </div>
              </div>

              <div className="bg-white p-4 rounded-lg border border-purple-200">
                <div className="flex items-center gap-2 mb-2">
                  <DollarSign className="w-5 h-5 text-purple-600" />
                  <h4 className="font-semibold">Optimal Pricing</h4>
                </div>
                <p className="text-sm text-muted-foreground mb-2">Target price bands with highest success rates and fastest sales</p>
                <div className="space-y-1">
                  {priceBands.slice(0, 2).map((band, idx) => (
                    <div key={idx} className="text-xs bg-purple-50 p-2 rounded">
                      <span className="font-medium">{band.price_range}</span>
                      <span className="text-muted-foreground ml-2">({band.success_rate}% success)</span>
                    </div>
                  ))}
                  {priceBands.length === 0 && (
                    <p className="text-xs text-gray-600">Analyze price trends</p>
                  )}
                </div>
              </div>

              <div className="bg-white p-4 rounded-lg border border-purple-200">
                <div className="flex items-center gap-2 mb-2">
                  <Clock className="w-5 h-5 text-purple-600" />
                  <h4 className="font-semibold">Market Velocity</h4>
                </div>
                <p className="text-sm text-muted-foreground mb-2">Average time to close deals across all markets</p>
                {marketSummary && (
                  <div className="text-center bg-purple-50 p-3 rounded">
                    <p className="text-3xl font-bold text-purple-900">{Math.round(marketSummary.seller_summary.avg_days_on_market)}</p>
                    <p className="text-xs text-muted-foreground">days average</p>
                  </div>
                )}
              </div>
            </div>
            
            <div className="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="bg-white p-4 rounded-lg border border-purple-200">
                <h4 className="font-semibold mb-2 flex items-center gap-2">
                  <Users className="w-4 h-4 text-purple-600" />
                  Client Advisory Tips
                </h4>
                <ul className="space-y-2 text-sm text-gray-700">
                  <li className="flex items-start gap-2">
                    <span className="text-purple-600 mt-0.5">•</span>
                    <span>Advise sellers in hot markets to maintain pricing power</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-purple-600 mt-0.5">•</span>
                    <span>Recommend competitive pricing in saturated areas</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-purple-600 mt-0.5">•</span>
                    <span>Focus on properties in fast-selling price ranges</span>
                  </li>
                </ul>
              </div>
              
              <div className="bg-white p-4 rounded-lg border border-purple-200">
                <h4 className="font-semibold mb-2 flex items-center gap-2">
                  <TrendingUpIcon className="w-4 h-4 text-purple-600" />
                  Deal Closing Strategy
                </h4>
                <ul className="space-y-2 text-sm text-gray-700">
                  <li className="flex items-start gap-2">
                    <span className="text-purple-600 mt-0.5">•</span>
                    <span>Target areas with demand-supply ratio above 1.2</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-purple-600 mt-0.5">•</span>
                    <span>Leverage seasonal trends for optimal listing timing</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-purple-600 mt-0.5">•</span>
                    <span>Monitor conversion rates to adjust strategy</span>
                  </li>
                </ul>
              </div>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
