import React, { useState, useEffect } from 'react';
import { Clock, CheckCircle, AlertTriangle, FileText, MessageSquare, Upload, Eye, Calendar, X, Send, Phone, Mail, MapPin, User, Building, DollarSign, Shield, Gavel, CreditCard } from 'lucide-react';
import { toast } from 'sonner';

interface NRIRequest {
  id: string;
  property_id: string;
  property_title: string;
  property_address: string;
  buyer_name: string;
  buyer_email: string;
  buyer_phone: string;
  assistance_category: string;
  description: string;
  priority_level: 'high' | 'medium' | 'low';
  status: 'pending_acknowledgment' | 'acknowledged' | 'documents_pending' | 'in_progress' | 'completed' | 'declined';
  created_at: string;
  approved_at: string;
}

interface ResponseTemplate {
  category: string;
  icon: any;
  color: string;
  templates: {
    title: string;
    content: string;
  }[];
}

export function NRIAssistanceAgentEnhanced() {
  const [requests, setRequests] = useState<NRIRequest[]>([]);
  const [selectedRequest, setSelectedRequest] = useState<NRIRequest | null>(null);
  const [loading, setLoading] = useState(true);
  const [showDetailsModal, setShowDetailsModal] = useState(false);
  const [showMessageModal, setShowMessageModal] = useState(false);
  const [showUploadModal, setShowUploadModal] = useState(false);
  const [showResponseModal, setShowResponseModal] = useState(false);
  const [messageText, setMessageText] = useState('');
  const [selectedTemplate, setSelectedTemplate] = useState('');

  const responseTemplates: ResponseTemplate[] = [
    {
      category: 'Property',
      icon: Building,
      color: 'blue',
      templates: [
        {
          title: 'Market Analysis',
          content: 'I can provide you with a comprehensive market analysis for this property, including recent comparable sales, price trends, and investment potential in this area.'
        },
        {
          title: 'Property Documentation',
          content: 'I will share all relevant property documents including title deed, approved plans, NOC certificates, and property tax receipts for your review.'
        }
      ]
    },
    {
      category: 'Legal',
      icon: Gavel,
      color: 'purple',
      templates: [
        {
          title: 'Legal Compliance',
          content: 'I can guide you through the legal requirements for NRI property purchase, including FEMA compliance, required documentation, and legal procedures.'
        },
        {
          title: 'Documentation Support',
          content: 'I will help you with Power of Attorney preparation, legal document verification, and connect you with trusted legal advisors.'
        }
      ]
    },
    {
      category: 'Banking',
      icon: CreditCard,
      color: 'green',
      templates: [
        {
          title: 'Loan Assistance',
          content: 'I can connect you with banks offering NRI home loans, help with documentation, and guide you through the loan approval process.'
        },
        {
          title: 'Banking Setup',
          content: 'I will assist you with NRE/NRO account setup, remittance procedures, and tax implications for your property investment.'
        }
      ]
    },
    {
      category: 'Verification',
      icon: Shield,
      color: 'orange',
      templates: [
        {
          title: 'Property Verification',
          content: 'I will conduct thorough verification of property documents, ownership history, legal clearances, and provide you with a detailed verification report.'
        },
        {
          title: 'Due Diligence',
          content: 'I can arrange for professional due diligence including site inspection, document verification, and legal status confirmation.'
        }
      ]
    }
  ];

  useEffect(() => {
    fetchNRIRequests();
  }, []);

  const fetchNRIRequests = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const accessToken = localStorage.getItem('access_token');
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      
      if (!accessToken) {
        setLoading(false);
        return;
      }

      let userId = user.id || user.user_id || user.userId || localStorage.getItem('user_id');
      
      // If still no userId, try to extract from user object properties
      if (!userId && user) {
        // Check all possible user ID fields
        const possibleIds = Object.values(user).filter(val => 
          typeof val === 'string' && val.includes('-') && val.length > 20
        );
        if (possibleIds.length > 0) {
          userId = possibleIds[0];
        }
      }
      if (!userId) {
        setLoading(false);
        return;
      }

      const response = await fetch(`${API_BASE}/nri-requests/my-properties/?userid=${userId}`, {
        headers: {
          'Authorization': `Bearer ${accessToken}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        const data = await response.json();
        const transformedRequests = data.requests?.map((req: any) => ({
          id: req.id.toString(),
          property_id: req.property_id,
          property_title: req.property_title || 'Property',
          property_address: req.property_address || '',
          buyer_name: req.buyer_name,
          buyer_email: req.buyer_email,
          buyer_phone: req.buyer_phone,
          assistance_category: req.assistance_category,
          description: req.description,
          priority_level: req.priority_level?.toLowerCase() || 'medium',
          status: 'pending_acknowledgment',
          created_at: req.created_at,
          approved_at: req.approved_at
        })) || [];
        setRequests(transformedRequests);
      }
    } catch (error) {
      console.error('Error fetching NRI requests:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending_acknowledgment': return 'bg-red-100 text-red-800 border-red-200';
      case 'acknowledged': return 'bg-blue-100 text-blue-800 border-blue-200';
      case 'documents_pending': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'in_progress': return 'bg-purple-100 text-purple-800 border-purple-200';
      case 'completed': return 'bg-green-100 text-green-800 border-green-200';
      case 'declined': return 'bg-gray-100 text-gray-800 border-gray-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high': return 'bg-red-500';
      case 'medium': return 'bg-yellow-500';
      case 'low': return 'bg-green-500';
      default: return 'bg-gray-500';
    }
  };

  const getCategoryIcon = (category: string) => {
    switch (category.toLowerCase()) {
      case 'property': return Building;
      case 'legal': return Gavel;
      case 'banking': return CreditCard;
      case 'verification': return Shield;
      default: return FileText;
    }
  };

  const handleStatusUpdate = (requestId: string, newStatus: string) => {
    setRequests(prev => prev.map(req => 
      req.id === requestId ? { ...req, status: newStatus as any } : req
    ));
    toast.success(`Request status updated to ${newStatus.replace('_', ' ')}`);
  };

  const handleSendMessage = () => {
    if (!messageText.trim()) {
      toast.error('Please enter a message');
      return;
    }
    
    toast.success('Message sent successfully to buyer');
    setMessageText('');
    setShowMessageModal(false);
  };

  const handleViewDetails = (request: NRIRequest) => {
    setSelectedRequest(request);
    setShowDetailsModal(true);
  };

  const getResponseTemplatesForCategory = (category: string) => {
    return responseTemplates.find(t => t.category.toLowerCase() === category.toLowerCase())?.templates || [];
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-white rounded-lg shadow-sm border p-6">
        <h1 className="text-2xl font-bold text-gray-900 mb-2">NRI Assistance Requests</h1>
        <p className="text-gray-600">Manage assistance requests from NRI buyers for properties</p>
      </div>

      {/* Active Requests */}
      <div className="grid gap-6">
        {requests.map((request) => {
          const CategoryIcon = getCategoryIcon(request.assistance_category);
          return (
            <div key={request.id} className="bg-white rounded-lg shadow-sm border overflow-hidden">
              {/* Request Header */}
              <div className="p-6 border-b bg-gradient-to-r from-blue-50 to-indigo-50">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    <div className={`w-3 h-3 rounded-full ${getPriorityColor(request.priority_level)}`}></div>
                    <div className="flex items-center space-x-3">
                      <CategoryIcon className="w-6 h-6 text-blue-600" />
                      <div>
                        <h3 className="font-semibold text-gray-900">{request.property_title}</h3>
                        <p className="text-sm text-gray-600">Request from {request.buyer_name}</p>
                      </div>
                    </div>
                  </div>
                  <div className="text-right">
                    <span className={`inline-flex px-3 py-1 text-xs font-medium rounded-full border ${getStatusColor(request.status)}`}>
                      {request.status.replace('_', ' ').toUpperCase()}
                    </span>
                    <p className="text-xs text-gray-500 mt-1">
                      Priority: <span className="capitalize font-medium">{request.priority_level}</span>
                    </p>
                  </div>
                </div>
              </div>

              {/* Request Content */}
              <div className="p-6">
                <div className="grid md:grid-cols-2 gap-6">
                  {/* Left Column - Request Info */}
                  <div className="space-y-4">
                    <div>
                      <h4 className="font-medium text-gray-900 mb-3 flex items-center">
                        <FileText className="w-4 h-4 mr-2" />
                        Request Details
                      </h4>
                      <div className="bg-gray-50 rounded-lg p-4 space-y-3">
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">Property ID:</span>
                          <span className="font-medium">{request.property_id}</span>
                        </div>
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">Category:</span>
                          <span className="font-medium">{request.assistance_category}</span>
                        </div>
                        <div className="text-sm">
                          <span className="text-gray-600">Description:</span>
                          <p className="mt-1 text-gray-900">{request.description}</p>
                        </div>
                      </div>
                    </div>

                    {/* Buyer Contact Info */}
                    <div>
                      <h4 className="font-medium text-gray-900 mb-3 flex items-center">
                        <User className="w-4 h-4 mr-2" />
                        Buyer Information
                      </h4>
                      <div className="bg-gray-50 rounded-lg p-4 space-y-2">
                        <div className="flex items-center text-sm">
                          <Mail className="w-4 h-4 mr-2 text-gray-500" />
                          <span>{request.buyer_email}</span>
                        </div>
                        <div className="flex items-center text-sm">
                          <Phone className="w-4 h-4 mr-2 text-gray-500" />
                          <span>{request.buyer_phone}</span>
                        </div>
                        {request.property_address && (
                          <div className="flex items-start text-sm">
                            <MapPin className="w-4 h-4 mr-2 text-gray-500 mt-0.5" />
                            <span className="text-gray-600">{request.property_address}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  {/* Right Column - Actions */}
                  <div className="space-y-4">
                    <div>
                      <h4 className="font-medium text-gray-900 mb-3">Quick Actions</h4>
                      <div className="space-y-3">
                        {request.status === 'pending_acknowledgment' && (
                          <>
                            <button
                              onClick={() => handleStatusUpdate(request.id, 'acknowledged')}
                              className="w-full bg-green-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-green-600 transition-colors flex items-center justify-center"
                            >
                              <CheckCircle className="w-4 h-4 mr-2" />
                              Accept Request
                            </button>
                            <button
                              onClick={() => handleStatusUpdate(request.id, 'declined')}
                              className="w-full bg-red-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-red-600 transition-colors flex items-center justify-center"
                            >
                              <X className="w-4 h-4 mr-2" />
                              Decline Request
                            </button>
                          </>
                        )}

                        {request.status === 'acknowledged' && (
                          <>
                            <button
                              onClick={() => {
                                setSelectedRequest(request);
                                setShowResponseModal(true);
                              }}
                              className="w-full bg-blue-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-600 transition-colors flex items-center justify-center"
                            >
                              <MessageSquare className="w-4 h-4 mr-2" />
                              Send Response
                            </button>
                            <button
                              onClick={() => setShowUploadModal(true)}
                              className="w-full bg-purple-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-purple-700 transition-colors flex items-center justify-center"
                            >
                              <Upload className="w-4 h-4 mr-2" />
                              Upload Documents
                            </button>
                            <button
                              onClick={() => handleStatusUpdate(request.id, 'in_progress')}
                              className="w-full bg-orange-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-orange-600 transition-colors"
                            >
                              Mark In Progress
                            </button>
                          </>
                        )}

                        {request.status === 'in_progress' && (
                          <>
                            <button
                              onClick={() => {
                                setSelectedRequest(request);
                                setShowMessageModal(true);
                              }}
                              className="w-full bg-blue-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-600 transition-colors flex items-center justify-center"
                            >
                              <MessageSquare className="w-4 h-4 mr-2" />
                              Send Update
                            </button>
                            <button
                              onClick={() => handleStatusUpdate(request.id, 'completed')}
                              className="w-full bg-green-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-green-600 transition-colors flex items-center justify-center"
                            >
                              <CheckCircle className="w-4 h-4 mr-2" />
                              Mark Completed
                            </button>
                          </>
                        )}

                        {request.status === 'completed' && (
                          <div className="bg-green-50 border border-green-200 rounded-lg p-3">
                            <div className="flex items-center text-green-800 text-sm">
                              <CheckCircle className="w-4 h-4 mr-2" />
                              Request completed successfully
                            </div>
                          </div>
                        )}

                        {request.status === 'declined' && (
                          <div className="bg-gray-50 border border-gray-200 rounded-lg p-3">
                            <div className="flex items-center text-gray-800 text-sm">
                              <X className="w-4 h-4 mr-2" />
                              Request declined
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>
                </div>

                {/* Bottom Actions */}
                <div className="mt-6 pt-4 border-t flex items-center justify-between">
                  <div className="flex items-center space-x-4 text-sm text-gray-600">
                    <span className="flex items-center">
                      <Calendar className="w-4 h-4 mr-1" />
                      Created {new Date(request.created_at).toLocaleDateString()}
                    </span>
                    {request.approved_at && (
                      <span className="flex items-center">
                        <CheckCircle className="w-4 h-4 mr-1" />
                        Approved {new Date(request.approved_at).toLocaleDateString()}
                      </span>
                    )}
                  </div>
                  <button 
                    onClick={() => handleViewDetails(request)}
                    className="text-blue-600 hover:text-blue-700 text-sm font-medium flex items-center"
                  >
                    <Eye className="w-4 h-4 mr-1" />
                    View Full Details
                  </button>
                </div>
              </div>
            </div>
          );
        })}
      </div>

      {requests.length === 0 && (
        <div className="bg-white rounded-lg shadow-sm border p-12 text-center">
          <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No NRI Assistance Requests</h3>
          <p className="text-gray-600">You don't have any active NRI assistance requests at the moment.</p>
        </div>
      )}

      {/* Details Modal */}
      {showDetailsModal && selectedRequest && (
        <div className="fixed inset-0 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl p-6 max-w-2xl w-full max-h-[90vh] overflow-y-auto border">
            <div className="flex justify-between items-center mb-6">
              <h3 className="text-xl font-semibold">Request Details</h3>
              <button onClick={() => setShowDetailsModal(false)}>
                <X className="w-5 h-5" />
              </button>
            </div>
            
            <div className="space-y-6">
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Property Information</h4>
                  <div className="space-y-2 text-sm">
                    <div><span className="font-medium">Title:</span> {selectedRequest.property_title}</div>
                    <div><span className="font-medium">ID:</span> {selectedRequest.property_id}</div>
                    {selectedRequest.property_address && (
                      <div><span className="font-medium">Address:</span> {selectedRequest.property_address}</div>
                    )}
                  </div>
                </div>
                
                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Buyer Information</h4>
                  <div className="space-y-2 text-sm">
                    <div><span className="font-medium">Name:</span> {selectedRequest.buyer_name}</div>
                    <div><span className="font-medium">Email:</span> {selectedRequest.buyer_email}</div>
                    <div><span className="font-medium">Phone:</span> {selectedRequest.buyer_phone}</div>
                  </div>
                </div>
              </div>
              
              <div>
                <h4 className="font-medium text-gray-900 mb-3">Request Information</h4>
                <div className="bg-gray-50 rounded-lg p-4 space-y-3">
                  <div><span className="font-medium">Category:</span> {selectedRequest.assistance_category}</div>
                  <div><span className="font-medium">Priority:</span> <span className="capitalize">{selectedRequest.priority_level}</span></div>
                  <div><span className="font-medium">Status:</span> {selectedRequest.status.replace('_', ' ')}</div>
                  <div>
                    <span className="font-medium">Description:</span>
                    <p className="mt-1">{selectedRequest.description}</p>
                  </div>
                </div>
              </div>
            </div>
            
            <div className="mt-6 flex space-x-3">
              <button
                onClick={() => {
                  setShowDetailsModal(false);
                  setShowMessageModal(true);
                }}
                className="flex-1 bg-blue-500 text-white px-4 py-2 rounded-lg hover:bg-blue-600"
              >
                Send Message
              </button>
              <button
                onClick={() => setShowDetailsModal(false)}
                className="flex-1 bg-gray-500 text-white px-4 py-2 rounded-lg hover:bg-gray-600"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Response Templates Modal */}
      {showResponseModal && selectedRequest && (
        <div className="fixed inset-0 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl p-6 max-w-2xl w-full max-h-[90vh] overflow-y-auto border">
            <div className="flex justify-between items-center mb-6">
              <h3 className="text-xl font-semibold">Send Response - {selectedRequest.assistance_category}</h3>
              <button onClick={() => setShowResponseModal(false)}>
                <X className="w-5 h-5" />
              </button>
            </div>
            
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Response Templates</label>
                <div className="space-y-2">
                  {getResponseTemplatesForCategory(selectedRequest.assistance_category).map((template, index) => (
                    <div key={index} className="border rounded-lg p-3 hover:bg-gray-50 cursor-pointer"
                         onClick={() => setMessageText(template.content)}>
                      <h4 className="font-medium text-sm">{template.title}</h4>
                      <p className="text-xs text-gray-600 mt-1">{template.content.substring(0, 100)}...</p>
                    </div>
                  ))}
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Your Response</label>
                <textarea 
                  value={messageText}
                  onChange={(e) => setMessageText(e.target.value)}
                  className="w-full border border-gray-300 rounded-lg p-3 h-32 resize-none"
                  placeholder="Type your response here or select a template above..."
                />
              </div>
            </div>
            
            <div className="mt-6 flex space-x-3">
              <button
                onClick={() => {
                  handleSendMessage();
                  setShowResponseModal(false);
                }}
                className="flex-1 bg-blue-500 text-white px-4 py-2 rounded-lg hover:bg-blue-600 flex items-center justify-center"
              >
                <Send className="w-4 h-4 mr-2" />
                Send Response
              </button>
              <button
                onClick={() => setShowResponseModal(false)}
                className="flex-1 bg-white text-black px-4 py-2 rounded-lg hover:bg-gray-100 border border-gray-300"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Message Modal */}
      {showMessageModal && selectedRequest && (
        <div className="fixed inset-0 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl p-6 max-w-md w-full border">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-semibold">Message {selectedRequest.buyer_name}</h3>
              <button onClick={() => setShowMessageModal(false)}>
                <X className="w-5 h-5" />
              </button>
            </div>
            <textarea 
              value={messageText}
              onChange={(e) => setMessageText(e.target.value)}
              className="w-full border border-gray-300 rounded-lg p-3 h-32 resize-none"
              placeholder="Type your message here..."
            />
            <div className="mt-4 flex space-x-3">
              <button 
                onClick={handleSendMessage}
                className="flex-1 bg-blue-500 text-white px-4 py-2 rounded-lg hover:bg-blue-600 flex items-center justify-center"
              >
                <Send className="w-4 h-4 mr-2" />
                Send Message
              </button>
              <button 
                onClick={() => setShowMessageModal(false)}
                className="flex-1 bg-white text-black px-4 py-2 rounded-lg hover:bg-gray-100 border border-gray-300"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Upload Modal */}
      {showUploadModal && (
        <div className="fixed inset-0 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl p-6 max-w-md w-full border">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-semibold">Upload Documents</h3>
              <button onClick={() => setShowUploadModal(false)}>
                <X className="w-5 h-5" />
              </button>
            </div>
            <div className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center">
              <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <p className="text-gray-600 mb-4">Drag and drop files here or click to browse</p>
              <input type="file" multiple className="hidden" id="file-upload" />
              <label htmlFor="file-upload" className="bg-blue-500 text-white px-4 py-2 rounded-lg cursor-pointer hover:bg-blue-600">
                Choose Files
              </label>
            </div>
            <div className="mt-4 flex space-x-3">
              <button 
                onClick={() => { setShowUploadModal(false); toast.success('Documents uploaded successfully'); }}
                className="flex-1 bg-green-500 text-white px-4 py-2 rounded-lg hover:bg-green-600 font-medium"
              >
                Upload
              </button>
              <button 
                onClick={() => setShowUploadModal(false)}
                className="flex-1 bg-white text-black px-4 py-2 rounded-lg hover:bg-gray-100 border border-gray-300 font-medium"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}