import { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { negotiationApi } from '../../api/negotiation.api';

export default function Negotiations() {
  const navigate = useNavigate();
  const [negotiations, setNegotiations] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadNegotiations();
  }, []);

  const loadNegotiations = async () => {
    try {
      const data = await negotiationApi.listNegotiations();
      setNegotiations(data);
    } catch (error) {
      console.error('Failed to load negotiations:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-1/4"></div>
          <div className="h-32 bg-gray-200 rounded"></div>
          <div className="h-32 bg-gray-200 rounded"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900">Negotiations</h1>
        <p className="text-gray-600 mt-1">Manage all property negotiations</p>
      </div>

      {negotiations.length === 0 ? (
        <div className="bg-white rounded-lg border p-12 text-center">
          <p className="text-gray-500 text-lg">No negotiations yet</p>
          <p className="text-gray-400 text-sm mt-2">Negotiations will appear here when buyers make offers</p>
        </div>
      ) : (
        <div className="grid gap-4">
          {negotiations.map((negotiation) => (
            <div key={negotiation.negotiation_id} className="bg-white rounded-lg border p-6 hover:shadow-md transition-shadow">
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="text-lg font-semibold text-gray-900">
                      Property #{negotiation.property_id}
                    </h3>
                    <span className={`px-3 py-1 rounded-full text-xs font-medium ${
                      negotiation.status === 'ACCEPTED' ? 'bg-green-100 text-green-800' :
                      negotiation.status === 'REJECTED' ? 'bg-red-100 text-red-800' :
                      negotiation.status === 'OFFER_SENT' || negotiation.status === 'COUNTER_SENT' ? 'bg-yellow-100 text-yellow-800' :
                      'bg-blue-100 text-blue-800'
                    }`}>
                      {negotiation.status ? negotiation.status.replace('_', ' ') : 'PENDING'}
                    </span>
                  </div>
                  <p className="text-sm text-gray-600 mb-1">
                    Negotiation ID: {negotiation.negotiation_id}
                  </p>
                  <p className="text-sm text-gray-500">
                    Started: {new Date(negotiation.created_at).toLocaleDateString('en-IN')}
                  </p>
                </div>
                <button
                  onClick={() => navigate(`/agent/dashboard/negotiation/${negotiation.negotiation_id}`)}
                  style={{ backgroundColor: '#2563eb', color: '#ffffff', padding: '8px 16px', borderRadius: '6px', fontWeight: '600' }}
                  className="shadow-md hover:opacity-90"
                >
                  View Details
                </button>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
