import React, { useState } from 'react';
import { GitCompare, Plus, X, MapPin, Bed, Bath, Square, IndianRupee } from 'lucide-react';

const PropertyComparison: React.FC = () => {
  const [selectedProperties, setSelectedProperties] = useState([
    {
      id: '1',
      title: '3BHK Luxury Apartment',
      price: '₹50,00,000',
      location: 'Bandra West, Mumbai',
      bedrooms: 3,
      bathrooms: 2,
      area: '1200 sq ft',
      image: 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=1000&q=80',
      amenities: ['Swimming Pool', 'Gym', 'Parking', 'Security'],
      pricePerSqft: 4167
    },
    {
      id: '2',
      title: '2BHK Modern Flat',
      price: '₹35,00,000',
      location: 'Andheri West, Mumbai',
      bedrooms: 2,
      bathrooms: 2,
      area: '900 sq ft',
      image: 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=1000&q=80',
      amenities: ['Parking', 'Security', 'Garden'],
      pricePerSqft: 3889
    }
  ]);

  const availableProperties = [
    {
      id: '3',
      title: 'Villa with Garden',
      price: '₹85,00,000',
      location: 'Juhu, Mumbai',
      bedrooms: 4,
      bathrooms: 3,
      area: '2500 sq ft',
      image: 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=1000&q=80',
      amenities: ['Swimming Pool', 'Garden', 'Parking', 'Security', 'Gym'],
      pricePerSqft: 3400
    }
  ];

  const addProperty = (property: any) => {
    if (selectedProperties.length < 3) {
      setSelectedProperties([...selectedProperties, property]);
    }
  };

  const removeProperty = (id: string) => {
    setSelectedProperties(selectedProperties.filter(p => p.id !== id));
  };

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Comparison</h2>
        <p className="text-gray-600 mb-6">Compare properties side by side to make informed decisions</p>

        {/* Add Property Section */}
        {selectedProperties.length < 3 && (
          <div className="mb-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-3">Add Property to Compare</h3>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              {availableProperties.map((property) => (
                <div key={property.id} className="border border-gray-200 rounded-lg p-4">
                  <img src={property.image} alt={property.title} className="w-full h-32 object-cover rounded mb-3" />
                  <h4 className="font-semibold text-gray-900 mb-1">{property.title}</h4>
                  <p className="text-[#0056D2] font-bold mb-2">{property.price}</p>
                  <p className="text-gray-600 text-sm mb-3">{property.location}</p>
                  <button
                    onClick={() => addProperty(property)}
                    className="w-full flex items-center justify-center gap-2 px-3 py-2 bg-[#0056D2] text-white rounded hover:bg-[#0041A3] transition-colors"
                  >
                    <Plus className="w-4 h-4" />
                    Add to Compare
                  </button>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Comparison Table */}
        {selectedProperties.length > 0 && (
          <div className="overflow-x-auto">
            <table className="w-full border-collapse">
              <thead>
                <tr>
                  <td className="p-4 font-semibold text-gray-900 border-b">Property</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4 border-b">
                      <div className="relative">
                        <button
                          onClick={() => removeProperty(property.id)}
                          className="absolute -top-2 -right-2 w-6 h-6 bg-red-500 text-white rounded-full flex items-center justify-center hover:bg-red-600"
                        >
                          <X className="w-3 h-3" />
                        </button>
                        <img src={property.image} alt={property.title} className="w-full h-32 object-cover rounded mb-2" />
                        <h3 className="font-semibold text-gray-900">{property.title}</h3>
                      </div>
                    </td>
                  ))}
                </tr>
              </thead>
              <tbody>
                <tr>
                  <td className="p-4 font-medium text-gray-700 border-b">Price</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4 border-b">
                      <span className="text-[#0056D2] font-bold text-lg">{property.price}</span>
                    </td>
                  ))}
                </tr>
                <tr>
                  <td className="p-4 font-medium text-gray-700 border-b">Location</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4 border-b">
                      <div className="flex items-center">
                        <MapPin className="w-4 h-4 mr-1 text-gray-500" />
                        {property.location}
                      </div>
                    </td>
                  ))}
                </tr>
                <tr>
                  <td className="p-4 font-medium text-gray-700 border-b">Bedrooms</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4 border-b">
                      <div className="flex items-center">
                        <Bed className="w-4 h-4 mr-1 text-gray-500" />
                        {property.bedrooms}
                      </div>
                    </td>
                  ))}
                </tr>
                <tr>
                  <td className="p-4 font-medium text-gray-700 border-b">Bathrooms</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4 border-b">
                      <div className="flex items-center">
                        <Bath className="w-4 h-4 mr-1 text-gray-500" />
                        {property.bathrooms}
                      </div>
                    </td>
                  ))}
                </tr>
                <tr>
                  <td className="p-4 font-medium text-gray-700 border-b">Area</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4 border-b">
                      <div className="flex items-center">
                        <Square className="w-4 h-4 mr-1 text-gray-500" />
                        {property.area}
                      </div>
                    </td>
                  ))}
                </tr>
                <tr>
                  <td className="p-4 font-medium text-gray-700 border-b">Price per Sq Ft</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4 border-b">
                      <div className="flex items-center">
                        <IndianRupee className="w-4 h-4 mr-1 text-gray-500" />
                        {property.pricePerSqft}
                      </div>
                    </td>
                  ))}
                </tr>
                <tr>
                  <td className="p-4 font-medium text-gray-700">Amenities</td>
                  {selectedProperties.map((property) => (
                    <td key={property.id} className="p-4">
                      <div className="space-y-1">
                        {property.amenities.map((amenity, index) => (
                          <div key={index} className="flex items-center">
                            <div className="w-2 h-2 bg-green-500 rounded-full mr-2"></div>
                            <span className="text-sm">{amenity}</span>
                          </div>
                        ))}
                      </div>
                    </td>
                  ))}
                </tr>
              </tbody>
            </table>
          </div>
        )}

        {selectedProperties.length === 0 && (
          <div className="text-center py-12">
            <GitCompare className="w-16 h-16 text-gray-300 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No properties to compare</h3>
            <p className="text-gray-500">Add properties to start comparing</p>
          </div>
        )}
      </div>
    </div>
  );
};

export default PropertyComparison;