import React, { createContext, useContext, useState, useEffect, ReactNode } from 'react';

interface Property {
  id: string;
  title: string;
  price: string;
  location: string;
  area: string;
  bedrooms: number;
  bathrooms: number;
  image: string;
  status: string;
}

interface PropertyContextType {
  properties: Property[];
  loading: boolean;
  addProperty: (property: Property) => void;
  updateProperty: (id: string, property: Partial<Property>) => void;
  deleteProperty: (id: string) => void;
  refreshProperties: () => void;
}

const PropertyContext = createContext<PropertyContextType | undefined>(undefined);

export const PropertyProvider: React.FC<{ children: ReactNode }> = ({ children }) => {
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);

  const fetchAgentProperties = async () => {
    try {
      setLoading(true);
      const agentId = localStorage.getItem('agentId');
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const seller = JSON.parse(localStorage.getItem('seller') || '{}');
      const userId = agentId || user.user_id || seller.user_id || seller.id || user.id;
      
      if (!userId) {
        console.error('No user ID found');
        setLoading(false);
        return;
      }
      
      // Get auth token
      const token = localStorage.getItem('agentToken') || localStorage.getItem('token');
      const headers: HeadersInit = {
        'Content-Type': 'application/json'
      };
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }

      // Fetch all properties listed by this agent
      console.log('🔄 Fetching agent properties for user_id:', userId);
      const response = await fetch(`http://localhost:8090/api/properties?listed_by_id=${userId}`, { headers });
      if (response.ok) {
        const propertiesData = await response.json();
        const propertiesArray = propertiesData.properties || propertiesData.results || propertiesData || [];
        console.log('✅ Agent properties after filtering:', propertiesArray.length, 'properties');
        console.log('Properties:', propertiesArray.map((p: any) => ({ id: p.id, title: p.title, listed_by_id: p.listed_by_id, status: p.status })));
        const mappedProperties = propertiesArray.map((prop: any) => ({
          id: prop.id.toString(),
          title: prop.title,
          price: `₹${prop.sale_price || prop.monthly_rent || 0}`,
          location: `${prop.locality || prop.address}, ${prop.city}`,
          area: `${prop.built_up_area} sq ft`,
          bedrooms: prop.bedrooms || 0,
          bathrooms: prop.bathrooms || 2,
          image: prop.primary_image || prop.image_urls?.[0] || prop.photo || 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=1000&q=80',
          status: prop.status === 'active' ? 'Available' : prop.status === 'pending' ? 'Pending' : prop.status === 'sold' ? 'Sold' : prop.status === 'rejected' ? 'Rejected' : prop.status
        }));
        setProperties(mappedProperties);
      }
    } catch (error) {
      console.error('Error fetching agent properties:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchAgentProperties();
  }, []);

  const addProperty = (property: Property) => {
    setProperties(prev => [...prev, property]);
    // Refresh from backend to get the actual saved property
    setTimeout(() => fetchAgentProperties(), 1000);
  };

  const updateProperty = (id: string, updatedProperty: Partial<Property>) => {
    setProperties(prev => 
      prev.map(property => 
        property.id === id ? { ...property, ...updatedProperty } : property
      )
    );
  };

  const deleteProperty = (id: string) => {
    setProperties(prev => prev.filter(property => property.id !== id));
  };

  const refreshProperties = () => {
    fetchAgentProperties();
  };

  return (
    <PropertyContext.Provider value={{
      properties,
      loading,
      addProperty,
      updateProperty,
      deleteProperty,
      refreshProperties
    }}>
      {children}
    </PropertyContext.Provider>
  );
};

export const useProperty = () => {
  const context = useContext(PropertyContext);
  if (!context) {
    throw new Error('useProperty must be used within PropertyProvider');
  }
  return context;
};