import React, { useState } from 'react';
import { Plus, MapPin, Building, Bath, Home, Users, Square, X } from 'lucide-react';
import PropertyDetailView from './PropertyDetailView';
import CollaborationModal from './CollaborationModal';
import { useProperty } from './PropertyContext';
import { ListingForm } from '../../pages/ListingForm';
import { useAzureBlobMedia } from '../../hooks/useAzureBlobMedia';

// Property Card Component with Azure Blob Storage
const PropertyCard: React.FC<{ property: Property; onViewDetails: () => void; onCollaborate: () => void; getStatusColor: (status: string) => string }> = ({ 
  property, 
  onViewDetails, 
  onCollaborate, 
  getStatusColor 
}) => {
  const { images: azureImages } = useAzureBlobMedia(property.id);
  
  const getImageUrl = () => {
    if (azureImages && azureImages.length > 0) {
      return azureImages[0].url;
    }
    return property.image || 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=1000&q=80';
  };

  return (
    <div className="bg-white rounded-xl shadow-sm overflow-hidden hover:shadow-lg transition-all duration-300 border border-gray-100">
      <div className="relative">
        <img
          src={getImageUrl()}
          alt={property.title || 'Property'}
          className="w-full h-48 object-cover"
          onError={(e) => {
            e.currentTarget.src = 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=1000&q=80';
          }}
        />
        <div className="absolute top-3 left-3">
          <div className="flex items-center space-x-1 bg-white/90 backdrop-blur-sm px-2 py-1 rounded-full">
            <div className="w-2 h-2 bg-green-500 rounded-full"></div>
            <span className="text-xs font-medium text-gray-700">NAL Verified</span>
          </div>
        </div>
        <div className="absolute top-3 right-3">
          <span className={`px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(property.status)}`}>
            {property.status}
          </span>
        </div>
      </div>
      
      <div className="p-4">
        <h3 className="font-bold text-lg text-black mb-2 line-clamp-1">{property.title}</h3>
        
        <div className="text-2xl font-bold text-black mb-3">
          {property.price}
        </div>
        
        <div className="flex items-center text-black mb-4">
          <MapPin className="w-4 h-4 mr-1 flex-shrink-0 text-gray-500" />
          <span className="text-sm truncate font-medium">{property.location}</span>
        </div>
        
        <div className="flex items-center justify-between text-sm text-black mb-4">
          <div className="flex items-center">
            <Home className="w-4 h-4 mr-1 text-blue-500" />
            <span className="text-black font-medium">{property.bedrooms || 0} Beds</span>
          </div>
          <div className="flex items-center">
            <Bath className="w-4 h-4 mr-1 text-blue-500" />
            <span className="text-black font-medium">{property.bathrooms || 0} Baths</span>
          </div>
          <div className="flex items-center">
            <Square className="w-4 h-4 mr-1 text-blue-500" />
            <span className="text-black font-medium">{property.area}</span>
          </div>
        </div>
        
        <div className="flex space-x-2">
          <button
            onClick={onViewDetails}
            className="flex-1 py-2 bg-blue-600 text-black rounded-lg hover:bg-blue-700 transition-colors font-semibold text-sm"
          >
            View Details
          </button>
          <button
            onClick={onCollaborate}
            className="px-3 py-2 border border-blue-600 text-blue-600 rounded-lg hover:bg-blue-50 transition-colors"
            title="Collaborate"
          >
            <Users className="w-4 h-4" />
          </button>
        </div>
      </div>
    </div>
  );
};

interface Property {
  id: string;
  title: string;
  price: string;
  location: string;
  area: string;
  bedrooms: number;
  bathrooms: number;
  image: string;
  status: string;
  addedDate?: string;
  description?: string;
  amenities?: string[];
  nearbyLocations?: Array<{
    name: string;
    type: string;
    distance: string;
  }>;
}

interface PropertyListingsProps {
  onScheduleVisit?: (property: Property) => void;
}

const PropertyListings: React.FC<PropertyListingsProps> = ({ onScheduleVisit }) => {
  const { properties, loading, deleteProperty } = useProperty();
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('All');
  const [selectedProperty, setSelectedProperty] = useState<Property | null>(null);
  const [showAddProperty, setShowAddProperty] = useState(false);
  const [editingProperty, setEditingProperty] = useState<Property | null>(null);
  const [collaborationModal, setCollaborationModal] = useState<{ open: boolean; propertyId: string | null }>({ open: false, propertyId: null });

  const filteredProperties = properties.filter(property => {
    const matchesSearch = property.title?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         property.location?.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = statusFilter === 'All' || property.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  const handleCloseDetailView = () => {
    setSelectedProperty(null);
  };

  const handleScheduleVisit = (property: Property) => {
    if (onScheduleVisit) {
      onScheduleVisit(property);
    }
    setSelectedProperty(null);
  };

  const handleAddProperty = () => {
    setShowAddProperty(true);
  };

  const handleCloseAddProperty = () => {
    setShowAddProperty(false);
  };

  const handleEditProperty = (property: Property) => {
    setEditingProperty(property);
    setShowAddProperty(true);
  };

  const handleDeleteProperty = (propertyId: string) => {
    deleteProperty(propertyId);
    setSelectedProperty(null);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Available':
        return 'bg-green-500 text-white';
      case 'Pending':
        return 'bg-yellow-500 text-white';
      case 'Sold':
        return 'bg-gray-600 text-white';
      default:
        return 'bg-gray-500 text-white';
    }
  };

  if (showAddProperty) {
    return (
      <div className="space-y-6">
        <div className="bg-white rounded-xl shadow-sm p-6">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-2xl font-bold text-gray-900">Add New Property</h2>
              <p className="text-gray-600">Fill in the details to list your property</p>
            </div>
            <button
              onClick={handleCloseAddProperty}
              className="flex items-center space-x-2 px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors"
            >
              <X className="w-5 h-5" />
              <span>Back to Properties</span>
            </button>
          </div>
        </div>
        <ListingForm />
      </div>
    );
  }

  return (
    <div className="space-y-3">
      <div className="bg-white rounded-lg shadow-sm p-4">
        <div className="flex items-center justify-between">
          <div>
            <h2 className="text-xl font-bold text-gray-900">Property Listings</h2>
            <p className="text-sm text-gray-600">Manage and view all your properties</p>
          </div>
          <button
            onClick={handleAddProperty}
            style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
            className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm"
          >
            <Plus className="w-4 h-4" />
            <span>Add Property</span>
          </button>
        </div>
      </div>

      <div className="bg-white rounded-lg shadow-sm p-3">
        <div className="flex items-center gap-3">
          <div className="flex-1">
            <input
              type="text"
              placeholder="Search properties..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-1 focus:ring-blue-500 focus:border-blue-500 transition-colors text-sm bg-white"
            />
          </div>
          <div className="min-w-[140px]">
            <select
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-1 focus:ring-blue-500 focus:border-blue-500 transition-colors bg-white text-sm"
            >
              <option value="All">All Status</option>
              <option value="Available">Available</option>
              <option value="Pending">Pending</option>
              <option value="Sold">Sold</option>
            </select>
          </div>
        </div>
      </div>

      {loading ? (
        <div className="bg-white rounded-lg shadow-sm p-8 text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-3"></div>
          <p className="text-gray-600 text-sm">Loading agent properties...</p>
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
          {filteredProperties.map((property) => (
            <PropertyCard
              key={property.id}
              property={property}
              onViewDetails={() => setSelectedProperty(property)}
              onCollaborate={() => setCollaborationModal({ open: true, propertyId: property.id })}
              getStatusColor={getStatusColor}
            />
          ))}
        </div>
      )}

      {!loading && filteredProperties.length === 0 && (
        <div className="bg-white rounded-lg shadow-sm p-8 text-center">
          <Building className="w-12 h-12 text-gray-400 mx-auto mb-3" />
          <h3 className="text-base font-medium text-gray-900 mb-1">No properties found</h3>
          <p className="text-gray-600 mb-4 text-sm">
            {searchTerm || statusFilter !== 'All' 
              ? 'Try adjusting your search or filters'
              : 'Get started by adding your first property'
            }
          </p>
          {!searchTerm && statusFilter === 'All' && (
            <button
              onClick={handleAddProperty}
              className="flex items-center space-x-1 px-3 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors mx-auto text-sm"
            >
              <Plus className="w-4 h-4" />
              <span>Add Property</span>
            </button>
          )}
        </div>
      )}

      {selectedProperty && (
        <PropertyDetailView
          property={selectedProperty}
          onClose={handleCloseDetailView}
          onScheduleVisit={handleScheduleVisit}
          onEdit={handleEditProperty}
          onDelete={handleDeleteProperty}
        />
      )}
      
      <CollaborationModal
        open={collaborationModal.open}
        onClose={() => setCollaborationModal({ open: false, propertyId: null })}
        propertyId={collaborationModal.propertyId}
      />
    </div>
  );
};

export default PropertyListings;