import React, { useState, useEffect } from 'react';
import { Building, User, Phone, Calendar, Clock, CheckCircle, XCircle, Eye } from 'lucide-react';
import { getAgentVisits, agentApproveVisit, agentRejectVisit, updateAgentVisitStatus, AgentVisitSchedule } from '../../api/agentVisitSchedules';
import { getUserId } from '../../utils/userUtils';
import { toast } from 'sonner';

const PropertyVisitRequests: React.FC = () => {
  const [activeTab, setActiveTab] = useState('pending');
  const [visitRequests, setVisitRequests] = useState<AgentVisitSchedule[]>([]);
  const [loading, setLoading] = useState(true);
  const [processingAction, setProcessingAction] = useState<{visitId: number, action: 'approve' | 'reject' | 'complete'} | null>(null);
  
  useEffect(() => {
    fetchVisitRequests();
  }, [activeTab]);

  const fetchVisitRequests = async () => {
    try {
      setLoading(true);
      const agentId = getUserId();
      if (!agentId) {
        console.log('No agent ID found');
        return;
      }
      
      const statusFilter = activeTab === 'all' ? undefined : activeTab;
      const response = await getAgentVisits(agentId, 1, 50, statusFilter);
      setVisitRequests(response.visits);
    } catch (error) {
      console.error('Error fetching visit requests:', error);
      toast.error('Failed to fetch visit requests');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'approve'});
      const agentId = getUserId();
      if (!agentId) return;
      
      await agentApproveVisit(visitId, agentId);
      toast.success('Visit approved successfully');
      fetchVisitRequests();
    } catch (error) {
      console.error('Error approving visit:', error);
      toast.error('Failed to approve visit');
    } finally {
      setProcessingAction(null);
    }
  };

  const handleReject = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'reject'});
      const agentId = getUserId();
      if (!agentId) return;
      
      await agentRejectVisit(visitId, agentId);
      toast.success('Visit rejected successfully');
      fetchVisitRequests();
    } catch (error) {
      console.error('Error rejecting visit:', error);
      toast.error('Failed to reject visit');
    } finally {
      setProcessingAction(null);
    }
  };

  const handleComplete = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'complete'});
      
      await updateAgentVisitStatus(visitId, 'completed');
      toast.success('Visit marked as completed');
      fetchVisitRequests();
    } catch (error) {
      console.error('Error completing visit:', error);
      toast.error('Failed to complete visit');
    } finally {
      setProcessingAction(null);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'seller_approved':
        return 'bg-blue-100 text-blue-800';
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'seller_rejected':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending':
        return <Clock className="w-4 h-4" />;
      case 'seller_approved':
        return <Calendar className="w-4 h-4" />;
      case 'completed':
        return <CheckCircle className="w-4 h-4" />;
      case 'seller_rejected':
        return <XCircle className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  const filteredRequests = visitRequests.filter(request => {
    if (activeTab === 'all') return true;
    if (activeTab === 'confirmed') return request.status === 'seller_approved';
    if (activeTab === 'cancelled') return request.status === 'seller_rejected';
    return request.status === activeTab;
  });

  if (loading) {
    return (
      <div className="flex justify-center py-12">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Visit Requests</h2>
        <p className="text-gray-600 mb-6">Manage client visit requests and schedule appointments</p>

        {/* Tabs */}
        <div className="border-b border-gray-200 mb-6">
          <nav className="flex space-x-8">
            {[
              { id: 'all', label: 'All Requests', count: visitRequests.length },
              { id: 'pending', label: 'Pending', count: visitRequests.filter(r => r.status === 'pending').length },
              { id: 'confirmed', label: 'Confirmed', count: visitRequests.filter(r => r.status === 'seller_approved').length },
              { id: 'completed', label: 'Completed', count: visitRequests.filter(r => r.status === 'completed').length },
              { id: 'cancelled', label: 'Cancelled', count: visitRequests.filter(r => r.status === 'seller_rejected').length }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`py-2 px-1 border-b-2 font-medium text-sm ${
                  activeTab === tab.id
                    ? 'border-[#0056D2] text-[#0056D2]'
                    : 'border-transparent text-gray-500 hover:text-gray-700'
                }`}
              >
                {tab.label} ({tab.count})
              </button>
            ))}
          </nav>
        </div>

        {/* Visit Requests List */}
        <div className="space-y-4">
          {filteredRequests.map((request) => (
            <div key={request.id} className="bg-gray-50 border border-gray-200 rounded-lg p-4">
              <div className="flex items-start justify-between mb-4">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="font-semibold text-gray-900">{request.full_name}</h3>
                    <span className={`inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(request.status)}`}>
                      {getStatusIcon(request.status)}
                      {request.status === 'seller_approved' ? 'Confirmed' : 
                       request.status === 'seller_rejected' ? 'Cancelled' :
                       request.status.charAt(0).toUpperCase() + request.status.slice(1)}
                    </span>
                  </div>
                  
                  <div className="grid md:grid-cols-2 gap-4 mb-3">
                    <div>
                      <div className="flex items-center text-sm text-gray-600 mb-1">
                        <Phone className="w-4 h-4 mr-2" />
                        {request.mobile}
                      </div>
                      <div className="flex items-center text-sm text-gray-600">
                        <Building className="w-4 h-4 mr-2" />
                        {request.property_title}
                      </div>
                      <div className="text-sm text-gray-500 ml-6">{request.property_city}</div>
                    </div>
                    
                    <div>
                      <div className="flex items-center text-sm text-gray-600 mb-1">
                        <Calendar className="w-4 h-4 mr-2" />
                        {formatDate(request.preferred_date)} at {request.preferred_time || 'Flexible'}
                      </div>
                      {request.alternate_date && (
                        <div className="text-sm text-gray-500 ml-6">
                          Alternate: {formatDate(request.alternate_date)} at {request.alternate_time || 'Flexible'}
                        </div>
                      )}
                      <div className="text-sm text-gray-500 ml-6">
                        Visitors: {request.number_of_visitors} | {request.visit_mode}
                      </div>
                    </div>
                  </div>
                  
                  {request.comments && (
                    <div className="bg-white p-3 rounded border mb-3">
                      <p className="text-sm text-gray-700">{request.comments}</p>
                    </div>
                  )}
                </div>
                
                <div className="text-right text-sm text-gray-500">
                  <div>Requested on</div>
                  <div>{formatDate(request.created_at)}</div>
                </div>
              </div>
              
              <div className="flex gap-2">
                {request.status === 'pending' && (
                  <>
                    <button 
                      onClick={() => handleApprove(request.id)}
                      disabled={processingAction?.visitId === request.id && processingAction?.action === 'approve'}
                      className="flex items-center gap-1 px-3 py-1 bg-green-600 text-white rounded text-sm hover:bg-green-700 transition-colors disabled:bg-green-400"
                    >
                      <CheckCircle className="w-4 h-4" />
                      {processingAction?.visitId === request.id && processingAction?.action === 'approve' ? 'Confirming...' : 'Confirm'}
                    </button>
                    <button 
                      onClick={() => handleReject(request.id)}
                      disabled={processingAction?.visitId === request.id && processingAction?.action === 'reject'}
                      className="flex items-center gap-1 px-3 py-1 bg-red-600 text-white rounded text-sm hover:bg-red-700 transition-colors disabled:bg-red-400"
                    >
                      <XCircle className="w-4 h-4" />
                      {processingAction?.visitId === request.id && processingAction?.action === 'reject' ? 'Declining...' : 'Decline'}
                    </button>
                  </>
                )}
                {request.status === 'seller_approved' && (
                  <button 
                    onClick={() => handleComplete(request.id)}
                    disabled={processingAction?.visitId === request.id && processingAction?.action === 'complete'}
                    className="flex items-center gap-1 px-3 py-1 bg-[#0056D2] text-white rounded text-sm hover:bg-[#0041A3] transition-colors disabled:bg-blue-400"
                  >
                    <CheckCircle className="w-4 h-4" />
                    {processingAction?.visitId === request.id && processingAction?.action === 'complete' ? 'Completing...' : 'Mark Complete'}
                  </button>
                )}
                <button className="flex items-center gap-1 px-3 py-1 border border-gray-300 text-gray-700 rounded text-sm hover:bg-white transition-colors">
                  <Eye className="w-4 h-4" />
                  View Property
                </button>
                <button className="flex items-center gap-1 px-3 py-1 border border-gray-300 text-gray-700 rounded text-sm hover:bg-white transition-colors">
                  <Phone className="w-4 h-4" />
                  Call Client
                </button>
              </div>
            </div>
          ))}
        </div>

        {filteredRequests.length === 0 && (
          <div className="text-center py-12">
            <Building className="w-16 h-16 text-gray-300 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No visit requests found</h3>
            <p className="text-gray-500">Client visit requests will appear here</p>
          </div>
        )}
      </div>
    </div>
  );
};

export default PropertyVisitRequests;