import React, { useState, useMemo } from 'react';
import { Download, FileText, Calendar, Filter, RefreshCw, Share, MoreHorizontal, TrendingUp, Users, Building, DollarSign, Eye, Phone, MessageCircle, MapPin } from 'lucide-react';
import { useProperty } from './PropertyContext';

const Reports: React.FC = () => {
  const { properties } = useProperty();
  const [dateRange, setDateRange] = useState('30');
  const [selectedLocation, setSelectedLocation] = useState('All');
  const [selectedPropertyType, setSelectedPropertyType] = useState('All');
  const [reportType, setReportType] = useState('Sales Report');
  const [startDate, setStartDate] = useState('');
  const [endDate, setEndDate] = useState('');
  const [generating, setGenerating] = useState(false);
  const [filtersApplied, setFiltersApplied] = useState(false);

  // Mock data for detailed reports
  const mockData = {
    leads: [
      { id: 1, name: 'Rajesh Kumar', phone: '+91-9876543210', source: 'Website', status: 'Hot', property: '3BHK Apartment', date: '2024-01-15', location: 'Bandra West' },
      { id: 2, name: 'Priya Sharma', phone: '+91-9876543211', source: 'Social Media', status: 'Warm', property: '2BHK Villa', date: '2024-01-20', location: 'Andheri West' },
      { id: 3, name: 'Amit Patel', phone: '+91-9876543212', source: 'Referral', status: 'Cold', property: 'Commercial Plot', date: '2024-01-25', location: 'Whitefield' },
    ],
    sales: [
      { id: 1, property: '3BHK Luxury Apartment', buyer: 'Rajesh Kumar', price: 8500000, commission: 85000, date: '2024-01-15', location: 'Bandra West', type: 'Apartment' },
      { id: 2, property: 'Independent Villa', buyer: 'Priya Sharma', price: 12000000, commission: 120000, date: '2024-01-20', location: 'Andheri West', type: 'Villa' },
      { id: 3, property: 'Commercial Plot', buyer: 'Amit Patel', price: 25000000, commission: 250000, date: '2024-01-25', location: 'Whitefield', type: 'Plot' },
    ],
    visits: [
      { id: 1, property: '3BHK Apartment', client: 'Rajesh Kumar', date: '2024-01-10', time: '10:00 AM', status: 'Completed', feedback: 'Interested', location: 'Bandra West' },
      { id: 2, property: '2BHK Villa', client: 'Priya Sharma', date: '2024-01-15', time: '2:00 PM', status: 'Completed', feedback: 'Considering', location: 'Andheri West' },
      { id: 3, property: 'Commercial Plot', client: 'Amit Patel', date: '2024-01-20', time: '11:00 AM', status: 'Scheduled', feedback: 'Pending', location: 'Whitefield' },
    ],
    commissions: [
      { id: 1, property: '3BHK Apartment', amount: 85000, status: 'Paid', date: '2024-01-30', type: 'Sale', location: 'Bandra West' },
      { id: 2, property: '2BHK Villa', amount: 120000, status: 'Pending', date: '2024-02-05', type: 'Sale', location: 'Andheri West' },
      { id: 3, property: 'Commercial Plot', amount: 250000, status: 'Processing', date: '2024-02-10', type: 'Sale', location: 'Whitefield' },
    ],
    marketing: [
      { id: 1, campaign: 'Social Media Ads', spend: 15000, leads: 25, conversions: 3, roi: '200%', date: '2024-01-01' },
      { id: 2, campaign: 'Google Ads', spend: 20000, leads: 35, conversions: 5, roi: '250%', date: '2024-01-15' },
      { id: 3, campaign: 'Email Campaign', spend: 5000, leads: 15, conversions: 2, roi: '180%', date: '2024-01-20' },
    ]
  };

  // Filter data based on selected filters
  const filteredData = useMemo(() => {
    const filterByLocation = (item: any) => selectedLocation === 'All' || item.location === selectedLocation;
    const filterByType = (item: any) => selectedPropertyType === 'All' || item.type === selectedPropertyType;
    const filterByDate = (item: any) => {
      if (dateRange === 'custom' && startDate && endDate) {
        const itemDate = new Date(item.date);
        return itemDate >= new Date(startDate) && itemDate <= new Date(endDate);
      }
      const days = parseInt(dateRange);
      const cutoffDate = new Date();
      cutoffDate.setDate(cutoffDate.getDate() - days);
      return new Date(item.date) >= cutoffDate;
    };

    return {
      leads: mockData.leads.filter(item => filterByLocation(item) && filterByDate(item)),
      sales: mockData.sales.filter(item => filterByLocation(item) && filterByType(item) && filterByDate(item)),
      visits: mockData.visits.filter(item => filterByLocation(item) && filterByDate(item)),
      commissions: mockData.commissions.filter(item => filterByLocation(item) && filterByDate(item)),
      marketing: mockData.marketing.filter(item => filterByDate(item))
    };
  }, [dateRange, selectedLocation, selectedPropertyType, startDate, endDate]);

  // Calculate summary statistics
  const summaryStats = useMemo(() => {
    const totalSales = filteredData.sales.reduce((sum, sale) => sum + sale.price, 0);
    const totalCommission = filteredData.commissions.reduce((sum, comm) => sum + comm.amount, 0);
    const totalLeads = filteredData.leads.length;
    const hotLeads = filteredData.leads.filter(lead => lead.status === 'Hot').length;
    const completedVisits = filteredData.visits.filter(visit => visit.status === 'Completed').length;
    const conversionRate = totalLeads > 0 ? ((filteredData.sales.length / totalLeads) * 100).toFixed(1) : '0';

    return {
      totalProperties: properties.length,
      totalSales,
      totalCommission,
      totalLeads,
      hotLeads,
      completedVisits,
      conversionRate
    };
  }, [filteredData, properties]);

  const handleDownloadReport = async () => {
    setGenerating(true);
    
    // Simulate report generation delay
    await new Promise(resolve => setTimeout(resolve, 2000));
    
    const reportData = generateReportData();
    const blob = new Blob([reportData], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `${reportType.replace(/\s+/g, '_')}_${new Date().toISOString().split('T')[0]}.txt`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
    
    setGenerating(false);
  };

  const generateReportData = () => {
    const dateRangeText = dateRange === 'custom' && startDate && endDate 
      ? `${startDate} to ${endDate}` 
      : `Last ${dateRange} days`;

    let reportContent = `
${reportType.toUpperCase()}
Generated on: ${new Date().toLocaleDateString()}
Date Range: ${dateRangeText}
Location: ${selectedLocation}
Property Type: ${selectedPropertyType}

`;

    reportContent += `SUMMARY STATISTICS:
`;
    reportContent += `- Total Properties: ${summaryStats.totalProperties}\n`;
    reportContent += `- Total Sales Value: ₹${summaryStats.totalSales.toLocaleString()}\n`;
    reportContent += `- Total Commission: ₹${summaryStats.totalCommission.toLocaleString()}\n`;
    reportContent += `- Total Leads: ${summaryStats.totalLeads}\n`;
    reportContent += `- Hot Leads: ${summaryStats.hotLeads}\n`;
    reportContent += `- Completed Visits: ${summaryStats.completedVisits}\n`;
    reportContent += `- Conversion Rate: ${summaryStats.conversionRate}%\n\n`;

    if (reportType === 'Sales Report') {
      reportContent += `SALES DETAILS:\n`;
      filteredData.sales.forEach((sale, index) => {
        reportContent += `${index + 1}. ${sale.property} - ${sale.buyer} - ₹${sale.price.toLocaleString()} - ${sale.date}\n`;
      });
    } else if (reportType === 'Leads Report') {
      reportContent += `LEADS DETAILS:\n`;
      filteredData.leads.forEach((lead, index) => {
        reportContent += `${index + 1}. ${lead.name} - ${lead.phone} - ${lead.status} - ${lead.source} - ${lead.date}\n`;
      });
    } else if (reportType === 'Commission Report') {
      reportContent += `COMMISSION DETAILS:\n`;
      filteredData.commissions.forEach((comm, index) => {
        reportContent += `${index + 1}. ${comm.property} - ₹${comm.amount.toLocaleString()} - ${comm.status} - ${comm.date}\n`;
      });
    } else if (reportType === 'Visit Report') {
      reportContent += `VISIT DETAILS:\n`;
      filteredData.visits.forEach((visit, index) => {
        reportContent += `${index + 1}. ${visit.property} - ${visit.client} - ${visit.date} ${visit.time} - ${visit.status}\n`;
      });
    } else if (reportType === 'Marketing Report') {
      reportContent += `MARKETING DETAILS:\n`;
      filteredData.marketing.forEach((campaign, index) => {
        reportContent += `${index + 1}. ${campaign.campaign} - Spend: ₹${campaign.spend.toLocaleString()} - Leads: ${campaign.leads} - ROI: ${campaign.roi}\n`;
      });
    }

    reportContent += `\n\nGenerated by NAL Agent Dashboard`;
    return reportContent;
  };

  const handleApplyFilters = () => {
    setFiltersApplied(true);
  };

  const clearFilters = () => {
    setDateRange('30');
    setSelectedLocation('All');
    setSelectedPropertyType('All');
    setReportType('Sales Report');
    setStartDate('');
    setEndDate('');
    setFiltersApplied(false);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Reports</h2>
          <p className="text-gray-600 mt-1">Generate and download detailed reports</p>
        </div>
      </div>

      {/* Report Filters */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Generate Report</h3>
        <div className="flex flex-wrap items-end gap-4">
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Report Type</label>
            <select 
              value={reportType}
              onChange={(e) => setReportType(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            >
              <option>Sales Report</option>
              <option>Leads Report</option>
              <option>Commission Report</option>
              <option>Visit Report</option>
              <option>Marketing Report</option>
            </select>
          </div>
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Date Range</label>
            <select 
              value={dateRange} 
              onChange={(e) => setDateRange(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            >
              <option value="7">Last 7 days</option>
              <option value="30">Last 30 days</option>
              <option value="90">Last 3 months</option>
              <option value="365">Last year</option>
              <option value="custom">Custom Range</option>
            </select>
          </div>
          {dateRange === 'custom' && (
            <>
              <div className="flex-1 min-w-[120px]">
                <label className="block text-sm font-medium text-gray-700 mb-2">Start Date</label>
                <input
                  type="date"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
                />
              </div>
              <div className="flex-1 min-w-[120px]">
                <label className="block text-sm font-medium text-gray-700 mb-2">End Date</label>
                <input
                  type="date"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
                />
              </div>
            </>
          )}
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Location</label>
            <select 
              value={selectedLocation} 
              onChange={(e) => setSelectedLocation(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            >
              <option value="All">All Locations</option>
              <option value="Bandra West">Bandra West</option>
              <option value="Andheri West">Andheri West</option>
              <option value="Whitefield">Whitefield</option>
              <option value="Electronic City">Electronic City</option>
              <option value="Koramangala">Koramangala</option>
            </select>
          </div>
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
            <select 
              value={selectedPropertyType} 
              onChange={(e) => setSelectedPropertyType(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            >
              <option value="All">All Types</option>
              <option value="Apartment">Apartment</option>
              <option value="Villa">Villa</option>
              <option value="Plot">Plot</option>
              <option value="Commercial">Commercial</option>
            </select>
          </div>
        </div>
        <div className="mt-6 flex flex-wrap gap-3">
          <button 
            onClick={handleDownloadReport}
            disabled={generating}
            className="flex items-center space-x-2 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-medium shadow-sm"
          >
            {generating ? (
              <>
                <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                <span>Generating...</span>
              </>
            ) : (
              <>
                <Download className="w-4 h-4" />
                <span>Download Report</span>
              </>
            )}
          </button>
          <button 
            onClick={filtersApplied ? clearFilters : handleApplyFilters}
            className="flex items-center space-x-2 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors font-medium shadow-sm"
          >
            <Filter className="w-4 h-4" />
            <span>{filtersApplied ? 'Clear Filters' : 'Apply Filters'}</span>
          </button>
        </div>
      </div>

      {/* Report Summary */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Report Summary</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <div className="bg-gradient-to-r from-blue-50 to-blue-100 p-4 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-600 font-medium">Total Properties</p>
                <p className="text-2xl font-bold text-blue-900">{summaryStats.totalProperties}</p>
              </div>
              <Building className="w-8 h-8 text-blue-600" />
            </div>
          </div>
          <div className="bg-gradient-to-r from-green-50 to-green-100 p-4 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-600 font-medium">Total Sales</p>
                <p className="text-2xl font-bold text-green-900">₹{summaryStats.totalSales.toLocaleString()}</p>
              </div>
              <TrendingUp className="w-8 h-8 text-green-600" />
            </div>
          </div>
          <div className="bg-gradient-to-r from-purple-50 to-purple-100 p-4 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-600 font-medium">Total Commission</p>
                <p className="text-2xl font-bold text-purple-900">₹{summaryStats.totalCommission.toLocaleString()}</p>
              </div>
              <DollarSign className="w-8 h-8 text-purple-600" />
            </div>
          </div>
          <div className="bg-gradient-to-r from-orange-50 to-orange-100 p-4 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-600 font-medium">Conversion Rate</p>
                <p className="text-2xl font-bold text-orange-900">{summaryStats.conversionRate}%</p>
              </div>
              <Users className="w-8 h-8 text-orange-600" />
            </div>
          </div>
        </div>
      </div>

      {/* Detailed Report Data */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">{reportType} Details</h3>
        
        {reportType === 'Leads Report' && (
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Name</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Phone</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Source</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Property Interest</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Date</th>
                </tr>
              </thead>
              <tbody>
                {filteredData.leads.map((lead) => (
                  <tr key={lead.id} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4 font-medium text-gray-900">{lead.name}</td>
                    <td className="py-3 px-4 text-gray-600">{lead.phone}</td>
                    <td className="py-3 px-4 text-gray-600">{lead.source}</td>
                    <td className="py-3 px-4">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        lead.status === 'Hot' ? 'bg-red-100 text-red-800' :
                        lead.status === 'Warm' ? 'bg-yellow-100 text-yellow-800' :
                        'bg-gray-100 text-gray-800'
                      }`}>
                        {lead.status}
                      </span>
                    </td>
                    <td className="py-3 px-4 text-gray-600">{lead.property}</td>
                    <td className="py-3 px-4 text-gray-600">{lead.date}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {reportType === 'Commission Report' && (
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Property</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Amount</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Type</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Date</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Location</th>
                </tr>
              </thead>
              <tbody>
                {filteredData.commissions.map((comm) => (
                  <tr key={comm.id} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4 font-medium text-gray-900">{comm.property}</td>
                    <td className="py-3 px-4 text-green-600 font-medium">₹{comm.amount.toLocaleString()}</td>
                    <td className="py-3 px-4">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        comm.status === 'Paid' ? 'bg-green-100 text-green-800' :
                        comm.status === 'Pending' ? 'bg-yellow-100 text-yellow-800' :
                        'bg-blue-100 text-blue-800'
                      }`}>
                        {comm.status}
                      </span>
                    </td>
                    <td className="py-3 px-4 text-gray-600">{comm.type}</td>
                    <td className="py-3 px-4 text-gray-600">{comm.date}</td>
                    <td className="py-3 px-4 text-gray-600">{comm.location}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {reportType === 'Visit Report' && (
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Property</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Client</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Date & Time</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Feedback</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Location</th>
                </tr>
              </thead>
              <tbody>
                {filteredData.visits.map((visit) => (
                  <tr key={visit.id} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4 font-medium text-gray-900">{visit.property}</td>
                    <td className="py-3 px-4 text-gray-600">{visit.client}</td>
                    <td className="py-3 px-4 text-gray-600">{visit.date} {visit.time}</td>
                    <td className="py-3 px-4">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        visit.status === 'Completed' ? 'bg-green-100 text-green-800' :
                        visit.status === 'Scheduled' ? 'bg-blue-100 text-blue-800' :
                        'bg-red-100 text-red-800'
                      }`}>
                        {visit.status}
                      </span>
                    </td>
                    <td className="py-3 px-4 text-gray-600">{visit.feedback}</td>
                    <td className="py-3 px-4 text-gray-600">{visit.location}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {reportType === 'Marketing Report' && (
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Campaign</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Spend</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Leads</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Conversions</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">ROI</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Date</th>
                </tr>
              </thead>
              <tbody>
                {filteredData.marketing.map((campaign) => (
                  <tr key={campaign.id} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4 font-medium text-gray-900">{campaign.campaign}</td>
                    <td className="py-3 px-4 text-red-600 font-medium">₹{campaign.spend.toLocaleString()}</td>
                    <td className="py-3 px-4 text-blue-600 font-medium">{campaign.leads}</td>
                    <td className="py-3 px-4 text-green-600 font-medium">{campaign.conversions}</td>
                    <td className="py-3 px-4 text-purple-600 font-medium">{campaign.roi}</td>
                    <td className="py-3 px-4 text-gray-600">{campaign.date}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Additional Insights */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Lead Performance</h3>
          <div className="space-y-3">
            <div className="flex justify-between items-center">
              <span className="text-gray-600">Total Leads</span>
              <span className="font-medium text-gray-900">{summaryStats.totalLeads}</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-gray-600">Hot Leads</span>
              <span className="font-medium text-red-600">{summaryStats.hotLeads}</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-gray-600">Completed Visits</span>
              <span className="font-medium text-green-600">{summaryStats.completedVisits}</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-gray-600">Conversion Rate</span>
              <span className="font-medium text-blue-600">{summaryStats.conversionRate}%</span>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Recent Reports</h3>
          <div className="space-y-3">
            {[
              { name: 'Monthly Sales Report', date: '2024-06-01', type: 'Sales' },
              { name: 'Q2 Commission Summary', date: '2024-04-01', type: 'Commission' },
              { name: 'Lead Source Analysis', date: '2024-05-15', type: 'Leads' },
            ].map((report, index) => (
              <div key={index} className="flex items-center justify-between p-3 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors">
                <div className="flex items-center space-x-3">
                  <FileText className="w-5 h-5 text-blue-600" />
                  <div>
                    <p className="text-sm font-medium text-gray-900">{report.name}</p>
                    <p className="text-xs text-gray-600">{report.date} • {report.type}</p>
                  </div>
                </div>
                <button className="p-1 text-gray-400 hover:text-gray-600 transition-colors">
                  <Download className="w-4 h-4" />
                </button>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default Reports;