import React, { useState, useEffect } from 'react';
import { Calendar, Clock, User, Phone, MapPin, Plus, CheckCircle, XCircle, Eye, Video } from 'lucide-react';
import { getAgentVisits, updateAgentVisitStatus, agentApproveVisit, agentRejectVisit, sellerApproveVisit, AgentVisitSchedule } from '../../api/agentVisitSchedules';
import { getUserId } from '../../utils/userUtils';
import { toast } from 'sonner';

const ScheduleVisits: React.FC = () => {
  const [visits, setVisits] = useState<AgentVisitSchedule[]>([]);
  const [loading, setLoading] = useState(true);
  const [processingAction, setProcessingAction] = useState<{visitId: number, action: 'complete' | 'approve' | 'reject'} | null>(null);

  useEffect(() => {
    fetchVisits();
  }, []);

  const fetchVisits = async () => {
    try {
      setLoading(true);
      const agentId = getUserId();
      if (!agentId) {
        console.log('No agent ID found');
        return;
      }
      
      const response = await getAgentVisits(agentId, 1, 50);
      setVisits(response.visits);
    } catch (error) {
      console.error('Error fetching visits:', error);
      toast.error('Failed to fetch visits');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'approve'});
      const agentId = getUserId();
      if (!agentId) {
        toast.error('Agent ID not found. Please log in again.');
        return;
      }
      
      console.log('🔍 Attempting to approve visit:', { visitId, agentId });
      
      // Find the visit to check its status
      const visit = visits.find(v => v.id === visitId);
      if (!visit) {
        toast.error('Visit not found');
        return;
      }
      
      // Use different endpoints based on visit status
      if (visit.status === 'pending') {
        await agentApproveVisit(visitId, agentId);
      } else if (visit.status === 'admin_approved') {
        await sellerApproveVisit(visitId, agentId);
      } else {
        toast.error(`Cannot approve visit with status: ${visit.status}`);
        return;
      }
      
      toast.success('Visit approved successfully');
      fetchVisits();
    } catch (error) {
      console.error('Error approving visit:', error);
      const errorMessage = error instanceof Error ? error.message : 'Failed to approve visit';
      toast.error(errorMessage);
    } finally {
      setProcessingAction(null);
    }
  };

  const handleReject = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'reject'});
      const agentId = getUserId();
      if (!agentId) return;
      
      await agentRejectVisit(visitId, agentId);
      toast.success('Visit rejected successfully');
      fetchVisits();
    } catch (error) {
      console.error('Error rejecting visit:', error);
      toast.error('Failed to reject visit');
    } finally {
      setProcessingAction(null);
    }
  };

  const handleComplete = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'complete'});
      
      await updateAgentVisitStatus(visitId, 'completed');
      toast.success('Visit marked as completed');
      fetchVisits();
    } catch (error) {
      console.error('Error completing visit:', error);
      toast.error('Failed to complete visit');
    } finally {
      setProcessingAction(null);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'seller_approved':
        return 'bg-[#2B256D]/10 text-[#2B256D]';
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'seller_rejected':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status.toLowerCase()) {
      case 'pending':
        return <Clock className="w-4 h-4" />;
      case 'seller_approved':
        return <CheckCircle className="w-4 h-4" />;
      case 'completed':
        return <CheckCircle className="w-4 h-4" />;
      case 'seller_rejected':
        return <XCircle className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case 'seller_approved': return 'Scheduled';
      case 'seller_rejected': return 'Cancelled';
      default: return status.charAt(0).toUpperCase() + status.slice(1);
    }
  };

  const isVideoCallTime = (visit: AgentVisitSchedule) => {
    if (!visit.preferred_date || !visit.preferred_time) return false;
    
    const now = new Date();
    const visitDate = new Date(visit.preferred_date);
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const scheduleDate = new Date(visitDate.getFullYear(), visitDate.getMonth(), visitDate.getDate());
    
    // Only show on the scheduled date
    if (today.getTime() !== scheduleDate.getTime()) return false;
    
    // Parse time range (e.g., "2–4 PM")
    const timeRange = visit.preferred_time;
    const currentHour = now.getHours();
    
    if (timeRange.includes('9–11 AM')) return currentHour >= 9 && currentHour < 11;
    if (timeRange.includes('11–1 PM')) return currentHour >= 11 && currentHour < 13;
    if (timeRange.includes('2–4 PM')) return currentHour >= 14 && currentHour < 16;
    if (timeRange.includes('4–6 PM')) return currentHour >= 16 && currentHour < 18;
    
    return true; // Default to show if time format is different
  };

  if (loading) {
    return (
      <div className="flex justify-center py-12">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-6">
          <div>
            <h2 className="text-2xl font-bold text-gray-900">Schedule Visits</h2>
            <p className="text-gray-600 mt-1">Manage your property visit schedules</p>
          </div>
        </div>

        {/* Visits List */}
        <div className="space-y-4">
          {visits.map((visit) => (
            <div key={visit.id} className="bg-white border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow">
              <div className="flex flex-col gap-4">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="font-semibold text-gray-900">{visit.property_title}</h3>
                    <span className={`inline-flex items-center gap-1 px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(visit.status)}`}>
                      {getStatusIcon(visit.status)}
                      {getStatusLabel(visit.status)}
                    </span>
                  </div>
                  
                  <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm text-gray-600">
                    <div className="flex items-center">
                      <User className="w-4 h-4 mr-2" />
                      <span>{visit.full_name}</span>
                    </div>
                    
                    <div className="flex items-center">
                      <Phone className="w-4 h-4 mr-2" />
                      <span>{visit.mobile}</span>
                    </div>
                    
                    <div className="flex items-center">
                      <Calendar className="w-4 h-4 mr-2" />
                      <span>{formatDate(visit.preferred_date)}</span>
                    </div>
                    
                    <div className="flex items-center">
                      <Clock className="w-4 h-4 mr-2" />
                      <span>{visit.preferred_time || 'Flexible'}</span>
                    </div>
                  </div>
                  
                  <div className="mt-2 flex flex-wrap gap-2">
                    <span className="inline-block px-2 py-1 text-xs font-medium rounded-full bg-blue-100 text-blue-800">
                      {visit.buyer_type}
                    </span>
                    <span className="inline-block px-2 py-1 text-xs font-medium rounded-full bg-purple-100 text-purple-800">
                      {visit.visit_mode}
                    </span>
                    {visit.number_of_visitors > 1 && (
                      <span className="inline-block px-2 py-1 text-xs font-medium rounded-full bg-gray-100 text-gray-800">
                        {visit.number_of_visitors} visitors
                      </span>
                    )}
                    {visit.video_call_enabled && (
                      <span className="inline-flex items-center px-2 py-1 text-xs font-medium rounded-full bg-blue-100 text-blue-800">
                        <Video className="w-3 h-3 mr-1" />
                        Video Call
                      </span>
                    )}
                  </div>
                </div>
                
                <div className="flex flex-wrap items-center gap-3 pt-4 border-t border-gray-200">
                  <button 
                    style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
                    className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm"
                  >
                    <Eye className="w-4 h-4" />
                    <span>View Details</span>
                  </button>
                  
                  {(visit.status === 'pending' || visit.status === 'admin_approved') && (
                    <>
                      <button 
                        onClick={() => handleApprove(visit.id)}
                        disabled={processingAction?.visitId === visit.id && processingAction?.action === 'approve'}
                        style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
                        className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm disabled:bg-blue-400"
                      >
                        <CheckCircle className="w-4 h-4" />
                        <span>Accept</span>
                      </button>
                      <button 
                        onClick={() => handleReject(visit.id)}
                        disabled={processingAction?.visitId === visit.id && processingAction?.action === 'reject'}
                        style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
                        className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm disabled:bg-blue-400"
                      >
                        <XCircle className="w-4 h-4" />
                        <span>Reject</span>
                      </button>
                    </>
                  )}
                  
                  {visit.status === 'seller_approved' && (
                    <>
                      <button 
                        onClick={() => handleComplete(visit.id)}
                        disabled={processingAction?.visitId === visit.id}
                        style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
                        className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm disabled:bg-blue-400"
                      >
                        <CheckCircle className="w-4 h-4" />
                        <span>Complete</span>
                      </button>
                      {(visit.visit_mode === 'Virtual Tour' || visit.video_call_enabled) && visit.status === 'seller_approved' && isVideoCallTime(visit) && (
                        <button 
                          onClick={() => {
                            const roomId = visit.video_call_room_id || `NAL_Property_${visit.property_id}_Visit_${visit.id}`;
                            window.open(`https://meet.jit.si/${roomId}`, '_blank');
                          }}
                          style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
                          className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm"
                        >
                          <Video className="w-4 h-4" />
                          <span>Join Call</span>
                        </button>
                      )}
                    </>
                  )}
                </div>
              </div>
            </div>
          ))}
        </div>
        
        {visits.length === 0 && (
          <div className="text-center py-12">
            <Calendar className="w-16 h-16 text-gray-300 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No visits scheduled</h3>
            <p className="text-gray-500 mb-4">Visit requests from clients will appear here once they book visits for your properties.</p>
          </div>
        )}
      </div>
    </div>
  );
};

export default ScheduleVisits;