import React, { useState, useEffect } from 'react';
import { Users, Clock, CheckCircle, XCircle, Send } from 'lucide-react';
import { collaborationApi, Collaboration } from '../../api/collaborations';

const SentCollaborations: React.FC = () => {
  const [sentInvitations, setSentInvitations] = useState<Collaboration[]>([]);
  const [loading, setLoading] = useState(true);

  const fetchSentInvitations = async () => {
    try {
      const userEmail = localStorage.getItem('agentEmail') || localStorage.getItem('sellerEmail');
      
      if (!userEmail) {
        setSentInvitations([]);
        setLoading(false);
        return;
      }
      
      const response = await collaborationApi.getMyInvitations();
      const allInvitations = response.collaborations || [];
      
      // Filter for invitations sent by current user
      const mySentInvitations = allInvitations.filter(
        (invitation: any) => invitation.primary_agent_name === userEmail
      );
      
      setSentInvitations(mySentInvitations);
    } catch (error) {
      console.error('Error fetching sent invitations:', error);
      setSentInvitations([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchSentInvitations();
  }, []);

  if (loading) {
    return (
      <div className="bg-white rounded-lg shadow-sm p-6">
        <div className="animate-pulse">
          <div className="h-4 bg-gray-200 rounded w-1/4 mb-4"></div>
          <div className="space-y-3">
            <div className="h-20 bg-gray-200 rounded"></div>
          </div>
        </div>
      </div>
    );
  }

  const userEmail = localStorage.getItem('agentEmail') || localStorage.getItem('sellerEmail');
  if (!userEmail) {
    return (
      <div className="bg-white rounded-lg shadow-sm p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-gray-900">Sent Invitations</h3>
        </div>
        <div className="text-center py-8">
          <Send className="w-12 h-12 text-gray-400 mx-auto mb-3" />
          <p className="text-gray-500 mb-2">Please log in to view sent invitations</p>
        </div>
      </div>
    );
  }

  if (sentInvitations.length === 0) {
    return (
      <div className="bg-white rounded-lg shadow-sm p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-gray-900">Sent Invitations</h3>
        </div>
        <div className="text-center py-8">
          <Send className="w-12 h-12 text-gray-400 mx-auto mb-3" />
          <p className="text-gray-500">No collaboration invitations sent</p>
        </div>
      </div>
    );
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'active':
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case 'declined':
        return <XCircle className="w-4 h-4 text-red-600" />;
      default:
        return <Clock className="w-4 h-4 text-yellow-600" />;
    }
  };

  const getStatusText = (status: string, secondaryAgentEmail: string) => {
    switch (status) {
      case 'active':
        return `Accepted by ${secondaryAgentEmail}`;
      case 'declined':
        return `Declined by ${secondaryAgentEmail}`;
      default:
        return `Sent to ${secondaryAgentEmail} - Pending`;
    }
  };

  return (
    <div className="bg-white rounded-lg shadow-sm p-6">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-semibold text-gray-900">Sent Invitations</h3>
        <span className="bg-blue-500 text-white text-xs px-2 py-1 rounded-full">
          {sentInvitations.length}
        </span>
      </div>
      
      <div className="space-y-4">
        {sentInvitations.map((invitation) => (
          <div
            key={invitation.id}
            className="border border-blue-200 bg-blue-50 rounded-lg p-4"
          >
            <div className="flex items-start justify-between">
              <div className="flex-1">
                <div className="flex items-center mb-2">
                  <Send className="w-4 h-4 text-blue-600 mr-2" />
                  <p className="text-sm font-medium text-gray-900">
                    Collaboration invitation sent
                  </p>
                </div>
                
                <p className="text-sm text-gray-700 mb-2">
                  <strong>Property:</strong> {invitation.property_title}
                </p>
                
                <div className="flex items-center space-x-4 text-xs text-gray-600 mb-3">
                  <div className="flex items-center">
                    {getStatusIcon(invitation.status)}
                    <span className="ml-1">{getStatusText(invitation.status, invitation.secondary_agent_email)}</span>
                  </div>
                  <div className="flex items-center">
                    <Clock className="w-3 h-3 mr-1" />
                    <span>{new Date(invitation.created_at).toLocaleDateString()}</span>
                  </div>
                </div>

                {invitation.invitation_message && (
                  <div className="text-xs text-gray-600 mb-3">
                    <strong>Message:</strong> {invitation.invitation_message}
                  </div>
                )}
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
};

export default SentCollaborations;