import React, { useState, useRef, useEffect } from 'react';
import { 
  User, 
  Shield, 
  Bell, 
  Palette, 
  Link, 
  CreditCard, 
  Lock, 
  HelpCircle,
  Camera,
  Edit,
  Save,
  X,
  Check,
  Eye,
  EyeOff,
  Download,
  Upload,
  Globe,
  Smartphone,
  Mail,
  MessageCircle,
  Calendar,
  Cloud,
  Key,
  Settings as SettingsIcon,
  ChevronRight,
  AlertTriangle,
  Info
} from 'lucide-react';
import { getAgentProfile, updateAgentProfile, type AgentProfile } from '../../api/agentProfile';

interface SettingsProps {}

const Settings: React.FC<SettingsProps> = () => {
  const [activeTab, setActiveTab] = useState('profile');
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [isEditing, setIsEditing] = useState(true);
  const [darkMode, setDarkMode] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);
  
  const [profile, setProfile] = useState({
    name: '',
    email: '',
    phone: '',
    licenseId: '',
    reraId: '',
    companyName: '',
    businessTagline: '',
    profilePhoto: null as string | null,
  });
  
  const [agentProfile, setAgentProfile] = useState<AgentProfile | null>(null);

  const [loading, setLoading] = useState(false);
  const [saveMessage, setSaveMessage] = useState('');
  const [initialLoading, setInitialLoading] = useState(true);
  
  useEffect(() => {
    loadAgentProfile();
  }, []);
  
  const loadAgentProfile = async () => {
    try {
      const profileData = await getAgentProfile();
      console.log('Loaded agent profile data:', profileData);
      console.log('Office address from API:', profileData.office_address);
      console.log('All profile fields:', Object.keys(profileData));
      console.log('Full profile object:', JSON.stringify(profileData, null, 2));
      setAgentProfile(profileData);
      setProfile({
        name: profileData.full_name || '',
        email: profileData.email || '',
        phone: profileData.phone_number || '',
        licenseId: profileData.license_number || '',
        reraId: profileData.business_registration_number || '',
        companyName: profileData.agency_name || '',
        businessTagline: profileData.bio || '',
        profilePhoto: profileData.profile_image_url || null,
      });
      
      // Update agentProfile state to trigger re-render of professional fields
      setAgentProfile(profileData);
    } catch (error) {
      console.error('Error loading agent profile:', error);
    } finally {
      setInitialLoading(false);
    }
  };
  
  // Reload profile data when component becomes visible
  useEffect(() => {
    const handleVisibilityChange = () => {
      if (!document.hidden && agentProfile) {
        loadAgentProfile();
      }
    };
    
    document.addEventListener('visibilitychange', handleVisibilityChange);
    return () => document.removeEventListener('visibilitychange', handleVisibilityChange);
  }, [agentProfile]);

  const updateProfile = (data: Partial<typeof profile>) => {
    setProfile(prev => ({ ...prev, ...data }));
  };

  const updateProfilePhoto = (photo: string | null) => {
    setProfile(prev => ({ ...prev, profilePhoto: photo }));
  };
  
  const updateAgentField = (field: string, value: any) => {
    setAgentProfile(prev => prev ? { ...prev, [field]: value } : null);
  };

  const handleSave = async () => {
    setLoading(true);
    setSaveMessage('');
    
    try {
      const updateData = {
        full_name: profile.name,
        email: profile.email,
        phone_number: profile.phone,
        license_number: profile.licenseId,
        business_registration_number: profile.reraId,
        agency_name: profile.companyName,
        bio: profile.businessTagline,
        profile_image_url: profile.profilePhoto,
        office_address: agentProfile?.office_address,
        experience_years: agentProfile?.experience_years,
        commission_rate: agentProfile?.commission_rate,
        preferred_contact_method: agentProfile?.preferred_contact_method,
        education: agentProfile?.education,
        website_url: agentProfile?.website_url,
        languages_spoken: agentProfile?.languages_spoken,
        certifications: agentProfile?.certifications,
        availability_hours: agentProfile?.availability_hours,
      };
      
      const updatedProfile = await updateAgentProfile(updateData);
      setAgentProfile(updatedProfile);
      
      setIsEditing(false);
      setSaveMessage('Profile updated successfully!');
      setTimeout(() => setSaveMessage(''), 3000);
    } catch (error) {
      setSaveMessage('Failed to save profile. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const resizeAndCropImage = (file: File): Promise<string> => {
    return new Promise((resolve) => {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d')!;
      const img = new Image();
      
      img.onload = () => {
        const size = 400; // Professional size
        canvas.width = size;
        canvas.height = size;
        
        // Calculate crop dimensions for square aspect ratio
        const minDim = Math.min(img.width, img.height);
        const cropX = (img.width - minDim) / 2;
        const cropY = (img.height - minDim) / 2;
        
        // Draw cropped and resized image
        ctx.drawImage(img, cropX, cropY, minDim, minDim, 0, 0, size, size);
        
        // Convert to high-quality JPEG
        resolve(canvas.toDataURL('image/jpeg', 0.9));
      };
      
      img.src = URL.createObjectURL(file);
    });
  };

  const handlePhotoUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      try {
        const resizedImage = await resizeAndCropImage(file);
        updateProfilePhoto(resizedImage);
      } catch (error) {
        console.error('Error processing image:', error);
      }
    }
  };

  const getInitials = (name: string) => {
    return name.split(' ').map(n => n[0]).join('').toUpperCase();
  };

  const tabs = [
    { id: 'profile', label: 'Profile & Personal Info', icon: User },
    { id: 'security', label: 'Account & Security', icon: Shield },
    { id: 'notifications', label: 'Notification Preferences', icon: Bell },
    { id: 'dashboard', label: 'Dashboard Customization', icon: Palette },
    { id: 'payment', label: 'Payment & Billing', icon: CreditCard },
  ];

  const renderProfileTab = () => {
    if (initialLoading) {
      return (
        <div className="p-6 flex items-center justify-center">
          <div className="text-gray-500">Loading profile...</div>
        </div>
      );
    }
    
    return (
    <div className="p-6">
      {/* Header */}
      <div className="mb-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Profile Settings</h2>
        <p className="text-gray-600">Manage your personal information and account details</p>
        {saveMessage && (
          <div className={`mt-3 p-3 rounded-lg text-sm font-medium ${
            saveMessage.includes('successfully') 
              ? 'bg-green-50 text-green-700 border border-green-200' 
              : 'bg-red-50 text-red-700 border border-red-200'
          }`}>
            {saveMessage}
          </div>
        )}
      </div>

      {/* Profile Photo */}
      <div className="bg-gray-50 rounded-lg p-6 mb-6">
        <div className="flex items-center space-x-4">
          <div className="relative">
            {profile.profilePhoto ? (
              <img src={profile.profilePhoto} alt="Profile" className="w-20 h-20 rounded-full object-cover" />
            ) : (
              <div className="w-20 h-20 rounded-full bg-gray-300 flex items-center justify-center">
                <User className="w-8 h-8 text-gray-600" />
              </div>
            )}
          </div>
          <div className="flex-1">
            <h3 className="font-semibold text-gray-900 mb-1">Profile Photo</h3>
            <p className="text-sm text-gray-600 mb-3">Upload a professional photo</p>
            <div className="flex space-x-2">
              <button 
                onClick={() => fileInputRef.current?.click()}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm"
              >
                Upload Photo
              </button>
              {profile.profilePhoto && (
                <button 
                  onClick={() => updateProfilePhoto(null)}
                  className="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors text-sm"
                >
                  Remove
                </button>
              )}
            </div>
          </div>
        </div>
        <input
          ref={fileInputRef}
          type="file"
          accept="image/*"
          onChange={handlePhotoUpload}
          className="hidden"
        />
      </div>

      {/* Form Fields */}
      <div className="space-y-6">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Full Name *</label>
            <input
              type="text"
              value={profile.name}
              onChange={(e) => updateProfile({ name: e.target.value })}
              placeholder="Enter your full name"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Email Address *</label>
            <input
              type="email"
              value={profile.email}
              onChange={(e) => updateProfile({ email: e.target.value })}
              placeholder="Enter your email"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Phone Number *</label>
            <input
              type="tel"
              value={profile.phone}
              onChange={(e) => updateProfile({ phone: e.target.value })}
              placeholder="+91 98765 43210"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Agent License ID</label>
            <input
              type="text"
              value={profile.licenseId}
              onChange={(e) => updateProfile({ licenseId: e.target.value })}
              placeholder="Enter license number"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">RERA ID</label>
            <input
              type="text"
              value={profile.reraId}
              onChange={(e) => updateProfile({ reraId: e.target.value })}
              placeholder="Enter RERA ID"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Company Name</label>
            <input
              type="text"
              value={profile.companyName}
              onChange={(e) => updateProfile({ companyName: e.target.value })}
              placeholder="Enter company name"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Business Tagline</label>
          <input
            type="text"
            value={profile.businessTagline}
            onChange={(e) => updateProfile({ businessTagline: e.target.value })}
            placeholder="Enter your business tagline"
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          />
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Office Address</label>
            <textarea
              value={agentProfile?.office_address || ''}
              onChange={(e) => updateAgentField('office_address', e.target.value)}
              placeholder="Enter your office address"
              rows={2}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Experience Years</label>
            <select
              value={agentProfile?.experience_years || 0}
              onChange={(e) => updateAgentField('experience_years', parseInt(e.target.value))}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value={0}>0-1 years</option>
              <option value={2}>2-5 years</option>
              <option value={6}>6-10 years</option>
              <option value={11}>11-20 years</option>
              <option value={21}>20+ years</option>
            </select>
          </div>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Commission Rate (%)</label>
            <input
              type="number"
              step="0.1"
              min="0"
              max="10"
              value={agentProfile?.commission_rate || ''}
              onChange={(e) => updateAgentField('commission_rate', e.target.value ? parseFloat(e.target.value) : 0)}
              placeholder="2.5"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Preferred Contact Method</label>
            <select
              value={agentProfile?.preferred_contact_method || ''}
              onChange={(e) => updateAgentField('preferred_contact_method', e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="">Select method</option>
              <option value="Phone">Phone</option>
              <option value="Email">Email</option>
              <option value="WhatsApp">WhatsApp</option>
              <option value="Video Call">Video Call</option>
            </select>
          </div>
        </div>
        
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Education Background</label>
          <input
            type="text"
            value={agentProfile?.education || ''}
            onChange={(e) => updateAgentField('education', e.target.value)}
            placeholder="e.g., MBA in Real Estate, Bachelor's in Business"
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          />
        </div>
        
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Website URL</label>
          <input
            type="url"
            value={agentProfile?.website_url || ''}
            onChange={(e) => updateAgentField('website_url', e.target.value)}
            placeholder="https://your-website.com"
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          />
        </div>
        
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Availability Hours</label>
          <input
            type="text"
            value={agentProfile?.availability_hours || ''}
            onChange={(e) => updateAgentField('availability_hours', e.target.value)}
            placeholder="Mon-Fri 9AM-6PM, Sat 10AM-4PM"
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          />
        </div>
        
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Languages Spoken</label>
          <div className="text-sm text-gray-600 p-2 bg-gray-50 rounded-lg">
            {agentProfile?.languages_spoken && agentProfile.languages_spoken.length > 0 
              ? agentProfile.languages_spoken.join(', ') 
              : 'No languages specified'}
          </div>
        </div>
        
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Specialization</label>
          <div className="text-sm text-gray-600 p-2 bg-gray-50 rounded-lg">
            {agentProfile?.specialization && agentProfile.specialization.length > 0 
              ? agentProfile.specialization.join(', ') 
              : 'No specialization specified'}
          </div>
        </div>
      </div>

      {/* Save Button */}
      <div className="flex justify-end mt-8 pt-6 border-t border-gray-200">
        <button
          onClick={handleSave}
          disabled={loading}
          className="flex items-center space-x-2 px-6 py-3 bg-blue-600 text-black rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          <Save className="w-4 h-4" />
          <span>{loading ? 'Saving...' : 'Save Changes'}</span>
        </button>
      </div>
    </div>
    );
  };

  const renderSecurityTab = () => (
    <div className="p-6">
      <div className="bg-gray-50 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Change Password</h3>
        <div className="space-y-4 max-w-md">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Current Password</label>
            <div className="relative">
              <input
                type={showPassword ? "text" : "password"}
                placeholder="Enter current password"
                className="w-full px-3 py-2 pr-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              />
              <button
                onClick={() => setShowPassword(!showPassword)}
                className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
              >
                {showPassword ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
              </button>
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">New Password</label>
            <div className="relative">
              <input
                type={showConfirmPassword ? "text" : "password"}
                placeholder="Enter new password"
                className="w-full px-3 py-2 pr-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              />
              <button
                onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
              >
                {showConfirmPassword ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
              </button>
            </div>
          </div>

          <button className="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
            Update Password
          </button>
        </div>
      </div>
    </div>
  );

  const renderNotificationsTab = () => (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Notification Channels</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <Mail className="w-5 h-5 text-blue-600" />
              <div>
                <p className="font-medium text-gray-900">Email Notifications</p>
                <p className="text-sm text-gray-600">Receive notifications via email</p>
              </div>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <Smartphone className="w-5 h-5 text-green-600" />
              <div>
                <p className="font-medium text-gray-900">SMS Notifications</p>
                <p className="text-sm text-gray-600">Receive notifications via SMS</p>
              </div>
            </div>
            <input type="checkbox" className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <MessageCircle className="w-5 h-5 text-green-500" />
              <div>
                <p className="font-medium text-gray-900">WhatsApp Notifications</p>
                <p className="text-sm text-gray-600">Receive notifications via WhatsApp</p>
              </div>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Notification Types</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">New Leads</p>
              <p className="text-sm text-gray-600">When someone shows interest in your properties</p>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Visit Reminders</p>
              <p className="text-sm text-gray-600">Reminders for scheduled property visits</p>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Payment Updates</p>
              <p className="text-sm text-gray-600">Commission payments and financial updates</p>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
        </div>
      </div>
    </div>
  );

  const renderDashboardTab = () => (
    <div className="p-6 space-y-6">
      <div className="bg-gray-50 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Appearance</h3>
        <div className="flex items-center justify-between">
          <div>
            <p className="font-medium text-gray-900">Dark Mode</p>
            <p className="text-sm text-gray-600">Switch between light and dark themes</p>
          </div>
          <button
            onClick={() => setDarkMode(!darkMode)}
            className={`px-4 py-2 rounded-lg transition-colors ${
              darkMode 
                ? 'bg-blue-600 text-white' 
                : 'bg-gray-200 text-gray-700 hover:bg-gray-300'
            }`}
          >
            {darkMode ? 'Enabled' : 'Disabled'}
          </button>
        </div>
      </div>

      <div className="bg-gray-50 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Dashboard Widgets</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Quick Stats</p>
              <p className="text-sm text-gray-600">Show property statistics on dashboard</p>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Top Properties</p>
              <p className="text-sm text-gray-600">Display your best performing properties</p>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Recent Activities</p>
              <p className="text-sm text-gray-600">Show recent property activities</p>
            </div>
            <input type="checkbox" className="rounded" />
          </div>
        </div>
      </div>
    </div>
  );

  const renderIntegrationsTab = () => (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Calendar Integration</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center space-x-3">
              <Calendar className="w-5 h-5 text-blue-600" />
              <div>
                <p className="font-medium text-gray-900">Google Calendar</p>
                <p className="text-sm text-gray-600">Sync your property visits with Google Calendar</p>
              </div>
            </div>
            <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
              Connect
            </button>
          </div>
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center space-x-3">
              <Calendar className="w-5 h-5 text-blue-700" />
              <div>
                <p className="font-medium text-gray-900">Outlook Calendar</p>
                <p className="text-sm text-gray-600">Sync your property visits with Outlook</p>
              </div>
            </div>
            <button className="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
              Connect
            </button>
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Communication Platforms</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center space-x-3">
              <MessageCircle className="w-5 h-5 text-green-500" />
              <div>
                <p className="font-medium text-gray-900">WhatsApp Business</p>
                <p className="text-sm text-gray-600">Connect your WhatsApp Business account</p>
              </div>
            </div>
            <button className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors">
              Connect
            </button>
          </div>
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center space-x-3">
              <Mail className="w-5 h-5 text-blue-600" />
              <div>
                <p className="font-medium text-gray-900">Gmail</p>
                <p className="text-sm text-gray-600">Connect your Gmail account</p>
              </div>
            </div>
            <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
              Connect
            </button>
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Cloud Storage</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center space-x-3">
              <Cloud className="w-5 h-5 text-blue-500" />
              <div>
                <p className="font-medium text-gray-900">Google Drive</p>
                <p className="text-sm text-gray-600">Backup documents to Google Drive</p>
              </div>
            </div>
            <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
              Connect
            </button>
          </div>
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center space-x-3">
              <Cloud className="w-5 h-5 text-blue-600" />
              <div>
                <p className="font-medium text-gray-900">Dropbox</p>
                <p className="text-sm text-gray-600">Backup documents to Dropbox</p>
              </div>
            </div>
            <button className="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
              Connect
            </button>
          </div>
        </div>
      </div>
    </div>
  );

  const renderPaymentTab = () => (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Bank Details</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Account Holder Name</label>
            <input
              type="text"
              defaultValue="Suresh Kumar"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Account Number</label>
            <input
              type="text"
              defaultValue="1234567890"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">IFSC Code</label>
            <input
              type="text"
              defaultValue="SBIN0001234"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Bank Name</label>
            <input
              type="text"
              defaultValue="State Bank of India"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            />
          </div>
        </div>
        <button className="mt-6 px-4 py-2 bg-[#2B256D] text-white rounded-lg hover:bg-[#4A3F8C] transition-colors">
          Update Bank Details
        </button>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Subscription Plan</h3>
        <div className="border border-gray-200 rounded-lg p-6">
          <div className="flex items-center justify-between mb-4">
            <div>
              <h4 className="text-lg font-semibold text-gray-900">Professional Plan</h4>
              <p className="text-gray-600">₹999/month</p>
            </div>
            <span className="px-3 py-1 bg-green-100 text-green-800 rounded-full text-sm font-medium">
              Active
            </span>
          </div>
          <ul className="space-y-2 text-sm text-gray-600">
            <li>• Unlimited property listings</li>
            <li>• Advanced analytics</li>
            <li>• Priority support</li>
            <li>• Custom branding</li>
          </ul>
          <div className="mt-6 flex space-x-3">
            <button className="px-4 py-2 bg-[#2B256D] text-white rounded-lg hover:bg-[#4A3F8C] transition-colors">
              Upgrade Plan
            </button>
            <button className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
              Cancel Subscription
            </button>
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Billing History</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div>
              <p className="font-medium text-gray-900">Professional Plan - March 2024</p>
              <p className="text-sm text-gray-600">March 1, 2024</p>
            </div>
            <div className="flex items-center space-x-3">
              <span className="font-medium text-gray-900">₹999</span>
              <button className="flex items-center space-x-1 text-[#2B256D] hover:text-[#4A3F8C]">
                <Download className="w-4 h-4" />
                <span>Download</span>
              </button>
            </div>
          </div>
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div>
              <p className="font-medium text-gray-900">Professional Plan - February 2024</p>
              <p className="text-sm text-gray-600">February 1, 2024</p>
            </div>
            <div className="flex items-center space-x-3">
              <span className="font-medium text-gray-900">₹999</span>
              <button className="flex items-center space-x-1 text-[#2B256D] hover:text-[#4A3F8C]">
                <Download className="w-4 h-4" />
                <span>Download</span>
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );

  const renderPrivacyTab = () => (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Data Sharing</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Share with Clients</p>
              <p className="text-sm text-gray-600">Allow clients to view your contact information</p>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Share with Admins</p>
              <p className="text-sm text-gray-600">Allow administrators to view your data</p>
            </div>
            <input type="checkbox" defaultChecked className="rounded" />
          </div>
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">Analytics Sharing</p>
              <p className="text-sm text-gray-600">Share anonymous usage data for improvements</p>
            </div>
            <input type="checkbox" className="rounded" />
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">API Access</h3>
        <div className="space-y-4">
          <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
            <div>
              <p className="font-medium text-gray-900">API Key</p>
              <p className="text-sm text-gray-600">sk_live_1234567890abcdef</p>
            </div>
            <button className="text-[#2B256D] hover:text-[#4A3F8C]">Regenerate</button>
          </div>
          <div className="flex items-center justify-between">
            <div>
              <p className="font-medium text-gray-900">API Access</p>
              <p className="text-sm text-gray-600">Enable API access for third-party integrations</p>
            </div>
            <input type="checkbox" className="rounded" />
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Data Export</h3>
        <div className="space-y-4">
          <button className="w-full flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors">
            <div>
              <p className="font-medium text-gray-900">Export My Data</p>
              <p className="text-sm text-gray-600">Download all your data in JSON format</p>
            </div>
            <Download className="w-5 h-5 text-[#2B256D]" />
          </button>
          <button className="w-full flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors">
            <div>
              <p className="font-medium text-gray-900">Delete My Account</p>
              <p className="text-sm text-gray-600">Permanently delete your account and all data</p>
            </div>
            <AlertTriangle className="w-5 h-5 text-red-600" />
          </button>
        </div>
      </div>
    </div>
  );

  const renderSupportTab = () => (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Contact Support</h3>
        <div className="space-y-4">
          <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg">
            <Mail className="w-5 h-5 text-[#2B256D]" />
            <div>
              <p className="font-medium text-gray-900">Email Support</p>
              <p className="text-sm text-gray-600">support@agentdashboard.com</p>
            </div>
          </div>
          <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg">
            <Smartphone className="w-5 h-5 text-[#2B256D]" />
            <div>
              <p className="font-medium text-gray-900">Phone Support</p>
              <p className="text-sm text-gray-600">+91 1800-123-4567 (9 AM - 6 PM IST)</p>
            </div>
          </div>
          <div className="flex items-center space-x-3 p-4 border border-gray-200 rounded-lg">
            <MessageCircle className="w-5 h-5 text-[#2B256D]" />
            <div>
              <p className="font-medium text-gray-900">Live Chat</p>
              <p className="text-sm text-gray-600">Available 24/7 for urgent issues</p>
            </div>
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Frequently Asked Questions</h3>
        <div className="space-y-4">
          <div className="border border-gray-200 rounded-lg">
            <button className="w-full flex items-center justify-between p-4 text-left">
              <span className="font-medium text-gray-900">How do I add a new property?</span>
              <ChevronRight className="w-5 h-5 text-gray-400" />
            </button>
          </div>
          <div className="border border-gray-200 rounded-lg">
            <button className="w-full flex items-center justify-between p-4 text-left">
              <span className="font-medium text-gray-900">How do I schedule a site visit?</span>
              <ChevronRight className="w-5 h-5 text-gray-400" />
            </button>
          </div>
          <div className="border border-gray-200 rounded-lg">
            <button className="w-full flex items-center justify-between p-4 text-left">
              <span className="font-medium text-gray-900">How do I track my commissions?</span>
              <ChevronRight className="w-5 h-5 text-gray-400" />
            </button>
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Report a Problem</h3>
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Subject</label>
            <input
              type="text"
              placeholder="Brief description of the issue"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Description</label>
            <textarea
              rows={4}
              placeholder="Detailed description of the problem..."
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Priority</label>
            <select className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]">
              <option>Low</option>
              <option>Medium</option>
              <option>High</option>
              <option>Critical</option>
            </select>
          </div>
          <button className="w-full px-4 py-2 bg-[#2B256D] text-white rounded-lg hover:bg-[#4A3F8C] transition-colors">
            Submit Report
          </button>
        </div>
      </div>
    </div>
  );

  const renderTabContent = () => {
    switch (activeTab) {
      case 'profile':
        return renderProfileTab();
      case 'security':
        return renderSecurityTab();
      case 'notifications':
        return renderNotificationsTab();
      case 'dashboard':
        return renderDashboardTab();
      case 'payment':
        return renderPaymentTab();
      default:
        return renderProfileTab();
    }
  };

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm">
        {/* Header */}
        <div className="border-b border-gray-200 p-6">
          <h1 className="text-2xl font-bold text-gray-900">Profile Settings</h1>
          <p className="text-gray-600 mt-1">Manage your account information and preferences</p>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            {tabs.map((tab) => {
              const Icon = tab.icon;
              return (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === tab.id
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700'
                  }`}
                >
                  <Icon className="w-4 h-4" />
                  <span>{tab.label}</span>
                </button>
              );
            })}
          </nav>
        </div>

        {/* Content */}
        {renderTabContent()}
      </div>
    </div>
  );
};

export default Settings;
