import React, { useState } from "react";
import logoImage from "../../assets/logo.jpg";
import { useNavigate, useLocation } from "react-router-dom";
import {
  Home,
  Building,
  Users,
  Calendar,
  BarChart3,
  Settings,
  TrendingUp,
  LogOut,
  ChevronLeft,
  ChevronRight,
  FileText,
  Heart,
  Bookmark,
  History,
  MessageSquare,
  GitCompare,
  User,
  LifeBuoy,
  DollarSign,
  Calculator,
  Globe,
  Megaphone,
  Monitor,
  Package,
  Wallet,
} from "lucide-react";

interface SidebarProps {
  collapsed?: boolean;
  activeView?: string;
  onViewChange?: (view: string) => void;
  onToggle?: (collapsed: boolean) => void;
}

const Sidebar: React.FC<SidebarProps> = ({
  collapsed = false,
  activeView = "dashboard",
  onViewChange,
  onToggle,
}) => {
  const [isCollapsed, setIsCollapsed] = useState(collapsed);

  React.useEffect(() => {
    setIsCollapsed(collapsed);
  }, [collapsed]);

  const handleToggle = () => {
    const newState = !isCollapsed;
    setIsCollapsed(newState);
    if (onToggle) onToggle(newState);
  };
  const navigate = useNavigate();
  const location = useLocation();
  const menuItems = [
    { id: "dashboard", label: "Dashboard Overview", icon: Home },
    { id: "properties", label: "My Properties", icon: Building },
    { id: "campaigns", label: "Campaign Management", icon: Megaphone },
    { id: "listing-packages", label: "Listing Packages", icon: Package },
    { id: "saved", label: "Saved Properties", icon: Bookmark },
    { id: "price-prediction", label: "Price Prediction", icon: Calculator },
    { id: "dynamic-pricing", label: "Dynamic Pricing", icon: TrendingUp },
    { id: "negotiations", label: "Negotiations", icon: FileText },
    { id: "inquiries", label: "Leads/Enquiries", icon: MessageSquare },
    { id: "visits", label: "Scheduled Visits", icon: Calendar },
    { id: "reports", label: "Reports", icon: FileText },
    { id: "property-analytics", label: "Property Analytics", icon: BarChart3 },
    { id: "market-analysis", label: "Market Analysis", icon: TrendingUp },
    { id: "analytics", label: "Analytics & Market Insights", icon: BarChart3 },
    { id: "nri-assistance", label: "NRI Assistance", icon: Globe },
    { id: "system-health", label: "System Health & Monitoring", icon: Monitor },
    { id: "earnings", label: "Commission Center", icon: () => <span className="text-lg font-bold mr-4">₹</span> },
    { id: "wallet", label: "Wallet", icon: Wallet },
    { id: "profile", label: "Profile Settings", icon: User },
    { id: "support", label: "Support Center", icon: LifeBuoy },
  ];

  const [logoError, setLogoError] = useState(false);
  const [logoIdx, setLogoIdx] = useState(0);
  const logoCandidates = ["/nal.jpg", "/Public/nal.jpg", "/nal-logo.png"];
  const currentLogo = logoCandidates[logoIdx];

  return (
    <div
      className={`bg-white shadow-sm border-r fixed left-0 top-0 h-full min-h-screen z-50 transition-all duration-300 flex flex-col ${
        isCollapsed ? "w-20" : "w-64"
      }`}
    >
      <div className="p-4">
        <div
          className="flex items-center gap-3 cursor-pointer"
          onClick={() => navigate("/")}
        >
          {!isCollapsed ? (
            <>
              <img
                src={logoImage}
                alt="NAL Logo"
                className="w-12 h-12 rounded-lg object-cover"
              />
              <div>
                <h1 className="text-lg font-bold text-gray-900">
                  New Age Land
                </h1>
                <p className="text-xs text-gray-600">Agent Panel</p>
              </div>
            </>
          ) : (
            <img
              src={logoImage}
              alt="NAL Logo"
              className="w-12 h-12 rounded-lg object-cover mx-auto"
            />
          )}
        </div>
      </div>

      <nav
        className="px-2 flex-1 overflow-y-auto"
        style={{ scrollbarWidth: "none", msOverflowStyle: "none" }}
      >
        <div className="space-y-1">
          {menuItems.map((item, index) => {
            const Icon = item.icon;
            const isActive =
              location.pathname ===
                `/agent/dashboard/${item.id === "dashboard" ? "" : item.id}` ||
              (item.id === "dashboard" &&
                location.pathname === "/agent/dashboard");
            const isComparisonItem = item.id === "analytics";

            return (
              <div
                key={item.id}
                className={isComparisonItem ? "flex items-center" : ""}
              >
                <button
                  onClick={() => {
                    navigate(
                      `/agent/dashboard/${
                        item.id === "dashboard" ? "" : item.id
                      }`
                    );
                    if (onViewChange) onViewChange(item.id);
                  }}
                  className={`${
                    isComparisonItem ? "flex-1" : "w-full"
                  } flex items-center px-3 py-4 text-sm font-medium rounded-lg transition-colors group relative ${
                    isActive
                      ? "bg-[#0056D2] text-white shadow-sm"
                      : "text-gray-700 hover:bg-gray-100 hover:text-[#0056D2]"
                  }`}
                >
                  <Icon
                    className={`w-5 h-5 ${isCollapsed ? "mx-auto" : "mr-3"}`}
                  />
                  {!isCollapsed && (
                    <span className="whitespace-nowrap">{item.label}</span>
                  )}
                  {isActive && !isCollapsed && (
                    <div className="absolute right-0 top-0 bottom-0 w-1 bg-[#00BFA6] rounded-l"></div>
                  )}
                </button>
                {isComparisonItem && (
                  <button
                    onClick={handleToggle}
                    className="ml-2 w-8 h-8 bg-[#0056D2] hover:bg-[#0041A3] rounded-full flex items-center justify-center transition-colors shadow-sm"
                  >
                    {isCollapsed ? (
                      <ChevronRight className="w-4 h-4 text-white" />
                    ) : (
                      <ChevronLeft className="w-4 h-4 text-white" />
                    )}
                  </button>
                )}
              </div>
            );
          })}
        </div>
      </nav>
    </div>
  );
};

export default Sidebar;
