import React, { useState } from "react";
import { LifeBuoy, MessageSquare, Phone, Mail, FileText, Search, Send, Clock, CheckCircle } from "lucide-react";

const SupportCenter: React.FC = () => {
  const [activeTab, setActiveTab] = useState("faq");
  const [searchQuery, setSearchQuery] = useState("");
  const [newTicket, setNewTicket] = useState({
    subject: "",
    category: "",
    message: ""
  });

  const faqs = [
    {
      id: 1,
      question: "How do I add a new property listing?",
      answer: "You can add a new property by clicking on the 'Add Property' button in your dashboard. Fill in all the required details, upload images, and submit for approval."
    },
    {
      id: 2,
      question: "How can I track my commission earnings?",
      answer: "Visit the 'Commissions' section in your dashboard to view detailed commission reports, pending payments, and transaction history."
    },
    {
      id: 3,
      question: "How do I manage property visits?",
      answer: "Go to 'Site Visits' section to view scheduled visits, confirm appointments, and manage your visit calendar with clients."
    },
    {
      id: 4,
      question: "What documents are required for property listing?",
      answer: "You need property ownership documents, NOC certificates, approved building plans, and property tax receipts. Our team will guide you through the verification process."
    }
  ];

  const [supportTickets, setSupportTickets] = useState<any[]>([]);
  
  const loadUserTickets = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/agent-tickets/', {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });
      
      if (response.ok) {
        const tickets = await response.json();
        const formattedTickets = tickets.map(ticket => ({
          id: `T${ticket.id.toString().padStart(3, '0')}`,
          subject: ticket.subject,
          status: ticket.status === 'open' ? 'open' : 'resolved',
          createdDate: ticket.created_at.split('T')[0],
          lastUpdate: ticket.updated_at.split('T')[0]
        }));
        setSupportTickets(formattedTickets);
      } else {
        console.error('Failed to load tickets:', response.status);
        setSupportTickets([]);
      }
    } catch (error) {
      console.error('Error loading tickets:', error);
      setSupportTickets([]);
    }
  };
  
  React.useEffect(() => {
    loadUserTickets();
  }, []);

  const handleTicketSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      const token = localStorage.getItem('access_token');
      
      const response = await fetch('http://localhost:8090/api/create-agent-support-ticket/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
          subject: newTicket.subject,
          category: newTicket.category,
          message: newTicket.message
        })
      });
      
      if (response.ok) {
        alert('Ticket submitted successfully!');
        setNewTicket({ subject: "", category: "", message: "" });
        setActiveTab("tickets");
        loadUserTickets();
      } else {
        const errorText = await response.text();
        alert(`Failed to submit ticket: ${response.status} - ${errorText}`);
      }
    } catch (error: any) {
      console.error('Error submitting ticket:', error);
      alert(`Network error: ${error.message}. Please ensure Django server is running.`);
    }
  };

  const getUserName = () => {
    try {
      const token = localStorage.getItem('access_token');
      if (token) {
        const payload = JSON.parse(atob(token.split('.')[1]));
        return payload.name || payload.first_name || 'Agent';
      }
    } catch {}
    return 'Agent';
  };

  return (
    <div className="space-y-6">
      <div>
        <h2 className="text-2xl font-bold text-gray-900">Welcome back, {getUserName()}!</h2>
        <p className="text-gray-600">Get help and support for your property search journey</p>
      </div>

      {/* Support Center Header */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-xl font-semibold text-gray-900 mb-2">Support Center</h3>
        <p className="text-gray-600 mb-6">Get help and support for your property search journey</p>

        {/* Contact Options */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 text-center">
            <Phone className="w-8 h-8 text-[#0056D2] mx-auto mb-3" />
            <h3 className="font-semibold text-gray-900 mb-2">Call Us</h3>
            <p className="text-sm text-gray-600 mb-3">Speak with our experts</p>
            <button className="px-4 py-2 bg-[#0056D2] hover:bg-[#0041A3] text-white rounded-lg text-sm font-medium transition-colors">
              +91-8068447416
            </button>
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 text-center">
            <Mail className="w-8 h-8 text-[#0056D2] mx-auto mb-3" />
            <h3 className="font-semibold text-gray-900 mb-2">Email Support</h3>
            <p className="text-sm text-gray-600 mb-3">Get help via email</p>
            <button className="px-4 py-2 bg-[#0056D2] hover:bg-[#0041A3] text-white rounded-lg text-sm font-medium transition-colors">
              info@newagelandindia.com
            </button>
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            <button
              onClick={() => setActiveTab("faq")}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === "faq"
                  ? "border-[#0056D2] text-[#0056D2]"
                  : "border-transparent text-gray-500 hover:text-gray-700"
              }`}
            >
              FAQ
            </button>
            <button
              onClick={() => setActiveTab("tickets")}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === "tickets"
                  ? "border-[#0056D2] text-[#0056D2]"
                  : "border-transparent text-gray-500 hover:text-gray-700"
              }`}
            >
              Support Tickets
            </button>
            <button
              onClick={() => setActiveTab("new-ticket")}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === "new-ticket"
                  ? "border-[#0056D2] text-[#0056D2]"
                  : "border-transparent text-gray-500 hover:text-gray-700"
              }`}
            >
              Create Ticket
            </button>
          </nav>
        </div>

        <div className="p-6">
          {/* FAQ Tab */}
          {activeTab === "faq" && (
            <div className="space-y-4">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  placeholder="Search frequently asked questions..."
                  className="w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                />
              </div>

              <div className="space-y-4">
                {faqs
                  .filter(faq => 
                    searchQuery === "" || 
                    faq.question.toLowerCase().includes(searchQuery.toLowerCase().split(' ')[0])
                  )
                  .map((faq) => (
                    <div key={faq.id} className="border border-gray-200 rounded-lg p-4">
                      <h3 className="font-semibold text-gray-900 mb-2">{faq.question}</h3>
                      <p className="text-gray-700 text-sm">{faq.answer}</p>
                    </div>
                  ))
                }
              </div>
            </div>
          )}

          {/* Support Tickets Tab */}
          {activeTab === "tickets" && (
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">Your Support Tickets</h3>
                <button 
                  onClick={() => setActiveTab("new-ticket")}
                  className="px-4 py-2 bg-[#0056D2] hover:bg-[#0041A3] text-white rounded-lg text-sm font-medium transition-colors"
                >
                  Create New Ticket
                </button>
              </div>

              <div className="space-y-3">
                {supportTickets.length > 0 ? supportTickets.map((ticket) => (
                  <div key={ticket.id} className="border border-gray-200 rounded-lg p-4">
                    <div className="flex items-center justify-between mb-2">
                      <div className="flex items-center gap-3">
                        <span className="font-medium text-gray-900">#{ticket.id}</span>
                        <span className={`inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs font-medium ${
                          ticket.status === 'open' 
                            ? 'bg-yellow-100 text-yellow-800' 
                            : 'bg-green-100 text-green-800'
                        }`}>
                          {ticket.status === 'open' ? <Clock className="w-3 h-3" /> : <CheckCircle className="w-3 h-3" />}
                          {ticket.status.charAt(0).toUpperCase() + ticket.status.slice(1)}
                        </span>
                      </div>
                      <button className="px-3 py-1 border border-gray-300 text-gray-700 rounded text-sm hover:bg-gray-50 transition-colors">
                        View Details
                      </button>
                    </div>
                    <h4 className="font-medium text-gray-900 mb-1">{ticket.subject}</h4>
                    <div className="text-sm text-gray-600">
                      <span>Created: {ticket.createdDate}</span>
                      <span className="mx-2">•</span>
                      <span>Last Update: {ticket.lastUpdate}</span>
                    </div>
                  </div>
                )) : (
                  <p className="text-gray-500 text-center py-8">No support tickets found</p>
                )}
              </div>
            </div>
          )}

          {/* Create Ticket Tab */}
          {activeTab === "new-ticket" && (
            <form onSubmit={handleTicketSubmit} className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Subject</label>
                <input
                  type="text"
                  value={newTicket.subject}
                  onChange={(e) => setNewTicket(prev => ({ ...prev, subject: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  placeholder="Brief description of your issue"
                  required
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Category</label>
                <select
                  value={newTicket.category}
                  onChange={(e) => setNewTicket(prev => ({ ...prev, category: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  required
                >
                  <option value="">Select a category</option>
                  <option value="property-listing">Property Listing</option>
                  <option value="commission-inquiry">Commission Inquiry</option>
                  <option value="technical-issue">Technical Issue</option>
                  <option value="client-management">Client Management</option>
                  <option value="documentation">Documentation</option>
                  <option value="other">Other</option>
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Message</label>
                <textarea
                  value={newTicket.message}
                  onChange={(e) => setNewTicket(prev => ({ ...prev, message: e.target.value }))}
                  rows={6}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  placeholder="Describe your issue in detail..."
                  required
                />
              </div>

              <div className="flex justify-end">
                <button 
                  type="submit" 
                  className="flex items-center gap-2 px-4 py-2 bg-[#0056D2] hover:bg-[#0041A3] text-white rounded-lg font-medium transition-colors"
                >
                  <Send className="w-4 h-4" />
                  Submit Ticket
                </button>
              </div>
            </form>
          )}
        </div>
      </div>
    </div>
  );
}

export default SupportCenter;