import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';
import { Badge } from '../ui/badge';
import { Plus, Upload, Download } from 'lucide-react';
import { unitInventoryAPI } from '../../api/unitInventory';
import { UnitAvailabilityVisualization } from '../unit-availability';

interface Project {
  id: string;
  name: string;
  location: string;
  status: string;
}

const UnitInventoryPage: React.FC = () => {
  const [selectedProject, setSelectedProject] = useState<string>('');
  const [projects, setProjects] = useState<Project[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadProjects();
  }, []);

  const loadProjects = async () => {
    try {
      setLoading(true);
      console.log('🔄 Loading projects from API...');
      
      // Try to get real projects from API
      const realProjects = await unitInventoryAPI.getProjects();
      
      setProjects(realProjects);
      if (realProjects.length > 0) {
        setSelectedProject(realProjects[0].id);
      }
    } catch (error) {
      console.error('Error loading projects:', error);
    } finally {
      setLoading(false);
    }
  };

  const selectedProjectData = projects.find(p => p.id === selectedProject);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p>Loading projects...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Unit Inventory Management</h1>
          <p className="text-gray-600">
            Visualize and manage unit availability across your projects
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Button variant="outline" size="sm">
            <Upload className="w-4 h-4 mr-2" />
            Bulk Upload
          </Button>
          <Button variant="outline" size="sm">
            <Download className="w-4 h-4 mr-2" />
            Export
          </Button>
          <Button size="sm">
            <Plus className="w-4 h-4 mr-2" />
            Add Unit
          </Button>
        </div>
      </div>

      {/* Project Selection */}
      <Card>
        <CardHeader>
          <CardTitle>Select Project</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center gap-4">
            <div className="flex-1">
              <Select value={selectedProject} onValueChange={setSelectedProject}>
                <SelectTrigger>
                  <SelectValue placeholder="Select a project" />
                </SelectTrigger>
                <SelectContent>
                  {projects.map(project => (
                    <SelectItem key={project.id} value={project.id}>
                      <div className="flex items-center justify-between w-full">
                        <div>
                          <div className="font-medium">{project.name}</div>
                          <div className="text-sm text-gray-500">{project.location}</div>
                        </div>
                        <Badge variant="outline" className="ml-2">
                          {project.status}
                        </Badge>
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <Button variant="outline" onClick={loadProjects}>
              Refresh
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Unit Availability Visualization */}
      {selectedProject && selectedProjectData && (
        <UnitAvailabilityVisualization
          projectId={selectedProject}
          projectName={selectedProjectData.name}
        />
      )}

      {/* Help Section */}
      <Card>
        <CardHeader>
          <CardTitle>How to Use Unit Availability Visualization</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <h4 className="font-semibold mb-2">Color Coding</h4>
              <ul className="text-sm space-y-1 text-gray-600">
                <li>• <span className="text-green-600 font-medium">Green</span> - Available units</li>
                <li>• <span className="text-blue-600 font-medium">Blue</span> - Booked units</li>
                <li>• <span className="text-red-600 font-medium">Red</span> - Sold units</li>
                <li>• <span className="text-yellow-600 font-medium">Yellow</span> - On hold units</li>
                <li>• <span className="text-gray-600 font-medium">Gray</span> - Blocked/Reserved units</li>
              </ul>
            </div>
            <div>
              <h4 className="font-semibold mb-2">Features</h4>
              <ul className="text-sm space-y-1 text-gray-600">
                <li>• Click on any unit to view detailed information</li>
                <li>• Use filters to narrow down your view</li>
                <li>• Switch between floor-wise, tower-wise, and grid views</li>
                <li>• Update unit status directly from the interface</li>
                <li>• Real-time synchronization with booking system</li>
              </ul>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default UnitInventoryPage;