import React, { useState } from 'react';
import { CheckCircle, ArrowRight, ArrowLeft, AlertCircle } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Card } from '../ui/card';
import { toast } from 'sonner';
import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8090';

export const AgentKYCForm: React.FC = () => {
  const [currentStep, setCurrentStep] = useState(1);
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    full_name: '',
    pan_number: '',
    aadhaar_number: '',
    date_of_birth: '',
    phone_number: '',
    email: '',
    address: '',
    city: '',
    state: '',
    pincode: '',
    agency_name: '',
    rera_id: '',
    years_of_experience: '',
    pan_document_url: '',
    aadhaar_document_url: '',
    photo_url: '',
    agency_certificate_url: '',
    rera_certificate_url: ''
  });

  const getToken = () => localStorage.getItem('token');

  const validateStep1 = () => {
    const { full_name, pan_number, date_of_birth, phone_number, email } = formData;
    if (!full_name || full_name.length < 2) {
      toast.error('Full name is required');
      return false;
    }
    if (!/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/.test(pan_number.toUpperCase())) {
      toast.error('Invalid PAN format (e.g., ABCDE1234F)');
      return false;
    }
    if (!date_of_birth) {
      toast.error('Date of birth is required');
      return false;
    }
    if (!/^[0-9]{10}$/.test(phone_number)) {
      toast.error('Invalid phone number (10 digits)');
      return false;
    }
    if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
      toast.error('Invalid email format');
      return false;
    }
    return true;
  };

  const validateStep2 = () => {
    const { address, city, state, pincode } = formData;
    if (!address || address.length < 10) {
      toast.error('Complete address is required');
      return false;
    }
    if (!city || !state) {
      toast.error('City and state are required');
      return false;
    }
    if (!/^[0-9]{6}$/.test(pincode)) {
      toast.error('Invalid pincode (6 digits)');
      return false;
    }
    return true;
  };

  const validateStep3 = () => {
    const { agency_name, rera_id, years_of_experience } = formData;
    if (!agency_name || agency_name.length < 2) {
      toast.error('Agency name is required');
      return false;
    }
    if (!rera_id || rera_id.length < 5) {
      toast.error('Valid RERA ID is required');
      return false;
    }
    if (!years_of_experience || parseInt(years_of_experience) < 0) {
      toast.error('Years of experience is required');
      return false;
    }
    return true;
  };

  const handleNext = () => {
    if (currentStep === 1 && !validateStep1()) return;
    if (currentStep === 2 && !validateStep2()) return;
    if (currentStep === 3 && !validateStep3()) return;
    setCurrentStep(currentStep + 1);
  };

  const uploadFile = async (file: File) => {
    // For now, create a local URL for the file
    // In production, this should upload to your storage service
    return URL.createObjectURL(file);
  };

  const handleFileChange = async (type: string, e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    setLoading(true);
    try {
      const url = await uploadFile(file);
      setFormData(prev => ({ ...prev, [`${type}_url`]: url }));
      toast.success(`${type.replace(/_/g, ' ')} uploaded successfully`);
    } catch (error) {
      toast.error('Upload failed');
    } finally {
      setLoading(false);
    }
  };

  const submitKYC = async () => {
    if (!formData.pan_document_url || !formData.agency_certificate_url || !formData.rera_certificate_url) {
      toast.error('PAN, Agency Certificate, and RERA Certificate are mandatory');
      return;
    }

    setLoading(true);
    try {
      const token = getToken();
      await axios.post(`${API_BASE_URL}/api/agent-kyc/submit`, formData, {
        headers: token ? { Authorization: `Bearer ${token}` } : {}
      });

      toast.success('Agent KYC submitted successfully!');
      setTimeout(() => window.location.href = '/agent/dashboard', 2000);
    } catch (error: any) {
      toast.error(error.response?.data?.detail || 'Submission failed');
    } finally {
      setLoading(false);
    }
  };

  const steps = [
    { number: 1, title: 'Basic Info' },
    { number: 2, title: 'Address' },
    { number: 3, title: 'Agent Details' },
    { number: 4, title: 'Documents' }
  ];

  return (
    <div className="max-w-4xl mx-auto p-6">
      {/* Progress Indicator */}
      <div className="flex justify-between mb-12">
        {steps.map((step, idx) => (
          <div key={step.number} className="flex items-center flex-1">
            <div className="flex flex-col items-center flex-1">
              <div className={`w-10 h-10 rounded-full flex items-center justify-center font-semibold
                ${currentStep > step.number ? 'bg-green-500 text-white' : 
                  currentStep === step.number ? 'bg-blue-500 text-white ring-4 ring-blue-200' : 
                  'bg-gray-200 text-gray-600'}`}>
                {currentStep > step.number ? <CheckCircle className="w-6 h-6" /> : step.number}
              </div>
              <span className="text-sm mt-2 font-medium">{step.title}</span>
            </div>
            {idx < steps.length - 1 && (
              <div className={`h-1 flex-1 mx-2 ${currentStep > step.number ? 'bg-green-500' : 'bg-gray-200'}`} />
            )}
          </div>
        ))}
      </div>

      <Card className="p-6">
        {/* Step 1: Basic Info */}
        {currentStep === 1 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Basic Identity Information</h2>
            <div>
              <Label className="mb-2 block">Full Name *</Label>
              <Input className="h-10" value={formData.full_name} onChange={(e) => setFormData(prev => ({ ...prev, full_name: e.target.value }))} placeholder="Enter your full name" />
            </div>
            <div>
              <Label className="mb-2 block">PAN Number *</Label>
              <Input className="h-10" value={formData.pan_number} onChange={(e) => setFormData(prev => ({ ...prev, pan_number: e.target.value.toUpperCase() }))} placeholder="ABCDE1234F" maxLength={10} />
            </div>
            <div>
              <Label className="mb-2 block">Aadhaar Number (Optional)</Label>
              <Input className="h-10" value={formData.aadhaar_number} onChange={(e) => setFormData(prev => ({ ...prev, aadhaar_number: e.target.value }))} placeholder="123456789012" maxLength={12} />
            </div>
            <div>
              <Label className="mb-2 block">Date of Birth *</Label>
              <Input className="h-10" type="date" value={formData.date_of_birth} onChange={(e) => setFormData(prev => ({ ...prev, date_of_birth: e.target.value }))} />
            </div>
            <div>
              <Label className="mb-2 block">Phone Number *</Label>
              <Input className="h-10" value={formData.phone_number} onChange={(e) => setFormData(prev => ({ ...prev, phone_number: e.target.value }))} placeholder="9876543210" maxLength={10} />
            </div>
            <div>
              <Label className="mb-2 block">Email *</Label>
              <Input className="h-10" type="email" value={formData.email} onChange={(e) => setFormData(prev => ({ ...prev, email: e.target.value }))} placeholder="your@email.com" />
            </div>
          </div>
        )}

        {/* Step 2: Address */}
        {currentStep === 2 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Address Information</h2>
            <div>
              <Label className="mb-2 block">Complete Address *</Label>
              <Input className="h-10" value={formData.address} onChange={(e) => setFormData(prev => ({ ...prev, address: e.target.value }))} placeholder="House No, Street, Area" />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label className="mb-2 block">City *</Label>
                <Input className="h-10" value={formData.city} onChange={(e) => setFormData(prev => ({ ...prev, city: e.target.value }))} placeholder="Mumbai" />
              </div>
              <div>
                <Label className="mb-2 block">State *</Label>
                <Input className="h-10" value={formData.state} onChange={(e) => setFormData(prev => ({ ...prev, state: e.target.value }))} placeholder="Maharashtra" />
              </div>
            </div>
            <div>
              <Label className="mb-2 block">Pincode *</Label>
              <Input className="h-10" value={formData.pincode} onChange={(e) => setFormData(prev => ({ ...prev, pincode: e.target.value }))} placeholder="400001" maxLength={6} />
            </div>
          </div>
        )}

        {/* Step 3: Agent Details */}
        {currentStep === 3 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold mb-4">Agent & Agency Details</h2>
            <div>
              <Label className="mb-2 block">Agency Name *</Label>
              <Input className="h-10" value={formData.agency_name} onChange={(e) => setFormData(prev => ({ ...prev, agency_name: e.target.value }))} placeholder="ABC Realty" />
            </div>
            <div>
              <Label className="mb-2 block">RERA ID *</Label>
              <Input className="h-10" value={formData.rera_id} onChange={(e) => setFormData(prev => ({ ...prev, rera_id: e.target.value }))} placeholder="RERA123456" />
            </div>
            <div>
              <Label className="mb-2 block">Years of Experience *</Label>
              <Input className="h-10" type="number" min="0" value={formData.years_of_experience} onChange={(e) => setFormData(prev => ({ ...prev, years_of_experience: e.target.value }))} placeholder="5" />
            </div>
          </div>
        )}

        {/* Step 4: Document Upload */}
        {currentStep === 4 && (
          <div className="space-y-6">
            <h2 className="text-xl font-semibold mb-4">Upload Documents</h2>
            <div className="space-y-4">
              <div className="border rounded-lg p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base font-semibold">PAN Card * (Required)</Label>
                    <p className="text-sm text-gray-500">PDF, JPG, or PNG</p>
                  </div>
                  <div className="flex items-center gap-2">
                    {formData.pan_document_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                    <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('pan_document', e)} className="w-auto" />
                  </div>
                </div>
              </div>

              <div className="border rounded-lg p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base font-semibold">Agency Certificate * (Required)</Label>
                    <p className="text-sm text-gray-500">PDF, JPG, or PNG</p>
                  </div>
                  <div className="flex items-center gap-2">
                    {formData.agency_certificate_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                    <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('agency_certificate', e)} className="w-auto" />
                  </div>
                </div>
              </div>

              <div className="border rounded-lg p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base font-semibold">RERA Certificate * (Required)</Label>
                    <p className="text-sm text-gray-500">PDF, JPG, or PNG</p>
                  </div>
                  <div className="flex items-center gap-2">
                    {formData.rera_certificate_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                    <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('rera_certificate', e)} className="w-auto" />
                  </div>
                </div>
              </div>

              <div className="border rounded-lg p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base font-semibold">Your Photo (Recommended)</Label>
                    <p className="text-sm text-gray-500">JPG or PNG</p>
                  </div>
                  <div className="flex items-center gap-2">
                    {formData.photo_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                    <Input type="file" accept=".jpg,.jpeg,.png" onChange={(e) => handleFileChange('photo', e)} className="w-auto" />
                  </div>
                </div>
              </div>

              {formData.aadhaar_number && (
                <div className="border rounded-lg p-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-base font-semibold">Aadhaar Card (Optional)</Label>
                      <p className="text-sm text-gray-500">PDF, JPG, or PNG</p>
                    </div>
                    <div className="flex items-center gap-2">
                      {formData.aadhaar_document_url && <CheckCircle className="w-5 h-5 text-green-500" />}
                      <Input type="file" accept=".pdf,.jpg,.jpeg,.png" onChange={(e) => handleFileChange('aadhaar_document', e)} className="w-auto" />
                    </div>
                  </div>
                </div>
              )}
            </div>

            {(!formData.pan_document_url || !formData.agency_certificate_url || !formData.rera_certificate_url) && (
              <div className="flex items-center gap-2 text-amber-600 bg-amber-50 p-3 rounded">
                <AlertCircle className="w-5 h-5" />
                <span className="text-sm">PAN, Agency Certificate, and RERA Certificate are mandatory</span>
              </div>
            )}
          </div>
        )}

        {/* Navigation Buttons */}
        <div className="flex justify-between mt-8">
          {currentStep > 1 && (
            <Button variant="outline" onClick={() => setCurrentStep(currentStep - 1)} disabled={loading} className="border-[#0056D2] text-[#0056D2] hover:bg-[#0056D2] hover:text-white">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Previous
            </Button>
          )}

          <div className="ml-auto">
            {currentStep < 4 ? (
              <Button 
                onClick={handleNext} 
                disabled={loading}
                className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
              >
                Next
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            ) : (
              <Button 
                onClick={submitKYC} 
                disabled={loading || !formData.pan_document_url || !formData.agency_certificate_url || !formData.rera_certificate_url}
                className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white"
              >
                {loading ? 'Submitting...' : 'Submit for Review'}
              </Button>
            )}
          </div>
        </div>
      </Card>
    </div>
  );
};
