import React, { useState, useEffect } from 'react';
import { agreementsApi } from '@/api/agreements';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { toast } from '@/utils/toast';

interface AgreementCreatorProps {
  propertyId?: number;
  onCreated?: (agreement: any) => void;
}

export const AgreementCreator: React.FC<AgreementCreatorProps> = ({ propertyId, onCreated }) => {
  const [templates, setTemplates] = useState<any[]>([]);
  const [selectedTemplate, setSelectedTemplate] = useState<any>(null);
  const [parties, setParties] = useState<any[]>([]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    loadTemplates();
  }, []);

  const loadTemplates = async () => {
    try {
      const data = await agreementsApi.getTemplates();
      setTemplates(data);
    } catch (error) {
      toast.error('Failed to load templates');
    }
  };

  const handleCreate = async () => {
    if (!selectedTemplate) {
      toast.error('Please select a template');
      return;
    }

    if (parties.length === 0) {
      toast.error('Please add at least one party');
      return;
    }

    setLoading(true);
    try {
      const agreement = await agreementsApi.create({
        template_id: selectedTemplate.id,
        agreement_type: selectedTemplate.template_type,
        property_id: propertyId,
        parties: parties,
        content: selectedTemplate.content
      });

      toast.success('Agreement created! Waiting for approval.');
      
      // Show next step hint
      setTimeout(() => {
        toast.info('Next: Both parties need to approve this agreement');
      }, 2000);
      
      onCreated?.(agreement);
    } catch (error) {
      toast.error('Failed to create agreement');
    } finally {
      setLoading(false);
    }
  };

  const addParty = () => {
    const userStr = localStorage.getItem('user');
    if (userStr) {
      const user = JSON.parse(userStr);
      setParties([...parties, {
        user_id: user.user_id,
        role: 'buyer',
        name: user.full_name || user.email,
        email: user.email
      }]);
    }
  };

  return (
    <div className="space-y-4">
      <h2 className="text-2xl font-bold">Create Agreement</h2>

      <div>
        <label className="block text-sm font-medium mb-2">Select Template</label>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {templates.map((template) => (
            <Card
              key={template.id}
              className={`p-4 cursor-pointer hover:border-blue-500 ${
                selectedTemplate?.id === template.id ? 'border-blue-500 bg-blue-50' : ''
              }`}
              onClick={() => setSelectedTemplate(template)}
            >
              <h3 className="font-semibold">{template.template_name}</h3>
              <p className="text-sm text-gray-600">{template.template_type}</p>
            </Card>
          ))}
        </div>
      </div>

      {selectedTemplate && (
        <div>
          <label className="block text-sm font-medium mb-2">Parties</label>
          <Button onClick={addParty} variant="outline" size="sm">
            Add Yourself as Party
          </Button>
          <div className="mt-2 space-y-2">
            {parties.map((party, idx) => (
              <div key={idx} className="p-2 bg-gray-50 rounded">
                {party.name} ({party.role})
              </div>
            ))}
          </div>
        </div>
      )}

      <Button onClick={handleCreate} disabled={loading} className="w-full">
        {loading ? 'Creating...' : 'Create Agreement'}
      </Button>
    </div>
  );
};
