import React, { useState, useEffect } from 'react';
import { agreementsApi } from '@/api/agreements';
import { contractsApi } from '@/api/contracts';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { toast } from '@/utils/toast';

export const AgreementList: React.FC = () => {
  const [agreements, setAgreements] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [activeTab, setActiveTab] = useState<'created' | 'pending'>('created');
  const [expandedAgreements, setExpandedAgreements] = useState<Set<string>>(new Set());

  useEffect(() => {
    loadAgreements();
    // Auto-refresh every 5 seconds to catch new agreements
    const interval = setInterval(loadAgreements, 5000);
    return () => clearInterval(interval);
  }, []);

  useEffect(() => {
    // Force re-render when tab changes
    console.log('Active tab:', activeTab);
    console.log('Filtered agreements:', getFilteredAgreements());
  }, [activeTab, agreements]);

  const loadAgreements = async () => {
    try {
      console.log('Loading agreements...');
      
      // Load from localStorage instead of API
      const purchaseIntents = JSON.parse(localStorage.getItem('purchase_intents') || '[]');
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      
      // Convert purchase intents to agreement format
      const agreementsData = purchaseIntents.map((intent: any) => ({
        id: intent.id,
        agreement_number: `AGR-${intent.id.split('_')[1]}`,
        agreement_type: 'purchase',
        property_id: intent.property_id,
        user_id: intent.buyer_id,
        agreement_status: intent.status === 'initiated' ? 'pending_approval' : intent.status,
        content: `Property Purchase Agreement for ${intent.property_title}`,
        parties: [
          {
            user_id: intent.buyer_id,
            role: 'buyer',
            name: intent.buyer_name,
            email: intent.buyer_email
          },
          {
            user_id: 'seller_' + intent.property_id,
            role: 'seller',
            name: 'Property Owner',
            email: intent.seller_email
          }
        ],
        metadata: {
          property_title: intent.property_title,
          property_address: intent.property_address,
          sale_price: intent.sale_price
        },
        created_at: intent.created_at,
        myApprovalStatus: 'pending'
      }));
      
      console.log('Agreements loaded:', agreementsData);
      setAgreements(agreementsData);
    } catch (error) {
      console.error('Failed to load agreements:', error);
      toast.error('Failed to load agreements');
    } finally {
      setLoading(false);
    }
  };

  const getFilteredAgreements = () => {
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    
    if (activeTab === 'created') {
      // Show agreements created by this user
      return agreements.filter(a => a.user_id === user.user_id);
    } else {
      // Show agreements where user needs to approve (is in parties but not creator)
      return agreements.filter(a => {
        const isInParties = a.parties?.some((p: any) => p.user_id === user.user_id);
        const isNotCreator = a.user_id !== user.user_id;
        return isInParties && isNotCreator;
      });
    }
  };

  const handleConvertToContract = async (agreement: any) => {
    if (agreement.agreement_status !== 'approved') {
      toast.error('Agreement must be approved by all parties first');
      return;
    }

    try {
      const buyer = agreement.parties.find((p: any) => p.role === 'buyer');
      const seller = agreement.parties.find((p: any) => p.role === 'seller');

      if (!buyer) {
        toast.error('Buyer information missing in agreement');
        return;
      }

      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const sellerId = seller?.user_id || user.user_id;
      const contractAmount = agreement.metadata?.sale_price || 5000000;

      const contract = await contractsApi.create({
        agreement_id: agreement.id,
        property_id: agreement.property_id,
        buyer_id: buyer.user_id,
        seller_id: sellerId,
        contract_type: 'purchase',
        contract_terms: agreement.content,
        contract_amount: contractAmount
      });

      if (!seller) {
        toast.success('Contract created! Note: Seller info was missing, using your account as placeholder.');
      } else {
        toast.success('Contract created! Go to Contracts section to sign.');
      }
      loadAgreements();
    } catch (error: any) {
      console.error('Contract creation error:', error);
      toast.error('Failed to create contract: ' + (error.message || 'Unknown error'));
    }
  };

  const handleApprove = async (agreementId: string) => {
    try {
      console.log('Approving agreement:', agreementId);
      
      // Update local storage
      const purchaseIntents = JSON.parse(localStorage.getItem('purchase_intents') || '[]');
      const updatedIntents = purchaseIntents.map((intent: any) => 
        intent.id === agreementId 
          ? { ...intent, status: 'approved' }
          : intent
      );
      localStorage.setItem('purchase_intents', JSON.stringify(updatedIntents));
      
      // Reload agreements
      await loadAgreements();
      
      toast.success('Agreement approved successfully!');
    } catch (error: any) {
      console.error('Failed to approve agreement:', error);
      toast.error('Failed to approve: ' + (error.message || 'Unknown error'));
    }
  };

  if (loading) return <div>Loading...</div>;

  const filteredAgreements = getFilteredAgreements();

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold">Digital Agreements</h2>
      </div>

      {/* Tabs */}
      <div className="flex gap-2 border-b">
        <button
          onClick={() => setActiveTab('created')}
          className={`px-4 py-2 font-medium transition-colors ${
            activeTab === 'created'
              ? 'border-b-2 border-blue-600 text-blue-600'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          My Agreements ({agreements.filter(a => {
            const user = JSON.parse(localStorage.getItem('user') || '{}');
            return a.user_id === user.user_id;
          }).length})
        </button>
        <button
          onClick={() => setActiveTab('pending')}
          className={`px-4 py-2 font-medium transition-colors ${
            activeTab === 'pending'
              ? 'border-b-2 border-orange-600 text-orange-600'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          Pending My Approval ({agreements.filter(a => {
            const user = JSON.parse(localStorage.getItem('user') || '{}');
            const isInParties = a.parties?.some((p: any) => p.user_id === user.user_id);
            const isNotCreator = a.user_id !== user.user_id;
            return isInParties && isNotCreator;
          }).length})
        </button>
      </div>

      {filteredAgreements.length === 0 ? (
        <Card className="p-8 text-center text-gray-500">
          {activeTab === 'created' 
            ? 'No agreements created yet. Make a payment to create one.'
            : 'No agreements waiting for your approval.'}
        </Card>
      ) : (
        filteredAgreements.map((agreement) => {
          const isExpanded = expandedAgreements.has(agreement.id);
          const toggleExpand = () => {
            setExpandedAgreements(prev => {
              const newSet = new Set(prev);
              if (newSet.has(agreement.id)) {
                newSet.delete(agreement.id);
              } else {
                newSet.add(agreement.id);
              }
              return newSet;
            });
          };
          
          return (
          <Card key={agreement.id} className="p-6">
            <div className="flex justify-between items-start mb-4">
              <div>
                <h3 className="text-lg font-semibold">{agreement.agreement_number}</h3>
                <p className="text-sm text-gray-600">Type: {agreement.agreement_type}</p>
                <p className="text-sm text-gray-600">Property ID: #{agreement.property_id}</p>
              </div>
              <Badge className={
                agreement.agreement_status === 'approved' ? 'bg-green-100 text-green-800' :
                agreement.agreement_status === 'pending_approval' ? 'bg-yellow-100 text-yellow-800' :
                'bg-gray-100 text-gray-800'
              }>
                {agreement.agreement_status.toUpperCase()}
              </Badge>
            </div>

            <div className="mb-4 flex items-center justify-between">
              <button
                onClick={toggleExpand}
                className="flex items-center gap-2 text-sm font-medium text-blue-600 hover:text-blue-800 transition-colors"
              >
                <svg 
                  className={`w-4 h-4 transition-transform duration-200 ${isExpanded ? 'rotate-180' : ''}`}
                  fill="none" 
                  stroke="currentColor" 
                  viewBox="0 0 24 24"
                >
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
                {isExpanded ? 'Hide Details' : 'View Full Details'}
              </button>
            </div>

            {isExpanded && (
              <div className="mb-4 bg-gray-50 p-4 rounded-lg border border-gray-200 transition-all duration-300">
                <h4 className="text-sm font-semibold text-gray-900 mb-3">Agreement Details</h4>
                <div className="space-y-2">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <span className="text-xs text-gray-600">Agreement Number:</span>
                      <p className="text-sm font-medium text-gray-900">{agreement.agreement_number}</p>
                    </div>
                    <div>
                      <span className="text-xs text-gray-600">Type:</span>
                      <p className="text-sm font-medium text-gray-900 capitalize">{agreement.agreement_type}</p>
                    </div>
                    <div>
                      <span className="text-xs text-gray-600">Property ID:</span>
                      <p className="text-sm font-medium text-gray-900">#{agreement.property_id}</p>
                    </div>
                    <div>
                      <span className="text-xs text-gray-600">Status:</span>
                      <p className="text-sm font-medium text-gray-900 capitalize">{agreement.agreement_status.replace('_', ' ')}</p>
                    </div>
                  </div>
                  
                  {agreement.parties && agreement.parties.length > 0 && (
                    <div className="mt-3 pt-3 border-t border-gray-300">
                      <span className="text-xs text-gray-600 block mb-2">Parties Involved:</span>
                      <div className="grid grid-cols-2 gap-3">
                        {agreement.parties.map((party: any, idx: number) => (
                          <div key={idx} className="bg-white p-3 rounded border border-gray-200">
                            <p className="text-xs font-bold text-gray-900 uppercase mb-2">{party.role}</p>
                            <div className="space-y-1">
                              <p className="text-xs text-gray-700">
                                <span className="font-medium">Name:</span> {party.name}
                              </p>
                              <p className="text-xs text-gray-700">
                                <span className="font-medium">Email:</span> {party.email}
                              </p>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                  
                  {agreement.metadata && (
                    <div className="mt-3 pt-3 border-t border-gray-300">
                      <span className="text-xs text-gray-600 block mb-2">Additional Information:</span>
                      <div className="grid grid-cols-2 gap-2">
                        {agreement.metadata.sale_price && (
                          <div>
                            <span className="text-xs text-gray-600">Sale Price:</span>
                            <p className="text-sm font-semibold text-green-600">₹{agreement.metadata.sale_price.toLocaleString()}</p>
                          </div>
                        )}
                        {agreement.metadata.property_address && (
                          <div className="col-span-2">
                            <span className="text-xs text-gray-600">Property Address:</span>
                            <p className="text-sm text-gray-900">{agreement.metadata.property_address}</p>
                          </div>
                        )}
                      </div>
                    </div>
                  )}
                  
                  <div className="mt-3 pt-3 border-t border-gray-300">
                    <span className="text-xs text-gray-600 block mb-2">Agreement Terms & Conditions:</span>
                    <div className="bg-white p-4 rounded border border-gray-200 max-h-60 overflow-y-auto">
                      <div className="prose prose-sm max-w-none">
                        <p className="text-sm text-gray-900 font-semibold mb-3">PROPERTY PURCHASE AGREEMENT</p>
                        <div className="text-sm text-gray-700 space-y-2 leading-relaxed">
                          <p className="mb-2">
                            This Agreement is entered into on <span className="font-medium">{new Date(agreement.created_at).toLocaleDateString('en-IN', { day: 'numeric', month: 'long', year: 'numeric' })}</span> between the parties mentioned above.
                          </p>
                          
                          <p className="font-medium mt-3 mb-1">1. Property Details:</p>
                          <p className="ml-4">
                            Property ID: #{agreement.property_id}<br />
                            {agreement.metadata?.property_address && `Location: ${agreement.metadata.property_address}`}
                          </p>
                          
                          <p className="font-medium mt-3 mb-1">2. Purchase Terms:</p>
                          <p className="ml-4">
                            {agreement.metadata?.sale_price && `Sale Price: ₹${agreement.metadata.sale_price.toLocaleString()}`}<br />
                            Payment Method: As per escrow agreement<br />
                            The Buyer agrees to purchase the property from the Seller at the agreed price.
                          </p>
                          
                          <p className="font-medium mt-3 mb-1">3. Obligations:</p>
                          <p className="ml-4">
                            • The Seller warrants that they have clear title to the property<br />
                            • The Buyer agrees to complete due diligence within the stipulated timeframe<br />
                            • Both parties agree to cooperate in good faith to complete the transaction
                          </p>
                          
                          <p className="font-medium mt-3 mb-1">4. Escrow & Payment:</p>
                          <p className="ml-4">
                            • Payment shall be held in escrow until all conditions are met<br />
                            • Funds will be released to Seller upon successful verification<br />
                            • Refund provisions apply as per platform policy
                          </p>
                          
                          <p className="font-medium mt-3 mb-1">5. Additional Terms:</p>
                          <div className="ml-4 bg-gray-50 p-2 rounded text-xs">
                            {agreement.content}
                          </div>
                          
                          <p className="mt-4 text-xs text-gray-600 italic">
                            By approving this agreement, both parties acknowledge and accept all terms and conditions stated herein.
                          </p>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            <div className="space-y-2">
              <div className="flex gap-2 items-center">
                {(() => {
                  const user = JSON.parse(localStorage.getItem('user') || '{}');
                  const isCreator = agreement.user_id === user.user_id;
                  const myApprovalStatus = agreement.myApprovalStatus || 'pending';
                  
                  if (agreement.agreement_status === 'approved') {
                    return (
                      <>
                        <Badge className="bg-green-100 text-green-800">
                          ✓ Approved by both parties
                        </Badge>
                        {agreement.agreement_type === 'purchase' && isCreator && (
                          <Button 
                            onClick={() => handleConvertToContract(agreement)}
                            style={{ backgroundColor: '#16a34a', color: 'white', display: 'inline-flex' }}
                            size="sm"
                          >
                            Convert to Contract →
                          </Button>
                        )}
                      </>
                    );
                  }
                  
                  if (myApprovalStatus === 'approved' && agreement.agreement_status === 'pending_approval') {
                    const otherParty = agreement.parties?.find((p: any) => p.user_id !== user.user_id);
                    return (
                      <>
                        <Badge className="bg-blue-100 text-blue-800">
                          ✓ Approved by you
                        </Badge>
                        <span className="text-sm text-yellow-600">
                          Waiting for {otherParty?.role || 'other party'} approval...
                        </span>
                      </>
                    );
                  }
                  
                  if (myApprovalStatus === 'pending') {
                    return (
                      <Button 
                        onClick={() => handleApprove(agreement.id)} 
                        size="sm" 
                        className={activeTab === 'pending' ? 'bg-orange-600 hover:bg-orange-700' : ''}
                      >
                        Approve Agreement
                      </Button>
                    );
                  }
                  
                  return null;
                })()}
              </div>
            </div>

            <p className="text-xs text-gray-500 mt-4">
              Created: {new Date(agreement.created_at).toLocaleString()}
            </p>
          </Card>
          );
        })
      )}
    </div>
  );
};
