import React, { useState, useEffect, useMemo } from 'react';
import { behaviorAnalysisApi } from '../../api/behaviorAnalysis';
import { TrendingUp, MapPin, Home, Clock, Activity, Eye, MousePointer, Bookmark, Search, Download, Filter, LineChart, BarChart3, PieChart, Calendar, Target, Brain, Zap, Shield, Lightbulb } from 'lucide-react';
import { PrivacyControls } from '../PrivacyControls';

export const BehaviorAnalytics: React.FC = () => {
  const [insights, setInsights] = useState<any>(null);
  const [events, setEvents] = useState<any[]>([]);
  const [recommendations, setRecommendations] = useState<any[]>([]);
  const [marketInsights, setMarketInsights] = useState<any>({});
  const [journeyAnalysis, setJourneyAnalysis] = useState<any>({});
  const [userSegments, setUserSegments] = useState<any>({});
  const [loading, setLoading] = useState(true);
  const [dateRange, setDateRange] = useState<'7d' | '30d' | '90d' | 'all'>('30d');
  const [selectedEventType, setSelectedEventType] = useState<string>('all');
  const [chartView, setChartView] = useState<'bar' | 'line' | 'pie'>('line');
  const [showPrivacyControls, setShowPrivacyControls] = useState(false);
  const [activeTab, setActiveTab] = useState<'overview' | 'journey' | 'recommendations' | 'insights'>('overview');

  const filteredEvents = useMemo(() => {
    return events.filter(e => {
      const matchesType = selectedEventType === 'all' || e.event_type === selectedEventType;
      return matchesType;
    });
  }, [events, selectedEventType]);

  const dayWiseData = useMemo(() => {
    const grouped: Record<string, Record<string, number>> = {};
    events.forEach(event => {
      const date = new Date(event.created_at).toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
      if (!grouped[date]) grouped[date] = {};
      grouped[date][event.event_type] = (grouped[date][event.event_type] || 0) + 1;
    });
    return Object.entries(grouped).map(([date, types]) => ({
      date,
      total: Object.values(types).reduce((a, b) => a + b, 0),
      ...types
    })).slice(-14);
  }, [events]);

  useEffect(() => {
    loadData();
  }, [dateRange]);

  useEffect(() => {
    // Auto-refresh every 30 seconds
    const interval = setInterval(() => {
      loadData();
    }, 30000);
    
    return () => clearInterval(interval);
  }, [dateRange]);

  const loadData = async () => {
    try {
      const token = localStorage.getItem('access_token');
      if (!token) {
        console.error('No access token found');
        setLoading(false);
        return;
      }
      
      console.log('Loading behavior analytics data...');
      const days = dateRange === '7d' ? 7 : dateRange === '30d' ? 30 : dateRange === '90d' ? 90 : undefined;
      
      const [insightsData, eventsData, recsData, marketData, journeyData, segmentsData] = await Promise.all([
        behaviorAnalysisApi.getUserInsights(days).catch(err => {
          console.error('Insights error:', err);
          return null;
        }),
        behaviorAnalysisApi.getUserEvents(100, days).catch(err => {
          console.error('Events error:', err);
          return [];
        }),
        fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/behavior/recommendations`, {
          headers: { 'Authorization': `Bearer ${token}` }
        }).then(r => {
          if (!r.ok) {
            console.error('Recommendations failed:', r.status);
            return [];
          }
          return r.json();
        }).catch(err => {
          console.error('Recommendations error:', err);
          return [];
        }),
        fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/behavior/market-insights`, {
          headers: { 'Authorization': `Bearer ${token}` }
        }).then(r => {
          if (!r.ok) {
            console.error('Market insights failed:', r.status);
            return {};
          }
          return r.json();
        }).catch(err => {
          console.error('Market insights error:', err);
          return {};
        }),
        fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/behavior/journey-analysis`, {
          headers: { 'Authorization': `Bearer ${token}` }
        }).then(r => {
          if (!r.ok) {
            console.error('Journey analysis failed:', r.status);
            return {};
          }
          return r.json();
        }).catch(err => {
          console.error('Journey analysis error:', err);
          return {};
        }),
        fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/behavior/segments`, {
          headers: { 'Authorization': `Bearer ${token}` }
        }).then(r => {
          if (!r.ok) {
            console.error('Segments failed:', r.status);
            return {};
          }
          return r.json();
        }).catch(err => {
          console.error('Segments error:', err);
          return {};
        })
      ]);
      
      console.log('Data loaded:', {
        insights: !!insightsData,
        events: eventsData?.length || 0,
        recommendations: recsData?.length || 0
      });
      console.log('📊 All events received:', eventsData);
      console.log('📊 Event types:', eventsData?.map(e => e.event_type));
      console.log('📊 Search events:', eventsData?.filter(e => e.event_type === 'search'));
      
      setInsights(insightsData);
      setEvents(eventsData || []);
      setRecommendations(recsData || []);
      setMarketInsights(marketData || {});
      setJourneyAnalysis(journeyData || {});
      setUserSegments(segmentsData || {});
    } catch (error) {
      console.error('Error loading data:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="text-gray-500">Loading analytics...</div>
      </div>
    );
  }

  if (!insights) {
    return (
      <div className="bg-white rounded-lg shadow p-8 text-center">
        <Activity className="w-16 h-16 text-gray-300 mx-auto mb-4" />
        <p className="text-gray-500 font-medium">No analytics data available yet</p>
        <p className="text-sm text-gray-400 mt-2">Start browsing properties to see your behavior insights</p>
        <button
          onClick={loadData}
          className="mt-4 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
        >
          Retry Loading
        </button>
      </div>
    );
  }

  return (
    <div className="h-[calc(100vh-120px)] overflow-y-auto space-y-6 pr-2">
      <div className="flex items-center justify-between sticky top-0 bg-gray-50 py-4 z-10">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Behavior Analytics</h2>
          <p className="text-sm text-gray-500 mt-1">Interactive insights dashboard</p>
        </div>
        <button
          onClick={() => setShowPrivacyControls(!showPrivacyControls)}
          style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
          className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm disabled:bg-blue-400"
        >
          <Shield className="w-4 h-4" />
          <span>Privacy Settings</span>
        </button>
      </div>

      {/* Tab Navigation */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 mb-6">
        <div className="flex border-b border-gray-200">
          {[
            { id: 'overview', label: 'Overview', icon: BarChart3 },
            { id: 'journey', label: 'User Journey', icon: Target },
            { id: 'recommendations', label: 'AI Recommendations', icon: Brain },
            { id: 'insights', label: 'Market Insights', icon: Zap }
          ].map(tab => (
            <button
              key={tab.id}
              onClick={() => setActiveTab(tab.id as any)}
              className={`flex items-center gap-2 px-6 py-4 font-medium transition-colors ${
                activeTab === tab.id
                  ? 'text-blue-600 border-b-2 border-blue-600 bg-blue-50'
                  : 'text-gray-600 hover:text-gray-900 hover:bg-gray-50'
              }`}
            >
              <tab.icon className="w-4 h-4" />
              {tab.label}
            </button>
          ))}
        </div>
      </div>

      {/* Privacy Controls */}
      {showPrivacyControls && (
        <div className="mb-6">
          <PrivacyControls
            onDataDeleted={() => {
              setInsights(null);
              setEvents([]);
              setShowPrivacyControls(false);
            }}
            onDataAnonymized={() => {
              setInsights(null);
              setEvents([]);
              setShowPrivacyControls(false);
            }}
          />
        </div>
      )}

      {/* Tab Content */}
      {activeTab === 'overview' && (
        <>
          {/* Overview Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
            <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg shadow-sm p-6 border border-blue-200">
              <div className="flex items-center gap-3 mb-2">
                <div className="bg-blue-500 rounded-lg p-2">
                  <Activity className="w-5 h-5 text-white" />
                </div>
                <div className="text-3xl font-bold text-blue-900">{insights?.total_events || 0}</div>
              </div>
              <div className="text-sm text-blue-700 font-medium">Total Activity</div>
            </div>
            
            <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg shadow-sm p-6 border border-green-200">
              <div className="flex items-center gap-3 mb-2">
                <div className="bg-green-500 rounded-lg p-2">
                  <TrendingUp className="w-5 h-5 text-white" />
                </div>
                <div className="text-3xl font-bold text-green-900">{insights?.engagement_score?.toFixed(1) || '0.0'}</div>
              </div>
              <div className="text-sm text-green-700 font-medium">Engagement Score</div>
            </div>
            
            <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg shadow-sm p-6 border border-blue-200">
              <div className="flex items-center gap-3 mb-2">
                <div className="bg-blue-500 rounded-lg p-2">
                  <Lightbulb className="w-5 h-5 text-blue" />
                </div>
                <div className="text font-bold text-green-900 ">{journeyAnalysis?.current_stage || 'awareness'}</div>
              </div>
              <div className="text-sm text-green-700 font-medium">Journey Stage</div>
            </div>
            
            <div className="bg-gradient-to-br from-orange-50 to-orange-100 rounded-lg shadow-sm p-6 border border-orange-200">
              <div className="flex items-center gap-3 mb-2">
                <div className="bg-orange-500 rounded-lg p-2">
                  <Brain className="w-5 h-5 text-white" />
                </div>
                <div className="text-2xl font-bold text-orange-900">{recommendations?.length || 0}</div>
              </div>
              <div className="text-sm text-orange-700 font-medium">AI Recommendations</div>
            </div>
          </div>

          {/* Activity Breakdown */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
            <h3 className="font-semibold text-lg text-gray-900 mb-4">Activity Breakdown</h3>
            <ActivityChart events={filteredEvents} />
          </div>

          {/* Recent Activity */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="font-semibold text-lg text-gray-900">Recent Activity Timeline</h3>
              <span className="text-xs text-gray-500">{filteredEvents.length} events</span>
            </div>
            <div className="h-80 overflow-y-auto pr-2">
              <RecentActivityList events={filteredEvents} />
            </div>
          </div>
        </>
      )}

      {activeTab === 'journey' && (
        <JourneyAnalysisView journeyData={journeyAnalysis} />
      )}

      {activeTab === 'recommendations' && (
        <RecommendationsView recommendations={recommendations} />
      )}

      {activeTab === 'insights' && (
        <MarketInsightsView insights={marketInsights} />
      )}
    </div>
  );
};

const ActivityChart: React.FC<{ events: any[] }> = ({ events }) => {
  console.log('📊 ActivityChart - Received events:', events.length, events.map(e => e.event_type));
  
  const eventCounts = events.filter(e => e.event_type !== 'exit_intent' && e.event_type !== 'page_view' && e.event_type !== 'time_spent').reduce((acc, event) => {
    const type = event.event_type.replace(/_/g, ' ');
    acc[type] = (acc[type] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);
  
  console.log('📊 ActivityChart - Event counts:', eventCounts);

  const maxCount = Math.max(...(Object.values(eventCounts) as number[]), 1);
  const topEvents = Object.entries(eventCounts).sort(([, a], [, b]) => (b as number) - (a as number)).slice(0, 6);

  const getIcon = (type: string) => {
    const lowerType = type.toLowerCase();
    if (lowerType.includes('view')) return <Eye className="w-5 h-5" />;
    if (lowerType.includes('click')) return <MousePointer className="w-5 h-5" />;
    if (lowerType.includes('wishlist')) return (
      <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
        <path fillRule="evenodd" d="M3.172 5.172a4 4 0 015.656 0L10 6.343l1.172-1.171a4 4 0 115.656 5.656L10 17.657l-6.828-6.829a4 4 0 010-5.656z" clipRule="evenodd" />
      </svg>
    );
    if (lowerType.includes('save')) return <Bookmark className="w-5 h-5" />;
    if (lowerType.includes('search')) return <Search className="w-5 h-5" />;
    if (lowerType.includes('visit')) return <Calendar className="w-5 h-5" />;
    return <Activity className="w-5 h-5" />;
  };

  const getColor = (index: number) => {
    const colors = [
      { bg: 'bg-blue-500', text: 'text-blue-600', light: 'bg-blue-50' },
      { bg: 'bg-green-500', text: 'text-green-600', light: 'bg-green-50' },
      { bg: 'bg-purple-500', text: 'text-purple-600', light: 'bg-purple-50' },
      { bg: 'bg-orange-500', text: 'text-orange-600', light: 'bg-orange-50' },
      { bg: 'bg-pink-500', text: 'text-pink-600', light: 'bg-pink-50' },
      { bg: 'bg-indigo-500', text: 'text-indigo-600', light: 'bg-indigo-50' },
    ];
    return colors[index % colors.length];
  };

  return (
    <div className="space-y-4">
      {topEvents.map(([type, count], index) => {
        const barWidth = ((count as number) / maxCount) * 100;
        const color = getColor(index);
        
        return (
          <div key={type}>
            <div className="flex items-center justify-between mb-2">
              <div className="flex items-center gap-3">
                <div className={`${color.light} ${color.text} p-2 rounded-lg`}>
                  {getIcon(type)}
                </div>
                <span className="text-sm font-semibold text-gray-800 capitalize">{type}</span>
              </div>
              <div className="text-lg font-bold text-gray-900">{count as React.ReactNode}</div>
            </div>
            <div className="relative w-full bg-gray-100 rounded-full h-3">
              <div 
                className={`${color.bg} h-3 rounded-full transition-all duration-1000`}
                style={{ width: `${barWidth}%` }}
              />
            </div>
          </div>
        );
      })}
      {topEvents.length === 0 && (
        <div className="text-center py-8">
          <Activity className="w-12 h-12 text-gray-300 mx-auto mb-2" />
          <p className="text-gray-400 text-sm">No activity data yet</p>
        </div>
      )}
    </div>
  );
};

const RecentActivityList: React.FC<{ events: any[] }> = ({ events }) => {
  const formatTime = (timestamp: string) => {
    const date = new Date(timestamp);
    const now = new Date();
    const diff = now.getTime() - date.getTime();
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);
    if (days > 0) return `${days}d ago`;
    if (hours > 0) return `${hours}h ago`;
    if (minutes > 0) return `${minutes}m ago`;
    return 'Just now';
  };

  const getEventIcon = (type: string) => {
    if (type === 'property_view') return { icon: <Eye className="w-4 h-4" />, color: 'bg-blue-100 text-blue-600' };
    if (type === 'search') return { icon: <Search className="w-4 h-4" />, color: 'bg-orange-100 text-orange-600' };
    if (type === 'schedule_visit') return { icon: <Calendar className="w-4 h-4" />, color: 'bg-pink-100 text-pink-600' };
    if (type === 'save_property') return { icon: <Bookmark className="w-4 h-4" />, color: 'bg-green-100 text-green-600' };
    if (type === 'wishlist_add') return { 
      icon: (
        <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
          <path fillRule="evenodd" d="M3.172 5.172a4 4 0 015.656 0L10 6.343l1.172-1.171a4 4 0 115.656 5.656L10 17.657l-6.828-6.829a4 4 0 010-5.656z" clipRule="evenodd" />
        </svg>
      ), 
      color: 'bg-red-100 text-red-600' 
    };
    return { icon: <Activity className="w-4 h-4" />, color: 'bg-gray-100 text-gray-600' };
  };

  const getEventText = (event: any) => {
    const type = event.event_type;
    const data = event.event_data || {};
    if (type === 'property_view') return `Viewed ${data.title || 'property #' + event.property_id}`;
    if (type === 'search') return `Searched for "${data.query || data.filters?.query || 'properties'}"`;
    if (type === 'schedule_visit') return `Scheduled visit for property`;
    if (type === 'save_property') return `Saved ${data.title || 'property #' + event.property_id}`;
    if (type === 'wishlist_add') return `Added ${data.title || 'property #' + event.property_id} to wishlist`;
    return type.replace(/_/g, ' ');
  };

  const filteredEvents = events.filter(e => ['property_view', 'search', 'schedule_visit', 'save_property', 'wishlist_add'].includes(e.event_type));
  
  console.log('📊 RecentActivityList - All events:', events.length);
  console.log('📊 RecentActivityList - Filtered events:', filteredEvents.length);
  console.log('📊 RecentActivityList - Event types:', events.map(e => e.event_type));
  console.log('📊 RecentActivityList - Wishlist events:', events.filter(e => e.event_type === 'wishlist_add'));

  return (
    <div className="space-y-3">
      {filteredEvents.map((event, index) => {
        const { icon, color } = getEventIcon(event.event_type);
        return (
          <div key={index} className="flex items-start gap-3 p-3 hover:bg-gray-50 rounded-lg transition-colors">
            <div className={`${color} p-2 rounded-lg flex-shrink-0`}>{icon}</div>
            <div className="flex-1 min-w-0">
              <p className="text-sm font-medium text-gray-900 truncate">{getEventText(event)}</p>
              <p className="text-xs text-gray-500 mt-0.5">{formatTime(event.created_at)}</p>
            </div>
          </div>
        );
      })}
      {filteredEvents.length === 0 && <p className="text-gray-400 text-sm text-center py-8">No recent activity</p>}
    </div>
  );
};
const JourneyAnalysisView: React.FC<{ journeyData: any }> = ({ journeyData }) => {
  if (!journeyData.journey_stages) {
    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8 text-center">
        <Target className="w-16 h-16 text-gray-300 mx-auto mb-4" />
        <p className="text-gray-500">No journey data available yet</p>
      </div>
    );
  }

  const stages = ['awareness', 'interest', 'consideration', 'intent', 'action'];
  const stageColors = {
    awareness: 'bg-blue-500',
    interest: 'bg-green-500', 
    consideration: 'bg-yellow-500',
    intent: 'bg-orange-500',
    action: 'bg-red-500'
  };

  return (
    <div className="space-y-6">
      {/* Journey Funnel */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="font-semibold text-lg text-gray-900 mb-6">User Journey Funnel</h3>
        <div className="space-y-4">
          {stages.map((stage, index) => {
            const count = journeyData.journey_stages[stage] || 0;
            const maxCount = Math.max(...Object.values(journeyData.journey_stages));
            const width = maxCount > 0 ? (count / maxCount) * 100 : 0;
            
            return (
              <div key={stage} className="flex items-center gap-4">
                <div className="w-24 text-sm font-medium capitalize text-gray-700">{stage}</div>
                <div className="flex-1 bg-gray-200 rounded-full h-8 relative">
                  <div 
                    className={`${stageColors[stage]} h-8 rounded-full flex items-center justify-center text-white font-medium transition-all duration-1000`}
                    style={{ width: `${width}%` }}
                  >
                    {count > 0 && <span className="text-sm">{count}</span>}
                  </div>
                </div>
                <div className="w-16 text-sm text-gray-600">{count}</div>
              </div>
            );
          })}
        </div>
      </div>

      {/* Conversion Rates */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="font-semibold text-lg text-gray-900 mb-6">Conversion Rates</h3>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
          {Object.entries(journeyData.conversion_funnel || {}).map(([key, rate]) => (
            <div key={key} className="text-center p-4 bg-gray-50 rounded-lg">
              <div className="text-2xl font-bold text-gray-900">{rate}%</div>
              <div className="text-sm text-gray-600 capitalize">{key.replace(/_/g, ' ')}</div>
            </div>
          ))}
        </div>
      </div>

      {/* Journey Insights */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="font-semibold text-lg text-gray-900 mb-4">Journey Insights</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="p-4 bg-blue-50 rounded-lg">
            <div className="text-lg font-bold text-blue-900">{journeyData.journey_length_days || 0} days</div>
            <div className="text-sm text-blue-700">Journey Length</div>
          </div>
          <div className="p-4 bg-green-50 rounded-lg">
            <div className="text-lg font-bold text-green-900">{journeyData.total_touchpoints || 0}</div>
            <div className="text-sm text-green-700">Total Touchpoints</div>
          </div>
          <div className="p-4 bg-purple-50 rounded-lg">
            <div className="text-lg font-bold text-purple-900 capitalize">{journeyData.next_likely_action || 'N/A'}</div>
            <div className="text-sm text-purple-700">Next Likely Action</div>
          </div>
        </div>
      </div>
    </div>
  );
};

const RecommendationsView: React.FC<{ recommendations: any[] }> = ({ recommendations }) => {
  console.log('🔍 RecommendationsView - Received recommendations:', recommendations);
  console.log('🔍 RecommendationsView - Recommendations length:', recommendations?.length);
  
  if (!recommendations || !Array.isArray(recommendations) || recommendations.length === 0) {
    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8 text-center">
        <Brain className="w-16 h-16 text-gray-300 mx-auto mb-4" />
        <p className="text-gray-500">No AI recommendations available yet</p>
        <p className="text-sm text-gray-400 mt-2">Browse more properties to get personalized recommendations</p>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="font-semibold text-lg text-gray-900 mb-6">AI-Powered Recommendations</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {recommendations.map((rec, index) => (
            <div key={index} className="border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow">
              <div className="flex items-start justify-between mb-3">
                <div className="flex-1">
                  <h4 className="font-medium text-gray-900">Property #{rec.property_id}</h4>
                  <p className="text-sm text-gray-600">{rec.location}</p>
                  <p className="text-xs text-gray-500 capitalize">{rec.property_type}</p>
                </div>
                <div className="text-right">
                  <div className="text-lg font-bold text-blue-600">{rec.recommendation_score?.toFixed(1)}</div>
                  <div className="text-xs text-gray-500">Score</div>
                </div>
              </div>
              <div className="text-xs text-gray-600 bg-gray-50 p-2 rounded">
                {rec.reason}
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};

const MarketInsightsView: React.FC<{ insights: any }> = ({ insights }) => {
  if (!insights.personalized_locations?.length) {
    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8 text-center">
        <Zap className="w-16 h-16 text-gray-300 mx-auto mb-4" />
        <p className="text-gray-500">No market insights available yet</p>
        <p className="text-sm text-gray-400 mt-2">Search for properties to get personalized market insights</p>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="font-semibold text-lg text-gray-900 mb-6">Personalized Market Insights</h3>
        <div className="space-y-4">
          {insights.personalized_locations.map((location: any, index: number) => (
            <div key={index} className="border border-gray-200 rounded-lg p-4">
              <div className="flex items-center justify-between mb-3">
                <h4 className="font-medium text-gray-900">{location.location}</h4>
                {location.trending && (
                  <span className="bg-green-100 text-green-800 text-xs px-2 py-1 rounded-full">Trending</span>
                )}
              </div>
              <div className="grid grid-cols-3 gap-4 text-center">
                <div>
                  <div className="text-lg font-bold text-blue-600">{location.user_interest}</div>
                  <div className="text-xs text-gray-500">Your Interest</div>
                </div>
                <div>
                  <div className="text-lg font-bold text-green-600">{location.market_activity}</div>
                  <div className="text-xs text-gray-500">Market Activity</div>
                </div>
                <div>
                  <div className="text-lg font-bold text-purple-600">{location.demand_score}%</div>
                  <div className="text-xs text-gray-500">Demand Score</div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};